# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.


from typing import Optional


from .._uniffi import CommonMetricData
from .._uniffi import UrlMetric
from ..testing import ErrorType


class UrlMetricType:
    """
    This implements the developer facing API for recording URL metrics.

    Instances of this class type are automatically generated by
    `glean.load_metrics`, allowing developers to record values that were
    previously registered in the metrics.yaml file.

    The url API only exposes the `UrlMetricType.set` method, which
    takes care of validating the input data and making sure that limits are
    enforced.
    See https://mozilla.github.io/glean/book/reference/metrics/url.html#limits
    """

    def __init__(self, common_metric_data: CommonMetricData):
        self._inner = UrlMetric(common_metric_data)

    def set(self, value: str) -> None:
        """
        Set a URL value.

        Args:
            value (str): This is a user-defined URL value. If the length of
                the URL exceeds the maximum length, it will not be recorded.
        """
        if value is None:
            return

        self._inner.set(value)

    def test_get_value(self, ping_name: Optional[str] = None) -> Optional[str]:
        return self._inner.test_get_value(ping_name)

    def test_get_num_recorded_errors(self, error_type: ErrorType) -> int:
        return self._inner.test_get_num_recorded_errors(error_type)


__all__ = ["UrlMetricType"]
