# coding: latin-1
import unittest
import ctypes

try:
    ctypes.c_wchar
except AttributeError:
    pass
else:
    import _ctypes_test
    dll = ctypes.CDLL(_ctypes_test.__file__)
    wcslen = dll.my_wcslen
    wcslen.argtypes = [ctypes.c_wchar_p]


    class UnicodeTestCase(unittest.TestCase):
        def setUp(self):
            self.prev_conv_mode = ctypes.set_conversion_mode("ascii", "strict")

        def tearDown(self):
            ctypes.set_conversion_mode(*self.prev_conv_mode)

        def test_ascii_strict(self):
            ctypes.set_conversion_mode("ascii", "strict")
            # no conversions take place with unicode arguments
            self.assertEqual(wcslen(u"abc"), 3)
            self.assertEqual(wcslen(u"ab\u2070"), 3)
            # string args are converted
            self.assertEqual(wcslen("abc"), 3)
            self.assertRaises(ctypes.ArgumentError, wcslen, "ab")

        def test_ascii_replace(self):
            ctypes.set_conversion_mode("ascii", "replace")
            self.assertEqual(wcslen(u"abc"), 3)
            self.assertEqual(wcslen(u"ab\u2070"), 3)
            self.assertEqual(wcslen("abc"), 3)
            self.assertEqual(wcslen("ab"), 3)

        def test_ascii_ignore(self):
            ctypes.set_conversion_mode("ascii", "ignore")
            self.assertEqual(wcslen(u"abc"), 3)
            self.assertEqual(wcslen(u"ab\u2070"), 3)
            # ignore error mode skips non-ascii characters
            self.assertEqual(wcslen("abc"), 3)
            self.assertEqual(wcslen(""), 0)

        def test_latin1_strict(self):
            ctypes.set_conversion_mode("latin-1", "strict")
            self.assertEqual(wcslen(u"abc"), 3)
            self.assertEqual(wcslen(u"ab\u2070"), 3)
            self.assertEqual(wcslen("abc"), 3)
            self.assertEqual(wcslen(""), 4)

        def test_buffers(self):
            ctypes.set_conversion_mode("ascii", "strict")
            buf = ctypes.create_unicode_buffer("abc")
            self.assertEqual(len(buf), 3+1)

            ctypes.set_conversion_mode("ascii", "replace")
            buf = ctypes.create_unicode_buffer("ab")
            self.assertEqual(buf[:], u"ab\uFFFD\uFFFD\uFFFD\0")
            self.assertEqual(buf[::], u"ab\uFFFD\uFFFD\uFFFD\0")
            self.assertEqual(buf[::-1], u"\0\uFFFD\uFFFD\uFFFDba")
            self.assertEqual(buf[::2], u"a\uFFFD\uFFFD")
            self.assertEqual(buf[6:5:-1], u"")

            ctypes.set_conversion_mode("ascii", "ignore")
            buf = ctypes.create_unicode_buffer("ab")
            # is that correct? not sure.  But with 'ignore', you get what you pay for..
            self.assertEqual(buf[:], u"ab\0\0\0\0")
            self.assertEqual(buf[::], u"ab\0\0\0\0")
            self.assertEqual(buf[::-1], u"\0\0\0\0ba")
            self.assertEqual(buf[::2], u"a\0\0")
            self.assertEqual(buf[6:5:-1], u"")

    import _ctypes_test
    func = ctypes.CDLL(_ctypes_test.__file__)._testfunc_p_p

    class StringTestCase(UnicodeTestCase):
        def setUp(self):
            self.prev_conv_mode = ctypes.set_conversion_mode("ascii", "strict")
            func.argtypes = [ctypes.c_char_p]
            func.restype = ctypes.c_char_p

        def tearDown(self):
            ctypes.set_conversion_mode(*self.prev_conv_mode)
            func.argtypes = None
            func.restype = ctypes.c_int

        def test_ascii_replace(self):
            ctypes.set_conversion_mode("ascii", "strict")
            self.assertEqual(func("abc"), "abc")
            self.assertEqual(func(u"abc"), "abc")
            self.assertRaises(ctypes.ArgumentError, func, u"ab")

        def test_ascii_ignore(self):
            ctypes.set_conversion_mode("ascii", "ignore")
            self.assertEqual(func("abc"), "abc")
            self.assertEqual(func(u"abc"), "abc")
            self.assertEqual(func(u""), "")

        def test_ascii_replace(self):
            ctypes.set_conversion_mode("ascii", "replace")
            self.assertEqual(func("abc"), "abc")
            self.assertEqual(func(u"abc"), "abc")
            self.assertEqual(func(u""), "????")

        def test_buffers(self):
            ctypes.set_conversion_mode("ascii", "strict")
            buf = ctypes.create_string_buffer(u"abc")
            self.assertEqual(len(buf), 3+1)

            ctypes.set_conversion_mode("ascii", "replace")
            buf = ctypes.create_string_buffer(u"ab")
            self.assertEqual(buf[:], "ab???\0")
            self.assertEqual(buf[::], "ab???\0")
            self.assertEqual(buf[::-1], "\0???ba")
            self.assertEqual(buf[::2], "a??")
            self.assertEqual(buf[6:5:-1], "")

            ctypes.set_conversion_mode("ascii", "ignore")
            buf = ctypes.create_string_buffer(u"ab")
            # is that correct? not sure.  But with 'ignore', you get what you pay for..
            self.assertEqual(buf[:], "ab\0\0\0\0")
            self.assertEqual(buf[::], "ab\0\0\0\0")
            self.assertEqual(buf[::-1], "\0\0\0\0ba")

if __name__ == '__main__':
    unittest.main()
