/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

package mozilla.telemetry.glean.private

import androidx.annotation.VisibleForTesting
import mozilla.telemetry.glean.internal.CustomDistributionMetric
import mozilla.telemetry.glean.testing.ErrorType

/**
 * This implements the developer facing API for recording custom distribution metrics.
 *
 * Custom distributions are histograms with the following parameters that are settable on a
 * per-metric basis:
 *
 *    - `rangeMin`/`rangeMax`: The minimum and maximum values
 *    - `bucketCount`: The number of histogram buckets
 *    - `histogramType`: Whether the bucketing is linear or exponential
 *
 * This metric exists primarily for backward compatibility with histograms in
 * legacy (pre-Glean) telemetry, and its use is not recommended for newly-created
 * metrics.
 *
 * Instances of this class type are automatically generated by the parsers at build time,
 * allowing developers to record values that were previously registered in the metrics.yaml file.
 */
class CustomDistributionMetricType(
    meta: CommonMetricData,
    rangeMin: Long,
    rangeMax: Long,
    bucketCount: Long,
    histogramType: HistogramType
) : HistogramBase {
    val inner = CustomDistributionMetric(meta, rangeMin, rangeMax, bucketCount, histogramType)

    /**
     * Delegate common methods to the underlying type directly.
     */
    override fun accumulateSamples(samples: List<Long>) = inner.accumulateSamples(samples)

    /**
     * Testing-only methods get an annotation
     */

    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    @JvmOverloads
    fun testGetValue(pingName: String? = null) = inner.testGetValue(pingName)

    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    fun testGetNumRecordedErrors(error: ErrorType) = inner.testGetNumRecordedErrors(error)
}
