# This file contains the ReadFileChunk class implementation, by which we can stream the contents of the file in the
# parts/chunks. This class will allow you to specify the size and stream the data without using a chunked
# transfer-encoding. This implementation has been taken from https://github.com/boto/s3transfer/blob/develop/s3transfer/utils.py

import os


def invoke_progress_callbacks(callbacks, bytes_transferred):
    """Calls all progress callbacks
    :param callbacks: A list of progress callbacks to invoke
    :param bytes_transferred: The number of bytes transferred. This is passed
        to the callbacks. If no bytes were transferred the callbacks will not
        be invoked because no progress was achieved. It is also possible
        to receive a negative amount which comes from retrying a transfer
        request.
    """
    # Only invoke the callbacks if bytes were actually transferred.
    if bytes_transferred:
        # for callback in callbacks:
        callbacks(bytes_amount=bytes_transferred)


class ReadFileChunk(object):
    def __init__(self, fileobj, chunk_size, full_file_size,
                 callbacks=None, enable_callbacks=True, close_callbacks=None):
        """
        Given a file object shown below::
            |___________________________________________________|
            0          |                 |                 full_file_size
                       |----chunk_size---|
                    f.tell()
        :type fileobj: file
        :param fileobj: File like object
        :type chunk_size: int
        :param chunk_size: The max chunk size to read.  Trying to read
            pass the end of the chunk size will behave like you've
            reached the end of the file.
        :type full_file_size: int
        :param full_file_size: The entire content length associated
            with ``fileobj``.
        :type callbacks: A list of function(amount_read)
        :param callbacks: Called whenever data is read from this object in the
            order provided.
        :type enable_callbacks: boolean
        :param enable_callbacks: True if to run callbacks. Otherwise, do not
            run callbacks
        :type close_callbacks: A list of function()
        :param close_callbacks: Called when close is called. The function
            should take no arguments.
        """
        self._fileobj = fileobj
        self._start_byte = self._fileobj.tell()
        self._size = self._calculate_file_size(
            self._fileobj, requested_size=chunk_size,
            start_byte=self._start_byte, actual_file_size=full_file_size)
        # _amount_read represents the position in the chunk and may exceed
        # the chunk size, but won't allow reads out of bounds.
        self._amount_read = 0
        self._callbacks = callbacks
        if callbacks is None:
            self._callbacks = []
        self._callbacks_enabled = enable_callbacks
        self._close_callbacks = close_callbacks
        if close_callbacks is None:
            self._close_callbacks = close_callbacks

    @classmethod
    def from_filename(cls, filename, start_byte, chunk_size, callbacks=None,
                      enable_callbacks=True):
        """Convenience factory function to create from a filename.
        :type start_byte: int
        :param start_byte: The first byte from which to start reading.
        :type chunk_size: int
        :param chunk_size: The max chunk size to read.  Trying to read
            pass the end of the chunk size will behave like you've
            reached the end of the file.
        :type full_file_size: int
        :param full_file_size: The entire content length associated
            with ``fileobj``.
        :type callbacks: function(amount_read)
        :param callbacks: Called whenever data is read from this object.
        :type enable_callbacks: bool
        :param enable_callbacks: Indicate whether to invoke callback
            during read() calls.
        :rtype: ``ReadFileChunk``
        :return: A new instance of ``ReadFileChunk``
        """
        f = open(filename, 'rb')
        f.seek(start_byte)
        file_size = os.fstat(f.fileno()).st_size
        return cls(f, chunk_size, file_size, callbacks, enable_callbacks)

    def _calculate_file_size(self, fileobj, requested_size, start_byte,
                             actual_file_size):
        max_chunk_size = actual_file_size - start_byte
        return min(max_chunk_size, requested_size)

    def read(self, amount=None):

        amount_left = max(self._size - self._amount_read, 0)
        if amount is None:
            amount_to_read = amount_left
        else:
            amount_to_read = min(amount_left, amount)
        data = self._fileobj.read(amount_to_read)
        self._amount_read += len(data)
        if self._callbacks is not None and self._callbacks_enabled:
            invoke_progress_callbacks(self._callbacks, len(data))
        return data

    def enable_callback(self):
        self._callbacks_enabled = True

    def disable_callback(self):
        self._callbacks_enabled = False

    def seek(self, where, whence=0):
        if whence not in (0, 1, 2):
            # Mimic io's error for invalid whence values
            raise ValueError(
                u"invalid whence (%s, should be 0, 1 or 2)" % whence)

        # Recalculate where based on chunk attributes so seek from file
        # start (whence=0) is always used
        where += self._start_byte
        if whence == 1:
            where += self._amount_read
        elif whence == 2:
            where += self._size

        self._fileobj.seek(max(where, self._start_byte))
        if self._callbacks is not None and self._callbacks_enabled:
            # To also rewind the callback() for an accurate progress report
            bounded_where = max(min(where - self._start_byte, self._size), 0)
            bounded_amount_read = min(self._amount_read, self._size)
            amount = bounded_where - bounded_amount_read
            invoke_progress_callbacks(
                self._callbacks, bytes_transferred=amount)
        self._amount_read = max(where - self._start_byte, 0)

    def close(self):
        if self._close_callbacks is not None and self._callbacks_enabled:
            for callback in self._close_callbacks:
                callback()
        self._fileobj.close()

    def tell(self):
        return self._amount_read

    def __len__(self):
        # __len__ is defined because requests will try to determine the length
        # of the stream to set a content length.  In the normal case
        # of the file it will just stat the file, but we need to change that
        # behavior.  By providing a __len__, requests will use that instead
        # of stat'ing the file.
        return self._size

    def __enter__(self):
        return self

    def __exit__(self, *args, **kwargs):
        self.close()

    def __iter__(self):
        # This is a workaround for http://bugs.python.org/issue17575
        # Basically httplib will try to iterate over the contents, even
        # if its a file like object.  This wasn't noticed because we've
        # already exhausted the stream so iterating over the file immediately
        # stops, which is what we're simulating here.
        return iter([])


def open_file_chunk_reader(filename, start_byte, size, callback):
    return ReadFileChunk.from_filename(filename, start_byte, size, callback)
