/* ====================================================================
 *
 *
 *   BSD LICENSE
 *
 *   Copyright(c) 2021-2023 Intel Corporation.
 *   All rights reserved.
 *
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in
 *       the documentation and/or other materials provided with the
 *       distribution.
 *     * Neither the name of Intel Corporation nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without specific prior written permission.
 *
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *
 * ====================================================================
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <openssl/err.h>
#include <openssl/async.h>
#include <openssl/objects.h>
#include <openssl/bn.h>
#include <openssl/dsa.h>
#include <openssl/engine.h>
#ifdef QAT_OPENSSL_PROVIDER
# include <openssl/param_build.h>
#endif

#include "tests.h"
#include "../qat_utils.h"

static unsigned char BN_p[] = {
    0xD7, 0x57, 0x26, 0x2C, 0x45, 0x84, 0xC4, 0x4C,
    0x21, 0x1F, 0x18, 0xBD, 0x96, 0xE5, 0xF0, 0x61,
    0xC4, 0xF0, 0xA4, 0x23, 0xF7, 0xFE, 0x6B, 0x6B,
    0x85, 0xB3, 0x4C, 0xEF, 0x72, 0xCE, 0x14, 0xA0,
    0xD3, 0xA5, 0x22, 0x2F, 0xE0, 0x8C, 0xEC, 0xE6,
    0x5B, 0xE6, 0xC2, 0x65, 0x85, 0x48, 0x89, 0xDC,
    0x1E, 0xDB, 0xD1, 0x3E, 0xC8, 0xB2, 0x74, 0xDA,
    0x9F, 0x75, 0xBA, 0x26, 0xCC, 0xB9, 0x87, 0x72,
    0x36, 0x02, 0x78, 0x7E, 0x92, 0x2B, 0xA8, 0x44,
    0x21, 0xF2, 0x2C, 0x3C, 0x89, 0xCB, 0x9B, 0x06,
    0xFD, 0x60, 0xFE, 0x01, 0x94, 0x1D, 0xDD, 0x77,
    0xFE, 0x6B, 0x12, 0x89, 0x3D, 0xA7, 0x6E, 0xEB,
    0xC1, 0xD1, 0x28, 0xD9, 0x7F, 0x06, 0x78, 0xD7,
    0x72, 0x2B, 0x53, 0x41, 0xC8, 0x50, 0x6F, 0x35,
    0x82, 0x14, 0xB1, 0x6A, 0x2F, 0xAC, 0x4B, 0x36,
    0x89, 0x50, 0x38, 0x78, 0x11, 0xC7, 0xDA, 0x33,
};

static unsigned char BN_q[] = {
    0xC7, 0x73, 0x21, 0x8C, 0x73, 0x7E, 0xC8, 0xEE,
    0x99, 0x3B, 0x4F, 0x2D, 0xED, 0x30, 0xF4, 0x8E,
    0xDA, 0xCE, 0x91, 0x5F,
};

static unsigned char BN_g[] = {
    0x82, 0x26, 0x90, 0x09, 0xE1, 0x4E, 0xC4, 0x74,
    0xBA, 0xF2, 0x93, 0x2E, 0x69, 0xD3, 0xB1, 0xF1,
    0x85, 0x17, 0xAD, 0x95, 0x94, 0x18, 0x4C, 0xCD,
    0xFC, 0xEA, 0xE9, 0x6E, 0xC4, 0xD5, 0xEF, 0x93,
    0x13, 0x3E, 0x84, 0xB4, 0x70, 0x93, 0xC5, 0x2B,
    0x20, 0xCD, 0x35, 0xD0, 0x24, 0x92, 0xB3, 0x95,
    0x9E, 0xC6, 0x49, 0x96, 0x25, 0xBC, 0x4F, 0xA5,
    0x08, 0x2E, 0x22, 0xC5, 0xB3, 0x74, 0xE1, 0x6D,
    0xD0, 0x01, 0x32, 0xCE, 0x71, 0xB0, 0x20, 0x21,
    0x70, 0x91, 0xAC, 0x71, 0x7B, 0x61, 0x23, 0x91,
    0xC7, 0x6C, 0x1F, 0xB2, 0xE8, 0x83, 0x17, 0xC1,
    0xBD, 0x81, 0x71, 0xD4, 0x1E, 0xCB, 0x83, 0xE2,
    0x10, 0xC0, 0x3C, 0xC9, 0xB3, 0x2E, 0x81, 0x05,
    0x61, 0xC2, 0x16, 0x21, 0xC7, 0x3D, 0x6D, 0xAA,
    0xC0, 0x28, 0xF4, 0xB1, 0x58, 0x5D, 0xA7, 0xF4,
    0x25, 0x19, 0x71, 0x8C, 0xC9, 0xB0, 0x9E, 0xEF,
};


static unsigned char Pub_key[] = {
    0x52, 0x3C, 0x3E, 0x53, 0x41, 0xC3, 0xC8, 0xDF,
    0x22, 0x4E, 0x07, 0x0C, 0x99, 0x76, 0xFC, 0x7D,
    0xF1, 0x95, 0xD3, 0xC5, 0x1D, 0x67, 0x1A, 0xF2,
    0xC9, 0x68, 0xA2, 0xA1, 0x41, 0x35, 0x1F, 0xFC,
    0x64, 0x47, 0x65, 0xAD, 0xEB, 0xC4, 0x71, 0xD5,
    0x1B, 0xC2, 0xEF, 0x76, 0x21, 0xE9, 0xED, 0x6A,
    0xA6, 0xD9, 0xDB, 0x5B, 0xBB, 0x81, 0x43, 0x8F,
    0xC0, 0x1E, 0xE6, 0x49, 0x2A, 0xB7, 0xEA, 0x8F,
    0xCB, 0x6B, 0x93, 0x1E, 0x94, 0x8C, 0x24, 0x48,
    0xE3, 0x80, 0xD3, 0x9B, 0x4F, 0xDC, 0xBF, 0x45,
    0xE3, 0xC0, 0xFC, 0xDA, 0x1C, 0x06, 0x74, 0xF9,
    0x02, 0x2A, 0x5A, 0xAA, 0x18, 0x58, 0x1D, 0x12,
    0x4D, 0x07, 0x0F, 0x00, 0xB4, 0x5E, 0xA3, 0x62,
    0xB1, 0xFA, 0xEE, 0x05, 0x04, 0x89, 0x27, 0x5A,
    0x68, 0xBF, 0x8B, 0x19, 0x90, 0x84, 0x21, 0xB8,
    0xFB, 0x3B, 0x37, 0x04, 0x4D, 0x6B, 0xB6, 0xD2,
};

static unsigned char Priv_key[] = {
    0x6E, 0xB5, 0x08, 0x61, 0xE5, 0x25, 0xD7, 0xEF,
    0x6C, 0x53, 0xBD, 0x77, 0x72, 0x0F, 0x98, 0xE1,
    0x60, 0x3D, 0x67, 0x67,
};

#ifdef QAT_OPENSSL_PROVIDER
static unsigned char dsa1024_priv[] = {
    0x7d, 0x21, 0xda, 0xbb, 0x62, 0x15, 0x47, 0x36, 0x07, 0x67, 0x12, 0xe8,
    0x8c, 0xaa, 0x1c, 0xcd, 0x38, 0x12, 0x61, 0x18,
};

static unsigned char dsa1024_pub[] = {
    0x3c, 0x4e, 0x9c, 0x2a, 0x7f, 0x16, 0xc1, 0x25, 0xeb, 0xac, 0x78, 0x63,
    0x90, 0x14, 0x8c, 0x8b, 0xf4, 0x68, 0x43, 0x3c, 0x2d, 0xee, 0x65, 0x50,
    0x7d, 0x9c, 0x8f, 0x8c, 0x8a, 0x51, 0xd6, 0x11, 0x2b, 0x99, 0xaf, 0x1e,
    0x90, 0x97, 0xb5, 0xd3, 0xa6, 0x20, 0x25, 0xd6, 0xfe, 0x43, 0x02, 0xd5,
    0x91, 0x7d, 0xa7, 0x8c, 0xdb, 0xc9, 0x85, 0xa3, 0x36, 0x48, 0xf7, 0x68,
    0xaa, 0x60, 0xb1, 0xf7, 0x05, 0x68, 0x3a, 0xa3, 0x3f, 0xd3, 0x19, 0x82,
    0xd8, 0x82, 0x7a, 0x77, 0xfb, 0xef, 0xf4, 0x15, 0x0a, 0xeb, 0x06, 0x04,
    0x7f, 0x53, 0x07, 0x0c, 0xbc, 0xcb, 0x2d, 0x83, 0xdb, 0x3e, 0xd1, 0x28,
    0xa5, 0xa1, 0x31, 0xe0, 0x67, 0xfa, 0x50, 0xde, 0x9b, 0x07, 0x83, 0x7e,
    0x2c, 0x0b, 0xc3, 0x13, 0x50, 0x61, 0xe5, 0xad, 0xbd, 0x36, 0xb8, 0x97,
    0x4e, 0x40, 0x7d, 0xe8, 0x83, 0x0d, 0xbc, 0x4b
};

static unsigned char dsa1024_p[] = {
    0xA7, 0x3F, 0x6E, 0x85, 0xBF, 0x41, 0x6A, 0x29, 0x7D, 0xF0, 0x9F, 0x47,
    0x19, 0x30, 0x90, 0x9A, 0x09, 0x1D, 0xDA, 0x6A, 0x33, 0x1E, 0xC5, 0x3D,
    0x86, 0x96, 0xB3, 0x15, 0xE0, 0x53, 0x2E, 0x8F, 0xE0, 0x59, 0x82, 0x73,
    0x90, 0x3E, 0x75, 0x31, 0x99, 0x47, 0x7A, 0x52, 0xFB, 0x85, 0xE4, 0xD9,
    0xA6, 0x7B, 0x38, 0x9B, 0x68, 0x8A, 0x84, 0x9B, 0x87, 0xC6, 0x1E, 0xB5,
    0x7E, 0x86, 0x4B, 0x53, 0x5B, 0x59, 0xCF, 0x71, 0x65, 0x19, 0x88, 0x6E,
    0xCE, 0x66, 0xAE, 0x6B, 0x88, 0x36, 0xFB, 0xEC, 0x28, 0xDC, 0xC2, 0xD7,
    0xA5, 0xBB, 0xE5, 0x2C, 0x39, 0x26, 0x4B, 0xDA, 0x9A, 0x70, 0x18, 0x95,
    0x37, 0x95, 0x10, 0x56, 0x23, 0xF6, 0x15, 0xED, 0xBA, 0x04, 0x5E, 0xDE,
    0x39, 0x4F, 0xFD, 0xB7, 0x43, 0x1F, 0xB5, 0xA4, 0x65, 0x6F, 0xCD, 0x80,
    0x11, 0xE4, 0x70, 0x95, 0x5B, 0x50, 0xCD, 0x49,
};

static unsigned char dsa1024_q[] = {
    0xF7, 0x07, 0x31, 0xED, 0xFA, 0x6C, 0x06, 0x03, 0xD5, 0x85, 0x8A, 0x1C,
    0xAC, 0x9C, 0x65, 0xE7, 0x50, 0x66, 0x65, 0x6F,
};

static unsigned char dsa1024_g[] = {
    0x4D, 0xDF, 0x4C, 0x03, 0xA6, 0x91, 0x8A, 0xF5, 0x19, 0x6F, 0x50, 0x46,
    0x25, 0x99, 0xE5, 0x68, 0x6F, 0x30, 0xE3, 0x69, 0xE1, 0xE5, 0xB3, 0x5D,
    0x98, 0xBB, 0x28, 0x86, 0x48, 0xFC, 0xDE, 0x99, 0x04, 0x3F, 0x5F, 0x88,
    0x0C, 0x9C, 0x73, 0x24, 0x0D, 0x20, 0x5D, 0xB9, 0x2A, 0x9A, 0x3F, 0x18,
    0x96, 0x27, 0xE4, 0x62, 0x87, 0xC1, 0x7B, 0x74, 0x62, 0x53, 0xFC, 0x61,
    0x27, 0xA8, 0x7A, 0x91, 0x09, 0x9D, 0xB6, 0xF1, 0x4D, 0x9C, 0x54, 0x0F,
    0x58, 0x06, 0xEE, 0x49, 0x74, 0x07, 0xCE, 0x55, 0x7E, 0x23, 0xCE, 0x16,
    0xF6, 0xCA, 0xDC, 0x5A, 0x61, 0x01, 0x7E, 0xC9, 0x71, 0xB5, 0x4D, 0xF6,
    0xDC, 0x34, 0x29, 0x87, 0x68, 0xF6, 0x5E, 0x20, 0x93, 0xB3, 0xDB, 0xF5,
    0xE4, 0x09, 0x6C, 0x41, 0x17, 0x95, 0x92, 0xEB, 0x01, 0xB5, 0x73, 0xA5,
    0x6A, 0x7E, 0xD8, 0x32, 0xED, 0x0E, 0x02, 0xB8,
};

static unsigned char dsa2048_priv[] = {
    0x32, 0x67, 0x92, 0xf6, 0xc4, 0xe2, 0xe2, 0xe8, 0xa0, 0x8b, 0x6b, 0x45,
    0x0c, 0x8a, 0x76, 0xb0, 0xee, 0xcf, 0x91, 0xa7,
};

static unsigned char dsa2048_pub[] = {
    0x17, 0x8f, 0xa8, 0x11, 0x84, 0x92, 0xec, 0x83, 0x47, 0xc7, 0x6a, 0xb0,
    0x92, 0xaf, 0x5a, 0x20, 0x37, 0xa3, 0x64, 0x79, 0xd2, 0xd0, 0x3d, 0xcd,
    0xe0, 0x61, 0x88, 0x88, 0x21, 0xcc, 0x74, 0x5d, 0xce, 0x4c, 0x51, 0x47,
    0xf0, 0xc5, 0x5c, 0x4c, 0x82, 0x7a, 0xaf, 0x72, 0xad, 0xb9, 0xe0, 0x53,
    0xf2, 0x78, 0xb7, 0xf0, 0xb5, 0x48, 0x7f, 0x8a, 0x3a, 0x18, 0xd1, 0x9f,
    0x8b, 0x7d, 0xa5, 0x47, 0xb7, 0x95, 0xab, 0x98, 0xf8, 0x7b, 0x74, 0x50,
    0x56, 0x8e, 0x57, 0xf0, 0xee, 0xf5, 0xb7, 0xba, 0xab, 0x85, 0x86, 0xf9,
    0x2b, 0xef, 0x41, 0x56, 0xa0, 0xa4, 0x9f, 0xb7, 0x38, 0x00, 0x46, 0x0a,
    0xa6, 0xf1, 0xfc, 0x1f, 0xd8, 0x4e, 0x85, 0x44, 0x92, 0x43, 0x21, 0x5d,
    0x6e, 0xcc, 0xc2, 0xcb, 0x26, 0x31, 0x0d, 0x21, 0xc4, 0xbd, 0x8d, 0x24,
    0xbc, 0xd9, 0x18, 0x19, 0xd7, 0xdc, 0xf1, 0xe7, 0x93, 0x50, 0x48, 0x03,
    0x2c, 0xae, 0x2e, 0xe7, 0x49, 0x88, 0x5f, 0x93, 0x57, 0x27, 0x99, 0x36,
    0xb4, 0x20, 0xab, 0xfc, 0xa7, 0x2b, 0xf2, 0xd9, 0x98, 0xd7, 0xd4, 0x34,
    0x9d, 0x96, 0x50, 0x58, 0x9a, 0xea, 0x54, 0xf3, 0xee, 0xf5, 0x63, 0x14,
    0xee, 0x85, 0x83, 0x74, 0x76, 0xe1, 0x52, 0x95, 0xc3, 0xf7, 0xeb, 0x04,
    0x04, 0x7b, 0xa7, 0x28, 0x1b, 0xcc, 0xea, 0x4a, 0x4e, 0x84, 0xda, 0xd8,
    0x9c, 0x79, 0xd8, 0x9b, 0x66, 0x89, 0x2f, 0xcf, 0xac, 0xd7, 0x79, 0xf9,
    0xa9, 0xd8, 0x45, 0x13, 0x78, 0xb9, 0x00, 0x14, 0xc9, 0x7e, 0x22, 0x51,
    0x86, 0x67, 0xb0, 0x9f, 0x26, 0x11, 0x23, 0xc8, 0x38, 0xd7, 0x70, 0x1d,
    0x15, 0x8e, 0x4d, 0x4f, 0x95, 0x97, 0x40, 0xa1, 0xc2, 0x7e, 0x01, 0x18,
    0x72, 0xf4, 0x10, 0xe6, 0x8d, 0x52, 0x16, 0x7f, 0xf2, 0xc9, 0xf8, 0x33,
    0x8b, 0x33, 0xb7, 0xce,
};

static unsigned char dsa2048_p[] = {
    0xA0, 0x25, 0xFA, 0xAD, 0xF4, 0x8E, 0xB9, 0xE5, 0x99, 0xF3, 0x5D, 0x6F,
    0x4F, 0x83, 0x34, 0xE2, 0x7E, 0xCF, 0x6F, 0xBF, 0x30, 0xAF, 0x6F, 0x81,
    0xEB, 0xF8, 0xC4, 0x13, 0xD9, 0xA0, 0x5D, 0x8B, 0x5C, 0x8E, 0xDC, 0xC2,
    0x1D, 0x0B, 0x41, 0x32, 0xB0, 0x1F, 0xFE, 0xEF, 0x0C, 0xC2, 0xA2, 0x7E,
    0x68, 0x5C, 0x28, 0x21, 0xE9, 0xF5, 0xB1, 0x58, 0x12, 0x63, 0x4C, 0x19,
    0x4E, 0xFF, 0x02, 0x4B, 0x92, 0xED, 0xD2, 0x07, 0x11, 0x4D, 0x8C, 0x58,
    0x16, 0x5C, 0x55, 0x8E, 0xAD, 0xA3, 0x67, 0x7D, 0xB9, 0x86, 0x6E, 0x0B,
    0xE6, 0x54, 0x6F, 0x40, 0xAE, 0x0E, 0x67, 0x4C, 0xF9, 0x12, 0x5B, 0x3C,
    0x08, 0x7A, 0xF7, 0xFC, 0x67, 0x86, 0x69, 0xE7, 0x0A, 0x94, 0x40, 0xBF,
    0x8B, 0x76, 0xFE, 0x26, 0xD1, 0xF2, 0xA1, 0x1A, 0x84, 0xA1, 0x43, 0x56,
    0x28, 0xBC, 0x9A, 0x5F, 0xD7, 0x3B, 0x69, 0x89, 0x8A, 0x36, 0x2C, 0x51,
    0xDF, 0x12, 0x77, 0x2F, 0x57, 0x7B, 0xA0, 0xAA, 0xDD, 0x7F, 0xA1, 0x62,
    0x3B, 0x40, 0x7B, 0x68, 0x1A, 0x8F, 0x0D, 0x38, 0xBB, 0x21, 0x5D, 0x18,
    0xFC, 0x0F, 0x46, 0xF7, 0xA3, 0xB0, 0x1D, 0x23, 0xC3, 0xD2, 0xC7, 0x72,
    0x51, 0x18, 0xDF, 0x46, 0x95, 0x79, 0xD9, 0xBD, 0xB5, 0x19, 0x02, 0x2C,
    0x87, 0xDC, 0xE7, 0x57, 0x82, 0x7E, 0xF1, 0x8B, 0x06, 0x3D, 0x00, 0xA5,
    0x7B, 0x6B, 0x26, 0x27, 0x91, 0x0F, 0x6A, 0x77, 0xE4, 0xD5, 0x04, 0xE4,
    0x12, 0x2C, 0x42, 0xFF, 0xD2, 0x88, 0xBB, 0xD3, 0x92, 0xA0, 0xF9, 0xC8,
    0x51, 0x64, 0x14, 0x5C, 0xD8, 0xF9, 0x6C, 0x47, 0x82, 0xB4, 0x1C, 0x7F,
    0x09, 0xB8, 0xF0, 0x25, 0x83, 0x1D, 0x3F, 0x3F, 0x05, 0xB3, 0x21, 0x0A,
    0x5D, 0xA7, 0xD8, 0x54, 0xC3, 0x65, 0x7D, 0xC3, 0xB0, 0x1D, 0xBF, 0xAE,
    0xF8, 0x68, 0xCF, 0x9B,
};

static unsigned char dsa2048_q[] = {
    0x97, 0xE7, 0x33, 0x4D, 0xD3, 0x94, 0x3E, 0x0B, 0xDB, 0x62, 0x74, 0xC6,
    0xA1, 0x08, 0xDD, 0x19, 0xA3, 0x75, 0x17, 0x1B,
};

static unsigned char dsa2048_g[] = {
    0x2C, 0x78, 0x16, 0x59, 0x34, 0x63, 0xF4, 0xF3, 0x92, 0xFC, 0xB5, 0xA5,
    0x4F, 0x13, 0xDE, 0x2F, 0x1C, 0xA4, 0x3C, 0xAE, 0xAD, 0x38, 0x3F, 0x7E,
    0x90, 0xBF, 0x96, 0xA6, 0xAE, 0x25, 0x90, 0x72, 0xF5, 0x8E, 0x80, 0x0C,
    0x39, 0x1C, 0xD9, 0xEC, 0xBA, 0x90, 0x5B, 0x3A, 0xE8, 0x58, 0x6C, 0x9E,
    0x30, 0x42, 0x37, 0x02, 0x31, 0x82, 0xBC, 0x6A, 0xDF, 0x6A, 0x09, 0x29,
    0xE3, 0xC0, 0x46, 0xD1, 0xCB, 0x85, 0xEC, 0x0C, 0x30, 0x5E, 0xEA, 0xC8,
    0x39, 0x8E, 0x22, 0x9F, 0x22, 0x10, 0xD2, 0x34, 0x61, 0x68, 0x37, 0x3D,
    0x2E, 0x4A, 0x5B, 0x9A, 0xF5, 0xC1, 0x48, 0xC6, 0xF6, 0xDC, 0x63, 0x1A,
    0xD3, 0x96, 0x64, 0xBA, 0x34, 0xC9, 0xD1, 0xA0, 0xD1, 0xAE, 0x6C, 0x2F,
    0x48, 0x17, 0x93, 0x14, 0x43, 0xED, 0xF0, 0x21, 0x30, 0x19, 0xC3, 0x1B,
    0x5F, 0xDE, 0xA3, 0xF0, 0x70, 0x78, 0x18, 0xE1, 0xA8, 0xE4, 0xEE, 0x2E,
    0x00, 0xA5, 0xE4, 0xB3, 0x17, 0xC8, 0x0C, 0x7D, 0x6E, 0x42, 0xDC, 0xB7,
    0x46, 0x00, 0x36, 0x4D, 0xD4, 0x46, 0xAA, 0x3D, 0x3C, 0x46, 0x89, 0x40,
    0xBF, 0x1D, 0x84, 0x77, 0x0A, 0x75, 0xF3, 0x87, 0x1D, 0x08, 0x4C, 0xA6,
    0xD1, 0xA9, 0x1C, 0x1E, 0x12, 0x1E, 0xE1, 0xC7, 0x30, 0x28, 0x76, 0xA5,
    0x7F, 0x6C, 0x85, 0x96, 0x2B, 0x6F, 0xDB, 0x80, 0x66, 0x26, 0xAE, 0xF5,
    0x93, 0xC7, 0x8E, 0xAE, 0x9A, 0xED, 0xE4, 0xCA, 0x04, 0xEA, 0x3B, 0x72,
    0xEF, 0xDC, 0x87, 0xED, 0x0D, 0xA5, 0x4C, 0x4A, 0xDD, 0x71, 0x22, 0x64,
    0x59, 0x69, 0x4E, 0x8E, 0xBF, 0x43, 0xDC, 0xAB, 0x8E, 0x66, 0xBB, 0x01,
    0xB6, 0xF4, 0xE7, 0xFD, 0xD2, 0xAD, 0x9F, 0x36, 0xC1, 0xA0, 0x29, 0x99,
    0xD1, 0x96, 0x70, 0x59, 0x06, 0x78, 0x35, 0xBD, 0x65, 0x55, 0x52, 0x9E,
    0xF8, 0xB2, 0xE5, 0x38,
};

typedef struct testdsa_st {
    unsigned char *priv;
    unsigned char *pub;
    unsigned char *p;
    unsigned char *g;
    unsigned char *q;
    int priv_l;
    int pub_l;
    int p_l;
    int g_l;
    int q_l;
} testdsa;

#define set_dsa_ptr(st, bits)                 \
    do                                        \
    {                                         \
        st.priv = dsa##bits##_priv;           \
        st.pub = dsa##bits##_pub;             \
        st.p = dsa##bits##_p;                 \
        st.g = dsa##bits##_g;                 \
        st.q = dsa##bits##_q;                 \
        st.priv_l = sizeof(dsa##bits##_priv); \
        st.pub_l = sizeof(dsa##bits##_pub);   \
        st.p_l = sizeof(dsa##bits##_p);       \
        st.g_l = sizeof(dsa##bits##_g);       \
        st.q_l = sizeof(dsa##bits##_q);       \
    } while (0)

EVP_PKEY *get_dsa(int dsa_bits) {
    EVP_PKEY *pkey = NULL;
    BIGNUM *priv_key, *pub_key, *p, *q, *g;
    EVP_PKEY_CTX *pctx;
    testdsa dsa_t;
    OSSL_PARAM_BLD *tmpl = NULL;
    OSSL_PARAM *params = NULL;

    switch (dsa_bits) {
    case 1024:
        set_dsa_ptr(dsa_t, 1024);
        break;
    case 2048:
        set_dsa_ptr(dsa_t, 2048);
        break;
    case 4096:
        dsa_t.priv = Priv_key;
        dsa_t.pub = Pub_key;
        dsa_t.p = BN_p;
        dsa_t.g = BN_g;
        dsa_t.q = BN_q;
        dsa_t.priv_l = sizeof(Priv_key);
        dsa_t.pub_l = sizeof(Pub_key);
        dsa_t.p_l = sizeof(BN_p);
        dsa_t.g_l = sizeof(BN_g);
        dsa_t.q_l = sizeof(BN_q);
        break;
    default:
        return NULL;
    }

    if ((pctx = EVP_PKEY_CTX_new_from_name(NULL, "DSA", NULL)) == NULL)
        return NULL;

    priv_key = BN_bin2bn(dsa_t.priv, dsa_t.priv_l, NULL);
    pub_key = BN_bin2bn(dsa_t.pub, dsa_t.pub_l, NULL);
    p = BN_bin2bn(dsa_t.p, dsa_t.p_l, NULL);
    q = BN_bin2bn(dsa_t.q, dsa_t.q_l, NULL);
    g = BN_bin2bn(dsa_t.g, dsa_t.g_l, NULL);
    if (priv_key == NULL || pub_key == NULL || p == NULL || q == NULL || g == NULL) {
        goto err;
    }
    if ((tmpl = OSSL_PARAM_BLD_new()) == NULL 
        || !OSSL_PARAM_BLD_push_BN(tmpl, "p", p)
        || !OSSL_PARAM_BLD_push_BN(tmpl, "q", q)
        || !OSSL_PARAM_BLD_push_BN(tmpl, "g", g)
        || !OSSL_PARAM_BLD_push_BN(tmpl, "priv", priv_key)
        || !OSSL_PARAM_BLD_push_BN(tmpl, "pub", pub_key)
        || (params = OSSL_PARAM_BLD_to_param(tmpl)) == NULL)
        goto err;

    if (EVP_PKEY_fromdata_init(pctx) <= 0
        || EVP_PKEY_fromdata(pctx, &pkey, EVP_PKEY_KEYPAIR, params) <= 0)
        pkey = NULL;
err:
    OSSL_PARAM_free(params);
    OSSL_PARAM_BLD_free(tmpl);
    BN_free(priv_key);
    BN_free(pub_key);
    BN_free(p);
    BN_free(q);
    BN_free(g);
    EVP_PKEY_CTX_free(pctx);
    return pkey;
}
#endif

/******************************************************************************
* function:
*           run_dsa(void *args)
*
* @param args [IN] - the test parameters
*
* description:
*   The function is design to test DSA sign and verify using qat engine.
*
*******************************************************************************/
static int run_dsa(void *args)
{
    TEST_PARAMS *temp_args = (TEST_PARAMS *)args;
    int size = temp_args->size;
    int ret = 0;
#ifndef QAT_OPENSSL_PROVIDER
    struct async_additional_args_dsa *extra_args =
        (struct async_additional_args_dsa *)temp_args->additional_args;
    int count = *(temp_args->count);
    int print_output = temp_args->print_output;
    int print_verify = temp_args->verify;
    int local_verify = extra_args->local_verify;

    DSA *dsa = NULL;
    BIGNUM *q = NULL;
    BIGNUM *p = NULL;
    BIGNUM *g = NULL;
    BIGNUM *pub_key = NULL;
    BIGNUM *priv_key = NULL;

    int i = 0;
    unsigned char sig[256] = {0};
    unsigned int siglen = 0;

    unsigned char *DgstData = OPENSSL_malloc(size);

    if (DgstData == NULL) {
        WARN("# FAIL DSA DgstData Initial malloc failed ! \n");
        exit(EXIT_FAILURE);
    }

    /* setup input hash message */
    for (i = 0; i < size; i++)
        DgstData[i] = i % 16;

    if ((dsa = DSA_new()) == NULL) {
        WARN("# FAIL DSA DSA_new failed ! \n");
        goto end;
    }

    if (((q = BN_bin2bn(BN_q, sizeof(BN_q), NULL)) == NULL) ||
        ((p = BN_bin2bn(BN_p, sizeof(BN_p), NULL)) == NULL) ||
        ((g = BN_bin2bn(BN_g, sizeof(BN_g), NULL)) == NULL) ||
        ((pub_key = BN_bin2bn(Pub_key, sizeof(Pub_key), NULL)) == NULL) ||
        ((priv_key = BN_bin2bn(Priv_key, sizeof(Priv_key), NULL)) == NULL)) {
        ret = 0;
        WARN("# FAIL DSA set DSA Parameters failed! \n");
        if (q) BN_free(q);
        if (p) BN_free(p);
        if (g) BN_free(g);
        if (pub_key) BN_free(pub_key);
        if (priv_key) BN_free(priv_key);
        goto end;
    }

    DSA_set0_pqg(dsa, p, q, g);
    DSA_set0_key(dsa, pub_key, priv_key);

    for (i = 0; i < count; i++) {
        if (i == 0 || !local_verify) {
            /*
             * DSA_sign() computes a digital signature on the size byte message
             * digest dgst using the private key dsa and places its ASN.1 DER
             * encoding at sig. The length of the signature is placed in
             * *siglen. sig must point to DSA_size(dsa) bytes of memory.
             */
            DEBUG("%s starting DSA sign \n", __func__);

            ret = DSA_sign(0, DgstData, size, sig, &siglen, dsa);
            if (ret <= 0) {
                WARN("# FAIL DSA signature failed! \n");
                goto end;
            } else
                DEBUG("# DSA signature completed! \n");

            if (print_output)
                tests_hexdump(" DSA Signature:", sig, siglen);
        }

        if ((i == 0 && print_verify) || local_verify) {
            /*
             * DSA_verify() verifies that the signature sig of size siglen
             * matches a given message digest dgst of length 'size'. dsa is the
             * signer's public key. Returns * 1: correct signature * 0:
             * incorrect signature * -1: error
             */
            DEBUG("%s starting DSA verify! \n", __func__);

            ret = DSA_verify(0, DgstData, size, sig, siglen, dsa);

            if (ret <= 0) {
                INFO("# FAIL %s for DSA.\n", local_verify ? "verify" : "sign");
                goto end;
            } else if (i == 0 && print_verify)
                    INFO("# PASS %s for DSA.\n",
                         local_verify ? "verify" : "sign");
        }
    }
#endif
#ifdef QAT_OPENSSL_PROVIDER
    EVP_PKEY *dsa_key = NULL;
    EVP_PKEY_CTX *dsa_sign_ctx;
    EVP_PKEY_CTX *dsa_verify_ctx;
    unsigned char *buf = NULL;
    unsigned char *buf2 = NULL;
    size_t buflen = 16 * 1024;
    size_t sigsize = buflen;

    buf = OPENSSL_malloc(buflen);
    if (buf == NULL){
        WARN("# FAIL DSA buf malloc failed! \n");
        goto end;
    }
    buf2 = OPENSSL_malloc(buflen);
    if (buf2 == NULL){
        WARN("# FAIL DSA buf2 malloc failed! \n");
        goto end;
    }

    memset(buf, 0, buflen);
    memset(buf2, 0, buflen);

    if (RAND_bytes(buf, 36) <= 0){
        WARN("# FAIL DSA get rand bytes failed! \n");
        goto end;
    }

    dsa_key = get_dsa(size);
    if (dsa_key == NULL){
        WARN("# FAIL DSA get dsa key failed! \n");
        goto end;
    }

    dsa_sign_ctx = EVP_PKEY_CTX_new(dsa_key, NULL);

    if (dsa_sign_ctx == NULL
        || EVP_PKEY_sign_init(dsa_sign_ctx) <= 0
        || EVP_PKEY_sign(dsa_sign_ctx, buf2, &sigsize, buf, 20) <= 0){
        WARN("# FAIL DSA PKEY sign failed! \n");
        goto end;
    }

    INFO("# PASS DSA sign.\n");

    dsa_verify_ctx = EVP_PKEY_CTX_new(dsa_key, NULL);
    if (dsa_verify_ctx == NULL
        || EVP_PKEY_verify_init(dsa_verify_ctx) <= 0
        || EVP_PKEY_verify(dsa_verify_ctx, buf2, sigsize, buf, 36) <= 0){
        WARN("# FAIL DSA verify init failed! \n");
        goto end;
    }
    INFO("# PASS DSA verify.\n");

#endif
end:
    DEBUG("%s start to clean up! \n", __func__);
#ifdef QAT_OPENSSL_PROVIDER
    OPENSSL_free(buf);
    OPENSSL_free(buf2);
#endif
#ifndef QAT_OPENSSL_PROVIDER
    if (DgstData)
        OPENSSL_free(DgstData);
    if (dsa != NULL)
        DSA_free(dsa);
#endif
    return ret;
}

static void dsa_tests_triage(TEST_PARAMS *args, int local_verify)
{
    struct async_additional_args_dsa extra_args;
    args->additional_args = &extra_args;
    extra_args.local_verify = local_verify;

    if (!args->enable_async)
        run_dsa(args);
    else
        start_async_job(args, run_dsa);
}

/******************************************************************************
* function:
*     tests_run_dsa_sign(TEST_PARAMS *args)
*
* @param args [IN] - the test parameters
*
* description:
*   dsa sign only plus 1 verify if verify flag is set
*
******************************************************************************/

void tests_run_dsa_sign(TEST_PARAMS *args)
{
    dsa_tests_triage(args, 0);
}


/******************************************************************************
* function:
*     tests_run_dsa_verify(TEST_PARAMS *args)
*
* @param args [IN] - the test parameters
*
* description:
*   dsa verify only plus 1 sign to setup.
*   If verify flag is set then it will only print pass on the first verify
*   operation.
*
******************************************************************************/

void tests_run_dsa_verify(TEST_PARAMS *args)
{
    dsa_tests_triage(args, 1);
}

/******************************************************************************
* function:
*     tests_run_dsa_verify(TEST_PARAMS *args)
*
* @param args [IN] - the test parameters
*
* description:
*   dsa verify only plus 1 sign to setup.
*   If verify flag is set then it will only print pass on the first verify
*   operation.
*
******************************************************************************/

void tests_run_dsa(TEST_PARAMS *args)
{
    int local_verify = 0;

    if (args->sign_only)
        local_verify = 0;
    else if (args->verify_only)
        local_verify = 1;
    dsa_tests_triage(args, local_verify);
}
