/* Copyright (c) 2023 Intel Corporation

Copyright 2017 The TensorFlow Authors. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================*/

// Utilities for working with XLA Literals.

#ifndef ITEX_CORE_COMPILER_MLIR_XLA_UTILS_LITERAL_UTIL_H_
#define ITEX_CORE_COMPILER_MLIR_XLA_UTILS_LITERAL_UTIL_H_

#include "absl/types/span.h"
#include "itex/core/compiler/xla/literal.h"
#include "itex/core/utils/status.h"
#include "protos/xla_data.pb.h"

namespace itex {

// Returns a BorrowingLiteral that utilizes the same underlying buffer owned by
// 'host_tensor'.
Status HostTensorToBorrowingLiteral(const Tensor& host_tensor,
                                    itex_xla::BorrowingLiteral* literal);
// Similar as above, except the literal shape is explicitly provided and used
// instead of obtaining it from the 'host_tensor'. The provided literal shape
// 'xla_shape' must be compatible with the shape of 'host_tensor'.
Status HostTensorToBorrowingLiteral(const itex_xla::Shape& xla_shape,
                                    const Tensor& host_tensor,
                                    itex_xla::BorrowingLiteral* literal);

// Returns a Literal with the contents of 'host_tensor', backed by its own
// storage (i.e., not reusing 'host_tensor's buffers.)
StatusOr<itex_xla::Literal> HostTensorToLiteral(const Tensor& host_tensor);

// Returns a MutableBorrowingLiteral that utilizes the same underlying buffer
// owned by 'host_tensor', but is mutable via the itex_xla::Literal methods.
Status HostTensorToMutableBorrowingLiteral(
    Tensor* host_tensor, itex_xla::MutableBorrowingLiteral* literal);
// Similar as above, except the literal shape is explicitly provided and used
// instead of obtaining it from the 'host_tensor'. The provided literal shape
// 'xla_shape' must be compatible with the shape of 'host_tensor'.
Status HostTensorToMutableBorrowingLiteral(
    const itex_xla::Shape& xla_shape, Tensor* host_tensor,
    itex_xla::MutableBorrowingLiteral* literal);

// Returns a BorrowingLiteral tuple that utilizes the same underlying buffers
// owned by 'host_tensors'.
Status HostTensorsToBorrowingLiteralTuple(absl::Span<const Tensor> host_tensors,
                                          itex_xla::BorrowingLiteral* literal);

// Copies 'literal' to freshly allocated 'host_tensor', which is allocated of
// type <target_type>.
// Fails if the literal's primitive type !=
// DataTypeToPrimitiveType(target_type). Note that <target_type> is not
// derivable from the type of <literal>, because multiple tensorflow types map
// to the same XLA type (e.g. INT32 and QINT32 both map to INT32 in
// XLA).
Status LiteralToHostTensor(const itex_xla::LiteralSlice& literal,
                           DataType target_type, Tensor* host_tensor);

// Copies the contents of 'literal' to a previously allocated tensor
// 'host_tensor'. The tensor and the literal must have the same number of
// elements and the same type.
Status CopyLiteralToHostTensor(const itex_xla::LiteralSlice& literal,
                               Tensor* host_tensor);

}  // namespace itex

#endif  // ITEX_CORE_COMPILER_MLIR_XLA_UTILS_LITERAL_UTIL_H_
