// SPDX-License-Identifier: Apache-2.0

use std::collections::HashSet;
use std::fmt::Write;

use tokio::io::{AsyncReadExt, AsyncWriteExt};

use crate::{
    BaseInterface, ErrorKind, InterfaceIdentifier, InterfaceState,
    InterfaceType, MergedInterfaces, NmstateError,
};

/// Comment added into our generated link files
const PERSIST_GENERATED_BY: &str = "# Generated by nmstate";
const SYSTEMD_LINK_FILE_PREFIX: &str = "98-nmstate-";

const SYSTEMD_NETWORK_LINK_FOLDER: &str = "/etc/systemd/network";

fn gen_nispor_iface_conf_for_alt_name(
    base_iface: &BaseInterface,
    cur_iface: Option<&BaseInterface>,
) -> Result<Option<nispor::IfaceConf>, NmstateError> {
    if let Some(alt_names) = base_iface.alt_names.as_ref() {
        let mut np_iface = nispor::IfaceConf::default();

        np_iface.state = match base_iface.state {
            InterfaceState::Up => nispor::IfaceState::Up,
            InterfaceState::Down => nispor::IfaceState::Down,
            _ => return Ok(None),
        };

        np_iface.name = base_iface.name.to_string();
        let cur_alt_names: HashSet<&str> = cur_iface
            .and_then(|i| i.alt_names.as_ref())
            .map(|v| v.as_slice())
            .unwrap_or(&[])
            .iter()
            .map(|entry| entry.name.as_str())
            .collect();
        let mut np_alt_names = Vec::new();
        for alt_name in alt_names {
            match (
                alt_name.is_absent(),
                cur_alt_names.contains(&alt_name.name.as_str()),
            ) {
                (true, true) | (false, false) => {
                    np_alt_names.push(nispor::AltNameConf::new(
                        alt_name.name.to_string(),
                        alt_name.is_absent(),
                    ));
                }
                (false, true) | (true, false) => {
                    // Already exist or already deleted, do nothing
                }
            }
        }
        if !np_alt_names.is_empty() {
            np_iface.alt_names = np_alt_names;
        }
        Ok(Some(np_iface))
    } else {
        Ok(None)
    }
}

pub(crate) async fn apply_ifaces_alt_names(
    merged_ifaces: &MergedInterfaces,
) -> Result<(), NmstateError> {
    let mut np_ifaces: Vec<nispor::IfaceConf> = Vec::new();

    for merged_iface in merged_ifaces.kernel_ifaces.values() {
        let apply_iface = if let Some(i) = merged_iface.for_apply.as_ref() {
            i.base_iface()
        } else {
            continue;
        };
        let cur_iface = merged_iface.current.as_ref().map(|i| i.base_iface());
        if let Some(np_iface) =
            gen_nispor_iface_conf_for_alt_name(apply_iface, cur_iface)?
        {
            np_ifaces.push(np_iface);
        }
    }

    if !np_ifaces.is_empty() {
        let mut net_conf = nispor::NetConf::default();
        net_conf.ifaces = Some(np_ifaces);

        if let Err(e) = net_conf.apply_async().await {
            return Err(NmstateError::new(
                ErrorKind::PluginFailure,
                format!("Failed to change alt-name: {} {}", e.kind, e.msg),
            ));
        }
    }
    Ok(())
}

pub(crate) async fn persist_alt_name_config(
    merged_ifaces: &MergedInterfaces,
) -> Result<(), NmstateError> {
    for merged_iface in merged_ifaces.kernel_ifaces.values() {
        let apply_iface = if let Some(i) = merged_iface.for_apply.as_ref() {
            i.base_iface()
        } else {
            continue;
        };
        if apply_iface.state == InterfaceState::Absent {
            remove_systemd_network_link_file(apply_iface.name.as_str()).await?
        } else if let Some(alt_names) = apply_iface.alt_names.as_ref() {
            let mut saved_alt_names =
                load_saved_alt_names(apply_iface.name.as_str()).await;
            for alt_name in alt_names {
                if alt_name.is_absent() {
                    saved_alt_names.remove(&alt_name.name);
                } else {
                    saved_alt_names.insert(alt_name.name.to_string());
                }
            }
            if saved_alt_names.is_empty() {
                remove_systemd_network_link_file(apply_iface.name.as_str())
                    .await?
            } else {
                save_systemd_network_link_file(
                    apply_iface,
                    merged_iface.current.as_ref().map(|i| i.base_iface()),
                    &saved_alt_names,
                )
                .await?;
            }
        }
    }
    Ok(())
}

// The systemd udev does not merge along all link files, the matched link
// file with smallest number will be solely used.
// So we use 98-nmstate-<iface_name>link which holds precedence than systemd
// fallback file 99-default.link.
fn gen_systemd_link_file_path(iface_name: &str) -> std::path::PathBuf {
    format!(
        "{SYSTEMD_NETWORK_LINK_FOLDER}/{SYSTEMD_LINK_FILE_PREFIX}{iface_name}.\
         link"
    )
    .into()
}

async fn load_saved_alt_names(iface_name: &str) -> HashSet<String> {
    let mut ret: HashSet<String> = HashSet::new();
    let file_path = gen_systemd_link_file_path(iface_name);

    let mut fd = match tokio::fs::File::open(&file_path).await {
        Ok(fd) => fd,
        Err(e) => {
            log::debug!("Failed to open {}: {e}", file_path.display());
            return ret;
        }
    };

    let mut content = String::new();
    if let Err(e) = fd.read_to_string(&mut content).await {
        log::debug!("Failed to read {}: {e}", file_path.display());
        return ret;
    }

    for line in content.lines() {
        if let Some(alt_name) = line.strip_prefix("AlternativeName=") {
            ret.insert(alt_name.trim().to_string());
        }
    }

    ret
}

async fn remove_systemd_network_link_file(
    iface_name: &str,
) -> Result<(), NmstateError> {
    let file_path = gen_systemd_link_file_path(iface_name);
    if file_path.exists() {
        log::debug!("Deleting {}", file_path.display());
        tokio::fs::remove_file(&file_path).await.map_err(|e| {
            NmstateError::new(
                ErrorKind::PluginFailure,
                format!("Failed to delete file {}: {e}", file_path.display()),
            )
        })
    } else {
        Ok(())
    }
}

// The `OriginalName` match rule is unstable because kernel interface
// name can not be predicted.
// For ethernet interface, using `OriginalName` match rule is unstable because
// kernel interface name cannot be predicted. Hence use try in these order:
//  * permanent MAC address + driver name[1]
//  * permanent MAC address
//  * PCI address
//  * MAC address + driver defined in current
//  * MAC address defined in desired
//  * Interface name defined in desired
// For non-ethernet interface, we use `OriginalName` match rule.
// [1]: Azure VM with `Microsoft Azure Network Adapter` has two
//      NICs holding the same MAC address. We need to make sure we matched
//      the correct by including driver information also.
fn gen_systemd_network_link_match_rules(
    des_iface: &BaseInterface,
    cur_iface: Option<&BaseInterface>,
) -> String {
    if let Some(cur_iface) = cur_iface {
        match (
            cur_iface.permanent_mac_address.as_ref(),
            cur_iface.driver.as_ref(),
            cur_iface.pci_address.as_ref(),
            cur_iface.mac_address.as_ref(),
        ) {
            (Some(perm_mac), Some(driver), _pci_addr, _mac) => {
                format!("PermanentMACAddress={perm_mac}\nDriver={driver}")
            }
            (Some(perm_mac), None, _pci_addr, _mac) => {
                format!("PermanentMACAddress={perm_mac}")
            }
            (None, _driver, Some(pci_addr), _mac) => {
                format!("Path=pci-{pci_addr}")
            }
            (None, Some(driver), None, Some(mac)) => {
                format!("MACAddress={mac}\nDriver={driver}")
            }
            (None, None, None, Some(mac)) => {
                format!("MACAddress={mac}")
            }
            _ => {
                format!("OriginalName={}", cur_iface.name)
            }
        }
    } else {
        // User is creating software interface or gen_conf mode
        if des_iface.iface_type == InterfaceType::Ethernet {
            match (des_iface.mac_address.as_ref(), des_iface.driver.as_ref()) {
                (Some(mac), Some(driver)) => {
                    format!("MACAddress={mac}\nDriver={driver}")
                }
                (Some(mac), None) => {
                    format!("MACAddress={mac}")
                }
                (None, _) => {
                    format!("OriginalName={}", des_iface.name)
                }
            }
        } else {
            format!("OriginalName={}", des_iface.name)
        }
    }
}

async fn save_systemd_network_link_file(
    des_base_iface: &BaseInterface,
    cur_base_iface: Option<&BaseInterface>,
    alt_names: &HashSet<String>,
) -> Result<(), NmstateError> {
    if !std::path::Path::new(SYSTEMD_NETWORK_LINK_FOLDER).exists() {
        tokio::fs::create_dir_all(SYSTEMD_NETWORK_LINK_FOLDER)
            .await
            .map_err(|e| {
                NmstateError::new(
                    ErrorKind::PluginFailure,
                    format!(
                        "Failed to create folder \
                         {SYSTEMD_NETWORK_LINK_FOLDER}: {e}"
                    ),
                )
            })?;
    }
    let iface_name = des_base_iface.name.as_str();

    let match_rules = match des_base_iface.identifier.as_ref() {
        Some(InterfaceIdentifier::MacAddress) => {
            let mut match_rules = if let Some(mac) =
                des_base_iface.permanent_mac_address.as_ref()
            {
                format!("PermanentMACAddress={mac}")
            } else if let Some(mac) = des_base_iface.mac_address.as_ref() {
                format!("MACAddress={mac}")
            } else {
                return Err(NmstateError::new(
                    ErrorKind::Bug,
                    format!(
                        "Got no MAC address for `identifier:mac-address` on \
                         interface {}, should be failed by previous checker",
                        des_base_iface.name
                    ),
                ));
            };
            // Include driver in match rule in case we have two NIC holding the
            // same MAC. (e.g. M$ Azure - Microsoft Azure Network Adapter)
            if let Some(cur_base_iface) = cur_base_iface.as_ref() {
                if let Some(driver) = cur_base_iface.driver.as_ref() {
                    write!(match_rules, "\nDriver={driver}").ok();
                }
            }
            match_rules
        }
        Some(InterfaceIdentifier::PciAddress) => {
            if let Some(pci_addr) = des_base_iface.pci_address.as_ref() {
                format!("Path=pci-{pci_addr}")
            } else {
                return Err(NmstateError::new(
                    ErrorKind::Bug,
                    format!(
                        "Got no PCI address for `identifier:pci-address` on \
                         interface {}, should be failed by previous checker",
                        des_base_iface.name
                    ),
                ));
            }
        }
        _ => {
            gen_systemd_network_link_match_rules(des_base_iface, cur_base_iface)
        }
    };

    let file_path = gen_systemd_link_file_path(iface_name);
    let mut content = {
        #[cfg_attr(any(), rustfmt::skip)]
        format!(
            "{PERSIST_GENERATED_BY}\n\
            [Match]\n\
            {match_rules}\n\n\
            [Link]\n"
        )
    };
    let alt_names: Vec<&str> = alt_names.iter().map(|s| s.as_str()).collect();
    for alt_name in alt_names.iter() {
        writeln!(content, "AlternativeName={alt_name}").ok();
    }
    // Since systemd stop using other naming policy after matched our link file
    // we should explicitly name this interface to current interface name,
    // otherwise, reboot will get NIC named as eth0.
    writeln!(content, "Name={iface_name}").ok();

    log::info!(
        "Creating {} for alt-names of interface {iface_name}",
        file_path.display(),
    );
    log::debug!(
        "Creating file {} with content: {}",
        file_path.display(),
        content
    );

    let mut fd = tokio::fs::File::create(&file_path).await.map_err(|e| {
        NmstateError::new(
            ErrorKind::PluginFailure,
            format!("Failed to create file {}: {e}", file_path.display()),
        )
    })?;
    fd.write_all(content.as_bytes()).await.map_err(|e| {
        NmstateError::new(
            ErrorKind::PluginFailure,
            format!("Failed to write file {}: {e}", file_path.display()),
        )
    })?;
    Ok(())
}
