/** BEGIN COPYRIGHT BLOCK
 * This Program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; version 2 of the License.
 *
 * This Program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this Program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Copyright (C) 2005 Red Hat, Inc.
 * All rights reserved.
 * END COPYRIGHT BLOCK **/

imports.gi.versions.Gtk = '3.0'
const Gtk = imports.gi.Gtk;
const Lang = imports.lang;
const Coolkey = imports.gi.Coolkey;
const Gio = imports.gi.Gio;
const GObject = imports.gi.GObject;
const Pango = imports.gi.Pango;
const GLib = imports.gi.GLib;

const CoolKeyNotifyIface = '<node> \
<interface name="com.jmagne.CoolKeyNotify"> \
<method name="notifyCoolKeyEvent"> \
    <arg type="x" direction="in" /> \
    <arg type="s" direction="in" /> \
    <arg type="t" direction="in" /> \
    <arg type="t" direction="in" /> \
    <arg type="s" direction="in" /> \
</method> \
</interface> \
</node>';

const CoolKeyNotify = new Lang.Class({
    Name: 'CoolKeyNotify',

    _init: function(client) {
        this._client = client;
        this._unique_name = "Unknown";
        this._dbusImpl = Gio.DBusExportedObject.wrapJSObject(CoolKeyNotifyIface, this);
        this._dbusImpl.export(Gio.DBus.session, '/com/jmagne/CoolKeyNotify');
        this._dbusId = Gio.DBus.session.own_name( 'com.rm5248', Gio.BusNameOwnerFlags.NONE, this._nameAcquired.bind(this), this._nameLost);
    },
    _nameAcquired: function( name ) {
         this._unique_name = name.unique_name;
         this._client._createCoolKeyMgr(this._unique_name);
    },
    _nameLost: function( name ) {
    },

    notifyCoolKeyEvent: function(aKeyType, aKeyID, aKeyState, aData, strData) {
        // print("In notifyCoolKeyEvent: aKeyType:  " + aKeyType + " aKeyID: " + aKeyID
        //    + " aKeyState: " + aKeyState + " aData: " + aData + " strData: " + strData) ;

        this._client._notifyCoolkey(aKeyType, aKeyID, aKeyState, aData, strData);
    },
   _getUniqueName() {
        return this._unique_name;
    }
   
});

class ESC {
    constructor() {
        this.application = new Gtk.Application();
        this.application.connect('activate', this._onActivate.bind(this));
        this.application.connect('startup', this._onStartup.bind(this));
    }

    _onActivate() {
        this._window.present();
    }
    _onStartup() {
         this._buildUI();
         this.notify = new CoolKeyNotify(this);
     }

    _onWindowDestroy() {
         if(this.mgr) {
             this.mgr.cleanup();
         }

         if(this.notify) {
             this.notify._dbusImpl.unexport();
         }
     }

    _onExit() {
        if(this.notify) {
            this.notify._dbusImpl.unexport();
            this.notify = null;
        }
        this.mgr.cleanup();
        this.mgr = null;
        this._window.destroy();
    }
  
    _notifyCoolkey(aKeyType, aKeyID, aKeyState, aData, strData) {
        switch(aKeyState) {
            case 1000:
                this._insertCoolkey(aKeyType, aKeyID, aKeyState, aData, strData);
            break; 
            case 1001:
               this._removeCoolkey(aKeyType, aKeyID, aKeyState, aData, strData);
            break;
            default:
            break;
        }
    }

    _getStatusString(intStatus) {
        switch(intStatus) {
            case 4:
                return "enrolled";
            break;

            case 2:
                return "unitialized";
            break;
            case 1:
                return "no applet";
            break;
            default:
                return "unknown";
            break;
        }
     }

    _insertCoolkey(aKeyType, aKeyID, aKeyState, aData, strData) {

         if(aKeyType && aKeyID) {

             let inserted = new Coolkey.Token({ key_type: String(aKeyType) ,
             cuid: aKeyID } );

             this.mgr.get_token_info(inserted);

             this._tokenStore.set (this._tokenStore.append(), [0, 1, 2, 3, 4],
                [inserted.issuer, inserted.issued_to, this._getStatusString(inserted.status), aKeyID,aKeyType]);
         }
     }

     _removeCoolkey(aKeyType, aKeyID, aKeyState, aData, strData) {
         let [ isSelected, iter]  = this._tokenStore.get_iter_first(); 

         while(isSelected) {
              if(aKeyID == this._tokenStore.get_value(iter,3)) {
                  this._tokenStore.remove(iter);
                  this._tokenInfoBuffer.text = "";
                  return;
              }
              isSelected  = this._tokenStore.iter_next(iter);
         }
     }

    _updateTokenInfoTextView(coolkey_token) {

        this._tokenInfoBuffer.text = "";

        let cuid = coolkey_token.cuid;
        let atr  = coolkey_token.atr;
        let issuer_info = coolkey_token.issuer_info;

        this._tokenInfoBuffer.text = "Token ID: " + cuid + "\n" 
             + "ATR: " + atr + "\n"
             + "Token Issuer URL: " + issuer_info + "\n";
     }

    _onTokenTreeViewSelectionChanged()  {

        let [ isSelected, model, iter ] = this.selection.get_selected();

        if(isSelected == false)
            return;

        let selected = new Coolkey.Token({ key_type: this._tokenStore.get_value(iter,4) , 
             cuid: this._tokenStore.get_value(iter,3) } );

        this.mgr.get_token_info(selected);
        this._updateTokenInfoTextView(selected);
    }

    _initConfig() {
         // See if it exists already, otherwise create
         this._configFile =  new GLib.KeyFile();

         this._configPath = GLib.get_user_config_dir() + "/esc";

         let configDir = Gio.File.new_for_path(this._configPath);

         try {
             configDir.make_directory(null);
         } catch (e) {
             if(e.matches (Gio.IOErrorEnum, Gio.IOErrorEnum.EXISTS)) {
             } else {
                this._onExitClicked();
             }
         }

         let config_name = this._configPath + "/esc.conf";
         let res = false;
         try {
             res = this._configFile.load_from_file (config_name,  0 );
         } catch(e) {
             // need to create config file
         }

         if (res == false) {
             this._configFile.set_string("ESC","name","Smart Card Utility");
             print("attmpeting to create: " + config_name);
             res = this._configFile.save_to_file(config_name); 
             print("res: " + res);
         } 
     }

    _buildUI() {
        // Create the application window
        this._window = new Gtk.ApplicationWindow({
            application: this.application,
            window_position: Gtk.WindowPosition.CENTER,
            border_width: 0});

            try {
                 this._window.set_icon_name('application-x-executable');
            } catch (err) {
                this._window.set_icon_name('application-x-executable');
            }

            this._hb = new Gtk.HeaderBar();
            this._hb.set_title("Smart Card Manager");
            this._hb.set_show_close_button(true);
            this._window.set_titlebar(this._hb);

            this._headerTokenBox = new Gtk.Box({orientation: Gtk.Orientation.HORIZONTAL,halign: Gtk.Align.CENTER});
            this._headerInfoBox = new Gtk.Box({orientation: Gtk.Orientation.HORIZONTAL,halign: Gtk.Align.CENTER});

            this._infoLabel = new Gtk.Label({label: "<b>Token Information</b>",use_markup: true});
            this._tokenLabel = new Gtk.Label({label: "<b>Tokens</b>",use_markup: true});

            this._headerInfoBox.add(this._infoLabel);

            this._headerTokenBox.add(this._tokenLabel);

            this._initConfig();
            this._outerGrid = new Gtk.Grid({orientation: Gtk.Orientation.HORIZONTAL, border_width: 0});
            this._tokenFrame = new Gtk.Frame({hexpand: true});
            this._infoFrame  = new Gtk.Frame({border_width: 0});
            
            this._tokenBox = new Gtk.Grid({orientation: Gtk.Orientation.VERTICAL, border_width: 0,hexpand: true});
            this._tokenFrame.add(this._tokenBox);

            this._infoBox = new Gtk.Grid({orientation: Gtk.Orientation.VERTICAL, border_width: 0,hexpand: true});
            this._infoFrame.add(this._infoBox);
             
            this._outerGrid.add(this._tokenFrame);
            this._outerGrid.add(this._infoFrame);
           
            this._window.add(this._outerGrid); 

            this._currentToken = null;

            // Create ListStore and Tree View

            this._tokenStore = new Gtk.ListStore ();
            this._tokenStore.set_column_types ([
                GObject.TYPE_STRING,
                GObject.TYPE_STRING,
                GObject.TYPE_STRING,
                GObject.TYPE_STRING,
                GObject.TYPE_STRING]);

             // Create the treeview
            this._tokenTreeView = new Gtk.TreeView ({
                expand: true,
                model: this._tokenStore });

            let issuer = new Gtk.TreeViewColumn ({ title: "Issuer" });
            let issuedTo = new Gtk.TreeViewColumn ({ title: "Issued To" });
            let status = new Gtk.TreeViewColumn ({ title: "Status" });
            let tokenId = new Gtk.TreeViewColumn({ title: "Token Id"});

            let bold = new Gtk.CellRendererText ({
              weight: Pango.Weight.BOLD });

            let normal = new Gtk.CellRendererText ();

            issuer.pack_start (bold, true);
            issuedTo.pack_start (normal, true);
            status.pack_start (normal, true);
            tokenId.pack_start(normal,true);
 
            issuer.add_attribute (bold, "text", 0);
            issuedTo.add_attribute (normal, "text", 1);
            status.add_attribute (normal, "text", 2);
            tokenId.add_attribute(normal,"text",3);

            this._tokenTreeView.insert_column (issuer, 0);
            this._tokenTreeView.insert_column (issuedTo, 1);
            this._tokenTreeView.insert_column (status, 2);
            this._tokenTreeView.insert_column (tokenId, 3);

            // Connect view slection to a method
            this._label = new Gtk.Label ({ label: "" });

            this.selection = this._tokenTreeView.get_selection();
            this.selection.set_mode(Gtk.SelectionMode.BROWSE);

            // When something new is selected, call _on_changed
            this.selection.connect ('changed', Lang.bind (this, this._onTokenTreeViewSelectionChanged));

            //Create a TextView with it's own TextBuffer

            this._tokenInfoBuffer = new Gtk.TextBuffer();
            this._tokenInfoTextView = new Gtk.TextView ({
            buffer: this._tokenInfoBuffer,
            editable: false,
            wrap_mode: Gtk.WrapMode.WORD, hexpand: true });

            this._scrolledTokenInfoWindow = new Gtk.ScrolledWindow ({
                hscrollbar_policy: Gtk.PolicyType.AUTOMATIC,
                vscrollbar_policy: Gtk.PolicyType.AUTOMATIC,
                shadow_type: Gtk.ShadowType.ETCHED_IN,
                height_request: 400,
                width_request: 400, });
           
            this._scrolledTokenInfoWindow.add_with_viewport (this._tokenInfoTextView);
            this._infoBox.add(this._headerInfoBox);
            this._infoBox.add(this._scrolledTokenInfoWindow);
            this._operationsGrid =  new Gtk.Grid({orientation: Gtk.Orientation.VERTICAL, 
                halign: Gtk.Align.CENTER,
                valign: Gtk.Align.CENTER,
                row_spacing: 20,
                border_width: 0});

            this._operationsLabel = new Gtk.Label({label: "<b>Operations</b>", use_markup: true});
            this._infoBox.add(this._operationsGrid); 
            this._operationsGrid.add(this._operationsLabel);

            this._exitButton = new Gtk.Button({ label: "Exit" });
            this._exitButton.connect("clicked", this._onExit.bind(this));
            this._operationsGrid.add(this._exitButton);
           
            this._tokenBox.add(this._headerTokenBox);
            this._tokenBox.add(this._tokenTreeView);

            this._window.set_default_size(750, 550);
            this._window.connect('destroy',this._onWindowDestroy.bind(this));
            this._window.show_all();
    } 
    _createCoolKeyMgr(dbusName) {
        this.mgr = new Coolkey.Mgr({ dbusUniqueName: dbusName, config_dir: this._configPath});
        //this.mgr.speak("Welcome to ESC...");
    }
}

let app = new ESC();
app.application.run (ARGV);
