// SPDX-License-Identifier: Apache-2.0

use std::panic;

use pretty_assertions::assert_eq;

use super::utils::assert_value_match;
use crate::{IfaceState, NetConf, NetState};

const IFACE_NAME: &str = "veth1";

const EXPECTED_VETH_INFO: &str = r#"---
peer: veth1.ep"#;

#[test]
fn test_get_veth_iface_yaml() {
    with_veth_iface(|| {
        let state = NetState::retrieve().unwrap();
        let iface = &state.ifaces[IFACE_NAME];
        let iface_type = &iface.iface_type;
        assert_eq!(iface_type, &crate::IfaceType::Veth);
        assert_value_match(EXPECTED_VETH_INFO, &iface.veth);
    });
}

fn with_veth_iface<T>(test: T)
where
    T: FnOnce() + panic::UnwindSafe,
{
    super::utils::set_network_environment("veth");

    let result = panic::catch_unwind(|| {
        test();
    });

    super::utils::clear_network_environment();
    assert!(result.is_ok())
}

const VETH_CREATE_YML: &str = r#"---
ifaces:
  - name: veth1
    type: veth
    mac-address: 00:23:45:67:89:1a
    veth:
      peer: veth1.ep
  - name: veth1.ep
    type: veth
    "#;

const VETH_CHANGE_MAC_YML: &str = r#"---
ifaces:
  - name: veth1
    type: veth
    mac-address: 00:23:45:67:89:2a"#;

const VETH_DOWN_YML: &str = r#"---
ifaces:
  - name: veth1
    type: veth
    state: down"#;

const VETH_DELETE_YML: &str = r#"---
ifaces:
  - name: veth1
    type: veth
    state: absent"#;

#[test]
fn test_create_down_delete_veth() {
    let net_conf: NetConf = serde_yaml::from_str(VETH_CREATE_YML).unwrap();
    net_conf.apply().unwrap();
    // Wait 1 second for veth to up
    std::thread::sleep(std::time::Duration::from_secs(1));
    let state = NetState::retrieve().unwrap();
    let iface = &state.ifaces[IFACE_NAME];
    assert_eq!(&iface.iface_type, &crate::IfaceType::Veth);
    assert_eq!(iface.veth.as_ref().unwrap().peer, "veth1.ep");
    assert_eq!(iface.state, IfaceState::Up);
    assert_eq!(iface.mac_address, "00:23:45:67:89:1a".to_string());

    // Change the MAC should have the interface as UP state
    let net_conf: NetConf = serde_yaml::from_str(VETH_CHANGE_MAC_YML).unwrap();
    net_conf.apply().unwrap();
    // Wait 1 second for veth to up
    std::thread::sleep(std::time::Duration::from_secs(1));
    let state = NetState::retrieve().unwrap();
    let iface = &state.ifaces[IFACE_NAME];
    assert_eq!(iface.state, IfaceState::Up);
    assert_eq!(iface.mac_address, "00:23:45:67:89:2a".to_string());

    let net_conf: NetConf = serde_yaml::from_str(VETH_DOWN_YML).unwrap();
    net_conf.apply().unwrap();
    let state = NetState::retrieve().unwrap();
    let iface = &state.ifaces[IFACE_NAME];
    assert_eq!(iface.state, IfaceState::Down);

    let net_conf: NetConf = serde_yaml::from_str(VETH_DELETE_YML).unwrap();
    net_conf.apply().unwrap();
    let state = NetState::retrieve().unwrap();
    assert_eq!(None, state.ifaces.get(IFACE_NAME));
}

#[test]
fn test_change_veth_mtu() {
    let net_conf: NetConf = serde_yaml::from_str(
        r#"---
        interfaces:
          - name: veth1
            type: veth
            mtu: 2000
            veth:
              peer: veth1.ep
          - name: veth1.ep
            type: veth
            "#,
    )
    .unwrap();
    net_conf.apply().unwrap();

    let state = NetState::retrieve().unwrap();
    let iface = &state.ifaces[IFACE_NAME];
    assert_eq!(iface.mtu, 2000);

    let net_conf: NetConf = serde_yaml::from_str(VETH_DELETE_YML).unwrap();
    net_conf.apply().unwrap();
    let state = NetState::retrieve().unwrap();
    assert_eq!(None, state.ifaces.get(IFACE_NAME));
}
