/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set ts=8 sts=2 et sw=2 tw=80: */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#include "ActiveElementManager.h"
#include "mozilla/EventStateManager.h"
#include "mozilla/EventStates.h"
#include "mozilla/PresShell.h"
#include "mozilla/StaticPrefs_ui.h"
#include "mozilla/dom/Element.h"
#include "mozilla/dom/Document.h"

static mozilla::LazyLogModule sApzAemLog("apz.activeelement");
#define AEM_LOG(...) MOZ_LOG(sApzAemLog, LogLevel::Debug, (__VA_ARGS__))

namespace mozilla {
namespace layers {

ActiveElementManager::ActiveElementManager()
    : mCanBePan(false), mCanBePanSet(false), mSetActiveTask(nullptr) {}

ActiveElementManager::~ActiveElementManager() = default;

void ActiveElementManager::SetTargetElement(dom::EventTarget* aTarget) {
  if (mTarget) {
    // Multiple fingers on screen (since HandleTouchEnd clears mTarget).
    AEM_LOG("Multiple fingers on-screen, clearing target element\n");
    CancelTask();
    ResetActive();
    ResetTouchBlockState();
    return;
  }

  mTarget = dom::Element::FromEventTargetOrNull(aTarget);
  AEM_LOG("Setting target element to %p\n", mTarget.get());
  TriggerElementActivation();
}

void ActiveElementManager::HandleTouchStart(bool aCanBePan) {
  AEM_LOG("Touch start, aCanBePan: %d\n", aCanBePan);
  if (mCanBePanSet) {
    // Multiple fingers on screen (since HandleTouchEnd clears mCanBePanSet).
    AEM_LOG("Multiple fingers on-screen, clearing touch block state\n");
    CancelTask();
    ResetActive();
    ResetTouchBlockState();
    return;
  }

  mCanBePan = aCanBePan;
  mCanBePanSet = true;
  TriggerElementActivation();
}

void ActiveElementManager::TriggerElementActivation() {
  // Both HandleTouchStart() and SetTargetElement() call this. They can be
  // called in either order. One will set mCanBePanSet, and the other, mTarget.
  // We want to actually trigger the activation once both are set.
  if (!(mTarget && mCanBePanSet)) {
    return;
  }

  // If the touch cannot be a pan, make mTarget :active right away.
  // Otherwise, wait a bit to see if the user will pan or not.
  if (!mCanBePan) {
    SetActive(mTarget);
  } else {
    CancelTask();  // this is only needed because of bug 1169802. Fixing that
                   // bug properly should make this unnecessary.
    MOZ_ASSERT(mSetActiveTask == nullptr);

    RefPtr<CancelableRunnable> task =
        NewCancelableRunnableMethod<nsCOMPtr<dom::Element>>(
            "layers::ActiveElementManager::SetActiveTask", this,
            &ActiveElementManager::SetActiveTask, mTarget);
    mSetActiveTask = task;
    NS_GetCurrentThread()->DelayedDispatch(
        task.forget(), StaticPrefs::ui_touch_activation_delay_ms());
    AEM_LOG("Scheduling mSetActiveTask %p\n", mSetActiveTask.get());
  }
}

void ActiveElementManager::ClearActivation() {
  AEM_LOG("Clearing element activation\n");
  CancelTask();
  ResetActive();
}

void ActiveElementManager::HandleTouchEndEvent(bool aWasClick) {
  AEM_LOG("Touch end event, aWasClick: %d\n", aWasClick);

  // If the touch was a click, make mTarget :active right away.
  // nsEventStateManager will reset the active element when processing
  // the mouse-down event generated by the click.
  CancelTask();
  if (aWasClick) {
    // Scrollbar thumbs use a different mechanism for their active
    // highlight (the "active" attribute), so don't set the active state
    // on them because nothing will clear it.
    if (!(mTarget && mTarget->IsXULElement(nsGkAtoms::thumb))) {
      SetActive(mTarget);
    }
  } else {
    // We might reach here if mCanBePan was false on touch-start and
    // so we set the element active right away. Now it turns out the
    // action was not a click so we need to reset the active element.
    ResetActive();
  }

  ResetTouchBlockState();
}

void ActiveElementManager::HandleTouchEnd() {
  AEM_LOG("Touch end, clearing pan state\n");
  mCanBePanSet = false;
}

static nsPresContext* GetPresContextFor(nsIContent* aContent) {
  if (!aContent) {
    return nullptr;
  }
  PresShell* presShell = aContent->OwnerDoc()->GetPresShell();
  if (!presShell) {
    return nullptr;
  }
  return presShell->GetPresContext();
}

void ActiveElementManager::SetActive(dom::Element* aTarget) {
  AEM_LOG("Setting active %p\n", aTarget);

  if (nsPresContext* pc = GetPresContextFor(aTarget)) {
    pc->EventStateManager()->SetContentState(aTarget, NS_EVENT_STATE_ACTIVE);
  }
}

void ActiveElementManager::ResetActive() {
  AEM_LOG("Resetting active from %p\n", mTarget.get());

  // Clear the :active flag from mTarget by setting it on the document root.
  if (mTarget) {
    dom::Element* root = mTarget->OwnerDoc()->GetDocumentElement();
    if (root) {
      AEM_LOG("Found root %p, making active\n", root);
      SetActive(root);
    }
  }
}

void ActiveElementManager::ResetTouchBlockState() {
  mTarget = nullptr;
  mCanBePanSet = false;
}

void ActiveElementManager::SetActiveTask(
    const nsCOMPtr<dom::Element>& aTarget) {
  AEM_LOG("mSetActiveTask %p running\n", mSetActiveTask.get());

  // This gets called from mSetActiveTask's Run() method. The message loop
  // deletes the task right after running it, so we need to null out
  // mSetActiveTask to make sure we're not left with a dangling pointer.
  mSetActiveTask = nullptr;
  SetActive(aTarget);
}

void ActiveElementManager::CancelTask() {
  AEM_LOG("Cancelling task %p\n", mSetActiveTask.get());

  if (mSetActiveTask) {
    mSetActiveTask->Cancel();
    mSetActiveTask = nullptr;
  }
}

}  // namespace layers
}  // namespace mozilla
