// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use std::sync::RwLock;

use glean_core::metrics::MetricType;

use super::{BooleanMetric, CommonMetricData, ErrorType, StringMetric};

/// Sealed traits protect against downstream implementations.
///
/// We wrap it in a private module that is inaccessible outside of this module.
mod private {
    use super::{BooleanMetric, CommonMetricData, StringMetric};

    /// The sealed trait.
    ///
    /// This also allows us to hide methods, that are only used internally
    /// and should not be visible to users.
    pub trait Sealed {
        type Inner: glean_core::metrics::MetricType + Clone;

        /// Create a new metric object from the inner type.
        fn from_inner(metric: Self::Inner) -> Self;

        /// Create a new `glean_core `metric of the inner type.
        fn new_inner(meta: CommonMetricData) -> Self::Inner;
    }

    // `LabeledMetric<BooleanMetric>` is possible.
    //
    // See [Labeled Booleans](https://mozilla.github.io/glean/book/user/metrics/labeled_booleans.html).
    impl Sealed for BooleanMetric {
        type Inner = glean_core::metrics::BooleanMetric;

        fn from_inner(metric: Self::Inner) -> Self {
            BooleanMetric(metric)
        }

        fn new_inner(meta: CommonMetricData) -> Self::Inner {
            glean_core::metrics::BooleanMetric::new(meta)
        }
    }

    // `LabeledMetric<StringMetric>` is possible.
    //
    // See [Labeled Strings](https://mozilla.github.io/glean/book/user/metrics/labeled_strings.html).
    impl Sealed for StringMetric {
        type Inner = glean_core::metrics::StringMetric;

        fn from_inner(metric: Self::Inner) -> Self {
            StringMetric(metric)
        }

        fn new_inner(meta: CommonMetricData) -> Self::Inner {
            glean_core::metrics::StringMetric::new(meta)
        }
    }
}

/// Marker trait for metrics that can be nested inside a labeled metric.
///
/// This trait is sealed and cannot be implemented for types outside this crate.
pub trait AllowLabeled: private::Sealed {}

// Implement the trait for everything we marked as allowed.
impl<T> AllowLabeled for T where T: private::Sealed {}

/// A labeled metric.
///
/// Labeled metrics allow to record multiple sub-metrics of the same type under different string labels.
///
/// ## Example
///
/// The following piece of code will be generated by `glean_parser`:
///
/// ```rust,ignore
/// use glean::metrics::{LabeledMetric, BooleanMetric, CommonMetricData, Lifetime};
/// use once_cell::sync::Lazy;
///
/// mod error {
///     pub static seen_one: Lazy<LabeledMetric<BooleanMetric>> = Lazy::new(|| LabeledMetric::new(CommonMetricData {
///         name: "seen_one".into(),
///         category: "error".into(),
///         send_in_pings: vec!["ping".into()],
///         disabled: false,
///         lifetime: Lifetime::Ping,
///         ..Default::default()
///     }, None));
/// }
/// ```
///
/// It can then be used with:
///
/// ```rust,ignore
/// errro::seen_one.get("upload").set(true);
/// ```
#[derive(Debug)]
pub struct LabeledMetric<T: AllowLabeled> {
    /// Wrapping the underlying core metric.
    ///
    /// We delegate all functionality to this and wrap it up again in our own metric type.
    ///
    /// *Note*: Needs to be thread-safe mutable, as required by the underlying `glean_core` type.
    core: RwLock<glean_core::metrics::LabeledMetric<T::Inner>>,
}

impl<T> LabeledMetric<T>
where
    T: AllowLabeled,
{
    /// Create a new labeled metric from the given metric instance and optional list of labels.
    ///
    /// See [`get`](#method.get) for information on how static or dynamic labels are handled.
    pub fn new(meta: CommonMetricData, labels: Option<Vec<String>>) -> LabeledMetric<T> {
        let submetric = T::new_inner(meta);
        let core = glean_core::metrics::LabeledMetric::new(submetric, labels);

        LabeledMetric {
            core: RwLock::new(core),
        }
    }

    /// Get a specific metric for a given label.
    ///
    /// If a set of acceptable labels were specified in the `metrics.yaml` file,
    /// and the given label is not in the set, it will be recorded under the special `__other__` label.
    ///
    /// If a set of acceptable labels was not specified in the `metrics.yaml` file,
    /// only the first 16 unique labels will be used.
    /// After that, any additional labels will be recorded under the special `__other__` label.
    ///
    /// Labels must conform to the [label formatting regular expression](https://mozilla.github.io/glean/book/user/metrics/index.html#label-format).
    /// Labels must be made up of dot-separated identifiers with lowercase ASCII alphanumerics, containing underscores and dashes.
    ///
    /// If an invalid label is used, the metric will be recorded in the special `__other__` label.
    pub fn get(&self, label: &str) -> T {
        let mut core = self
            .core
            .write()
            .expect("lock of wrapped metric was poisoned");
        let inner = core.get(label);
        T::from_inner(inner)
    }

    /// **Test-only API.**
    ///
    /// Get the number of recorded errors for this labeled metric and error type.
    ///
    /// ## Arguments
    ///
    /// * error - The type of error
    /// * `storage_name` - the storage name to look into.
    ///
    /// ## Return value
    ///
    /// The number of errors reported if any.
    /// Otherwise an error.
    pub fn test_get_num_recorded_errors(
        &self,
        error_type: ErrorType,
        storage_name: Option<&str>,
    ) -> Result<i32, String> {
        crate::with_glean(move |glean| {
            let core = self
                .core
                .read()
                .expect("lock of wrapped metric was poisoned");
            glean_core::test_get_num_recorded_errors(
                &glean,
                &core.get_submetric().meta(),
                error_type,
                storage_name,
            )
        })
    }
}
