// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2021 Alibaba Cloud. or its affiliates. All Rights Reserved.
 *
 * This module introduces common logic for dragonball virtual platform
 * features.
 */

#define pr_fmt(fmt) "dragonball platform: " fmt

#include <linux/slab.h>
#include <linux/ioport.h>
#include <linux/platform_device.h>
#include <dragonball/platform_feature.h>
#include <dragonball/platform_feature_defs.h>

static DEFINE_RWLOCK(db_platform_feature_lock);
static LIST_HEAD(common_platform_feature_list);

static bool db_features_probed;

struct common_feature_entry {
	struct common_platform_feature *feature;
	struct platform_feature_driver driver;
	struct list_head list;
};

int register_db_platform_feature(struct common_platform_feature *feature,
				 struct platform_feature_driver driver)
{
	int r = 0;
	struct common_feature_entry *e, *entry;

	if (db_features_probed)
		return -EPERM;

	if (!has_dragonball_feature(driver.feature_type))
		return -ENOENT;

	entry = kzalloc(sizeof(*entry), GFP_KERNEL);
	if (!entry)
		return -ENOMEM;

	write_lock(&db_platform_feature_lock);
	list_for_each_entry(e, &common_platform_feature_list, list) {
		if (e->driver.feature_type == driver.feature_type) {
			r = -EEXIST;
			goto error;
		}
	}

	entry->feature = feature;
	entry->driver = driver;
	list_add_tail(&entry->list, &common_platform_feature_list);
	write_unlock(&db_platform_feature_lock);

	return 0;

error:
	write_unlock(&db_platform_feature_lock);
	kfree(entry);
	return r;
}

static int platform_allocate_msi_irq(struct dragonball_device *db_dev,
				     u32 nr_irq)
{
	u32 current_irq = db_dev->next_irq;
	u32 next_irq = current_irq + nr_irq - 1;

	if (next_irq > db_dev->max_irq)
		return -EINVAL;

	db_dev->next_irq = next_irq;
	return current_irq;
}

/*
 * Since dragonball_platform_device is simulated by dragonball, and the
 * _probe_db_platform_feature function performs serial read and write
 * operations on mmio space, dragonball can ensure that each time the
 * mmio space is written, the corresponding information can be modified
 * immediately without delay.
 */
static int _probe_db_platform_feature(struct dragonball_device *db_dev,
				      struct common_feature_entry *entry)
{
	int r;
	u32 feature_caps;
	u64 cfg_offset;
	void __iomem *cfg_base;
	struct resource *db_dev_mem;
	resource_size_t io_mem_base, io_mem_size;
	struct common_platform_feature *feature = entry->feature;
	struct platform_feature_driver *driver = &entry->driver;

	if (!(db_dev->acked_features & (1 << driver->feature_type))) {
		pr_info("feature %d is not negoitated, skip probe",
			driver->feature_type);
		return 0;
	}

	/* Send driver now feature */
	writel((u32)driver->feature_type,
	       db_dev->base + DB_DRIVER_NOW_FEAT_OFF);

	/* Get and check device's config space offset */
	cfg_offset = readl(db_dev->base + DB_DEVICE_NOW_FEAT_CFG_SPACE_HI_OFF);
	cfg_offset <<= 32;
	cfg_offset += readl(db_dev->base + DB_DEVICE_NOW_FEAT_CFG_SPACE_LO_OFF);
	if (!cfg_offset)
		return -EINVAL;

	/* check the config space's offset is valid or not */
	db_dev_mem = platform_get_resource(db_dev->pdev, IORESOURCE_MEM, 0);
	if (cfg_offset < DB_FEATURE_SPACE_OFF ||
	    cfg_offset >= resource_size(db_dev_mem))
		return -EINVAL;

	cfg_base = db_dev->base + cfg_offset;
	feature->dragonball_device = db_dev;
	feature->cfg_base = cfg_base + DB_FEAT_PRIVATE_OFF;

	/* Send status [ACK] */
	writel(DB_PLATFORM_DRIVER_ACK, cfg_base + DB_FEAT_STATUS_OFF);

	/* Send driver version */
	writel(driver->version, cfg_base + DB_FEAT_DRIVER_VER_OFF);

	/* Get and check feature's version */
	feature->version = readl(cfg_base + DB_FEAT_VER_OFF);
	if (!feature->version) {
		r = -EPERM;
		goto err_notify;
	}

	if (driver->check_version) {
		r = driver->check_version(feature);
		if (r) {
			pr_err("version %d for feature %d is not supported!\n",
			       feature->version, driver->feature_type);
			goto err_notify;
		}
	}

	/* Send driver capabilities */
	writel(driver->caps, cfg_base + DB_FEAT_DRIVER_CAP_OFF);

	/* Get feature's capabilities */
	feature_caps = readl(cfg_base + DB_FEAT_CAP_OFF);
	if (!feature_caps) {
		r = -EPERM;
		goto err_notify;
	}

	/* Check and set feature's capabilities */
	if (driver->ack_cap) {
		feature_caps = driver->ack_cap(feature, feature_caps);
		if (!feature_caps) {
			pr_err("feature %d's feature negotiation failed!\n",
			       driver->feature_type);
			r = -EPERM;
			goto err_notify;
		}
	}
	feature->acked_caps = feature_caps;

	/* Send status [FEATURE_OK] */
	writel(DB_PLATFORM_FEAT_OK, cfg_base + DB_FEAT_STATUS_OFF);

	/* Get io mem's info */
	io_mem_base = readl(cfg_base + DB_FEAT_MMIO_BASE_HI_OFF);
	io_mem_base <<= 32;
	io_mem_base |= readl(cfg_base + DB_FEAT_MMIO_BASE_LO_OFF);
	feature->phys_base = io_mem_base;

	io_mem_size = readl(cfg_base + DB_FEAT_MMIO_SIZE_HI_OFF);
	io_mem_size <<= 32;
	io_mem_size |= readl(cfg_base + DB_FEAT_MMIO_SIZE_LO_OFF);
	feature->mmio_size = io_mem_size;

	/*
	 * In some of simple features, they may depend only on resources provided by
	 * the common platform feature, and have no resources of their own, so we need
	 * to check the resource status of the feature at the time of probe and allow
	 * skipping some resource configuration stages.
	 */
	/* Remap feature's mmio space */
	if (io_mem_base && io_mem_size) {
		feature->mmio_base = ioremap(io_mem_base, io_mem_size);
		if (feature->mmio_base == NULL) {
			pr_err("feature %d's io remap failed\n",
			       driver->feature_type);
			goto err_notify;
		}
	}

	/* Setup msi irq */
	feature->msi_irq_nr = readl(cfg_base + DB_FEAT_MSI_NR_OFF);
	if (feature->msi_irq_nr) {
		feature->msi_irq_base =
			platform_allocate_msi_irq(db_dev, feature->msi_irq_nr);
		if (feature->msi_irq_base < 0) {
			pr_err("feature %d allocate msi irq failed: %d\n",
			       driver->feature_type, r);
			goto err_remap;
		}
		/* the irq base in dragonball start from 0 */
		writel(feature->msi_irq_base - db_dev->base_irq,
		       cfg_base + DB_FEAT_MSI_BASE_OFF);
	}

	/* Start feature's driver */
	if (driver->start) {
		r = driver->start(feature);
		if (r) {
			pr_err("feature %d start failed\n",
			       driver->feature_type);
			goto err_irq_alloc;
		}
	}

	/* Send status [DRIVER_OK] */
	writel(DB_PLATFORM_DRIVER_OK, cfg_base + DB_FEAT_STATUS_OFF);
	return 0;

err_irq_alloc:
	db_dev->next_irq -= feature->msi_irq_nr;
err_remap:
	iounmap(feature->mmio_base);
err_notify:
	writel(DB_PLATFORM_DRIVER_FAIL, cfg_base + DB_FEAT_STATUS_OFF);
	return r;
}

int probe_db_platform_features(struct dragonball_device *db_dev)
{
	int r;
	struct common_feature_entry *entry;

	read_lock(&db_platform_feature_lock);
	list_for_each_entry(entry, &common_platform_feature_list, list) {
		r = _probe_db_platform_feature(db_dev, entry);
		if (r < 0) {
			pr_err("probe feature %d failed: %d",
			       entry->driver.feature_type, r);
		}
	}

	db_features_probed = true;

	read_unlock(&db_platform_feature_lock);
	return 0;
}

irqreturn_t db_features_handle_interrupt(struct dragonball_device *db_dev)
{
	struct common_feature_entry *entry;
	int r = IRQ_NONE;

	read_lock(&db_platform_feature_lock);
	list_for_each_entry(entry, &common_platform_feature_list, list) {
		if (!(entry->driver.legacy_irq))
			continue;

		r = entry->driver.legacy_irq(entry->feature);
		if (r != IRQ_NONE)
			break;
	}

	read_unlock(&db_platform_feature_lock);
	return r;
}
