// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2019 Alibaba Cloud. or its affiliates. All Rights Reserved.
 *
 * Paravirtualized DMA operations for PCI devices on Intel x86 platform.
 */
#define pr_fmt(fmt) "pvdma-mmio[pci]: " fmt

#include <linux/mm.h>
#include <linux/dma-mapping.h>
#include <linux/pci.h>
#include <linux/pvdma.h>
#ifdef CONFIG_ARM64
#include <dragonball/dragonball.h>
#endif
#include <linux/dma-direct.h>

static const struct dma_map_ops *inner_ops;

static void *pv_dma_pci_alloc(struct device *dev, size_t size,
			dma_addr_t *dma_handle, gfp_t gfp,
			unsigned long attrs)
{
	void *vaddr;
	phys_addr_t phys_addr;

	if (!inner_ops->alloc)
		return NULL;

	vaddr = inner_ops->alloc(dev, size, dma_handle, gfp, attrs);

	if (vaddr) {
		phys_addr = dma_to_phys(dev, *dma_handle);
		pv_dma_map(phys_addr, size, true);
	}

	return vaddr;
}

static void pv_dma_pci_free(struct device *dev, size_t size, void *vaddr,
			dma_addr_t dma_handle, unsigned long attrs)
{
	phys_addr_t phys_addr;

	if (inner_ops->free)
		inner_ops->free(dev, size, vaddr, dma_handle, attrs);

	phys_addr = dma_to_phys(dev, dma_handle);
	pv_dma_map(phys_addr, size, false);

}

static dma_addr_t pv_dma_pci_map_page(struct device *dev, struct page *page,
				unsigned long offset, size_t size,
				enum dma_data_direction dir,
				unsigned long attrs)
{
	dma_addr_t dma_handle;
	phys_addr_t phys_addr;

	if (!inner_ops->map_page)
		return DMA_MAPPING_ERROR;

	dma_handle = inner_ops->map_page(dev, page, offset, size, dir, attrs);

	/*
	 * Compared with 4.19, kernel 5.10 deletes the mapping_error() function in
	 * dma_map_ops, uses dma_mapping_error uniformly, and the uniform map
	 * error address is DMA_MAPPING_ERROR.
	 */
	if (dma_mapping_error(dev, dma_handle))
		return DMA_MAPPING_ERROR;

	phys_addr = dma_to_phys(dev, dma_handle);
	pv_dma_map(phys_addr, size, true);

	return dma_handle;
}

static void pv_dma_pci_unmap_page(struct device *dev, dma_addr_t dma_handle,
			size_t size, enum dma_data_direction dir,
			unsigned long attrs)
{
	phys_addr_t phys_addr;

	if (inner_ops->unmap_page)
		inner_ops->unmap_page(dev, dma_handle, size, dir, attrs);

	phys_addr = dma_to_phys(dev, dma_handle);
	pv_dma_map(phys_addr, size, false);
}

static int pv_dma_pci_map_sg(struct device *dev, struct scatterlist *sgl,
			int nents, enum dma_data_direction dir,
			unsigned long attrs)
{

	if (!inner_ops->map_sg)
		return 0;

	nents = inner_ops->map_sg(dev, sgl, nents, dir, attrs);

	if (nents)
		pv_dma_map_sg(sgl, nents, true);

	return nents;
}

static void pv_dma_pci_unmap_sg(struct device *dev, struct scatterlist *sgl,
				int nents, enum dma_data_direction dir,
				unsigned long attrs)
{
	if (inner_ops->unmap_sg)
		inner_ops->unmap_sg(dev, sgl, nents, dir, attrs);

	pv_dma_map_sg(sgl, nents, false);
}

static int pv_dma_pci_supported(struct device *dev, u64 mask)
{
	if (inner_ops->dma_supported)
		return inner_ops->dma_supported(dev, mask);

	return 0;
}

/*
 * Compared with 4.19, kernel 5.10 has made the following two changes to dma_ops:
 *
 * 1. Integrate dma_ops under different architectures into dma_direct_ops. For
 * details, refer to commit id 55897af63091.
 *
 * 2. Bypass indirect calls for dma-direct, which causes the current global
 * variable dma_ops to be NULL, and the kernel will directly call the dma_direct
 * related functions instead of using the dev->dma_ops callback function. For
 * details, refer to commit id 356da6d0cde3.
 *
 * Therefore, we added an instance of pvdma_direct_ops, which records the
 * functions related to dma_direct as inner_ops of pvdma.
 *
 * Also, since dma_direct_map_page and dma_direct_unmap_page are inline functions,
 * new functions are added to encapsulate them as pvdma_direct_*.
 */
const struct dma_map_ops pvdma_direct_ops = {
	.alloc = dma_direct_alloc,
	.free = dma_direct_free,
	.map_page = pvdma_direct_map_page,
	.unmap_page = pvdma_direct_unmap_page,
	.map_sg = dma_direct_map_sg,
	.unmap_sg = dma_direct_unmap_sg,
	.dma_supported = dma_direct_supported,
};

const struct dma_map_ops pv_dma_pci_ops = {
	.alloc		= pv_dma_pci_alloc,
	.free		= pv_dma_pci_free,
	.map_page	= pv_dma_pci_map_page,
	.unmap_page	= pv_dma_pci_unmap_page,
	.map_sg		= pv_dma_pci_map_sg,
	.unmap_sg	= pv_dma_pci_unmap_sg,
	.dma_supported	= pv_dma_pci_supported,
};

void pvdma_pci_enable(void)
{
	/* wrap dma_ops */
	inner_ops = &pvdma_direct_ops;

	/*
	 * arm does not have the same dma_ops global variable as x86. The assignment
	 * of arm to dev->dma_ops is done by calling arch_setup_dma_ops().
	 */
#ifdef CONFIG_X86_64
	dma_ops = &pv_dma_pci_ops;
#endif
}

void pvdma_pci_disable(void)
{
#ifdef CONFIG_X86_64
	/* unwrap dma_ops */
	/*
	 * Kernel 5.10 bypasses indirect calls for dma-direct, which causes the current
	 * global variable dma_ops to be NULL. For details, refer to commit 356da6d0.
	 */
	dma_ops = NULL;
#endif
}
