// SPDX-License-Identifier: GPL-2.0
/*
 * YCC: User space drivers for Alibaba YCC (Yitian Crypto Complex) crypto
 *   accelerator. Enables the on-chip crypto accelerator of Alibaba
 *   Yitian710 SoCs.
 *
 * Copyright (C) 2020-2022 Alibaba Corporation. All rights reserved.
 * Author: Zelin Deng <zelin.deng@linux.alibaba.com>
 * Author: Guanjun <guanjun@linux.alibaba.com>
 * Author: Jiayu Ni <jiayu.ni@linux.alibaba.com>
 */

#define _GNU_SOURCE
#include <sys/sysinfo.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <libudev.h>
#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <dirent.h>
#include <signal.h>
#include <fcntl.h>
#include <stdio.h>
#include <poll.h>

#include "../utils/utils.h"
#include "udma_ulib.h"
#include "ycc_uio.h"
#include "pke.h"
#include "aead.h"
#include "ske.h"
#include "rng.h"
#include "kpp.h"

#define YCC_UIO_CLASS_PATH	"/sys/class/uio"

#define YCC_UIO_MAP_ALIGNMENT_SIZE	0x1000

static struct ycc_list_head head;
static struct ycc_list *pre;

static bool mon_polling = true;
static bool rings_polling = true;

/* Default polling interval is 100ms */
static unsigned int interval = 100000;
static pthread_t mon_thread;
static pthread_t polling_thread;
static struct udev_monitor *mon;

static int ycc_udev_create_device(struct udev_device **dev, char *path)
{
	struct udev *udev;

	if (!path)
		return -1;

	udev = udev_new();
	if (!udev)
		return -1;

	*dev = udev_device_new_from_syspath(udev, path);
	if (!*dev) {
		udev_unref(udev);
		return -1;
	}

	return 0;
}

static void ycc_udev_free_device(struct udev_device *dev)
{
	struct udev *udev = udev_device_get_udev(dev);

	udev_device_unref(dev);
	udev_unref(udev);
}

static int ycc_udev_read_attr_raw(struct udev_device *dev,
				  char *value,
				  uint32_t size,
				  const char *format,
				  va_list args)
{
	const char *attr_value = NULL;
	char attr[128] = {0};

	vsnprintf(attr, 128, format, args);
	attr_value = udev_device_get_sysattr_value(dev, attr);
	if (!attr_value)
		return -1;

	snprintf(value, size, "%s", attr_value);

	return 0;
}

static int ycc_udev_read_attr_str(struct udev_device *dev,
				  char *value,
				  uint32_t size,
				  const char *format,
				  ...)
{
	va_list args;
	int ret;

	va_start(args, format);
	ret = ycc_udev_read_attr_raw(dev, value, size, format, args);
	va_end(args);

	return ret;
}

static int ycc_udev_read_attr_ulong(struct udev_device *dev,
				    unsigned long *value,
				    const char *format,
				    ...)
{
	char attr_value[32] = {0};
	va_list args;
	int ret;

	va_start(args, format);
	ret = ycc_udev_read_attr_raw(dev, attr_value, 32, format, args);
	va_end(args);

	if (!ret)
		*value = strtoul(attr_value, NULL, 0);

	return ret;
}

static int ycc_udev_read_map_attr(struct udev_device *dev, struct ycc_uio_map *umap)
{
	int ret;

	ret = ycc_udev_read_attr_str(dev, umap->name, YCC_UIO_MAP_NAME_LEN, "maps/map0/name");
	if (ret)
		return -1;

	ret = ycc_udev_read_attr_ulong(dev, &umap->addr, "maps/map0/addr");
	if (ret)
		return -1;

	ret = ycc_udev_read_attr_ulong(dev, &umap->offset, "maps/map0/offset");
	if (ret)
		return -1;

	ret = ycc_udev_read_attr_ulong(dev, &umap->size, "maps/map0/size");
	if (ret)
		return -1;

	return ret;
}

static int ycc_uio_ring_init(struct ycc_list *elem, int dev_id, int ring_id)
{
	struct ycc_ring *ring;
	char path[32];
	int ret;

	if (!elem)
		return -1;

	ring = malloc(sizeof(struct ycc_ring));
	if (!ring)
		return -1;

	memset(ring, 0, sizeof(struct ycc_ring));

	snprintf(path, 32, "/dev/%s", elem->umap.uio_name);
	elem->umap.ufd = open(path, O_RDWR);
	if (elem->umap.ufd < 0)
		goto free_ring;

	ring->csr_addr = mmap(NULL,
			      ALIGN(elem->umap.size, YCC_UIO_MAP_ALIGNMENT_SIZE),
			      PROT_READ | PROT_WRITE,
			      MAP_SHARED,
			      elem->umap.ufd,
			      0);
	if (ring->csr_addr == MAP_FAILED)
		goto close_fd;

	ring->csr_addr += elem->umap.offset;
	ring->csr_paddr = elem->umap.addr + elem->umap.offset;

	ring->ring_id = ring_id;
	ring->dev_id = dev_id;

	ret = ycc_init_ring(ring, YCC_RING_DEFAULT_DESC);
	if (ret)
		goto munmap_ring_addr;

	elem->ring = ring;
	return 0;

munmap_ring_addr:
	munmap(ring->csr_addr - elem->umap.offset,
	       ALIGN(elem->umap.size, YCC_UIO_MAP_ALIGNMENT_SIZE));
close_fd:
	close(elem->umap.ufd);
free_ring:
	free(ring);
	return -1;
}

static void ycc_uio_ring_release(struct ycc_list *elem)
{
	if (!elem || !elem->ring)
		return;

	munmap(elem->ring->csr_addr - elem->umap.offset,
	       ALIGN(elem->umap.size, YCC_UIO_MAP_ALIGNMENT_SIZE));
	ycc_release_ring(elem->ring);
	close(elem->umap.ufd);
	free(elem->ring);
	elem->ring = NULL;
}

static int ycc_uio_elem_init(struct ycc_list *elem, struct dirent *dent)
{
	struct udev_device *udevice;
	char path[128] = {0};
	int ret;

	sprintf(path, "/sys/class/uio/%s", dent->d_name);
	ret = ycc_udev_create_device(&udevice, path);
	if (ret < 0)
		return ret;

	memset(elem, 0, sizeof(struct ycc_list));
	snprintf(elem->umap.uio_name, YCC_UIO_NAME_LEN, "%s", dent->d_name);

	ret = ycc_udev_read_map_attr(udevice, &elem->umap);
	/*
	 * YCC uio attributes have been established, udevice
	 * can free now.
	 */
	ycc_udev_free_device(udevice);
	return ret;
}

static int ycc_uio_rings_enum(int count)
{
	struct dirent *dent;
	struct ycc_list *elem;
	unsigned char dev[8] = {0};
	unsigned char ring[8] = {0};
	int ret = 0, i = 0;
	DIR *dir;

	YCC_LIST_INIT(&head);

	dir = opendir(YCC_UIO_CLASS_PATH);
	if (!dir)
		return -1;

	while ((dent = readdir(dir)) && i < count) {
		if (dent->d_name[0] == '.')
			continue;

		elem = malloc(sizeof(struct ycc_list));
		if (!elem)
			break;

		ret = ycc_uio_elem_init(elem, dent);
		if (ret < 0) {
			free(elem);
			break;
		}

		/*
		 * The format of elem->umap.name is "YCC_DEV_%02d_RING_%02d_MAP",
		 * which is defined in kernel space. Therefore device id is at
		 * the offset 8, ring id is at the offset 16. Both size of them
		 * is 2 bytes.
		 */
		strncpy(dev, elem->umap.name + 8, 2);
		strncpy(ring, elem->umap.name + 16, 2);

		ret = ycc_uio_ring_init(elem, atoi(dev), atoi(ring));
		if (ret < 0) {
			free(elem);
			continue;
		}

		YCC_LIST_ADD(elem, &head);
		i++;
	}

	closedir(dir);

	/*
	 * YCC userspace driver works well if only one ring has been added
	 * to YCC_LIST. So just check 'i' value to make sure if there are
	 * any rings have been successfully initialized.
	 */
	return i ? 0 : -1;
}

static void ycc_uio_rings_free(void)
{
	struct ycc_list *elem, *temp;

	YCC_LIST_FOREACH_SAFE(elem, temp, &head) {
		ycc_uio_ring_release(elem);
		YCC_LIST_REMOVE(elem);
		free(elem);
	}
}

static void ycc_udev_free_monitor(void)
{
	struct udev *udev;

	if (mon) {
		udev = udev_monitor_get_udev(mon);
		udev_monitor_unref(mon);
		udev_unref(udev);
	}
}

static int ycc_udev_create_monitor(void)
{
	struct udev *udev;

	assert(!mon);

	udev = udev_new();
	if (!udev)
		return -1;

	mon = udev_monitor_new_from_netlink(udev, "udev");
	if (!mon) {
		udev_unref(udev);
		return -1;
	}

	if (udev_monitor_filter_add_match_subsystem_devtype(mon, "pci", NULL))
		goto free_monitor;

	if (udev_monitor_enable_receiving(mon))
		goto free_monitor;

	return 0;

free_monitor:
	ycc_udev_free_monitor();
	return -1;
}

/*
 * Polling YCC status and device id
 * TODO: YCC status and device id will be used for error
 *       handling in the future.
 */
static int ycc_udev_poll_event(struct udev_monitor *mon)
{
	struct pollfd ycc_event_fd = {0, POLLIN, 0};
	struct udev_device *dev;
	const char *ycc_status;
	const char *dev_id;

	if (!mon)
		return -1;

	ycc_event_fd.fd = udev_monitor_get_fd(mon);
	if (ycc_event_fd.fd <= 0)
		return -1;

	if (poll(&ycc_event_fd, 1, 0) > 0) {
		dev = udev_monitor_receive_device(mon);
		if (dev) {
			ycc_status = udev_device_get_property_value(dev, "YCC_STATUS");
			if (!ycc_status)
				goto unref_udevice;

			dev_id = udev_device_get_property_value(dev, "YCC_DEVID");
			if (!dev_id)
				goto unref_udevice;

			ycc_info("YCC uevent, status: %s, id: %s\n", ycc_status, dev_id);
		}
	}

	return 0;

unref_udevice:
	udev_device_unref(dev);
	return -1;
}

static void *ycc_mon_thread(void *arg)
{
	struct udev_monitor *mon = (struct udev_monitor *)arg;

	while (mon_polling) {
		ycc_udev_poll_event(mon);
		usleep(interval);
	}

	pthread_exit(NULL);
}

static void *ycc_rings_polling(void *arg)
{
	struct ycc_list *elem;
	struct ycc_ring *ring = NULL;

	while (rings_polling) {
		YCC_LIST_FOREACH(elem, &head) {
			ring = elem->ring;
			if (atomic_read(&ring->refcnt))
				ycc_dequeue(ring);
		}

		/*
		 * TODO: Sleeping here is provisional. Change to
		 * epoll in the future.
		 */
		usleep(interval);
	}

	pthread_exit(NULL);
}

/*
 * Set CPU affinity for polling thread
 */
int ycc_set_polling_affinity(cpu_set_t *cpuset)
{
	return pthread_setaffinity_np(polling_thread, sizeof(cpu_set_t), cpuset);
}

/*
 * Get CPU affinity for polling thread
 */
int ycc_get_polling_affinity(cpu_set_t *cpuset)
{
	return pthread_getaffinity_np(polling_thread, sizeof(cpu_set_t), cpuset);
}

int ycc_set_polling_interval(unsigned int usec)
{
	int ret;

	ret = pthread_kill(polling_thread, 0);
	if (ret) /* polling thread is dead */
		return ret;

	interval = usec;
	return 0;
}

unsigned int ycc_get_polling_interval(void)
{
	return interval;
}

struct ycc_ring *ycc_select_ring(void)
{
	struct ycc_list *elem, *first = NULL;
	struct ycc_ring *ring = NULL;

	YCC_LIST_FOREACH(elem, &head) {
		if (!pre) {
			pre = elem;
			ring = elem->ring;
			break;
		}

		if (!first)
			first = elem;

		if (atomic_read(&elem->ring->refcnt) <
		    atomic_read(&pre->ring->refcnt)) {
			pre = elem;
			ring = elem->ring;
			break;
		}

		pre = elem;
	}

	if (!elem && first) {
		pre = first;
		ring = first->ring;
	}

	atomic_inc(&ring->refcnt);
	return ring;
}

int ycc_drv_init(int count)
{
	int ret;

	ret = ycc_udma_init();
	if (ret < 0)
		return ret;

	ret = ycc_uio_rings_enum(count);
	if (ret < 0)
		goto udma_exit;

	ret = ycc_udev_create_monitor();
	if (ret < 0)
		goto free_uio_rings;

	ret = pthread_create(&mon_thread, NULL, ycc_mon_thread, mon);
	if (ret < 0)
		goto free_monitor;

	/* create polling thread per process */
	ret = pthread_create(&polling_thread, NULL, ycc_rings_polling, NULL);
	if (ret)
		goto free_monitor;

	akcipher_register_algs();
	skcipher_register_algs();
	aead_register_algs();
	rng_register_algs();
	kpp_register_algs();

	return 0;

free_monitor:
	ycc_udev_free_monitor();
free_uio_rings:
	ycc_uio_rings_free();
udma_exit:
	ycc_udma_exit();
	return ret;
}

void ycc_drv_exit(void)
{
	mon_polling = false;
	rings_polling = false;
	pthread_join(mon_thread, NULL);
	pthread_join(polling_thread, NULL);

	kpp_unregister_algs();
	rng_unregister_algs();
	aead_unregister_algs();
	skcipher_unregister_algs();
	akcipher_unregister_algs();

	ycc_udev_free_monitor();
	ycc_uio_rings_free();
	ycc_udma_exit();
}
