// SPDX-License-Identifier: GPL-2.0
#ifndef __YCC_ALG_H
#define __YCC_ALG_H

#include <stdio.h>

#include "ycc_ring.h"

#define MAX_DIGEST_SIZE			64

#define MEM_ALIGNMENT_64		64

#define YCC_RSA_KEY_SZ_512		64
#define YCC_RSA_KEY_SZ_1536		192
#define YCC_RSA_CRT_PARAMS		5
#define YCC_RSA_E_SZ_MAX		8
#define YCC_CMD_DATA_ALIGN_SZ		64
#define YCC_PIN_SZ			16

#define YCC_EC_DIGITS_P192		24
#define YCC_EC_DIGITS_P224		28
#define YCC_EC_DIGITS_P256		32
#define YCC_EC_DIGITS_P384		48
#define YCC_EC_DIGITS_P521		66
#define YCC_EC_DIGITS_MAX		66

#define DES_KEY_SIZE		8
#define DES_EXPKEY_WORDS	32
#define DES_BLOCK_SIZE		8

#define DES3_EDE_KEY_SIZE	(3 * DES_KEY_SIZE)
#define DES3_EDE_EXPKEY_WORDS	(3 * DES_EXPKEY_WORDS)
#define DES3_EDE_BLOCK_SIZE	DES_BLOCK_SIZE

#define AES_MIN_KEY_SIZE	16
#define AES_MAX_KEY_SIZE	32
#define AES_KEYSIZE_128		16
#define AES_KEYSIZE_192		24
#define AES_KEYSIZE_256		32
#define AES_BLOCK_SIZE		16
#define AES_MAX_KEYLENGTH	(15 * 16)
#define AES_MAX_KEYLENGTH_U32	(AES_MAX_KEYLENGTH / sizeof(uint32_t))

#define SM4_KEY_SIZE	16
#define SM4_BLOCK_SIZE	16
#define SM4_RKEY_WORDS	32

enum ycc_cmd_id {
	YCC_CMD_RNG_GEN = 0x03,

	YCC_CMD_GEN_CMAC = 0x21,
	YCC_CMD_VERIFY_CMAC,
	YCC_CMD_SKE_ENC,
	YCC_CMD_SKE_DEC,
	YCC_CMD_GCM_ENC,
	YCC_CMD_GCM_DEC,
	YCC_CMD_CCM_ENC,
	YCC_CMD_CCM_DEC, /* 0x28 */

	YCC_CMD_KPP_GEN_KEY = 0x61,
	YCC_CMD_KPP_GEN_KEY_PAIR,

	YCC_CMD_ECDSA_SIGN = 0x63,
	YCC_CMD_ECDSA_VERIFY,

	YCC_CMD_RSA_ENC = 0x83,
	YCC_CMD_RSA_DEC,
	YCC_CMD_RSA_CRT_DEC,
	YCC_CMD_RSA_CRT_SIGN,
	YCC_CMD_RSA_SIGN,
	YCC_CMD_RSA_VERIFY,

	YCC_CMD_KPP_ECDH_SS = 0xc1, /* ecdh shared secret */
};

/* Elliptic Curve id */
enum ycc_ec_id {
	YCC_EC_P192 = 0,
	YCC_EC_P224,
	YCC_EC_P256,
	YCC_EC_P384,
	YCC_EC_P521,
};

enum ycc_gcm_mode {
	YCC_AES_128_GCM = 0,
	YCC_AES_192_GCM,
	YCC_AES_256_GCM,
	YCC_SM4_GCM,
};

enum ycc_ccm_mode {
	YCC_AES_128_CCM = 0,
	YCC_AES_192_CCM,
	YCC_AES_256_CCM,
	YCC_SM4_CCM,
};

enum ycc_ske_alg_mode {
	YCC_DES_ECB = 26,
	YCC_DES_CBC,
	YCC_DES_CFB,
	YCC_DES_OFB,
	YCC_DES_CTR,
	YCC_TDES_128_ECB,
	YCC_TDES_128_CBC,
	YCC_TDES_128_CFB,
	YCC_TDES_128_OFB,
	YCC_TDES_128_CTR,
	YCC_TDES_192_ECB,
	YCC_TDES_192_CBC,
	YCC_TDES_192_CFB,
	YCC_TDES_192_OFB,
	YCC_TDES_192_CTR, /* 40 */

	YCC_AES_128_ECB,
	YCC_AES_128_CBC,
	YCC_AES_128_CFB,
	YCC_AES_128_OFB,
	YCC_AES_128_CTR,
	YCC_AES_128_XTS, /* 46 */

	YCC_AES_192_ECB = 48,
	YCC_AES_192_CBC,
	YCC_AES_192_CFB,
	YCC_AES_192_OFB,
	YCC_AES_192_CTR, /* 52 */

	YCC_AES_256_ECB = 55,
	YCC_AES_256_CBC,
	YCC_AES_256_CFB,
	YCC_AES_256_OFB,
	YCC_AES_256_CTR,
	YCC_AES_256_XTS, /* 60 */

	YCC_SM4_ECB = 62,
	YCC_SM4_CBC,
	YCC_SM4_CFB,
	YCC_SM4_OFB,
	YCC_SM4_CTR,
	YCC_SM4_XTS, /* 67 */
};

enum {
	CMAC_AES_SET_KEY = 252,
	CMAC_SM4_SET_KEY,
	CBC_MAC_AES_SET_KEY,
	CBC_MAC_SM4_SET_KEY,
};

struct ycc_crypto_ctx {
	uint8_t mode;
	uint8_t *cipher_key;
	uint32_t keysize;
	uint8_t key_type; /* Reserved for kek */
	uint32_t key_dma_size; /* Dma memory size for key/key+iv */
	struct ycc_ring *ring;
};

struct ycc_crypto_req {
	void *key_vaddr;
	dma_addr_t key_paddr;
	struct ycc_cmd_desc desc;
	union {
		struct skcipher_req *ske_req;
		struct aead_req *aead_req;
	};
	void *src_vaddr;
	dma_addr_t src_paddr;
	void *dst_vaddr;
	dma_addr_t dst_paddr;
	int in_len;
	int out_len;
	int aad_offset;
	struct ycc_crypto_ctx *ctx;
	uint8_t last_block[16]; /* Store decrypt iv out */
};

struct ycc_rsa_ctx {
	struct rsa_key *rsa_key;
	void *priv_key_vaddr;
	dma_addr_t priv_key_paddr;
	void *pub_key_vaddr;
	dma_addr_t pub_key_paddr;
	unsigned int key_len;
	unsigned int e_len;
	bool crt_mode;
	bool is_asn1;
	struct ycc_ring *ring;
};

enum ycc_pke_type {
	YCC_PKE_ENC = 0,
	YCC_PKE_DEC,
	YCC_PKE_VERIFY,
	YCC_PKE_SIGN,
};

struct ycc_ecdsa_ctx {
	uint32_t curve_id;
	uint32_t ndigits;
	void *priv_key_vaddr;
	dma_addr_t priv_key_paddr;
	void *pub_key_vaddr;
	dma_addr_t pub_key_paddr;
	unsigned int key_len;
	struct ycc_ring *ring;
};

struct ycc_pke_req {
	void *src_vaddr;
	dma_addr_t src_paddr;
	void *dst_vaddr;
	dma_addr_t dst_paddr;
	struct ycc_cmd_desc desc;
	union {
		struct ycc_rsa_ctx *rsa_ctx;
		struct ycc_ecdsa_ctx *ecdsa_ctx;
	} ctx;
	struct akcipher_req *req;

	enum ycc_pke_type type; /* Used for cb,indicate enc, dec, verify or sign */
};

struct ycc_rng_ctx {
	void *src_vaddr;
	uint32_t src_len;
	void *dst_vaddr;
	uint32_t dst_len;

	struct ycc_cmd_desc desc;
	struct ycc_ring *ring;
	struct rng_ctx *ctx;
};

struct ycc_kpp_ctx {
	uint32_t curve_id;
	uint32_t ndigits;
	void *private_key_vaddr;
	dma_addr_t private_key_paddr;

	struct kpp_ctx *ctx;
	struct ycc_ring *ring;
};

struct ycc_kpp_req {
	void *src_vaddr;
	dma_addr_t src_paddr;
	void *dst_vaddr;
	dma_addr_t dst_paddr;

	struct ycc_cmd_desc desc;
	struct ycc_kpp_ctx *ctx;
	struct kpp_req *req;
};

int ycc_pke_register(void);
void ycc_pke_unregister(void);
int ycc_sym_register(void);
void ycc_sym_unregister(void);
int ycc_kpp_register(void);
void ycc_kpp_unregister(void);

#endif
