// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2020-2022 Alibaba Corporation. All rights reserved.
 * Author: Zelin Deng <zelin.deng@linux.alibaba.com>
 * Author: Guanjun <guanjun@linux.alibaba.com>
 * Author: Jiayu Ni <jiayu.ni@linux.alibaba.com>
 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "ycc_uio.h"
#include "utils.h"
#include "pke.h"

struct akcipher_test {
	const char *name;
	const char *key;
	const char *priv;
	const char *msg;
	const char *sig;
	unsigned int key_len;
	unsigned int priv_len;
	unsigned int msg_len;
	unsigned int sig_len;
};

static const struct akcipher_test ecdsa = {
	.name = "ecdsa-nist-p256",
	.key = "\xbf\x30\xd5\x26\xac\xa2\xb8\xec"
	       "\xbb\x4b\x28\xef\x13\xfe\xe4\xfd"
	       "\xc7\x9d\x3c\xda\xe5\x1a\xd8\xc4"
	       "\x15\xfc\xc3\x65\xce\xdd\x8a\xa8"
	       "\x9e\x24\x54\xe7\x6c\x52\x21\xe7"
	       "\x7c\x49\xdd\xab\x13\x0d\xa3\xb8"
	       "\x4c\xf4\x5b\x1f\xc1\xd8\x52\x99"
	       "\x2e\xf3\xce\xa3\x79\xb8\x77\x4d",
	.key_len = 64,
	.msg = "\xcd\xb9\xd2\x1c\xb7\x6f\xcd\x44"
	       "\xb3\xfd\x63\xea\xa3\x66\x7f\xae"
	       "\x63\x85\xe7\x82",
	.msg_len = 20,
	.priv = "\x89\x83\xde\xf5\xca\xd5\x0e\x21"
		"\xe6\x88\xb2\x7b\x0d\x93\x2e\x72"
		"\x5e\x1d\xc2\x41\xf7\x48\x0f\x46"
		"\x1a\x24\xe1\xb8\x98\x29\x7a\x69",
	.priv_len = 32,
};

static bool inited;

/*
 * Destructor: It is must if driver initialize successfully.
 */
static void __attribute__((destructor)) drv_exit(void)
{
	if (inited)
		ycc_drv_exit();
}

int main(int argc, char *argv[])
{
	struct akcipher_ctx *cipher;
	struct akcipher_req *req;
	unsigned char src[64];
	unsigned char dst[64];
	unsigned char pub[64];
	unsigned char priv[64];
	int ret;

	ret = ycc_drv_init(1);
	if (ret < 0)
		return ret;

	inited = true;
	ret = -1;
	cipher = akcipher_alloc_ctx("ecdsa-nist-p256", CRYPTO_SYNC);
	if (!cipher)
		goto out;

	req = akcipher_alloc_req(cipher);
	if (!req)
		goto free_ctx;

	memcpy(priv, ecdsa.priv, ecdsa.priv_len);
	ret = akcipher_set_priv_key(cipher, priv, ecdsa.priv_len);
	if (ret < 0)
		goto free_req;

	memcpy(src, ecdsa.msg, ecdsa.msg_len);
	akcipher_set_req(req, src, dst, ecdsa.msg_len, ecdsa.key_len, 0);

	ret = akcipher_sign(req);
	if (!ret)
		hex_dump("Dump ecdsa-nist-p256 dst:", dst, ecdsa.key_len);

	memcpy(pub, ecdsa.key, ecdsa.key_len);
	ret = akcipher_set_pub_key(cipher, pub, ecdsa.key_len);
	if (ret < 0)
		goto free_req;

	ret = akcipher_verify(req);
	if (!ret)
		printf("Sample code ecdsa-nist-p256 passed\n");

free_req:
	akcipher_free_req(req);
free_ctx:
	akcipher_free_ctx(cipher);
out:
	return ret;
}
