// SPDX-License-Identifier: GPL-2.0
#ifndef __UTILS_H
#define __UTILS_H

#include <semaphore.h>
#include <stdio.h>
#include <stdint.h>
#include <stdbool.h>
#include <errno.h>
#include <time.h>

#ifndef atomic_t
typedef uint32_t atomic_t;
#endif

#ifndef unlikely
#define unlikely(x)	__builtin_expect((x), 0)
#endif

#define CRYPTO_ASYNC			0x0
#define CRYPTO_SYNC			0x2

#define COMPLETION_TIMEOUT_SECS		5

#ifdef memcpy_sve
#define YCC_memcpy(dst, src, size) memcpy_sve(dst, src, size)
#else
#define YCC_memcpy(dst, src, size) memcpy(dst, src, size)
#endif

enum ycc_loglevel_t {
	YCC_TRACE,
	YCC_DEBUG,
	YCC_INFO,
	YCC_WARN,
	YCC_ERROR,
	YCC_FATAL,
};

/*
 * Print messages into stream stderr. 'level' here is for
 * expanding easily and is not used temporarily.
 */
#define YCC_LOG(level, file, line, fmt, ...)	\
	fprintf(stderr, "[%s, %d]:"fmt, file, line, ##__VA_ARGS__)

#define ycc_trace(fmt, ...)    YCC_LOG(YCC_TRACE, __FILE__, __LINE__, fmt, ##__VA_ARGS__)
#define ycc_dbg(fmt, ...)      YCC_LOG(YCC_DEBUG, __FILE__, __LINE__, fmt, ##__VA_ARGS__)
#define ycc_info(fmt, ...)     YCC_LOG(YCC_INFO, __FILE__, __LINE__, fmt, ##__VA_ARGS__)
#define ycc_warn(fmt, ...)     YCC_LOG(YCC_WARN, __FILE__, __LINE__, fmt, ##__VA_ARGS__)
#define ycc_err(fmt, ...)      YCC_LOG(YCC_ERROR, __FILE__, __LINE__, fmt, ##__VA_ARGS__)
#define ycc_fatal(fmt, ...)    YCC_LOG(YCC_FATAL, __FILE__, __LINE__, fmt, ##__VA_ARGS__)

#define ALIGN(x, n)	(((x) + (n) - 1) & (~((n) - 1)))

#define max(a, b)			\
	({__typeof__(a) _a = (a);	\
	  __typeof__(b) _b = (b);	\
	  _a > _b ? _a : _b; })

#define MAX_ERRNO	4095
#define IS_ERR_VALUE(x)	unlikely((unsigned long)(void *)(x) >= (unsigned long)-MAX_ERRNO)

static inline void *ERR_PTR(long err)
{
	return (void *)err;
}

static inline long PTR_ERR(const void *ptr)
{
	return (long)ptr;
}

static inline bool IS_ERR_OR_NULL(const void *ptr)
{
	return unlikely(!ptr) || IS_ERR_VALUE(ptr);
}

static inline void atomic_set(atomic_t *v, int i)
{
	__sync_lock_test_and_set(v, i);
}

static inline int atomic_read(atomic_t *v)
{
	return __sync_add_and_fetch(v, 0);
}

static inline int atomic_inc(atomic_t *v)
{
	return __sync_add_and_fetch(v, 1);
}

static inline bool atomic_inc_and_test(atomic_t *v)
{
	return __sync_add_and_fetch(v, 1) == 0;
}

static inline int atomic_dec(atomic_t *v)
{
	return __sync_add_and_fetch(v, -1);
}

static inline bool atomic_dec_and_test(atomic_t *v)
{
	return __sync_add_and_fetch(v, -1) == 0;
}

static inline void hex_dump(const char *name, uint8_t *ptr, uint32_t len)
{
	int i;

	printf("%s\n", name);
	for (i = 0; i < len; i++) {
		printf("%02x ", ptr[i]);
		if (i % 8 == 7)
			printf("\n");
	}
	printf("\n");
}

struct completion {
	sem_t sem;
};

static inline int init_completion(struct completion *completion)
{
	return sem_init(&completion->sem, 0, 0);
}

static inline int wait_for_completion(struct completion *completion)
{
	return sem_wait(&completion->sem);
}

static inline int wait_for_completion_timeout(struct completion *completion,
		unsigned int timeout)
{
	struct timespec ts;
	int ret;

	if (clock_gettime(CLOCK_REALTIME, &ts) == -1)
		return -1;

	ts.tv_sec += timeout;

	/*
	 * sem_timedwait() could be interrupted by a signal handler,
	 * in this case we need to catch the EINTR and restart the wait.
	 */
	while ((ret = sem_timedwait(&completion->sem, &ts)) == -1 && errno == EINTR)
		continue;

	return ret;
}

static inline int complete(struct completion *completion)
{
	return sem_post(&completion->sem);
}

static inline int destroy_completion(struct completion *completion)
{
	return sem_destroy(&completion->sem);
}

#endif
