#package_name: dbms_profiler
#author: heyongyi.hyy

CREATE OR REPLACE PACKAGE BODY DBMS_PROFILER
AS

  RUN_STATUS VARCHAR2(64) := NULL;

  USER_SCHEMA VARCHAR2(128) := NULL;

  PROFILER_PAUSED CONSTANT BINARY_INTEGER := 0;
  PROFILER_RESUMED CONSTANT BINARY_INTEGER := 1;


  FUNCTION INTERNAL_VERSION_CHECK_INNER(major IN BINARY_INTEGER,
                                        minor IN BINARY_INTEGER)
  RETURN BINARY_INTEGER;
  PRAGMA INTERFACE(c, DBMS_PROFILER_CHECK_VERSION);


  FUNCTION INTERNAL_VERSION_CHECK RETURN BINARY_INTEGER
  IS
  BEGIN
    RETURN INTERNAL_VERSION_CHECK_INNER(major_version, minor_version);
  END INTERNAL_VERSION_CHECK;



  FUNCTION START_PROFILER_INNER_C(RUN_COMMENT IN VARCHAR2,
                                  RUN_COMMENT1 IN VARCHAR2,
                                  RUN_NUMBER OUT BINARY_INTEGER)
  RETURN BINARY_INTEGER;
  PRAGMA INTERFACE(c, DBMS_PROFILER_START_PROFILER);



  FUNCTION START_PROFILER_INNER(RUN_COMMENT IN VARCHAR2,
                                RUN_COMMENT1 IN VARCHAR2,
                                RUN_NUMBER OUT BINARY_INTEGER) RETURN BINARY_INTEGER
  IS
  local_runid BINARY_INTEGER;
  ret BINARY_INTEGER;
  BEGIN
    IF (RUN_STATUS IS NULL) THEN
      ret := START_PROFILER_INNER_C(run_comment, run_comment1, local_runid);
      IF (ret = SUCCESS) THEN
        RUN_NUMBER := local_runid;
        RUN_STATUS := local_runid || '@' || PROFILER_RESUMED;
      END IF;
      RETURN ret;
    ELSE
      RETURN SUCCESS;
    END IF;
  END START_PROFILER_INNER;



  FUNCTION START_PROFILER(RUN_COMMENT IN VARCHAR2 := SYSDATE, RUN_COMMENT1 IN VARCHAR2 := '', RUN_NUMBER OUT BINARY_INTEGER) RETURN BINARY_INTEGER
  IS
  BEGIN
    IF (INTERNAL_VERSION_CHECK = 0) THEN
      RETURN START_PROFILER_INNER(RUN_COMMENT, RUN_COMMENT1, RUN_NUMBER);
    ELSE
      RETURN ERROR_VERSION;
    END IF;
  EXCEPTION WHEN OTHERS THEN
      RETURN ERROR_IO;
  END START_PROFILER;


  FUNCTION START_PROFILER(RUN_COMMENT IN VARCHAR2 := SYSDATE, RUN_COMMENT1 IN VARCHAR2 := '') RETURN BINARY_INTEGER
  IS
    RUN_NUM BINARY_INTEGER;
  BEGIN
    RETURN START_PROFILER(RUN_COMMENT, RUN_COMMENT1, RUN_NUM);
  END;


  PROCEDURE START_PROFILER(RUN_COMMENT IN VARCHAR2 := SYSDATE, RUN_COMMENT1 IN VARCHAR2 := '', RUN_NUMBER OUT BINARY_INTEGER)
  IS
  BEGIN
    IF (INTERNAL_VERSION_CHECK = 0) THEN
      BEGIN
        IF (START_PROFILER_INNER(RUN_COMMENT, RUN_COMMENT1, RUN_NUMBER) != SUCCESS) THEN
          RAISE PROFILER_ERROR;
        END IF;

        EXCEPTION WHEN OTHERS THEN
            RAISE PROFILER_ERROR;
      END;
    ELSE
      RAISE VERSION_MISMATCH;
    END IF;

  END START_PROFILER;



  PROCEDURE START_PROFILER(RUN_COMMENT IN VARCHAR2 := SYSDATE, RUN_COMMENT1 IN VARCHAR2 := '') IS
    RUN_NUM BINARY_INTEGER;
  BEGIN
    START_PROFILER(RUN_COMMENT, RUN_COMMENT1, RUN_NUM);
  END START_PROFILER;



  FUNCTION STOP_PROFILER_INNER_C RETURN BINARY_INTEGER;
  PRAGMA INTERFACE(c, DBMS_PROFILER_STOP_PROFILER);



  FUNCTION STOP_PROFILER_INNER RETURN BINARY_INTEGER
  IS
  ret BINARY_INTEGER;
  BEGIN
    IF (RUN_STATUS IS NOT NULL) THEN
      PAUSE_PROFILER();
      FLUSH_DATA();
      ret := STOP_PROFILER_INNER_C;
      RUN_STATUS := NULL;
      RETURN ret;
    ELSE
      RETURN SUCCESS;
    END IF;
  END;



  FUNCTION STOP_PROFILER RETURN BINARY_INTEGER
  IS
  BEGIN
    IF (INTERNAL_VERSION_CHECK = 0) THEN
      RETURN STOP_PROFILER_INNER;
    ELSE
      RETURN ERROR_VERSION;
    END IF;
  EXCEPTION WHEN OTHERS THEN
      RETURN ERROR_IO;
  END STOP_PROFILER;
  PROCEDURE STOP_PROFILER IS
  BEGIN
    IF (INTERNAL_VERSION_CHECK = 0) THEN
      BEGIN
        IF (STOP_PROFILER_INNER != SUCCESS) THEN
          RAISE PROFILER_ERROR;
        END IF;
      EXCEPTION WHEN OTHERS THEN
          RAISE PROFILER_ERROR;
      END;
    ELSE
      RAISE VERSION_MISMATCH;
    END IF;
  END STOP_PROFILER;



  FUNCTION CHANGE_PROFILER_STATE_INNER_C(new_state IN BINARY_INTEGER)
  RETURN BINARY_INTEGER;
  PRAGMA INTERFACE(c, DBMS_PROFILER_CHANGE_STATE);


  FUNCTION CHANGE_PROFILER_STATE_INNER(new_state IN BINARY_INTEGER)
  RETURN BINARY_INTEGER IS
  ret BINARY_INTEGER := success;
  local_runid BINARY_INTEGER := -1;
  BEGIN
    IF (RUN_STATUS IS NOT NULL) THEN
      ret := CHANGE_PROFILER_STATE_INNER_C(new_state);

      IF (ret = success) THEN
        local_runid := substr(RUN_STATUS, 1, INSTR(RUN_STATUS, '@') - 1);
        CASE new_state
          WHEN profiler_paused THEN
            RUN_STATUS := local_runid || '@' || PROFILER_PAUSED;
          WHEN profiler_resumed THEN
            RUN_STATUS := local_runid || '@' || PROFILER_RESUMED;
          ELSE
            ret := error_param;
        END CASE;
      END IF;
    END IF;

    RETURN ret;
  END;



  FUNCTION PAUSE_PROFILER RETURN BINARY_INTEGER
  IS
  BEGIN
    IF (INTERNAL_VERSION_CHECK = 0) THEN
      RETURN CHANGE_PROFILER_STATE_INNER(profiler_paused);
    ELSE
      RETURN ERROR_VERSION;
    END IF;
  EXCEPTION WHEN OTHERS THEN
      RETURN ERROR_IO;
  END;



  PROCEDURE PAUSE_PROFILER
  IS
  BEGIN
    IF (INTERNAL_VERSION_CHECK = 0) THEN
      BEGIN
        IF (CHANGE_PROFILER_STATE_INNER(profiler_paused) != SUCCESS) THEN
          RAISE PROFILER_ERROR;
        END IF;
      EXCEPTION WHEN OTHERS THEN
          RAISE PROFILER_ERROR;
      END;
    ELSE
      RAISE VERSION_MISMATCH;
    END IF;

  END PAUSE_PROFILER;



  FUNCTION RESUME_PROFILER RETURN BINARY_INTEGER
  IS
  BEGIN
    IF (INTERNAL_VERSION_CHECK = 0) THEN
      RETURN CHANGE_PROFILER_STATE_INNER(profiler_resumed);
    ELSE
      RETURN ERROR_VERSION;
    END IF;
  EXCEPTION WHEN OTHERS THEN
      RETURN ERROR_IO;
  END;



  PROCEDURE RESUME_PROFILER
  IS
  BEGIN
    IF (INTERNAL_VERSION_CHECK = 0) THEN
      BEGIN
        IF (CHANGE_PROFILER_STATE_INNER(profiler_resumed) != SUCCESS) THEN
          RAISE PROFILER_ERROR;
        END IF;
      EXCEPTION WHEN OTHERS THEN
          RAISE PROFILER_ERROR;
      END;
    ELSE
      RAISE VERSION_MISMATCH;
    END IF;

  END RESUME_PROFILER;



  FUNCTION FLUSH_DATA_INNER RETURN BINARY_INTEGER;
  PRAGMA INTERFACE(c, DBMS_PROFILER_FLUSH_DATA);



  FUNCTION FLUSH_DATA RETURN BINARY_INTEGER
  IS
  BEGIN
    IF (INTERNAL_VERSION_CHECK = 0) THEN
      RETURN FLUSH_DATA_INNER;
    ELSE
      RETURN ERROR_VERSION;
    END IF;
  EXCEPTION
    WHEN OTHERS THEN RETURN ERROR_IO;
  END FLUSH_DATA;
  PROCEDURE FLUSH_DATA IS
  BEGIN
    IF (INTERNAL_VERSION_CHECK = 0) THEN
      BEGIN
        IF (FLUSH_DATA_INNER != SUCCESS) THEN
          RAISE PROFILER_ERROR;
        END IF;
      EXCEPTION
        WHEN OTHERS THEN
          RAISE PROFILER_ERROR;
      END;
    ELSE
      RAISE VERSION_MISMATCH;
    END IF;

  END FLUSH_DATA;


  PROCEDURE ROLLUP_UNIT(RUN_NUMBER IN NUMBER, UNIT IN NUMBER)
  IS
    SQL_STR VARCHAR2(1000);
  BEGIN
    SQL_STR := 'UPDATE PLSQL_PROFILER_UNITS SET TOTAL_TIME = (SELECT SUM(TOTAL_TIME) FROM PLSQL_PROFILER_DATA WHERE RUNID = ' || RUN_NUMBER || ' AND UNIT_NUMBER = ' || UNIT || ') WHERE RUNID = ' || RUN_NUMBER || ' AND UNIT_NUMBER = ' || UNIT;
    EXECUTE IMMEDIATE SQL_STR;
    COMMIT;
  END ROLLUP_UNIT;



  PROCEDURE ROLLUP_RUN(RUN_NUMBER IN NUMBER)
  IS
    SQL_STR VARCHAR2(1000);
  BEGIN
    SQL_STR := 'UPDATE PLSQL_PROFILER_UNITS P1 SET TOTAL_TIME = (SELECT SUM(TOTAL_TIME) FROM  PLSQL_PROFILER_DATA P2 WHERE RUNID= ' || RUN_NUMBER || ' AND P2.UNIT_NUMBER=P1.UNIT_NUMBER) WHERE RUNID= ' || RUN_NUMBER;
    EXECUTE IMMEDIATE SQL_STR;
    COMMIT;
  END ROLLUP_RUN;



  PROCEDURE GET_VERSION(MAJOR OUT BINARY_INTEGER, MINOR OUT BINARY_INTEGER)
  IS
  BEGIN
    MINOR := MINOR_VERSION;
    MAJOR := MAJOR_VERSION;
  END GET_VERSION;


  PROCEDURE OB_INIT_OBJECTS(FORCE_CREATE IN BOOLEAN := FALSE);
  PRAGMA INTERFACE(c, DBMS_PROFILER_INIT_OBJECTS);


  PROCEDURE OB_DROP_OBJECTS;
  PRAGMA INTERFACE(c, DBMS_PROFILER_DROP_OBJECTS);
END;

//