/*
 * QEMU SEV support
 *
 * Copyright Advanced Micro Devices 2016-2018
 *
 * Author:
 *      Brijesh Singh <brijesh.singh@amd.com>
 *
 * This work is licensed under the terms of the GNU GPL, version 2 or later.
 * See the COPYING file in the top-level directory.
 *
 */

#include "qemu/osdep.h"

#include <linux/kvm.h>
#include <linux/psp-sev.h>

#include <sys/ioctl.h>

#include "qapi/error.h"
#include "qom/object_interfaces.h"
#include "qemu/base64.h"
#include "qemu/module.h"
#include "qemu/uuid.h"
#include "qemu/error-report.h"
#include "crypto/hash.h"
#include "sysemu/kvm.h"
#include "sev.h"
#include "csv.h"
#include "sysemu/sysemu.h"
#include "sysemu/runstate.h"
#include "trace.h"
#include "migration/blocker.h"
#include "migration/qemu-file.h"
#include "migration/misc.h"
#include "qom/object.h"
#include "monitor/monitor.h"
#include "monitor/hmp-target.h"
#include "qapi/qapi-commands-misc-target.h"
#include "exec/confidential-guest-support.h"
#include "hw/i386/pc.h"
#include "exec/address-spaces.h"

#define TYPE_SEV_GUEST "sev-guest"
OBJECT_DECLARE_SIMPLE_TYPE(SevGuestState, SEV_GUEST)

struct shared_region {
    unsigned long gfn_start, gfn_end;
    QTAILQ_ENTRY(shared_region) list;
};


/**
 * SevGuestState:
 *
 * The SevGuestState object is used for creating and managing a SEV
 * guest.
 *
 * # $QEMU \
 *         -object sev-guest,id=sev0 \
 *         -machine ...,memory-encryption=sev0
 */
struct SevGuestState {
    ConfidentialGuestSupport parent_obj;

    /* configuration parameters */
    char *sev_device;
    uint32_t policy;
    char *dh_cert_file;
    char *session_file;
    uint32_t cbitpos;
    uint32_t reduced_phys_bits;
    bool kernel_hashes;
    char *user_id;

    /* runtime state */
    uint32_t handle;
    uint8_t api_major;
    uint8_t api_minor;
    uint8_t build_id;
    int sev_fd;
    SevState state;
    gchar *measurement;
    guchar *remote_pdh;
    size_t remote_pdh_len;
    guchar *remote_plat_cert;
    size_t remote_plat_cert_len;
    guchar *amd_cert;
    size_t amd_cert_len;
    gchar *send_packet_hdr;
    size_t send_packet_hdr_len;

    /* needed by live migration of HYGON CSV2 guest */
    gchar *send_vmsa_packet_hdr;
    size_t send_vmsa_packet_hdr_len;

    uint32_t reset_cs;
    uint32_t reset_ip;
    bool reset_data_valid;

    QTAILQ_HEAD(, shared_region) shared_regions_list;

    /* link list used for HYGON CSV */
    CsvBatchCmdList *csv_batch_cmd_list;
};

#define DEFAULT_GUEST_POLICY    0x1 /* disable debug */
#define DEFAULT_SEV_DEVICE      "/dev/sev"

#define SEV_INFO_BLOCK_GUID     "00f771de-1a7e-4fcb-890e-68c77e2fb44e"
typedef struct __attribute__((__packed__)) SevInfoBlock {
    /* SEV-ES Reset Vector Address */
    uint32_t reset_addr;
} SevInfoBlock;

#define SEV_HASH_TABLE_RV_GUID  "7255371f-3a3b-4b04-927b-1da6efa8d454"
typedef struct QEMU_PACKED SevHashTableDescriptor {
    /* SEV hash table area guest address */
    uint32_t base;
    /* SEV hash table area size (in bytes) */
    uint32_t size;
} SevHashTableDescriptor;

/* hard code sha256 digest size */
#define HASH_SIZE 32

typedef struct QEMU_PACKED SevHashTableEntry {
    QemuUUID guid;
    uint16_t len;
    uint8_t hash[HASH_SIZE];
} SevHashTableEntry;

typedef struct QEMU_PACKED SevHashTable {
    QemuUUID guid;
    uint16_t len;
    SevHashTableEntry cmdline;
    SevHashTableEntry initrd;
    SevHashTableEntry kernel;
} SevHashTable;

/*
 * Data encrypted by sev_encrypt_flash() must be padded to a multiple of
 * 16 bytes.
 */
typedef struct QEMU_PACKED PaddedSevHashTable {
    SevHashTable ht;
    uint8_t padding[ROUND_UP(sizeof(SevHashTable), 16) - sizeof(SevHashTable)];
} PaddedSevHashTable;

QEMU_BUILD_BUG_ON(sizeof(PaddedSevHashTable) % 16 != 0);

static SevGuestState *sev_guest;
static Error *sev_mig_blocker;

bool sev_kvm_has_msr_ghcb;

static const char *const sev_fw_errlist[] = {
    [SEV_RET_SUCCESS]                = "",
    [SEV_RET_INVALID_PLATFORM_STATE] = "Platform state is invalid",
    [SEV_RET_INVALID_GUEST_STATE]    = "Guest state is invalid",
    [SEV_RET_INAVLID_CONFIG]         = "Platform configuration is invalid",
    [SEV_RET_INVALID_LEN]            = "Buffer too small",
    [SEV_RET_ALREADY_OWNED]          = "Platform is already owned",
    [SEV_RET_INVALID_CERTIFICATE]    = "Certificate is invalid",
    [SEV_RET_POLICY_FAILURE]         = "Policy is not allowed",
    [SEV_RET_INACTIVE]               = "Guest is not active",
    [SEV_RET_INVALID_ADDRESS]        = "Invalid address",
    [SEV_RET_BAD_SIGNATURE]          = "Bad signature",
    [SEV_RET_BAD_MEASUREMENT]        = "Bad measurement",
    [SEV_RET_ASID_OWNED]             = "ASID is already owned",
    [SEV_RET_INVALID_ASID]           = "Invalid ASID",
    [SEV_RET_WBINVD_REQUIRED]        = "WBINVD is required",
    [SEV_RET_DFFLUSH_REQUIRED]       = "DF_FLUSH is required",
    [SEV_RET_INVALID_GUEST]          = "Guest handle is invalid",
    [SEV_RET_INVALID_COMMAND]        = "Invalid command",
    [SEV_RET_ACTIVE]                 = "Guest is active",
    [SEV_RET_HWSEV_RET_PLATFORM]     = "Hardware error",
    [SEV_RET_HWSEV_RET_UNSAFE]       = "Hardware unsafe",
    [SEV_RET_UNSUPPORTED]            = "Feature not supported",
    [SEV_RET_INVALID_PARAM]          = "Invalid parameter",
    [SEV_RET_RESOURCE_LIMIT]         = "Required firmware resource depleted",
    [SEV_RET_SECURE_DATA_INVALID]    = "Part-specific integrity check failure",
};

#define SEV_FW_MAX_ERROR      ARRAY_SIZE(sev_fw_errlist)

#define SEV_FW_BLOB_MAX_SIZE            0x4000          /* 16KB */

#define SHARED_REGION_LIST_CONT     0x1
#define SHARED_REGION_LIST_END      0x2

#define ENCRYPTED_CPU_STATE_CONT    0x1
#define ENCRYPTED_CPU_STATE_END     0x2

static struct ConfidentialGuestMemoryEncryptionOps sev_memory_encryption_ops = {
    .save_setup = sev_save_setup,
    .save_outgoing_page = sev_save_outgoing_page,
    .load_incoming_page = sev_load_incoming_page,
    .is_gfn_in_unshared_region = sev_is_gfn_in_unshared_region,
    .save_outgoing_shared_regions_list = sev_save_outgoing_shared_regions_list,
    .load_incoming_shared_regions_list = sev_load_incoming_shared_regions_list,
    .queue_outgoing_page = csv_queue_outgoing_page,
    .save_queued_outgoing_pages = csv_save_queued_outgoing_pages,
    .queue_incoming_page = csv_queue_incoming_page,
    .load_queued_incoming_pages = csv_load_queued_incoming_pages,
    .save_outgoing_cpu_state = csv_save_outgoing_cpu_state,
    .load_incoming_cpu_state = csv_load_incoming_cpu_state,
};

static int
sev_ioctl(int fd, int cmd, void *data, int *error)
{
    int r;
    struct kvm_sev_cmd input;

    memset(&input, 0x0, sizeof(input));

    input.id = cmd;
    input.sev_fd = fd;
    input.data = (__u64)(unsigned long)data;

    r = kvm_vm_ioctl(kvm_state, KVM_MEMORY_ENCRYPT_OP, &input);

    if (error) {
        *error = input.error;
    }

    return r;
}

static int
sev_platform_ioctl(int fd, int cmd, void *data, int *error)
{
    int r;
    struct sev_issue_cmd arg;

    arg.cmd = cmd;
    arg.data = (unsigned long)data;
    r = ioctl(fd, SEV_ISSUE_CMD, &arg);
    if (error) {
        *error = arg.error;
    }

    return r;
}

static const char *
fw_error_to_str(int code)
{
    if (code < 0 || code >= SEV_FW_MAX_ERROR) {
        return "unknown error";
    }

    return sev_fw_errlist[code];
}

static bool
sev_check_state(const SevGuestState *sev, SevState state)
{
    assert(sev);
    return sev->state == state ? true : false;
}

static void
sev_set_guest_state(SevGuestState *sev, SevState new_state)
{
    assert(new_state < SEV_STATE__MAX);
    assert(sev);

    trace_kvm_sev_change_state(SevState_str(sev->state),
                               SevState_str(new_state));
    sev->state = new_state;
}

static void
sev_ram_block_added(RAMBlockNotifier *n, void *host, size_t size,
                    size_t max_size)
{
    int r;
    struct kvm_enc_region range;
    ram_addr_t offset;
    MemoryRegion *mr;

    /*
     * The RAM device presents a memory region that should be treated
     * as IO region and should not be pinned.
     */
    mr = memory_region_from_host(host, &offset);
    if (mr && memory_region_is_ram_device(mr)) {
        return;
    }

    range.addr = (__u64)(unsigned long)host;
    range.size = max_size;

    trace_kvm_memcrypt_register_region(host, max_size);
    r = kvm_vm_ioctl(kvm_state, KVM_MEMORY_ENCRYPT_REG_REGION, &range);
    if (r) {
        error_report("%s: failed to register region (%p+%#zx) error '%s'",
                     __func__, host, max_size, strerror(errno));
        exit(1);
    }
}

static void
sev_ram_block_removed(RAMBlockNotifier *n, void *host, size_t size,
                      size_t max_size)
{
    int r;
    struct kvm_enc_region range;
    ram_addr_t offset;
    MemoryRegion *mr;

    /*
     * The RAM device presents a memory region that should be treated
     * as IO region and should not have been pinned.
     */
    mr = memory_region_from_host(host, &offset);
    if (mr && memory_region_is_ram_device(mr)) {
        return;
    }

    range.addr = (__u64)(unsigned long)host;
    range.size = max_size;

    trace_kvm_memcrypt_unregister_region(host, max_size);
    r = kvm_vm_ioctl(kvm_state, KVM_MEMORY_ENCRYPT_UNREG_REGION, &range);
    if (r) {
        error_report("%s: failed to unregister region (%p+%#zx)",
                     __func__, host, max_size);
    }
}

static struct RAMBlockNotifier sev_ram_notifier = {
    .ram_block_added = sev_ram_block_added,
    .ram_block_removed = sev_ram_block_removed,
};

static void
sev_guest_finalize(Object *obj)
{
}

static char *
sev_guest_get_session_file(Object *obj, Error **errp)
{
    SevGuestState *s = SEV_GUEST(obj);

    return s->session_file ? g_strdup(s->session_file) : NULL;
}

static void
sev_guest_set_session_file(Object *obj, const char *value, Error **errp)
{
    SevGuestState *s = SEV_GUEST(obj);

    s->session_file = g_strdup(value);
}

static char *
sev_guest_get_dh_cert_file(Object *obj, Error **errp)
{
    SevGuestState *s = SEV_GUEST(obj);

    return g_strdup(s->dh_cert_file);
}

static void
sev_guest_set_dh_cert_file(Object *obj, const char *value, Error **errp)
{
    SevGuestState *s = SEV_GUEST(obj);

    s->dh_cert_file = g_strdup(value);
}

static char *
sev_guest_get_user_id(Object *obj, Error **errp)
{
    SevGuestState *s = SEV_GUEST(obj);

    return g_strdup(s->user_id);
}

static void
sev_guest_set_user_id(Object *obj, const char *value, Error **errp)
{
    SevGuestState *s = SEV_GUEST(obj);

    s->user_id = g_strdup(value);
}

static char *
sev_guest_get_sev_device(Object *obj, Error **errp)
{
    SevGuestState *sev = SEV_GUEST(obj);

    return g_strdup(sev->sev_device);
}

static void
sev_guest_set_sev_device(Object *obj, const char *value, Error **errp)
{
    SevGuestState *sev = SEV_GUEST(obj);

    sev->sev_device = g_strdup(value);
}

static bool sev_guest_get_kernel_hashes(Object *obj, Error **errp)
{
    SevGuestState *sev = SEV_GUEST(obj);

    return sev->kernel_hashes;
}

static void sev_guest_set_kernel_hashes(Object *obj, bool value, Error **errp)
{
    SevGuestState *sev = SEV_GUEST(obj);

    sev->kernel_hashes = value;
}

static void
sev_guest_class_init(ObjectClass *oc, void *data)
{
    object_class_property_add_str(oc, "sev-device",
                                  sev_guest_get_sev_device,
                                  sev_guest_set_sev_device);
    object_class_property_set_description(oc, "sev-device",
            "SEV device to use");
    object_class_property_add_str(oc, "dh-cert-file",
                                  sev_guest_get_dh_cert_file,
                                  sev_guest_set_dh_cert_file);
    object_class_property_set_description(oc, "dh-cert-file",
            "guest owners DH certificate (encoded with base64)");
    object_class_property_add_str(oc, "session-file",
                                  sev_guest_get_session_file,
                                  sev_guest_set_session_file);
    object_class_property_set_description(oc, "session-file",
            "guest owners session parameters (encoded with base64)");
    object_class_property_add_bool(oc, "kernel-hashes",
                                   sev_guest_get_kernel_hashes,
                                   sev_guest_set_kernel_hashes);
    object_class_property_set_description(oc, "kernel-hashes",
            "add kernel hashes to guest firmware for measured Linux boot");
    object_class_property_add_str(oc, "user-id",
                                  sev_guest_get_user_id,
                                  sev_guest_set_user_id);
    object_class_property_set_description(oc, "user-id",
            "user id of the guest owner");
}

static void
sev_guest_instance_init(Object *obj)
{
    SevGuestState *sev = SEV_GUEST(obj);

    sev->sev_device = g_strdup(DEFAULT_SEV_DEVICE);
    sev->policy = DEFAULT_GUEST_POLICY;
    object_property_add_uint32_ptr(obj, "policy", &sev->policy,
                                   OBJ_PROP_FLAG_READWRITE);
    object_property_add_uint32_ptr(obj, "handle", &sev->handle,
                                   OBJ_PROP_FLAG_READWRITE);
    object_property_add_uint32_ptr(obj, "cbitpos", &sev->cbitpos,
                                   OBJ_PROP_FLAG_READWRITE);
    object_property_add_uint32_ptr(obj, "reduced-phys-bits",
                                   &sev->reduced_phys_bits,
                                   OBJ_PROP_FLAG_READWRITE);
}

/* sev guest info */
static const TypeInfo sev_guest_info = {
    .parent = TYPE_CONFIDENTIAL_GUEST_SUPPORT,
    .name = TYPE_SEV_GUEST,
    .instance_size = sizeof(SevGuestState),
    .instance_finalize = sev_guest_finalize,
    .class_init = sev_guest_class_init,
    .instance_init = sev_guest_instance_init,
    .interfaces = (InterfaceInfo[]) {
        { TYPE_USER_CREATABLE },
        { }
    }
};

bool
sev_enabled(void)
{
    return !!sev_guest;
}

bool
sev_es_enabled(void)
{
    return sev_enabled() && (sev_guest->policy & SEV_POLICY_ES);
}

uint32_t
sev_get_cbit_position(void)
{
    return sev_guest ? sev_guest->cbitpos : 0;
}

uint32_t
sev_get_reduced_phys_bits(void)
{
    return sev_guest ? sev_guest->reduced_phys_bits : 0;
}

static SevInfo *sev_get_info(void)
{
    SevInfo *info;

    info = g_new0(SevInfo, 1);
    info->enabled = sev_enabled();

    if (info->enabled) {
        info->api_major = sev_guest->api_major;
        info->api_minor = sev_guest->api_minor;
        info->build_id = sev_guest->build_id;
        info->policy = sev_guest->policy;
        info->state = sev_guest->state;
        info->handle = sev_guest->handle;
    }

    return info;
}

SevInfo *qmp_query_sev(Error **errp)
{
    SevInfo *info;

    info = sev_get_info();
    if (!info) {
        error_setg(errp, "SEV feature is not available");
        return NULL;
    }

    return info;
}

void hmp_info_sev(Monitor *mon, const QDict *qdict)
{
    SevInfo *info = sev_get_info();

    if (info && info->enabled) {
        monitor_printf(mon, "handle: %d\n", info->handle);
        monitor_printf(mon, "state: %s\n", SevState_str(info->state));
        monitor_printf(mon, "build: %d\n", info->build_id);
        monitor_printf(mon, "api version: %d.%d\n",
                       info->api_major, info->api_minor);
        monitor_printf(mon, "debug: %s\n",
                       info->policy & SEV_POLICY_NODBG ? "off" : "on");
        monitor_printf(mon, "key-sharing: %s\n",
                       info->policy & SEV_POLICY_NOKS ? "off" : "on");
    } else {
        monitor_printf(mon, "SEV is not enabled\n");
    }

    qapi_free_SevInfo(info);
}

static int
sev_get_pdh_info(int fd, guchar **pdh, size_t *pdh_len, guchar **cert_chain,
                 size_t *cert_chain_len, Error **errp)
{
    guchar *pdh_data = NULL;
    guchar *cert_chain_data = NULL;
    struct sev_user_data_pdh_cert_export export = {};
    int err, r;

    /* query the certificate length */
    r = sev_platform_ioctl(fd, SEV_PDH_CERT_EXPORT, &export, &err);
    if (r < 0) {
        if (err != SEV_RET_INVALID_LEN) {
            error_setg(errp, "SEV: Failed to export PDH cert"
                             " ret=%d fw_err=%d (%s)",
                       r, err, fw_error_to_str(err));
            return 1;
        }
    }

    pdh_data = g_new(guchar, export.pdh_cert_len);
    cert_chain_data = g_new(guchar, export.cert_chain_len);
    export.pdh_cert_address = (unsigned long)pdh_data;
    export.cert_chain_address = (unsigned long)cert_chain_data;

    r = sev_platform_ioctl(fd, SEV_PDH_CERT_EXPORT, &export, &err);
    if (r < 0) {
        error_setg(errp, "SEV: Failed to export PDH cert ret=%d fw_err=%d (%s)",
                   r, err, fw_error_to_str(err));
        goto e_free;
    }

    *pdh = pdh_data;
    *pdh_len = export.pdh_cert_len;
    *cert_chain = cert_chain_data;
    *cert_chain_len = export.cert_chain_len;
    return 0;

e_free:
    g_free(pdh_data);
    g_free(cert_chain_data);
    return 1;
}

static int sev_get_cpu0_id(int fd, guchar **id, size_t *id_len, Error **errp)
{
    guchar *id_data;
    struct sev_user_data_get_id2 get_id2 = {};
    int err, r;

    /* query the ID length */
    r = sev_platform_ioctl(fd, SEV_GET_ID2, &get_id2, &err);
    if (r < 0 && err != SEV_RET_INVALID_LEN &&
        !(is_hygon_cpu() && err == SEV_RET_INVALID_PARAM)) {
        error_setg(errp, "SEV: Failed to get ID ret=%d fw_err=%d (%s)",
                   r, err, fw_error_to_str(err));
        return 1;
    }

    id_data = g_new(guchar, get_id2.length);
    get_id2.address = (unsigned long)id_data;

    r = sev_platform_ioctl(fd, SEV_GET_ID2, &get_id2, &err);
    if (r < 0) {
        error_setg(errp, "SEV: Failed to get ID ret=%d fw_err=%d (%s)",
                   r, err, fw_error_to_str(err));
        goto err;
    }

    *id = id_data;
    *id_len = get_id2.length;
    return 0;

err:
    g_free(id_data);
    return 1;
}

static SevCapability *sev_get_capabilities(Error **errp)
{
    SevCapability *cap = NULL;
    guchar *pdh_data = NULL;
    guchar *cert_chain_data = NULL;
    guchar *cpu0_id_data = NULL;
    size_t pdh_len = 0, cert_chain_len = 0, cpu0_id_len = 0;
    uint32_t ebx;
    int fd;

    if (!kvm_enabled()) {
        error_setg(errp, "KVM not enabled");
        return NULL;
    }
    if (kvm_vm_ioctl(kvm_state, KVM_MEMORY_ENCRYPT_OP, NULL) < 0) {
        error_setg(errp, "SEV is not enabled in KVM");
        return NULL;
    }

    fd = open(DEFAULT_SEV_DEVICE, O_RDWR);
    if (fd < 0) {
        error_setg_errno(errp, errno, "SEV: Failed to open %s",
                         DEFAULT_SEV_DEVICE);
        return NULL;
    }

    if (sev_get_pdh_info(fd, &pdh_data, &pdh_len,
                         &cert_chain_data, &cert_chain_len, errp)) {
        goto out;
    }

    if (sev_get_cpu0_id(fd, &cpu0_id_data, &cpu0_id_len, errp)) {
        goto out;
    }

    cap = g_new0(SevCapability, 1);
    cap->pdh = g_base64_encode(pdh_data, pdh_len);
    cap->cert_chain = g_base64_encode(cert_chain_data, cert_chain_len);
    cap->cpu0_id = g_base64_encode(cpu0_id_data, cpu0_id_len);

    host_cpuid(0x8000001F, 0, NULL, &ebx, NULL, NULL);
    cap->cbitpos = ebx & 0x3f;

    /*
     * When SEV feature is enabled, we loose one bit in guest physical
     * addressing.
     */
    cap->reduced_phys_bits = 1;

out:
    g_free(cpu0_id_data);
    g_free(pdh_data);
    g_free(cert_chain_data);
    close(fd);
    return cap;
}

SevCapability *qmp_query_sev_capabilities(Error **errp)
{
    return sev_get_capabilities(errp);
}

static SevAttestationReport *sev_get_attestation_report(const char *mnonce,
                                                        Error **errp)
{
    struct kvm_sev_attestation_report input = {};
    SevAttestationReport *report = NULL;
    SevGuestState *sev = sev_guest;
    g_autofree guchar *data = NULL;
    g_autofree guchar *buf = NULL;
    gsize len;
    int err = 0, ret;

    if (!sev_enabled()) {
        error_setg(errp, "SEV is not enabled");
        return NULL;
    }

    /* lets decode the mnonce string */
    buf = g_base64_decode(mnonce, &len);
    if (!buf) {
        error_setg(errp, "SEV: failed to decode mnonce input");
        return NULL;
    }

    /* verify the input mnonce length */
    if (len != sizeof(input.mnonce)) {
        error_setg(errp, "SEV: mnonce must be %zu bytes (got %" G_GSIZE_FORMAT ")",
                sizeof(input.mnonce), len);
        return NULL;
    }

    /* Query the report length */
    ret = sev_ioctl(sev->sev_fd, KVM_SEV_GET_ATTESTATION_REPORT,
            &input, &err);
    if (ret < 0) {
        if (err != SEV_RET_INVALID_LEN) {
            error_setg(errp, "SEV: Failed to query the attestation report"
                             " length ret=%d fw_err=%d (%s)",
                       ret, err, fw_error_to_str(err));
            return NULL;
        }
    }

    data = g_malloc(input.len);
    input.uaddr = (unsigned long)data;
    memcpy(input.mnonce, buf, sizeof(input.mnonce));

    /* Query the report */
    ret = sev_ioctl(sev->sev_fd, KVM_SEV_GET_ATTESTATION_REPORT,
            &input, &err);
    if (ret) {
        error_setg_errno(errp, errno, "SEV: Failed to get attestation report"
                " ret=%d fw_err=%d (%s)", ret, err, fw_error_to_str(err));
        return NULL;
    }

    report = g_new0(SevAttestationReport, 1);
    report->data = g_base64_encode(data, input.len);

    trace_kvm_sev_attestation_report(mnonce, report->data);

    return report;
}

SevAttestationReport *qmp_query_sev_attestation_report(const char *mnonce,
                                                       Error **errp)
{
    return sev_get_attestation_report(mnonce, errp);
}

static int
sev_read_file_base64(const char *filename, guchar **data, gsize *len)
{
    gsize sz;
    g_autofree gchar *base64 = NULL;
    GError *error = NULL;

    if (!g_file_get_contents(filename, &base64, &sz, &error)) {
        error_report("SEV: Failed to read '%s' (%s)", filename, error->message);
        g_error_free(error);
        return -1;
    }

    *data = g_base64_decode(base64, len);
    return 0;
}

static int
sev_launch_start(SevGuestState *sev)
{
    gsize sz;
    int ret = 1;
    int fw_error, rc;
    struct kvm_sev_launch_start start = {
        .handle = sev->handle, .policy = sev->policy
    };
    guchar *session = NULL, *dh_cert = NULL;

    if (sev->session_file) {
        if (sev_read_file_base64(sev->session_file, &session, &sz) < 0) {
            goto out;
        }
        start.session_uaddr = (unsigned long)session;
        start.session_len = sz;
    }

    if (sev->dh_cert_file) {
        if (sev_read_file_base64(sev->dh_cert_file, &dh_cert, &sz) < 0) {
            goto out;
        }
        start.dh_uaddr = (unsigned long)dh_cert;
        start.dh_len = sz;
    }

    trace_kvm_sev_launch_start(start.policy, session, dh_cert);
    rc = sev_ioctl(sev->sev_fd, KVM_SEV_LAUNCH_START, &start, &fw_error);
    if (rc < 0) {
        error_report("%s: LAUNCH_START ret=%d fw_error=%d '%s'",
                __func__, ret, fw_error, fw_error_to_str(fw_error));
        goto out;
    }

    sev_set_guest_state(sev, SEV_STATE_LAUNCH_UPDATE);
    sev->handle = start.handle;
    ret = 0;

out:
    g_free(session);
    g_free(dh_cert);
    return ret;
}

static int
sev_launch_update_data(SevGuestState *sev, uint8_t *addr, uint64_t len)
{
    int ret, fw_error;
    struct kvm_sev_launch_update_data update;

    if (!addr || !len) {
        return 1;
    }

    update.uaddr = (__u64)(unsigned long)addr;
    update.len = len;
    trace_kvm_sev_launch_update_data(addr, len);
    ret = sev_ioctl(sev->sev_fd, KVM_SEV_LAUNCH_UPDATE_DATA,
                    &update, &fw_error);
    if (ret) {
        error_report("%s: LAUNCH_UPDATE ret=%d fw_error=%d '%s'",
                __func__, ret, fw_error, fw_error_to_str(fw_error));
    }

    return ret;
}

static int
sev_launch_update_vmsa(SevGuestState *sev)
{
    int ret, fw_error;

    ret = sev_ioctl(sev->sev_fd, KVM_SEV_LAUNCH_UPDATE_VMSA, NULL, &fw_error);
    if (ret) {
        error_report("%s: LAUNCH_UPDATE_VMSA ret=%d fw_error=%d '%s'",
                __func__, ret, fw_error, fw_error_to_str(fw_error));
    }

    return ret;
}

static void
sev_launch_get_measure(Notifier *notifier, void *unused)
{
    SevGuestState *sev = sev_guest;
    int ret, error;
    g_autofree guchar *data = NULL;
    struct kvm_sev_launch_measure measurement = {};

    if (!sev_check_state(sev, SEV_STATE_LAUNCH_UPDATE)) {
        return;
    }

    if (sev_es_enabled()) {
        if (csv3_enabled()) {
            ret = csv3_launch_encrypt_vmcb();
        } else {
            /* measure all the VM save areas before getting launch_measure */
            ret = sev_launch_update_vmsa(sev);
        }
        if (ret) {
            exit(1);
        }
    }

    /* query the measurement blob length */
    ret = sev_ioctl(sev->sev_fd, KVM_SEV_LAUNCH_MEASURE,
                    &measurement, &error);
    if (!measurement.len) {
        error_report("%s: LAUNCH_MEASURE ret=%d fw_error=%d '%s'",
                     __func__, ret, error, fw_error_to_str(errno));
        return;
    }

    data = g_new0(guchar, measurement.len);
    measurement.uaddr = (unsigned long)data;

    /* get the measurement blob */
    ret = sev_ioctl(sev->sev_fd, KVM_SEV_LAUNCH_MEASURE,
                    &measurement, &error);
    if (ret) {
        error_report("%s: LAUNCH_MEASURE ret=%d fw_error=%d '%s'",
                     __func__, ret, error, fw_error_to_str(errno));
        return;
    }

    sev_set_guest_state(sev, SEV_STATE_LAUNCH_SECRET);

    /* encode the measurement value and emit the event */
    sev->measurement = g_base64_encode(data, measurement.len);
    trace_kvm_sev_launch_measurement(sev->measurement);
}

static char *sev_get_launch_measurement(void)
{
    if (sev_guest &&
        sev_guest->state >= SEV_STATE_LAUNCH_SECRET) {
        return g_strdup(sev_guest->measurement);
    }

    return NULL;
}

SevLaunchMeasureInfo *qmp_query_sev_launch_measure(Error **errp)
{
    char *data;
    SevLaunchMeasureInfo *info;

    data = sev_get_launch_measurement();
    if (!data) {
        error_setg(errp, "SEV launch measurement is not available");
        return NULL;
    }

    info = g_malloc0(sizeof(*info));
    info->data = data;

    return info;
}

static Notifier sev_machine_done_notify = {
    .notify = sev_launch_get_measure,
};

static void
sev_launch_finish(SevGuestState *sev)
{
    int ret, error;

    trace_kvm_sev_launch_finish();
    ret = sev_ioctl(sev->sev_fd, KVM_SEV_LAUNCH_FINISH, 0, &error);
    if (ret) {
        error_report("%s: LAUNCH_FINISH ret=%d fw_error=%d '%s'",
                     __func__, ret, error, fw_error_to_str(error));
        exit(1);
    }

    sev_set_guest_state(sev, SEV_STATE_RUNNING);

    /* add migration blocker */
    error_setg(&sev_mig_blocker,
               "SEV: Migration is not implemented");
    migrate_add_blocker(&sev_mig_blocker, &error_fatal);
}

void
sev_del_migrate_blocker(void)
{
    migrate_del_blocker(&sev_mig_blocker);
}

static int
sev_receive_finish(SevGuestState *s)
{
    int error, ret = 1;

    trace_kvm_sev_receive_finish();
    ret = sev_ioctl(s->sev_fd, KVM_SEV_RECEIVE_FINISH, 0, &error);
    if (ret) {
        error_report("%s: RECEIVE_FINISH ret=%d fw_error=%d '%s'",
                     __func__, ret, error, fw_error_to_str(error));
        goto err;
    }

    sev_set_guest_state(s, SEV_STATE_RUNNING);
err:
    return ret;
}

static void
sev_vm_state_change(void *opaque, bool running, RunState state)
{
    SevGuestState *sev = opaque;

    if (running) {
        if (sev_check_state(sev, SEV_STATE_RECEIVE_UPDATE)) {
            sev_receive_finish(sev);
        } else if (!sev_check_state(sev, SEV_STATE_RUNNING)) {
            sev_launch_finish(sev);
        }
    }
}

static inline bool check_blob_length(size_t value)
{
    if (value > SEV_FW_BLOB_MAX_SIZE) {
        error_report("invalid length max=%d got=%ld",
                     SEV_FW_BLOB_MAX_SIZE, value);
        return false;
    }

    return true;
}

int sev_save_setup(const char *pdh, const char *plat_cert,
                   const char *amd_cert)
{
    SevGuestState *s = sev_guest;

    if (is_hygon_cpu()) {
        if (sev_read_file_base64(pdh, &s->remote_pdh,
                                 &s->remote_pdh_len) < 0) {
            goto error;
        }
    } else {
        s->remote_pdh = g_base64_decode(pdh, &s->remote_pdh_len);
    }
    if (!check_blob_length(s->remote_pdh_len)) {
        goto error;
    }

    if (is_hygon_cpu()) {
        if (sev_read_file_base64(plat_cert, &s->remote_plat_cert,
                                 &s->remote_plat_cert_len) < 0) {
            goto error;
        }
    } else {
        s->remote_plat_cert = g_base64_decode(plat_cert,
                                              &s->remote_plat_cert_len);
    }
    if (!check_blob_length(s->remote_plat_cert_len)) {
        goto error;
    }

    if (is_hygon_cpu()) {
        if (sev_read_file_base64(amd_cert, &s->amd_cert,
                                 &s->amd_cert_len) < 0) {
            goto error;
        }
    } else {
        s->amd_cert = g_base64_decode(amd_cert, &s->amd_cert_len);
    }
    if (!check_blob_length(s->amd_cert_len)) {
        goto error;
    }

    return 0;

error:
    g_free(s->remote_pdh);
    g_free(s->remote_plat_cert);
    g_free(s->amd_cert);

    return 1;
}

static void
sev_send_finish(void)
{
    int ret, error;

    trace_kvm_sev_send_finish();
    ret = sev_ioctl(sev_guest->sev_fd, KVM_SEV_SEND_FINISH, 0, &error);
    if (ret) {
        error_report("%s: SEND_FINISH ret=%d fw_error=%d '%s'",
                     __func__, ret, error, fw_error_to_str(error));
    }

    g_free(sev_guest->send_packet_hdr);
    if (sev_es_enabled() && is_hygon_cpu()) {
        g_free(sev_guest->send_vmsa_packet_hdr);
    }
    sev_set_guest_state(sev_guest, SEV_STATE_RUNNING);
}

static void
sev_migration_state_notifier(Notifier *notifier, void *data)
{
    MigrationState *s = data;

    if (migration_has_finished(s) ||
        migration_in_postcopy_after_devices(s) ||
        migration_has_failed(s)) {
        if (sev_check_state(sev_guest, SEV_STATE_SEND_UPDATE)) {
            sev_send_finish();
        }
    }
}

static Notifier sev_migration_state;

int sev_kvm_init(ConfidentialGuestSupport *cgs, Error **errp)
{
    SevGuestState *sev
        = (SevGuestState *)object_dynamic_cast(OBJECT(cgs), TYPE_SEV_GUEST);
    char *devname;
    int ret, fw_error, cmd;
    uint32_t ebx;
    uint32_t host_cbitpos;
    struct sev_user_data_status status = {};

    if (!sev) {
        return 0;
    }

    ConfidentialGuestSupportClass *cgs_class =
        (ConfidentialGuestSupportClass *) object_get_class(OBJECT(cgs));

    ret = ram_block_discard_disable(true);
    if (ret) {
        error_report("%s: cannot disable RAM discard", __func__);
        return -1;
    }

    sev_guest = sev;
    sev->state = SEV_STATE_UNINIT;

    host_cpuid(0x8000001F, 0, NULL, &ebx, NULL, NULL);
    host_cbitpos = ebx & 0x3f;

    /*
     * The cbitpos value will be placed in bit positions 5:0 of the EBX
     * register of CPUID 0x8000001F. No need to verify the range as the
     * comparison against the host value accomplishes that.
     */
    if (host_cbitpos != sev->cbitpos) {
        error_setg(errp, "%s: cbitpos check failed, host '%d' requested '%d'",
                   __func__, host_cbitpos, sev->cbitpos);
        goto err;
    }

    /*
     * The reduced-phys-bits value will be placed in bit positions 11:6 of
     * the EBX register of CPUID 0x8000001F, so verify the supplied value
     * is in the range of 1 to 63.
     */
    if (sev->reduced_phys_bits < 1 || sev->reduced_phys_bits > 63) {
        error_setg(errp, "%s: reduced_phys_bits check failed,"
                   " it should be in the range of 1 to 63, requested '%d'",
                   __func__, sev->reduced_phys_bits);
        goto err;
    }

    devname = object_property_get_str(OBJECT(sev), "sev-device", NULL);
    sev->sev_fd = open(devname, O_RDWR);
    if (sev->sev_fd < 0) {
        error_setg(errp, "%s: Failed to open %s '%s'", __func__,
                   devname, strerror(errno));
        g_free(devname);
        goto err;
    }
    g_free(devname);

    ret = sev_platform_ioctl(sev->sev_fd, SEV_PLATFORM_STATUS, &status,
                             &fw_error);
    if (ret) {
        error_setg(errp, "%s: failed to get platform status ret=%d "
                   "fw_error='%d: %s'", __func__, ret, fw_error,
                   fw_error_to_str(fw_error));
        goto err;
    }
    sev->build_id = status.build;
    sev->api_major = status.api_major;
    sev->api_minor = status.api_minor;

    if (sev_es_enabled()) {
        if (!kvm_kernel_irqchip_allowed()) {
            error_report("%s: SEV-ES guests require in-kernel irqchip support",
                         __func__);
            goto err;
        }

        if (!(status.flags & SEV_STATUS_FLAGS_CONFIG_ES)) {
            error_report("%s: guest policy requires SEV-ES, but "
                         "host SEV-ES support unavailable",
                         __func__);
            goto err;
        }
        cmd = KVM_SEV_ES_INIT;
    } else {
        cmd = KVM_SEV_INIT;
    }

    trace_kvm_sev_init();

    /* Only support reuse asid for CSV/CSV2 guest */
    if (is_hygon_cpu() &&
        (sev_guest->policy & GUEST_POLICY_REUSE_ASID) &&
        !(sev_guest->policy & GUEST_POLICY_CSV3_BIT)) {
        char *user_id = NULL;
        struct kvm_csv_init *init_cmd_buf = NULL;

        user_id = object_property_get_str(OBJECT(sev), "user-id", NULL);
        if (user_id && strlen(user_id)) {
            init_cmd_buf = g_new0(struct kvm_csv_init, 1);
            init_cmd_buf->len = strlen(user_id);
            init_cmd_buf->userid_addr = (__u64)user_id;
        }
        ret = sev_ioctl(sev->sev_fd, cmd, init_cmd_buf, &fw_error);

        if (user_id) {
            g_free(user_id);
            g_free(init_cmd_buf);
        }
    } else {
        ret = sev_ioctl(sev->sev_fd, cmd, NULL, &fw_error);
    }

    if (ret) {
        error_setg(errp, "%s: failed to initialize ret=%d fw_error=%d '%s'",
                   __func__, ret, fw_error, fw_error_to_str(fw_error));
        goto err;
    }

    /* Support CSV3 */
    if (!ret && cmd == KVM_SEV_ES_INIT) {
        ret = csv3_init(sev_guest->policy, sev->sev_fd, (void *)&sev->state, &sev_ops);
        if (ret) {
            error_setg(errp, "%s: failed to init csv3 context", __func__);
            goto err;
        }
        /* The CSV3 guest is not resettable */
        if (csv3_enabled())
            csv_kvm_cpu_reset_inhibit = true;
    }

    /*
     * The LAUNCH context is used for new guest, if its an incoming guest
     * then RECEIVE context will be created after the connection is established.
     */
    if (!runstate_check(RUN_STATE_INMIGRATE)) {
        ret = sev_launch_start(sev);
        if (ret) {
            error_setg(errp, "%s: failed to create encryption context", __func__);
            goto err;
        }
    } else {
        /*
         * The CSV2 guest is not resettable after migrated to target machine,
         * set csv_kvm_cpu_reset_inhibit to true to indicate the CSV2 guest is
         * not resettable.
         */
        if (is_hygon_cpu() && sev_es_enabled()) {
            csv_kvm_cpu_reset_inhibit = true;
        }
    }

    /* CSV3 guest do not need notifier to reg/unreg memory */
    if (!csv3_enabled()) {
        ram_block_notifier_add(&sev_ram_notifier);
    }
    qemu_add_machine_init_done_notifier(&sev_machine_done_notify);
    qemu_add_vm_change_state_handler(sev_vm_state_change, sev);
    migration_add_notifier(&sev_migration_state, sev_migration_state_notifier);

    if (csv3_enabled()) {
        cgs_class->memory_encryption_ops = &csv3_memory_encryption_ops;
    } else {
        cgs_class->memory_encryption_ops = &sev_memory_encryption_ops;
    }
    QTAILQ_INIT(&sev->shared_regions_list);

    /* Determine whether support MSR_AMD64_SEV_ES_GHCB */
    if (sev_es_enabled()) {
        sev_kvm_has_msr_ghcb =
                kvm_vm_check_extension(kvm_state, KVM_CAP_SEV_ES_GHCB);
    } else {
        sev_kvm_has_msr_ghcb = false;
    }

    cgs->ready = true;

    return 0;
err:
    sev_guest = NULL;
    ram_block_discard_disable(false);
    return -1;
}

int
sev_encrypt_flash(uint8_t *ptr, uint64_t len, Error **errp)
{
    if (!sev_guest) {
        return 0;
    }

    /* if SEV is in update state then encrypt the data else do nothing */
    if (sev_check_state(sev_guest, SEV_STATE_LAUNCH_UPDATE)) {
        int ret = sev_launch_update_data(sev_guest, ptr, len);
        if (ret < 0) {
            error_setg(errp, "SEV: Failed to encrypt pflash rom");
            return ret;
        }
    }

    return 0;
}

int sev_inject_launch_secret(const char *packet_hdr, const char *secret,
                             uint64_t gpa, Error **errp)
{
    struct kvm_sev_launch_secret input;
    g_autofree guchar *data = NULL, *hdr = NULL;
    int error, ret = 1;
    void *hva;
    gsize hdr_sz = 0, data_sz = 0;
    MemoryRegion *mr = NULL;

    if (!sev_guest) {
        error_setg(errp, "SEV not enabled for guest");
        return 1;
    }

    /* secret can be injected only in this state */
    if (!sev_check_state(sev_guest, SEV_STATE_LAUNCH_SECRET)) {
        error_setg(errp, "SEV: Not in correct state. (LSECRET) %x",
                     sev_guest->state);
        return 1;
    }

    hdr = g_base64_decode(packet_hdr, &hdr_sz);
    if (!hdr || !hdr_sz) {
        error_setg(errp, "SEV: Failed to decode sequence header");
        return 1;
    }

    data = g_base64_decode(secret, &data_sz);
    if (!data || !data_sz) {
        error_setg(errp, "SEV: Failed to decode data");
        return 1;
    }

    hva = gpa2hva(&mr, gpa, data_sz, errp);
    if (!hva) {
        error_prepend(errp, "SEV: Failed to calculate guest address: ");
        return 1;
    }

    input.hdr_uaddr = (uint64_t)(unsigned long)hdr;
    input.hdr_len = hdr_sz;

    input.trans_uaddr = (uint64_t)(unsigned long)data;
    input.trans_len = data_sz;

    input.guest_uaddr = (uint64_t)(unsigned long)hva;
    input.guest_len = data_sz;

    trace_kvm_sev_launch_secret(gpa, input.guest_uaddr,
                                input.trans_uaddr, input.trans_len);

    ret = sev_ioctl(sev_guest->sev_fd, KVM_SEV_LAUNCH_SECRET,
                    &input, &error);
    if (ret) {
        error_setg(errp, "SEV: failed to inject secret ret=%d fw_error=%d '%s'",
                     ret, error, fw_error_to_str(error));
        return ret;
    }

    return 0;
}

#define SEV_SECRET_GUID "4c2eb361-7d9b-4cc3-8081-127c90d3d294"
struct sev_secret_area {
    uint32_t base;
    uint32_t size;
};

void qmp_sev_inject_launch_secret(const char *packet_hdr,
                                  const char *secret,
                                  bool has_gpa, uint64_t gpa,
                                  Error **errp)
{
    if (!sev_enabled()) {
        error_setg(errp, "SEV not enabled for guest");
        return;
    }
    if (!has_gpa) {
        uint8_t *data;
        struct sev_secret_area *area;

        if (!pc_system_ovmf_table_find(SEV_SECRET_GUID, &data, NULL)) {
            error_setg(errp, "SEV: no secret area found in OVMF,"
                       " gpa must be specified.");
            return;
        }
        area = (struct sev_secret_area *)data;
        gpa = area->base;
    }

    sev_inject_launch_secret(packet_hdr, secret, gpa, errp);
}

static int
sev_es_parse_reset_block(SevInfoBlock *info, uint32_t *addr)
{
    if (!info->reset_addr) {
        error_report("SEV-ES reset address is zero");
        return 1;
    }

    *addr = info->reset_addr;

    return 0;
}

static int
sev_es_find_reset_vector(void *flash_ptr, uint64_t flash_size,
                         uint32_t *addr)
{
    QemuUUID info_guid, *guid;
    SevInfoBlock *info;
    uint8_t *data;
    uint16_t *len;

    /*
     * Initialize the address to zero. An address of zero with a successful
     * return code indicates that SEV-ES is not active.
     */
    *addr = 0;

    /*
     * Extract the AP reset vector for SEV-ES guests by locating the SEV GUID.
     * The SEV GUID is located on its own (original implementation) or within
     * the Firmware GUID Table (new implementation), either of which are
     * located 32 bytes from the end of the flash.
     *
     * Check the Firmware GUID Table first.
     */
    if (pc_system_ovmf_table_find(SEV_INFO_BLOCK_GUID, &data, NULL)) {
        return sev_es_parse_reset_block((SevInfoBlock *)data, addr);
    }

    /*
     * SEV info block not found in the Firmware GUID Table (or there isn't
     * a Firmware GUID Table), fall back to the original implementation.
     */
    data = flash_ptr + flash_size - 0x20;

    qemu_uuid_parse(SEV_INFO_BLOCK_GUID, &info_guid);
    info_guid = qemu_uuid_bswap(info_guid); /* GUIDs are LE */

    guid = (QemuUUID *)(data - sizeof(info_guid));
    if (!qemu_uuid_is_equal(guid, &info_guid)) {
        error_report("SEV information block/Firmware GUID Table block not found in pflash rom");
        return 1;
    }

    len = (uint16_t *)((uint8_t *)guid - sizeof(*len));
    info = (SevInfoBlock *)(data - le16_to_cpu(*len));

    return sev_es_parse_reset_block(info, addr);
}

void sev_es_set_reset_vector(CPUState *cpu)
{
    X86CPU *x86;
    CPUX86State *env;

    /* Only update if we have valid reset information */
    if (!sev_guest || !sev_guest->reset_data_valid) {
        return;
    }

    /* Do not update the BSP reset state */
    if (cpu->cpu_index == 0) {
        return;
    }

    x86 = X86_CPU(cpu);
    env = &x86->env;

    cpu_x86_load_seg_cache(env, R_CS, 0xf000, sev_guest->reset_cs, 0xffff,
                           DESC_P_MASK | DESC_S_MASK | DESC_CS_MASK |
                           DESC_R_MASK | DESC_A_MASK);

    env->eip = sev_guest->reset_ip;
}

int sev_es_save_reset_vector(void *flash_ptr, uint64_t flash_size)
{
    CPUState *cpu;
    uint32_t addr;
    int ret;

    if (!sev_es_enabled()) {
        return 0;
    }

    addr = 0;
    ret = sev_es_find_reset_vector(flash_ptr, flash_size,
                                   &addr);
    if (ret) {
        return ret;
    }

    if (addr) {
        sev_guest->reset_cs = addr & 0xffff0000;
        sev_guest->reset_ip = addr & 0x0000ffff;
        sev_guest->reset_data_valid = true;

        CPU_FOREACH(cpu) {
            sev_es_set_reset_vector(cpu);
        }
    }

    return 0;
}

static int
sev_get_send_session_length(void)
{
    int ret, fw_err = 0;
    struct kvm_sev_send_start start = {};

    ret = sev_ioctl(sev_guest->sev_fd, KVM_SEV_SEND_START, &start, &fw_err);
    if (fw_err != SEV_RET_INVALID_LEN) {
        ret = -1;
        error_report("%s: failed to get session length ret=%d fw_error=%d '%s'",
                     __func__, ret, fw_err, fw_error_to_str(fw_err));
        goto err;
    }

    ret = start.session_len;
err:
    return ret;
}

static int
sev_send_start(SevGuestState *s, QEMUFile *f, uint64_t *bytes_sent)
{
    gsize pdh_len = 0, plat_cert_len;
    int session_len, ret, fw_error;
    struct kvm_sev_send_start start = { };
    guchar *pdh = NULL, *plat_cert = NULL, *session = NULL;
    Error *local_err = NULL;

    if (!s->remote_pdh || !s->remote_plat_cert || !s->amd_cert_len) {
        error_report("%s: missing remote PDH or PLAT_CERT", __func__);
        return 1;
    }

   start.pdh_cert_uaddr = (uintptr_t) s->remote_pdh;
   start.pdh_cert_len = s->remote_pdh_len;

   start.plat_certs_uaddr = (uintptr_t)s->remote_plat_cert;
   start.plat_certs_len = s->remote_plat_cert_len;

   start.amd_certs_uaddr = (uintptr_t)s->amd_cert;
   start.amd_certs_len = s->amd_cert_len;

    /* get the session length */
   session_len = sev_get_send_session_length();
   if (session_len < 0) {
       ret = 1;
       goto err;
   }

   session = g_new0(guchar, session_len);
   start.session_uaddr = (unsigned long)session;
   start.session_len = session_len;

   /* Get our PDH certificate */
   ret = sev_get_pdh_info(s->sev_fd, &pdh, &pdh_len,
                          &plat_cert, &plat_cert_len, &local_err);
   if (ret) {
       error_report("Failed to get our PDH cert");
       goto err;
   }

   trace_kvm_sev_send_start(start.pdh_cert_uaddr, start.pdh_cert_len,
                            start.plat_certs_uaddr, start.plat_certs_len,
                            start.amd_certs_uaddr, start.amd_certs_len);

   ret = sev_ioctl(s->sev_fd, KVM_SEV_SEND_START, &start, &fw_error);
   if (ret < 0) {
       error_report("%s: SEND_START ret=%d fw_error=%d '%s'",
               __func__, ret, fw_error, fw_error_to_str(fw_error));
       goto err;
   }

   qemu_put_be32(f, start.policy);
   qemu_put_be32(f, pdh_len);
   qemu_put_buffer(f, (uint8_t *)pdh, pdh_len);
   qemu_put_be32(f, start.session_len);
   qemu_put_buffer(f, (uint8_t *)start.session_uaddr, start.session_len);
   *bytes_sent = 12 + pdh_len + start.session_len;

   sev_set_guest_state(s, SEV_STATE_SEND_UPDATE);

err:
   g_free(pdh);
   g_free(plat_cert);
   return ret;
}

static int
sev_send_get_packet_len(int *fw_err)
{
    int ret;
    struct kvm_sev_send_update_data update = { 0, };

    ret = sev_ioctl(sev_guest->sev_fd, KVM_SEV_SEND_UPDATE_DATA,
                    &update, fw_err);
    if (*fw_err != SEV_RET_INVALID_LEN) {
        ret = 0;
        error_report("%s: failed to get session length ret=%d fw_error=%d '%s'",
                    __func__, ret, *fw_err, fw_error_to_str(*fw_err));
        goto err;
    }

    ret = update.hdr_len;

err:
    return ret;
}

static int
sev_send_update_data(SevGuestState *s, QEMUFile *f, uint8_t *ptr, uint32_t size,
                     uint64_t *bytes_sent)
{
    int ret, fw_error;
    guchar *trans;
    struct kvm_sev_send_update_data update = { };

    /*
     * If this is first call then query the packet header bytes and allocate
     * the packet buffer.
     */
    if (!s->send_packet_hdr) {
        s->send_packet_hdr_len = sev_send_get_packet_len(&fw_error);
        if (s->send_packet_hdr_len < 1) {
            error_report("%s: SEND_UPDATE fw_error=%d '%s'",
                         __func__, fw_error, fw_error_to_str(fw_error));
            return 1;
        }

        s->send_packet_hdr = g_new(gchar, s->send_packet_hdr_len);
    }

    /* allocate transport buffer */
    trans = g_new(guchar, size);

    update.hdr_uaddr = (uintptr_t)s->send_packet_hdr;
    update.hdr_len = s->send_packet_hdr_len;
    update.guest_uaddr = (uintptr_t)ptr;
    update.guest_len = size;
    update.trans_uaddr = (uintptr_t)trans;
    update.trans_len = size;

    trace_kvm_sev_send_update_data(ptr, trans, size);

    ret = sev_ioctl(s->sev_fd, KVM_SEV_SEND_UPDATE_DATA, &update, &fw_error);
    if (ret) {
        error_report("%s: SEND_UPDATE_DATA ret=%d fw_error=%d '%s'",
                     __func__, ret, fw_error, fw_error_to_str(fw_error));
        goto err;
    }

    qemu_put_be32(f, update.hdr_len);
    qemu_put_buffer(f, (uint8_t *)update.hdr_uaddr, update.hdr_len);
    *bytes_sent = 4 + update.hdr_len;

    qemu_put_be32(f, update.trans_len);
    qemu_put_buffer(f, (uint8_t *)update.trans_uaddr, update.trans_len);
    *bytes_sent += (4 + update.trans_len);

err:
    g_free(trans);
    return ret;
}

int sev_save_outgoing_page(QEMUFile *f, uint8_t *ptr,
                           uint32_t sz, uint64_t *bytes_sent)
{
    SevGuestState *s = sev_guest;

    /*
     * If this is a first buffer then create outgoing encryption context
     * and write our PDH, policy and session data.
     */
    if (!sev_check_state(s, SEV_STATE_SEND_UPDATE) &&
        sev_send_start(s, f, bytes_sent)) {
        error_report("Failed to create outgoing context");
        return 1;
    }

    return sev_send_update_data(s, f, ptr, sz, bytes_sent);
}

static int
sev_receive_start(SevGuestState *sev, QEMUFile *f)
{
    int ret = 1;
    int fw_error;
    struct kvm_sev_receive_start start = { };
    gchar *session = NULL, *pdh_cert = NULL;

    /* get SEV guest handle */
    start.handle = object_property_get_int(OBJECT(sev), "handle",
                                           &error_abort);

    /* get the source policy */
    start.policy = qemu_get_be32(f);

    /* get source PDH key */
    start.pdh_len = qemu_get_be32(f);
    if (!check_blob_length(start.pdh_len)) {
        return 1;
    }

    pdh_cert = g_new(gchar, start.pdh_len);
    qemu_get_buffer(f, (uint8_t *)pdh_cert, start.pdh_len);
    start.pdh_uaddr = (uintptr_t)pdh_cert;

    /* get source session data */
    start.session_len = qemu_get_be32(f);
    if (!check_blob_length(start.session_len)) {
        return 1;
    }
    session = g_new(gchar, start.session_len);
    qemu_get_buffer(f, (uint8_t *)session, start.session_len);
    start.session_uaddr = (uintptr_t)session;

    trace_kvm_sev_receive_start(start.policy, session, pdh_cert);

    ret = sev_ioctl(sev_guest->sev_fd, KVM_SEV_RECEIVE_START,
                    &start, &fw_error);
    if (ret < 0) {
        error_report("Error RECEIVE_START ret=%d fw_error=%d '%s'",
                      ret, fw_error, fw_error_to_str(fw_error));
        goto err;
    }

    object_property_set_int(OBJECT(sev), "handle", start.handle, &error_abort);
    sev_set_guest_state(sev, SEV_STATE_RECEIVE_UPDATE);
err:
    g_free(session);
    g_free(pdh_cert);

    return ret;
}

static int sev_receive_update_data(QEMUFile *f, uint8_t *ptr)
{
    int ret = 1, fw_error = 0;
    gchar *hdr = NULL, *trans = NULL;
    struct kvm_sev_receive_update_data update = {};

    /* get packet header */
    update.hdr_len = qemu_get_be32(f);
    if (!check_blob_length(update.hdr_len)) {
        return 1;
    }

    hdr = g_new(gchar, update.hdr_len);
    qemu_get_buffer(f, (uint8_t *)hdr, update.hdr_len);
    update.hdr_uaddr = (uintptr_t)hdr;

    /* get transport buffer */
    update.trans_len = qemu_get_be32(f);
    if (!check_blob_length(update.trans_len)) {
        goto err;
    }

    trans = g_new(gchar, update.trans_len);
    update.trans_uaddr = (uintptr_t)trans;
    qemu_get_buffer(f, (uint8_t *)update.trans_uaddr, update.trans_len);

    update.guest_uaddr = (uintptr_t) ptr;
    update.guest_len = update.trans_len;

    trace_kvm_sev_receive_update_data(trans, ptr, update.guest_len,
            hdr, update.hdr_len);

    ret = sev_ioctl(sev_guest->sev_fd, KVM_SEV_RECEIVE_UPDATE_DATA,
                    &update, &fw_error);
    if (ret) {
        error_report("Error RECEIVE_UPDATE_DATA ret=%d fw_error=%d '%s'",
                     ret, fw_error, fw_error_to_str(fw_error));
        goto err;
    }
err:
    g_free(trans);
    g_free(hdr);
    return ret;
}

int sev_load_incoming_page(QEMUFile *f, uint8_t *ptr)
{
    SevGuestState *s = sev_guest;

    /*
     * If this is first buffer and SEV is not in recieiving state then
     * use RECEIVE_START command to create a encryption context.
     */
    if (!sev_check_state(s, SEV_STATE_RECEIVE_UPDATE) &&
        sev_receive_start(s, f)) {
        return 1;
    }

    return sev_receive_update_data(f, ptr);
}

int sev_remove_shared_regions_list(unsigned long start, unsigned long end)
{
    SevGuestState *s = sev_guest;
    struct shared_region *pos, *next_pos;

    QTAILQ_FOREACH_SAFE(pos, &s->shared_regions_list, list, next_pos) {
        unsigned long l, r;
        unsigned long curr_gfn_end = pos->gfn_end;

        /*
         * Find if any intersection exists ?
         * left bound for intersecting segment
         */
        l = MAX(start, pos->gfn_start);
        /* right bound for intersecting segment */
        r = MIN(end, pos->gfn_end);
        if (l <= r) {
            if (pos->gfn_start == l && pos->gfn_end == r) {
                QTAILQ_REMOVE(&s->shared_regions_list, pos, list);
                g_free(pos);
            } else if (l == pos->gfn_start) {
                pos->gfn_start = r;
            } else if (r == pos->gfn_end) {
                pos->gfn_end = l;
            } else {
                /* Do a de-merge -- split linked list nodes */
                struct shared_region *shrd_region;

                pos->gfn_end = l;
                shrd_region = g_malloc0(sizeof(*shrd_region));
                if (!shrd_region) {
                    return 0;
                }
                shrd_region->gfn_start = r;
                shrd_region->gfn_end = curr_gfn_end;
                QTAILQ_INSERT_AFTER(&s->shared_regions_list, pos,
                                    shrd_region, list);
            }
        }
        if (end <= curr_gfn_end) {
            break;
        }
    }
    return 0;
}

int sev_add_shared_regions_list(unsigned long start, unsigned long end)
{
    struct shared_region *shrd_region;
    struct shared_region *pos;
    SevGuestState *s = sev_guest;

    if (QTAILQ_EMPTY(&s->shared_regions_list)) {
        shrd_region = g_malloc0(sizeof(*shrd_region));
        if (!shrd_region) {
            return -1;
        }
        shrd_region->gfn_start = start;
        shrd_region->gfn_end = end;
        QTAILQ_INSERT_TAIL(&s->shared_regions_list, shrd_region, list);
        return 0;
    }

    /*
     * shared regions list is a sorted list in ascending order
     * of guest PA's and also merges consecutive range of guest PA's
     */
    QTAILQ_FOREACH(pos, &s->shared_regions_list, list) {
        /* handle duplicate overlapping regions */
        if (start >= pos->gfn_start && end <= pos->gfn_end) {
            return 0;
        }
        if (pos->gfn_end < start) {
            continue;
        }
        /* merge consecutive guest PA(s) -- forward merge */
        if (pos->gfn_start <= start && pos->gfn_end >= start) {
            pos->gfn_end = end;
            return 0;
        }
        break;
    }
    /*
     * Add a new node
     */
    shrd_region = g_malloc0(sizeof(*shrd_region));
    if (!shrd_region) {
        return -1;
    }
    shrd_region->gfn_start = start;
    shrd_region->gfn_end = end;
    if (pos) {
        QTAILQ_INSERT_BEFORE(pos, shrd_region, list);
    } else {
        QTAILQ_INSERT_TAIL(&s->shared_regions_list, shrd_region, list);
    }
    return 1;
}

int sev_save_outgoing_shared_regions_list(QEMUFile *f, uint64_t *bytes_sent)
{
    SevGuestState *s = sev_guest;
    struct shared_region *pos;

    QTAILQ_FOREACH(pos, &s->shared_regions_list, list) {
        qemu_put_be32(f, SHARED_REGION_LIST_CONT);
        qemu_put_be32(f, pos->gfn_start);
        qemu_put_be32(f, pos->gfn_end);
        *bytes_sent += 12;
    }

    qemu_put_be32(f, SHARED_REGION_LIST_END);
    *bytes_sent += 4;
    return 0;
}

int sev_load_incoming_shared_regions_list(QEMUFile *f)
{
    SevGuestState *s = sev_guest;
    struct shared_region *shrd_region;
    int status;

    status = qemu_get_be32(f);
    while (status == SHARED_REGION_LIST_CONT) {

        shrd_region = g_malloc0(sizeof(*shrd_region));
        if (!shrd_region) {
            return 0;
        }
        shrd_region->gfn_start = qemu_get_be32(f);
        shrd_region->gfn_end = qemu_get_be32(f);

        QTAILQ_INSERT_TAIL(&s->shared_regions_list, shrd_region, list);

        status = qemu_get_be32(f);
    }
    return 0;
}

bool sev_is_gfn_in_unshared_region(unsigned long gfn)
{
    SevGuestState *s = sev_guest;
    struct shared_region *pos;

    QTAILQ_FOREACH(pos, &s->shared_regions_list, list) {
        if (gfn >= pos->gfn_start && gfn < pos->gfn_end) {
            return false;
        }
    }
    return true;
}

static CsvBatchCmdList *
csv_batch_cmd_list_create(struct kvm_csv_batch_list_node *head,
                          CsvDestroyCmdNodeFn func)
{
    CsvBatchCmdList *csv_batch_cmd_list =
                        g_malloc0(sizeof(*csv_batch_cmd_list));

    if (!csv_batch_cmd_list) {
        return NULL;
    }

    csv_batch_cmd_list->head = head;
    csv_batch_cmd_list->tail = head;
    csv_batch_cmd_list->destroy_fn = func;

    return csv_batch_cmd_list;
}

static int
csv_batch_cmd_list_add_after(CsvBatchCmdList *list,
                             struct kvm_csv_batch_list_node *new_node)
{
    list->tail->next_cmd_addr = (__u64)new_node;
    list->tail = new_node;

    return 0;
}

static struct kvm_csv_batch_list_node *
csv_batch_cmd_list_node_create(uint64_t cmd_data_addr, uint64_t addr)
{
    struct kvm_csv_batch_list_node *new_node =
                        g_malloc0(sizeof(struct kvm_csv_batch_list_node));

    if (!new_node) {
        return NULL;
    }

    new_node->cmd_data_addr = cmd_data_addr;
    new_node->addr = addr;
    new_node->next_cmd_addr = 0;

    return new_node;
}

static int csv_batch_cmd_list_destroy(CsvBatchCmdList *list)
{
    struct kvm_csv_batch_list_node *node = list->head;

    while (node != NULL) {
        if (list->destroy_fn != NULL)
            list->destroy_fn((void *)node->cmd_data_addr);

        list->head = (struct kvm_csv_batch_list_node *)node->next_cmd_addr;
        g_free(node);
        node = list->head;
    }

    g_free(list);
    return 0;
}

static void send_update_data_free(void *data)
{
    struct kvm_sev_send_update_data *update =
                        (struct kvm_sev_send_update_data *)data;
    g_free((guchar *)update->hdr_uaddr);
    g_free((guchar *)update->trans_uaddr);
    g_free(update);
}

static void receive_update_data_free(void *data)
{
    struct kvm_sev_receive_update_data *update =
                        (struct kvm_sev_receive_update_data *)data;
    g_free((guchar *)update->hdr_uaddr);
    g_free((guchar *)update->trans_uaddr);
    g_free(update);
}

static int
csv_send_queue_data(SevGuestState *s, uint8_t *ptr,
                    uint32_t size, uint64_t addr)
{
    int ret = 0;
    int fw_error;
    guchar *trans;
    guchar *packet_hdr;
    struct kvm_sev_send_update_data *update;
    struct kvm_csv_batch_list_node *new_node = NULL;

    /* If this is first call then query the packet header bytes and allocate
     * the packet buffer.
     */
    if (s->send_packet_hdr_len < 1) {
        s->send_packet_hdr_len = sev_send_get_packet_len(&fw_error);
        if (s->send_packet_hdr_len < 1) {
            error_report("%s: SEND_UPDATE fw_error=%d '%s'",
                    __func__, fw_error, fw_error_to_str(fw_error));
            return 1;
        }
    }

    packet_hdr = g_new(guchar, s->send_packet_hdr_len);
    memset(packet_hdr, 0, s->send_packet_hdr_len);

    update = g_new0(struct kvm_sev_send_update_data, 1);

    /* allocate transport buffer */
    trans = g_new(guchar, size);

    update->hdr_uaddr = (unsigned long)packet_hdr;
    update->hdr_len = s->send_packet_hdr_len;
    update->guest_uaddr = (unsigned long)ptr;
    update->guest_len = size;
    update->trans_uaddr = (unsigned long)trans;
    update->trans_len = size;

    new_node = csv_batch_cmd_list_node_create((uint64_t)update, addr);
    if (!new_node) {
        ret = -ENOMEM;
        goto err;
    }

    if (s->csv_batch_cmd_list == NULL) {
        s->csv_batch_cmd_list = csv_batch_cmd_list_create(new_node,
                                                send_update_data_free);
        if (s->csv_batch_cmd_list == NULL) {
            ret = -ENOMEM;
            goto err;
        }
    } else {
        /* Add new_node's command address to the last_node */
        csv_batch_cmd_list_add_after(s->csv_batch_cmd_list, new_node);
    }

    trace_kvm_sev_send_update_data(ptr, trans, size);

    return ret;

err:
    g_free(trans);
    g_free(update);
    g_free(packet_hdr);
    g_free(new_node);
    if (s->csv_batch_cmd_list) {
        csv_batch_cmd_list_destroy(s->csv_batch_cmd_list);
        s->csv_batch_cmd_list = NULL;
    }
    return ret;
}

static int
csv_receive_queue_data(SevGuestState *s, QEMUFile *f, uint8_t *ptr)
{
    int ret = 0;
    gchar *hdr = NULL, *trans = NULL;
    struct kvm_sev_receive_update_data *update;
    struct kvm_csv_batch_list_node *new_node = NULL;

    update = g_new0(struct kvm_sev_receive_update_data, 1);
    /* get packet header */
    update->hdr_len = qemu_get_be32(f);
    hdr = g_new(gchar, update->hdr_len);
    qemu_get_buffer(f, (uint8_t *)hdr, update->hdr_len);
    update->hdr_uaddr = (unsigned long)hdr;

    /* get transport buffer */
    update->trans_len = qemu_get_be32(f);
    trans = g_new(gchar, update->trans_len);
    update->trans_uaddr = (unsigned long)trans;
    qemu_get_buffer(f, (uint8_t *)update->trans_uaddr, update->trans_len);

    /* set guest address,guest len is page_size */
    update->guest_uaddr = (uint64_t)ptr;
    update->guest_len = TARGET_PAGE_SIZE;

    new_node = csv_batch_cmd_list_node_create((uint64_t)update, 0);
    if (!new_node) {
        ret = -ENOMEM;
        goto err;
    }

    if (s->csv_batch_cmd_list == NULL) {
        s->csv_batch_cmd_list = csv_batch_cmd_list_create(new_node,
                                                receive_update_data_free);
        if (s->csv_batch_cmd_list == NULL) {
            ret = -ENOMEM;
            goto err;
        }
    } else {
        /* Add new_node's command address to the last_node */
        csv_batch_cmd_list_add_after(s->csv_batch_cmd_list, new_node);
    }

    trace_kvm_sev_receive_update_data(trans, (void *)ptr, update->guest_len,
            (void *)hdr, update->hdr_len);

    return ret;

err:
    g_free(trans);
    g_free(update);
    g_free(hdr);
    g_free(new_node);
    if (s->csv_batch_cmd_list) {
        csv_batch_cmd_list_destroy(s->csv_batch_cmd_list);
        s->csv_batch_cmd_list = NULL;
    }
    return ret;
}

static int
csv_command_batch(uint32_t cmd_id, uint64_t head_uaddr, int *fw_err)
{
    int ret;
    struct kvm_csv_command_batch command_batch = { };

    command_batch.command_id = cmd_id;
    command_batch.csv_batch_list_uaddr = head_uaddr;

    ret = sev_ioctl(sev_guest->sev_fd, KVM_CSV_COMMAND_BATCH,
                    &command_batch, fw_err);
    if (ret) {
        error_report("%s: COMMAND_BATCH ret=%d fw_err=%d '%s'",
                __func__, ret, *fw_err, fw_error_to_str(*fw_err));
    }

    return ret;
}

static int
csv_send_update_data_batch(SevGuestState *s, QEMUFile *f, uint64_t *bytes_sent)
{
    int ret, fw_error = 0;
    struct kvm_sev_send_update_data *update;
    struct kvm_csv_batch_list_node *node;

    ret = csv_command_batch(KVM_SEV_SEND_UPDATE_DATA,
                            (uint64_t)s->csv_batch_cmd_list->head, &fw_error);
    if (ret) {
        error_report("%s: csv_command_batch ret=%d fw_error=%d '%s'",
                __func__, ret, fw_error, fw_error_to_str(fw_error));
        goto err;
    }

    for (node = s->csv_batch_cmd_list->head;
         node != NULL;
         node = (struct kvm_csv_batch_list_node *)node->next_cmd_addr) {
        if (node != s->csv_batch_cmd_list->head) {
            /* head's page header is saved before send_update_data */
            qemu_put_be64(f, node->addr);
            *bytes_sent += 8;
            if (node->next_cmd_addr != 0)
                qemu_put_be32(f, RAM_SAVE_ENCRYPTED_PAGE_BATCH);
            else
                qemu_put_be32(f, RAM_SAVE_ENCRYPTED_PAGE_BATCH_END);
            *bytes_sent += 4;
        }
        update = (struct kvm_sev_send_update_data *)node->cmd_data_addr;
        qemu_put_be32(f, update->hdr_len);
        qemu_put_buffer(f, (uint8_t *)update->hdr_uaddr, update->hdr_len);
        *bytes_sent += (4 + update->hdr_len);

        qemu_put_be32(f, update->trans_len);
        qemu_put_buffer(f, (uint8_t *)update->trans_uaddr, update->trans_len);
        *bytes_sent += (4 + update->trans_len);
    }

err:
    csv_batch_cmd_list_destroy(s->csv_batch_cmd_list);
    s->csv_batch_cmd_list = NULL;
    return ret;
}

static int
csv_receive_update_data_batch(SevGuestState *s)
{
    int ret;
    int fw_error;

    ret = csv_command_batch(KVM_SEV_RECEIVE_UPDATE_DATA,
                            (uint64_t)s->csv_batch_cmd_list->head, &fw_error);
    if (ret) {
        error_report("%s: csv_command_batch ret=%d fw_error=%d '%s'",
                __func__, ret, fw_error, fw_error_to_str(fw_error));
    }

    csv_batch_cmd_list_destroy(s->csv_batch_cmd_list);
    s->csv_batch_cmd_list = NULL;
    return ret;
}

int
csv_queue_outgoing_page(uint8_t *ptr, uint32_t sz, uint64_t addr)
{
    SevGuestState *s = sev_guest;

    /* Only support for HYGON CSV */
    if (!is_hygon_cpu()) {
        error_report("Only support enqueue pages for HYGON CSV");
        return -EINVAL;
    }

    return csv_send_queue_data(s, ptr, sz, addr);
}

int csv_queue_incoming_page(QEMUFile *f, uint8_t *ptr)
{
    SevGuestState *s = sev_guest;

    /* Only support for HYGON CSV */
    if (!is_hygon_cpu()) {
        error_report("Only support enqueue received pages for HYGON CSV");
        return -EINVAL;
    }

    /*
     * If this is first buffer and SEV is not in recieiving state then
     * use RECEIVE_START command to create a encryption context.
     */
    if (!sev_check_state(s, SEV_STATE_RECEIVE_UPDATE) &&
        sev_receive_start(s, f)) {
        return 1;
    }

    return csv_receive_queue_data(s, f, ptr);
}

int
csv_save_queued_outgoing_pages(QEMUFile *f, uint64_t *bytes_sent)
{
    SevGuestState *s = sev_guest;

    /* Only support for HYGON CSV */
    if (!is_hygon_cpu()) {
        error_report("Only support transfer queued pages for HYGON CSV");
        return -EINVAL;
    }

    /*
     * If this is a first buffer then create outgoing encryption context
     * and write our PDH, policy and session data.
     */
    if (!sev_check_state(s, SEV_STATE_SEND_UPDATE) &&
        sev_send_start(s, f, bytes_sent)) {
        error_report("Failed to create outgoing context");
        return 1;
    }

    return csv_send_update_data_batch(s, f, bytes_sent);
}

int csv_load_queued_incoming_pages(QEMUFile *f)
{
    SevGuestState *s = sev_guest;

    /* Only support for HYGON CSV */
    if (!is_hygon_cpu()) {
        error_report("Only support load queued pages for HYGON CSV");
        return -EINVAL;
    }

    return csv_receive_update_data_batch(s);
}

static int
sev_send_vmsa_get_packet_len(int *fw_err)
{
    int ret;
    struct kvm_sev_send_update_vmsa update = { 0, };

    ret = sev_ioctl(sev_guest->sev_fd, KVM_SEV_SEND_UPDATE_VMSA,
                    &update, fw_err);
    if (*fw_err != SEV_RET_INVALID_LEN) {
        ret = 0;
        error_report("%s: failed to get session length ret=%d fw_error=%d '%s'",
                     __func__, ret, *fw_err, fw_error_to_str(*fw_err));
        goto err;
    }

    ret = update.hdr_len;

err:
    return ret;
}

static int
sev_send_update_vmsa(SevGuestState *s, QEMUFile *f, uint32_t cpu_id,
                     uint32_t cpu_index, uint32_t size, uint64_t *bytes_sent)
{
    int ret, fw_error;
    guchar *trans = NULL;
    struct kvm_sev_send_update_vmsa update = {};

    /*
     * If this is first call then query the packet header bytes and allocate
     * the packet buffer.
     */
    if (!s->send_vmsa_packet_hdr) {
        s->send_vmsa_packet_hdr_len = sev_send_vmsa_get_packet_len(&fw_error);
        if (s->send_vmsa_packet_hdr_len < 1) {
            error_report("%s: SEND_UPDATE_VMSA fw_error=%d '%s'",
                         __func__, fw_error, fw_error_to_str(fw_error));
            return 1;
        }

        s->send_vmsa_packet_hdr = g_new(gchar, s->send_vmsa_packet_hdr_len);
    }

    /* allocate transport buffer */
    trans = g_new(guchar, size);

    update.vcpu_id = cpu_id;
    update.hdr_uaddr = (uintptr_t)s->send_vmsa_packet_hdr;
    update.hdr_len = s->send_vmsa_packet_hdr_len;
    update.trans_uaddr = (uintptr_t)trans;
    update.trans_len = size;

    trace_kvm_sev_send_update_vmsa(cpu_id, cpu_index, trans, size);

    ret = sev_ioctl(s->sev_fd, KVM_SEV_SEND_UPDATE_VMSA, &update, &fw_error);
    if (ret) {
        error_report("%s: SEND_UPDATE_VMSA ret=%d fw_error=%d '%s'",
                     __func__, ret, fw_error, fw_error_to_str(fw_error));
        goto err;
    }

    /*
     * Migration of vCPU's VMState according to the instance_id
     * (i.e. CPUState.cpu_index)
     */
    qemu_put_be32(f, sizeof(uint32_t));
    qemu_put_buffer(f, (uint8_t *)&cpu_index, sizeof(uint32_t));
    *bytes_sent += 4 + sizeof(uint32_t);

    qemu_put_be32(f, update.hdr_len);
    qemu_put_buffer(f, (uint8_t *)update.hdr_uaddr, update.hdr_len);
    *bytes_sent += 4 + update.hdr_len;

    qemu_put_be32(f, update.trans_len);
    qemu_put_buffer(f, (uint8_t *)update.trans_uaddr, update.trans_len);
    *bytes_sent += 4 + update.trans_len;

err:
    g_free(trans);
    return ret;
}

int csv_save_outgoing_cpu_state(QEMUFile *f, uint64_t *bytes_sent)
{
    SevGuestState *s = sev_guest;
    CPUState *cpu;
    int ret = 0;

    /* Only support migrate VMSAs for HYGON CSV2 guest */
    if (!sev_es_enabled() || !is_hygon_cpu()) {
        return 0;
    }

    CPU_FOREACH(cpu) {
        qemu_put_be32(f, ENCRYPTED_CPU_STATE_CONT);
        *bytes_sent += 4;
        ret = sev_send_update_vmsa(s, f, kvm_arch_vcpu_id(cpu),
                                   cpu->cpu_index, TARGET_PAGE_SIZE, bytes_sent);
        if (ret) {
            goto err;
        }
    }

    qemu_put_be32(f, ENCRYPTED_CPU_STATE_END);
    *bytes_sent += 4;

err:
    return ret;
}

static int sev_receive_update_vmsa(QEMUFile *f)
{
    int ret = 1, fw_error = 0;
    CPUState *cpu;
    uint32_t cpu_index, cpu_id = 0;
    gchar *hdr = NULL, *trans = NULL;
    struct kvm_sev_receive_update_vmsa update = {};

    /* get cpu index buffer */
    assert(qemu_get_be32(f) == sizeof(uint32_t));
    qemu_get_buffer(f, (uint8_t *)&cpu_index, sizeof(uint32_t));

    CPU_FOREACH(cpu) {
        if (cpu->cpu_index == cpu_index) {
            cpu_id = kvm_arch_vcpu_id(cpu);
            break;
        }
    }
    update.vcpu_id = cpu_id;

    /* get packet header */
    update.hdr_len = qemu_get_be32(f);
    if (!check_blob_length(update.hdr_len)) {
        return 1;
    }

    hdr = g_new(gchar, update.hdr_len);
    qemu_get_buffer(f, (uint8_t *)hdr, update.hdr_len);
    update.hdr_uaddr = (uintptr_t)hdr;

    /* get transport buffer */
    update.trans_len = qemu_get_be32(f);
    if (!check_blob_length(update.trans_len)) {
        goto err;
    }

    trans = g_new(gchar, update.trans_len);
    update.trans_uaddr = (uintptr_t)trans;
    qemu_get_buffer(f, (uint8_t *)update.trans_uaddr, update.trans_len);

    trace_kvm_sev_receive_update_vmsa(cpu_id, cpu_index,
                trans, update.trans_len, hdr, update.hdr_len);

    ret = sev_ioctl(sev_guest->sev_fd, KVM_SEV_RECEIVE_UPDATE_VMSA,
                    &update, &fw_error);
    if (ret) {
        error_report("Error RECEIVE_UPDATE_VMSA ret=%d fw_error=%d '%s'",
                     ret, fw_error, fw_error_to_str(fw_error));
    }

err:
    g_free(trans);
    g_free(hdr);
    return ret;
}

int csv_load_incoming_cpu_state(QEMUFile *f)
{
    int status, ret = 0;

    /* Only support migrate VMSAs for HYGON CSV2 guest */
    if (!sev_es_enabled() || !is_hygon_cpu()) {
        return 0;
    }

    status = qemu_get_be32(f);
    while (status == ENCRYPTED_CPU_STATE_CONT) {
        ret = sev_receive_update_vmsa(f);
        if (ret) {
            break;
        }

        status = qemu_get_be32(f);
    }

    return ret;
}

static const QemuUUID sev_hash_table_header_guid = {
    .data = UUID_LE(0x9438d606, 0x4f22, 0x4cc9, 0xb4, 0x79, 0xa7, 0x93,
                    0xd4, 0x11, 0xfd, 0x21)
};

static const QemuUUID sev_kernel_entry_guid = {
    .data = UUID_LE(0x4de79437, 0xabd2, 0x427f, 0xb8, 0x35, 0xd5, 0xb1,
                    0x72, 0xd2, 0x04, 0x5b)
};
static const QemuUUID sev_initrd_entry_guid = {
    .data = UUID_LE(0x44baf731, 0x3a2f, 0x4bd7, 0x9a, 0xf1, 0x41, 0xe2,
                    0x91, 0x69, 0x78, 0x1d)
};
static const QemuUUID sev_cmdline_entry_guid = {
    .data = UUID_LE(0x97d02dd8, 0xbd20, 0x4c94, 0xaa, 0x78, 0xe7, 0x71,
                    0x4d, 0x36, 0xab, 0x2a)
};

/*
 * Add the hashes of the linux kernel/initrd/cmdline to an encrypted guest page
 * which is included in SEV's initial memory measurement.
 */
bool sev_add_kernel_loader_hashes(SevKernelLoaderContext *ctx, Error **errp)
{
    uint8_t *data;
    SevHashTableDescriptor *area;
    SevHashTable *ht;
    PaddedSevHashTable *padded_ht;
    uint8_t cmdline_hash[HASH_SIZE];
    uint8_t initrd_hash[HASH_SIZE];
    uint8_t kernel_hash[HASH_SIZE];
    uint8_t *hashp;
    size_t hash_len = HASH_SIZE;
    hwaddr mapped_len = sizeof(*padded_ht);
    MemTxAttrs attrs = { 0 };
    bool ret = true;

    /*
     * Only add the kernel hashes if the sev-guest configuration explicitly
     * stated kernel-hashes=on.
     */
    if (!sev_guest->kernel_hashes) {
        return false;
    }

    if (!pc_system_ovmf_table_find(SEV_HASH_TABLE_RV_GUID, &data, NULL)) {
        error_setg(errp, "SEV: kernel specified but guest firmware "
                         "has no hashes table GUID");
        return false;
    }
    area = (SevHashTableDescriptor *)data;
    if (!area->base || area->size < sizeof(PaddedSevHashTable)) {
        error_setg(errp, "SEV: guest firmware hashes table area is invalid "
                         "(base=0x%x size=0x%x)", area->base, area->size);
        return false;
    }

    /*
     * Calculate hash of kernel command-line with the terminating null byte. If
     * the user doesn't supply a command-line via -append, the 1-byte "\0" will
     * be used.
     */
    hashp = cmdline_hash;
    if (qcrypto_hash_bytes(QCRYPTO_HASH_ALG_SHA256, ctx->cmdline_data,
                           ctx->cmdline_size, &hashp, &hash_len, errp) < 0) {
        return false;
    }
    assert(hash_len == HASH_SIZE);

    /*
     * Calculate hash of initrd. If the user doesn't supply an initrd via
     * -initrd, an empty buffer will be used (ctx->initrd_size == 0).
     */
    hashp = initrd_hash;
    if (qcrypto_hash_bytes(QCRYPTO_HASH_ALG_SHA256, ctx->initrd_data,
                           ctx->initrd_size, &hashp, &hash_len, errp) < 0) {
        return false;
    }
    assert(hash_len == HASH_SIZE);

    /* Calculate hash of the kernel */
    hashp = kernel_hash;
    struct iovec iov[2] = {
        { .iov_base = ctx->setup_data, .iov_len = ctx->setup_size },
        { .iov_base = ctx->kernel_data, .iov_len = ctx->kernel_size }
    };
    if (qcrypto_hash_bytesv(QCRYPTO_HASH_ALG_SHA256, iov, ARRAY_SIZE(iov),
                            &hashp, &hash_len, errp) < 0) {
        return false;
    }
    assert(hash_len == HASH_SIZE);

    /*
     * Populate the hashes table in the guest's memory at the OVMF-designated
     * area for the SEV hashes table
     */
    padded_ht = address_space_map(&address_space_memory, area->base,
                                  &mapped_len, true, attrs);
    if (!padded_ht || mapped_len != sizeof(*padded_ht)) {
        error_setg(errp, "SEV: cannot map hashes table guest memory area");
        return false;
    }
    ht = &padded_ht->ht;

    ht->guid = sev_hash_table_header_guid;
    ht->len = sizeof(*ht);

    ht->cmdline.guid = sev_cmdline_entry_guid;
    ht->cmdline.len = sizeof(ht->cmdline);
    memcpy(ht->cmdline.hash, cmdline_hash, sizeof(ht->cmdline.hash));

    ht->initrd.guid = sev_initrd_entry_guid;
    ht->initrd.len = sizeof(ht->initrd);
    memcpy(ht->initrd.hash, initrd_hash, sizeof(ht->initrd.hash));

    ht->kernel.guid = sev_kernel_entry_guid;
    ht->kernel.len = sizeof(ht->kernel);
    memcpy(ht->kernel.hash, kernel_hash, sizeof(ht->kernel.hash));

    /* zero the excess data so the measurement can be reliably calculated */
    memset(padded_ht->padding, 0, sizeof(padded_ht->padding));

    if (sev_encrypt_flash((uint8_t *)padded_ht, sizeof(*padded_ht), errp) < 0) {
        ret = false;
    }

    address_space_unmap(&address_space_memory, padded_ht,
                        mapped_len, true, mapped_len);

    return ret;
}

static int _sev_send_start(QEMUFile *f, uint64_t *bytes_sent)
{
    SevGuestState *s = sev_guest;

    return sev_send_start(s, f, bytes_sent);
}

static int _sev_receive_start(QEMUFile *f)
{
    SevGuestState *s = sev_guest;

    return sev_receive_start(s, f);
}

struct sev_ops sev_ops = {
    .sev_ioctl = sev_ioctl,
    .fw_error_to_str = fw_error_to_str,
    .sev_send_start = _sev_send_start,
    .sev_receive_start = _sev_receive_start,
};

static void
sev_register_types(void)
{
    type_register_static(&sev_guest_info);
}

type_init(sev_register_types);
