/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.scanner.bootstrap;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;
import javax.annotation.CheckForNull;
import org.apache.commons.lang.StringUtils;
import org.sonar.api.notifications.AnalysisWarnings;
import org.sonar.api.utils.MessageException;
import org.sonar.api.utils.Preconditions;
import org.sonar.api.utils.log.Logger;
import org.sonar.api.utils.log.Loggers;
import org.sonar.api.utils.log.Profiler;
import org.sonar.scanner.bootstrap.GlobalAnalysisMode;
import org.sonar.scanner.bootstrap.ScannerWsClient;
import org.sonarqube.ws.client.HttpException;
import org.sonarqube.ws.client.WsClient;
import org.sonarqube.ws.client.WsConnector;
import org.sonarqube.ws.client.WsRequest;
import org.sonarqube.ws.client.WsResponse;

public class DefaultScannerWsClient
implements ScannerWsClient {
    private static final int MAX_ERROR_MSG_LEN = 128;
    private static final String SQ_TOKEN_EXPIRATION_HEADER = "SonarQube-Authentication-Token-Expiration";
    private static final DateTimeFormatter USER_FRIENDLY_DATETIME_FORMAT = DateTimeFormatter.ofPattern("MMMM dd, yyyy");
    private static final Logger LOG = Loggers.get(DefaultScannerWsClient.class);
    private final Set<String> warningMessages = new HashSet<String>();
    private final WsClient target;
    private final boolean hasCredentials;
    private final GlobalAnalysisMode globalMode;
    private final AnalysisWarnings analysisWarnings;

    public DefaultScannerWsClient(WsClient target, boolean hasCredentials, GlobalAnalysisMode globalMode, AnalysisWarnings analysisWarnings) {
        this.target = target;
        this.hasCredentials = hasCredentials;
        this.globalMode = globalMode;
        this.analysisWarnings = analysisWarnings;
    }

    @Override
    public WsResponse call(WsRequest request) {
        Preconditions.checkState(!this.globalMode.isMediumTest(), "No WS call should be made in medium test mode");
        Profiler profiler = Profiler.createIfDebug(LOG).start();
        WsResponse response = this.target.wsConnector().call(request);
        profiler.stopDebug(String.format("%s %d %s", new Object[]{request.getMethod(), response.code(), response.requestUrl()}));
        this.failIfUnauthorized(response);
        this.checkAuthenticationWarnings(response);
        return response;
    }

    @Override
    public String baseUrl() {
        return this.target.wsConnector().baseUrl();
    }

    WsConnector wsConnector() {
        return this.target.wsConnector();
    }

    private void failIfUnauthorized(WsResponse response) {
        String jsonMsg;
        int code = response.code();
        if (code == 401) {
            response.close();
            if (this.hasCredentials) {
                throw MessageException.of(String.format("Not authorized. Please check the properties %s and %s.", "sonar.login", "sonar.password"));
            }
            throw MessageException.of(String.format("Not authorized. Analyzing this project requires authentication. Please provide a user token in %s or other credentials in %s and %s.", "sonar.login", "sonar.login", "sonar.password"));
        }
        if (code == 403) {
            throw MessageException.of("You're not authorized to run analysis. Please contact the project administrator.");
        }
        if (code == 400 && (jsonMsg = DefaultScannerWsClient.tryParseAsJsonError(response.content())) != null) {
            throw MessageException.of(jsonMsg);
        }
        response.failIfNotSuccessful();
    }

    private void checkAuthenticationWarnings(WsResponse response) {
        if (response.code() == 200) {
            response.header(SQ_TOKEN_EXPIRATION_HEADER).ifPresent(expirationDate -> {
                ZonedDateTime datetimeInUTC = ZonedDateTime.from(DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ssZ").parse((CharSequence)expirationDate)).withZoneSameInstant(ZoneOffset.UTC);
                if (DefaultScannerWsClient.isTokenExpiringInOneWeek(datetimeInUTC)) {
                    this.addAnalysisWarning(datetimeInUTC);
                }
            });
        }
    }

    private static boolean isTokenExpiringInOneWeek(ZonedDateTime expirationDate) {
        ZonedDateTime localDateTime = ZonedDateTime.now(ZoneOffset.UTC);
        ZonedDateTime headerDateTime = expirationDate.minusDays(7L);
        return localDateTime.isAfter(headerDateTime);
    }

    private void addAnalysisWarning(ZonedDateTime tokenExpirationDate) {
        String warningMessage = "The token used for this analysis will expire on: " + tokenExpirationDate.format(USER_FRIENDLY_DATETIME_FORMAT);
        if (!this.warningMessages.contains(warningMessage)) {
            this.warningMessages.add(warningMessage);
            LOG.warn(warningMessage);
            LOG.warn("Analysis executed with this token will fail after the expiration date.");
        }
        this.analysisWarnings.addUnique(warningMessage + "\nAfter this date, the token can no longer be used to execute the analysis. Please consider generating a new token and updating it in the locations where it is in use.");
    }

    public static String createErrorMessage(HttpException exception) {
        String json = DefaultScannerWsClient.tryParseAsJsonError(exception.content());
        if (json != null) {
            return json;
        }
        String msg = "HTTP code " + exception.code();
        if (DefaultScannerWsClient.isHtml(exception.content())) {
            return msg;
        }
        return msg + ": " + StringUtils.left(exception.content(), 128);
    }

    @CheckForNull
    private static String tryParseAsJsonError(String responseContent) {
        try {
            JsonParser parser = new JsonParser();
            JsonObject obj = parser.parse(responseContent).getAsJsonObject();
            JsonArray errors = obj.getAsJsonArray("errors");
            ArrayList<String> errorMessages = new ArrayList<String>();
            for (JsonElement e : errors) {
                errorMessages.add(e.getAsJsonObject().get("msg").getAsString());
            }
            return String.join((CharSequence)", ", errorMessages);
        }
        catch (Exception e) {
            return null;
        }
    }

    private static boolean isHtml(String responseContent) {
        return StringUtils.stripToEmpty(responseContent).startsWith("<!DOCTYPE html>");
    }
}

