/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.scm.git;

import java.io.IOException;
import java.nio.file.Path;
import java.time.Instant;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.lang.math.NumberUtils;
import org.sonar.api.batch.scm.BlameLine;
import org.sonar.api.utils.Preconditions;
import org.sonar.api.utils.System2;
import org.sonar.api.utils.Version;
import org.sonar.api.utils.log.Logger;
import org.sonar.api.utils.log.Loggers;
import org.sonar.scm.git.ProcessWrapperFactory;
import org.springframework.beans.factory.annotation.Autowired;

public class GitBlameCommand {
    protected static final String BLAME_COMMAND = "blame";
    protected static final String GIT_DIR_FLAG = "--git-dir";
    protected static final String GIT_DIR_ARGUMENT = "%s/.git";
    protected static final String GIT_DIR_FORCE_FLAG = "-C";
    private static final Logger LOG = Loggers.get(GitBlameCommand.class);
    private static final Pattern EMAIL_PATTERN = Pattern.compile("<(.*?)>");
    private static final String COMMITTER_TIME = "committer-time ";
    private static final String AUTHOR_MAIL = "author-mail ";
    private static final String MINIMUM_REQUIRED_GIT_VERSION = "2.24.0";
    private static final String DEFAULT_GIT_COMMAND = "git";
    private static final String BLAME_LINE_PORCELAIN_FLAG = "--line-porcelain";
    private static final String END_OF_OPTIONS_FLAG = "--end-of-options";
    private static final String IGNORE_WHITESPACES = "-w";
    private static final Pattern whitespaceRegex = Pattern.compile("\\s+");
    private static final Pattern semanticVersionDelimiter = Pattern.compile("\\.");
    private final System2 system;
    private final ProcessWrapperFactory processWrapperFactory;
    private String gitCommand;

    @Autowired
    public GitBlameCommand(System2 system, ProcessWrapperFactory processWrapperFactory) {
        this.system = system;
        this.processWrapperFactory = processWrapperFactory;
    }

    GitBlameCommand(String gitCommand, System2 system, ProcessWrapperFactory processWrapperFactory) {
        this.gitCommand = gitCommand;
        this.system = system;
        this.processWrapperFactory = processWrapperFactory;
    }

    public boolean checkIfEnabled() {
        try {
            this.gitCommand = this.locateDefaultGit();
            MutableString stdOut = new MutableString();
            this.processWrapperFactory.create(null, l -> {
                stdOut.string = l;
            }, this.gitCommand, "--version").execute();
            return stdOut.string != null && stdOut.string.startsWith("git version") && GitBlameCommand.isCompatibleGitVersion(stdOut.string);
        }
        catch (Exception e) {
            LOG.debug("Failed to find git native client", (Object)e);
            return false;
        }
    }

    private String locateDefaultGit() throws IOException {
        if (this.gitCommand != null) {
            return this.gitCommand;
        }
        if (this.system.isOsWindows()) {
            return this.locateGitOnWindows();
        }
        return DEFAULT_GIT_COMMAND;
    }

    private String locateGitOnWindows() throws IOException {
        LOG.debug("Looking for git command in the PATH using where.exe (Windows)");
        LinkedList whereCommandResult = new LinkedList();
        this.processWrapperFactory.create(null, whereCommandResult::add, "C:\\Windows\\System32\\where.exe", "$PATH:git.exe").execute();
        if (!whereCommandResult.isEmpty()) {
            String out = ((String)whereCommandResult.get(0)).trim();
            LOG.debug("Found git.exe at {}", (Object)out);
            return out;
        }
        throw new IllegalStateException("git.exe not found in PATH. PATH value was: " + this.system.property("PATH"));
    }

    public List<BlameLine> blame(Path baseDir, String fileName) throws Exception {
        BlameOutputProcessor outputProcessor = new BlameOutputProcessor();
        try {
            this.processWrapperFactory.create(baseDir, outputProcessor::process, this.gitCommand, GIT_DIR_FLAG, String.format(GIT_DIR_ARGUMENT, baseDir), GIT_DIR_FORCE_FLAG, baseDir.toString(), BLAME_COMMAND, BLAME_LINE_PORCELAIN_FLAG, IGNORE_WHITESPACES, END_OF_OPTIONS_FLAG, fileName).execute();
        }
        catch (UncommittedLineException e) {
            LOG.debug("Unable to blame file '{}' - it has uncommitted changes", (Object)fileName);
            return Collections.emptyList();
        }
        return outputProcessor.getBlameLines();
    }

    private static boolean isCompatibleGitVersion(String gitVersionCommandOutput) {
        String gitVersion = whitespaceRegex.splitAsStream(gitVersionCommandOutput).skip(2L).findFirst().orElse("");
        String formattedGitVersion = GitBlameCommand.formatGitSemanticVersion(gitVersion);
        return Version.parse(formattedGitVersion).isGreaterThanOrEqual(Version.parse(MINIMUM_REQUIRED_GIT_VERSION));
    }

    private static String formatGitSemanticVersion(String version) {
        return semanticVersionDelimiter.splitAsStream(version).takeWhile(NumberUtils::isNumber).collect(Collectors.joining("."));
    }

    private static class MutableString {
        String string;

        private MutableString() {
        }
    }

    private static class BlameOutputProcessor {
        private final List<BlameLine> blameLines = new LinkedList<BlameLine>();
        private String sha1 = null;
        private String committerTime = null;
        private String authorMail = null;

        private BlameOutputProcessor() {
        }

        public List<BlameLine> getBlameLines() {
            return this.blameLines;
        }

        public void process(String line) {
            if (this.sha1 == null) {
                this.sha1 = line.split(" ")[0];
            } else if (line.startsWith("\t")) {
                this.saveEntry();
            } else if (line.startsWith(GitBlameCommand.COMMITTER_TIME)) {
                this.committerTime = line.substring(GitBlameCommand.COMMITTER_TIME.length());
            } else if (line.startsWith(GitBlameCommand.AUTHOR_MAIL)) {
                Matcher matcher = EMAIL_PATTERN.matcher(line);
                if (matcher.find(GitBlameCommand.AUTHOR_MAIL.length())) {
                    this.authorMail = matcher.group(1);
                }
                if (this.authorMail.equals("not.committed.yet")) {
                    throw new UncommittedLineException();
                }
            }
        }

        private void saveEntry() {
            Preconditions.checkState(this.authorMail != null, "Did not find an author email for an entry");
            Preconditions.checkState(this.committerTime != null, "Did not find a committer time for an entry");
            Preconditions.checkState(this.sha1 != null, "Did not find a commit sha1 for an entry");
            try {
                this.blameLines.add(new BlameLine().revision(this.sha1).author(this.authorMail).date(Date.from(Instant.ofEpochSecond(Long.parseLong(this.committerTime)))));
            }
            catch (NumberFormatException e) {
                throw new IllegalStateException("Invalid committer time found: " + this.committerTime);
            }
            this.authorMail = null;
            this.sha1 = null;
            this.committerTime = null;
        }
    }

    private static class UncommittedLineException
    extends RuntimeException {
        private UncommittedLineException() {
        }
    }
}

