use std::sync::LazyLock;

pub static OID_RSA_ENCRYPTION: LazyLock<ObjectIdentifier> =
    LazyLock::new(|| ObjectIdentifier::from_slice(&[1, 2, 840, 113549, 1, 1, 1]));

pub static OID_RSASSA_PSS: LazyLock<ObjectIdentifier> =
    LazyLock::new(|| ObjectIdentifier::from_slice(&[1, 2, 840, 113549, 1, 1, 10]));

pub static OID_SHA1: LazyLock<ObjectIdentifier> =
    LazyLock::new(|| ObjectIdentifier::from_slice(&[1, 3, 14, 3, 2, 26]));

pub static OID_SHA256: LazyLock<ObjectIdentifier> =
    LazyLock::new(|| ObjectIdentifier::from_slice(&[2, 16, 840, 1, 101, 3, 4, 2, 1]));

pub static OID_SHA384: LazyLock<ObjectIdentifier> =
    LazyLock::new(|| ObjectIdentifier::from_slice(&[2, 16, 840, 1, 101, 3, 4, 2, 2]));

pub static OID_SHA512: LazyLock<ObjectIdentifier> =
    LazyLock::new(|| ObjectIdentifier::from_slice(&[2, 16, 840, 1, 101, 3, 4, 2, 3]));

pub static OID_MGF1: LazyLock<ObjectIdentifier> =
    LazyLock::new(|| ObjectIdentifier::from_slice(&[1, 2, 840, 113549, 1, 1, 8]));

pub static OID_ID_EC_PUBLIC_KEY: LazyLock<ObjectIdentifier> =
    LazyLock::new(|| ObjectIdentifier::from_slice(&[1, 2, 840, 10045, 2, 1]));

pub static OID_PRIME256V1: LazyLock<ObjectIdentifier> =
    LazyLock::new(|| ObjectIdentifier::from_slice(&[1, 2, 840, 10045, 3, 1, 7]));

pub static OID_SECP384R1: LazyLock<ObjectIdentifier> =
    LazyLock::new(|| ObjectIdentifier::from_slice(&[1, 3, 132, 0, 34]));

pub static OID_SECP521R1: LazyLock<ObjectIdentifier> =
    LazyLock::new(|| ObjectIdentifier::from_slice(&[1, 3, 132, 0, 35]));

pub static OID_SECP256K1: LazyLock<ObjectIdentifier> =
    LazyLock::new(|| ObjectIdentifier::from_slice(&[1, 3, 132, 0, 10]));

pub static OID_ED25519: LazyLock<ObjectIdentifier> =
    LazyLock::new(|| ObjectIdentifier::from_slice(&[1, 3, 101, 112]));

pub static OID_ED448: LazyLock<ObjectIdentifier> =
    LazyLock::new(|| ObjectIdentifier::from_slice(&[1, 3, 101, 113]));

pub static OID_X25519: LazyLock<ObjectIdentifier> =
    LazyLock::new(|| ObjectIdentifier::from_slice(&[1, 3, 101, 110]));

pub static OID_X448: LazyLock<ObjectIdentifier> =
    LazyLock::new(|| ObjectIdentifier::from_slice(&[1, 3, 101, 111]));

#[derive(Debug, Eq, PartialEq)]
pub struct ObjectIdentifier {
    values: Vec<u64>,
}

impl ObjectIdentifier {
    pub fn from_slice(values: &[u64]) -> Self {
        ObjectIdentifier {
            values: values.to_vec(),
        }
    }
}

impl<'a> IntoIterator for &'a ObjectIdentifier {
    type Item = &'a u64;
    type IntoIter = std::slice::Iter<'a, u64>;

    fn into_iter(self) -> Self::IntoIter {
        self.values.iter()
    }
}

impl PartialEq<&str> for ObjectIdentifier {
    fn eq(&self, other: &&str) -> bool {
        let mut vec: Vec<u64> = Vec::new();
        for val in other.split(".") {
            match val.parse() {
                Ok(nval) => vec.push(nval),
                Err(_) => return false,
            }
        }
        vec == self.values
    }
}

impl std::fmt::Display for ObjectIdentifier {
    fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
        write!(
            f,
            "{}",
            self.values
                .iter()
                .map(|val| val.to_string())
                .collect::<Vec<String>>()
                .join(".")
        )
    }
}
