/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.server.ce.queue;

import com.google.common.base.Preconditions;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Map;
import java.util.Optional;
import javax.annotation.Nullable;
import org.apache.commons.lang.StringUtils;
import org.sonar.api.server.ServerSide;
import org.sonar.ce.queue.CeQueue;
import org.sonar.ce.queue.CeTaskSubmit;
import org.sonar.ce.task.CeTask;
import org.sonar.db.DbClient;
import org.sonar.db.DbSession;
import org.sonar.db.component.BranchDto;
import org.sonar.db.component.ComponentDto;
import org.sonar.db.organization.OrganizationDto;
import org.sonar.db.permission.OrganizationPermission;
import org.sonar.server.ce.queue.BranchSupport;
import org.sonar.server.component.ComponentUpdater;
import org.sonar.server.component.NewComponent;
import org.sonar.server.exceptions.BadRequestException;
import org.sonar.server.exceptions.NotFoundException;
import org.sonar.server.permission.PermissionTemplateService;
import org.sonar.server.user.AbstractUserSession;
import org.sonar.server.user.UserSession;

@ServerSide
public class ReportSubmitter {
    private final CeQueue queue;
    private final UserSession userSession;
    private final ComponentUpdater componentUpdater;
    private final PermissionTemplateService permissionTemplateService;
    private final DbClient dbClient;
    private final BranchSupport branchSupport;

    public ReportSubmitter(CeQueue queue, UserSession userSession, ComponentUpdater componentUpdater, PermissionTemplateService permissionTemplateService, DbClient dbClient, BranchSupport branchSupport) {
        this.queue = queue;
        this.userSession = userSession;
        this.componentUpdater = componentUpdater;
        this.permissionTemplateService = permissionTemplateService;
        this.dbClient = dbClient;
        this.branchSupport = branchSupport;
    }

    public CeTask submit(String organizationKey, String projectKey, @Nullable String deprecatedBranch, @Nullable String projectName, Map<String, String> characteristics, InputStream reportInput) {
        try (DbSession dbSession = this.dbClient.openSession(false);){
            OrganizationDto organizationDto = this.getOrganizationDtoOrFail(dbSession, organizationKey);
            BranchSupport.ComponentKey componentKey = this.branchSupport.createComponentKey(projectKey, deprecatedBranch, characteristics);
            Optional existingComponent = this.dbClient.componentDao().selectByKey(dbSession, componentKey.getDbKey());
            this.validateProject(dbSession, existingComponent, projectKey);
            ReportSubmitter.ensureOrganizationIsConsistent(existingComponent, organizationDto);
            ComponentDto component = existingComponent.orElseGet(() -> this.createComponent(dbSession, organizationDto, componentKey, projectName));
            this.checkScanPermission(component);
            CeTask ceTask = this.submitReport(dbSession, reportInput, component, characteristics);
            return ceTask;
        }
    }

    private void checkScanPermission(ComponentDto project) {
        if (!this.userSession.hasComponentPermission("scan", project) && !this.userSession.hasPermission(OrganizationPermission.SCAN, project.getOrganizationUuid())) {
            throw AbstractUserSession.insufficientPrivilegesException();
        }
    }

    private OrganizationDto getOrganizationDtoOrFail(DbSession dbSession, String organizationKey) {
        return (OrganizationDto)this.dbClient.organizationDao().selectByKey(dbSession, organizationKey).orElseThrow(() -> new NotFoundException(String.format("Organization with key '%s' does not exist", organizationKey)));
    }

    private void validateProject(DbSession dbSession, Optional<ComponentDto> project, String rawProjectKey) {
        ArrayList<String> errors = new ArrayList<String>();
        if (!project.isPresent()) {
            return;
        }
        ComponentDto component = project.get();
        if (!"TRK".equals(component.qualifier()) || !"PRJ".equals(component.scope())) {
            errors.add(String.format("Component '%s' is not a project", rawProjectKey));
        }
        if (!component.projectUuid().equals(component.uuid())) {
            ComponentDto anotherBaseProject = this.dbClient.componentDao().selectOrFailByUuid(dbSession, component.projectUuid());
            errors.add(String.format("The project '%s' is already defined in SonarQube but as a module of project '%s'. If you really want to stop directly analysing project '%s', please first delete it from SonarQube and then relaunch the analysis of project '%s'.", rawProjectKey, anotherBaseProject.getKey(), anotherBaseProject.getKey(), rawProjectKey));
        }
        if (!errors.isEmpty()) {
            throw BadRequestException.create(errors);
        }
    }

    private static void ensureOrganizationIsConsistent(Optional<ComponentDto> project, OrganizationDto organizationDto) {
        if (project.isPresent()) {
            Preconditions.checkArgument((boolean)project.get().getOrganizationUuid().equals(organizationDto.getUuid()), (String)"Organization of component with key '%s' does not match specified organization '%s'", (Object[])new Object[]{project.get().getDbKey(), organizationDto.getKey()});
        }
    }

    private ComponentDto createComponent(DbSession dbSession, OrganizationDto organization, BranchSupport.ComponentKey componentKey, @Nullable String projectName) {
        if (componentKey.isMainBranch() || componentKey.isDeprecatedBranch()) {
            ComponentDto project = this.createProject(dbSession, organization, componentKey, projectName);
            this.componentUpdater.commitAndIndex(dbSession, project);
            return project;
        }
        Optional existingMainComponent = this.dbClient.componentDao().selectByKey(dbSession, componentKey.getKey());
        ComponentDto mainComponentDto = existingMainComponent.orElseGet(() -> this.createProject(dbSession, organization, componentKey.getMainBranchComponentKey(), projectName));
        BranchDto mainComponentBranchDto = (BranchDto)this.dbClient.branchDao().selectByUuid(dbSession, mainComponentDto.uuid()).orElseThrow(() -> new IllegalStateException("Branch of main component does not exist"));
        ComponentDto branchComponent = this.branchSupport.createBranchComponent(dbSession, componentKey, organization, mainComponentDto, mainComponentBranchDto);
        if (existingMainComponent.isPresent()) {
            dbSession.commit();
        } else {
            this.componentUpdater.commitAndIndex(dbSession, mainComponentDto);
        }
        return branchComponent;
    }

    private ComponentDto createProject(DbSession dbSession, OrganizationDto organization, BranchSupport.ComponentKey componentKey, @Nullable String projectName) {
        this.userSession.checkPermission(OrganizationPermission.PROVISION_PROJECTS, organization);
        Integer userId = this.userSession.getUserId();
        boolean wouldCurrentUserHaveScanPermission = this.permissionTemplateService.wouldUserHaveScanPermissionWithDefaultTemplate(dbSession, organization.getUuid(), userId, componentKey.getDbKey());
        if (!wouldCurrentUserHaveScanPermission) {
            throw AbstractUserSession.insufficientPrivilegesException();
        }
        boolean newProjectPrivate = this.dbClient.organizationDao().getNewProjectPrivate(dbSession, organization);
        NewComponent newProject = NewComponent.newComponentBuilder().setOrganizationUuid(organization.getUuid()).setKey(componentKey.getKey()).setName(StringUtils.defaultIfBlank((String)projectName, (String)componentKey.getKey())).setDeprecatedBranch(componentKey.getDeprecatedBranchName().orElse(null)).setQualifier("TRK").setPrivate(newProjectPrivate).build();
        return this.componentUpdater.createWithoutCommit(dbSession, newProject, userId);
    }

    private CeTask submitReport(DbSession dbSession, InputStream reportInput, ComponentDto project, Map<String, String> characteristics) {
        CeTaskSubmit.Builder submit = this.queue.prepareSubmit();
        this.dbClient.ceTaskInputDao().insert(dbSession, submit.getUuid(), reportInput);
        dbSession.commit();
        submit.setType("REPORT");
        submit.setComponent(CeTaskSubmit.Component.fromDto((ComponentDto)project));
        submit.setSubmitterUuid(this.userSession.getUuid());
        submit.setCharacteristics(characteristics);
        return this.queue.submit(submit.build());
    }
}

