/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.server.rule;

import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import javax.annotation.CheckForNull;
import javax.annotation.Nullable;
import org.sonar.api.rule.RuleKey;
import org.sonar.api.rule.RuleStatus;
import org.sonar.api.rule.Severity;
import org.sonar.api.server.ServerSide;
import org.sonar.api.server.rule.RuleParamType;
import org.sonar.api.utils.System2;
import org.sonar.db.DbClient;
import org.sonar.db.DbSession;
import org.sonar.db.organization.OrganizationDto;
import org.sonar.db.rule.RuleDefinitionDto;
import org.sonar.db.rule.RuleDto;
import org.sonar.db.rule.RuleMetadataDto;
import org.sonar.db.rule.RuleParamDto;
import org.sonar.server.exceptions.BadRequestException;
import org.sonar.server.organization.DefaultOrganizationProvider;
import org.sonar.server.rule.NewCustomRule;
import org.sonar.server.rule.ReactivationException;
import org.sonar.server.rule.index.RuleIndexer;
import org.sonar.server.util.TypeValidations;
import org.sonar.server.ws.WsUtils;

@ServerSide
public class RuleCreator {
    private final System2 system2;
    private final RuleIndexer ruleIndexer;
    private final DbClient dbClient;
    private final TypeValidations typeValidations;
    private final DefaultOrganizationProvider defaultOrganizationProvider;

    public RuleCreator(System2 system2, RuleIndexer ruleIndexer, DbClient dbClient, TypeValidations typeValidations, DefaultOrganizationProvider defaultOrganizationProvider) {
        this.system2 = system2;
        this.ruleIndexer = ruleIndexer;
        this.dbClient = dbClient;
        this.typeValidations = typeValidations;
        this.defaultOrganizationProvider = defaultOrganizationProvider;
    }

    public RuleKey create(DbSession dbSession, NewCustomRule newRule) {
        RuleKey templateKey = newRule.templateKey();
        Preconditions.checkArgument((templateKey != null ? 1 : 0) != 0, (Object)"Rule template key should not be null");
        String defaultOrganizationUuid = this.defaultOrganizationProvider.get().getUuid();
        OrganizationDto defaultOrganization = (OrganizationDto)this.dbClient.organizationDao().selectByUuid(dbSession, defaultOrganizationUuid).orElseThrow(() -> new IllegalStateException(String.format("Could not find default organization for uuid '%s'", defaultOrganizationUuid)));
        RuleDto templateRule = (RuleDto)this.dbClient.ruleDao().selectByKey(dbSession, defaultOrganization.getUuid(), templateKey).orElseThrow(() -> new IllegalArgumentException(String.format("The template key doesn't exist: %s", templateKey)));
        Preconditions.checkArgument((boolean)templateRule.isTemplate(), (String)"This rule is not a template rule: %s", (Object[])new Object[]{templateKey.toString()});
        Preconditions.checkArgument((templateRule.getStatus() != RuleStatus.REMOVED ? 1 : 0) != 0, (String)"The template key doesn't exist: %s", (Object[])new Object[]{templateKey.toString()});
        this.validateCustomRule(newRule, dbSession, templateKey);
        RuleKey customRuleKey = RuleKey.of((String)templateRule.getRepositoryKey(), (String)newRule.ruleKey());
        Optional<RuleDefinitionDto> definition = this.loadRule(dbSession, customRuleKey);
        int customRuleId = definition.map(d -> this.updateExistingRule((RuleDefinitionDto)d, newRule, dbSession)).orElseGet(() -> this.createCustomRule(customRuleKey, newRule, templateRule, dbSession));
        this.ruleIndexer.commitAndIndex(dbSession, customRuleId);
        return customRuleKey;
    }

    private void validateCustomRule(NewCustomRule newRule, DbSession dbSession, RuleKey templateKey) {
        ArrayList<String> errors = new ArrayList<String>();
        RuleCreator.validateRuleKey(errors, newRule.ruleKey());
        RuleCreator.validateName(errors, newRule);
        RuleCreator.validateDescription(errors, newRule);
        String severity = newRule.severity();
        if (Strings.isNullOrEmpty((String)severity)) {
            errors.add("The severity is missing");
        } else if (!Severity.ALL.contains(severity)) {
            errors.add(String.format("Severity \"%s\" is invalid", severity));
        }
        if (newRule.status() == null) {
            errors.add("The status is missing");
        }
        for (RuleParamDto ruleParam : this.dbClient.ruleDao().selectRuleParamsByRuleKey(dbSession, templateKey)) {
            try {
                this.validateParam(ruleParam, newRule.parameter(ruleParam.getName()));
            }
            catch (BadRequestException validationError) {
                errors.addAll(validationError.errors());
            }
        }
        WsUtils.checkRequest(errors.isEmpty(), errors);
    }

    @CheckForNull
    private void validateParam(RuleParamDto ruleParam, @Nullable String value) {
        if (value != null) {
            RuleParamType ruleParamType = RuleParamType.parse((String)ruleParam.getType());
            if (ruleParamType.multiple()) {
                ArrayList values = Lists.newArrayList((Iterable)Splitter.on((String)",").split((CharSequence)value));
                this.typeValidations.validate(values, ruleParamType.type(), (List<String>)ruleParamType.values());
            } else {
                this.typeValidations.validate(value, ruleParamType.type(), (List<String>)ruleParamType.values());
            }
        }
    }

    private static void validateName(List<String> errors, NewCustomRule newRule) {
        if (Strings.isNullOrEmpty((String)newRule.name())) {
            errors.add("The name is missing");
        }
    }

    private static void validateDescription(List<String> errors, NewCustomRule newRule) {
        if (Strings.isNullOrEmpty((String)newRule.htmlDescription()) && Strings.isNullOrEmpty((String)newRule.markdownDescription())) {
            errors.add("The description is missing");
        }
    }

    private static void validateRuleKey(List<String> errors, String ruleKey) {
        if (!ruleKey.matches("^[\\w]+$")) {
            errors.add(String.format("The rule key \"%s\" is invalid, it should only contain: a-z, 0-9, \"_\"", ruleKey));
        }
    }

    private Optional<RuleDefinitionDto> loadRule(DbSession dbSession, RuleKey ruleKey) {
        return this.dbClient.ruleDao().selectDefinitionByKey(dbSession, ruleKey);
    }

    private int createCustomRule(RuleKey ruleKey, NewCustomRule newRule, RuleDto templateRuleDto, DbSession dbSession) {
        RuleDefinitionDto ruleDefinition = new RuleDefinitionDto().setRuleKey(ruleKey).setPluginKey(templateRuleDto.getPluginKey()).setTemplateId(templateRuleDto.getId()).setConfigKey(templateRuleDto.getConfigKey()).setName(newRule.name()).setDescription(newRule.markdownDescription()).setDescriptionFormat(RuleDto.Format.MARKDOWN).setSeverity(newRule.severity()).setStatus(newRule.status()).setType(newRule.type() == null ? templateRuleDto.getType() : newRule.type().getDbConstant()).setLanguage(templateRuleDto.getLanguage()).setDefRemediationFunction(templateRuleDto.getDefRemediationFunction()).setDefRemediationGapMultiplier(templateRuleDto.getDefRemediationGapMultiplier()).setDefRemediationBaseEffort(templateRuleDto.getDefRemediationBaseEffort()).setGapDescription(templateRuleDto.getGapDescription()).setScope(templateRuleDto.getScope()).setSystemTags(templateRuleDto.getSystemTags()).setSecurityStandards(templateRuleDto.getSecurityStandards()).setIsExternal(false).setIsAdHoc(false).setCreatedAt(this.system2.now()).setUpdatedAt(this.system2.now());
        this.dbClient.ruleDao().insert(dbSession, ruleDefinition);
        Set tags = templateRuleDto.getTags();
        if (!tags.isEmpty()) {
            RuleMetadataDto ruleMetadata = new RuleMetadataDto().setOrganizationUuid(this.defaultOrganizationProvider.get().getUuid()).setRuleId(ruleDefinition.getId().intValue()).setTags(tags).setCreatedAt(this.system2.now()).setUpdatedAt(this.system2.now());
            this.dbClient.ruleDao().insertOrUpdate(dbSession, ruleMetadata);
        }
        for (RuleParamDto templateRuleParamDto : this.dbClient.ruleDao().selectRuleParamsByRuleKey(dbSession, templateRuleDto.getKey())) {
            String customRuleParamValue = Strings.emptyToNull((String)newRule.parameter(templateRuleParamDto.getName()));
            this.createCustomRuleParams(customRuleParamValue, ruleDefinition, templateRuleParamDto, dbSession);
        }
        return ruleDefinition.getId();
    }

    private void createCustomRuleParams(@Nullable String paramValue, RuleDefinitionDto ruleDto, RuleParamDto templateRuleParam, DbSession dbSession) {
        RuleParamDto ruleParamDto = RuleParamDto.createFor((RuleDefinitionDto)ruleDto).setName(templateRuleParam.getName()).setType(templateRuleParam.getType()).setDescription(templateRuleParam.getDescription()).setDefaultValue(paramValue);
        this.dbClient.ruleDao().insertRuleParam(dbSession, ruleDto, ruleParamDto);
    }

    private int updateExistingRule(RuleDefinitionDto ruleDto, NewCustomRule newRule, DbSession dbSession) {
        if (ruleDto.getStatus().equals((Object)RuleStatus.REMOVED)) {
            if (newRule.isPreventReactivation()) {
                throw new ReactivationException(String.format("A removed rule with the key '%s' already exists", ruleDto.getKey().rule()), ruleDto.getKey());
            }
        } else {
            throw new IllegalArgumentException(String.format("A rule with the key '%s' already exists", ruleDto.getKey().rule()));
        }
        ruleDto.setStatus(RuleStatus.READY).setUpdatedAt(this.system2.now());
        this.dbClient.ruleDao().update(dbSession, ruleDto);
        return ruleDto.getId();
    }
}

