/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.server.rule;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import javax.annotation.Nonnull;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.builder.EqualsBuilder;
import org.sonar.api.rule.RuleStatus;
import org.sonar.api.rule.Severity;
import org.sonar.api.server.ServerSide;
import org.sonar.api.server.debt.DebtRemediationFunction;
import org.sonar.api.utils.System2;
import org.sonar.db.DbClient;
import org.sonar.db.DbSession;
import org.sonar.db.organization.OrganizationDto;
import org.sonar.db.qualityprofile.ActiveRuleDto;
import org.sonar.db.qualityprofile.ActiveRuleParamDto;
import org.sonar.db.qualityprofile.OrgActiveRuleDto;
import org.sonar.db.rule.RuleDto;
import org.sonar.db.rule.RuleParamDto;
import org.sonar.server.rule.RuleTagHelper;
import org.sonar.server.rule.RuleUpdate;
import org.sonar.server.rule.index.RuleIndexer;
import org.sonar.server.user.UserSession;

@ServerSide
public class RuleUpdater {
    private final DbClient dbClient;
    private final RuleIndexer ruleIndexer;
    private final System2 system;

    public RuleUpdater(DbClient dbClient, RuleIndexer ruleIndexer, System2 system) {
        this.dbClient = dbClient;
        this.ruleIndexer = ruleIndexer;
        this.system = system;
    }

    public boolean update(DbSession dbSession, RuleUpdate update, OrganizationDto organization, UserSession userSession) {
        if (update.isEmpty()) {
            return false;
        }
        RuleDto rule = this.getRuleDto(update);
        this.apply(update, rule, userSession);
        this.update(dbSession, rule);
        this.updateParameters(dbSession, organization, update, rule);
        this.ruleIndexer.commitAndIndex(dbSession, rule.getId().intValue(), organization);
        return true;
    }

    private RuleDto getRuleDto(RuleUpdate change) {
        try (DbSession dbSession = this.dbClient.openSession(false);){
            RuleDto rule = this.dbClient.ruleDao().selectOrFailByKey(dbSession, change.getOrganization(), change.getRuleKey());
            if (RuleStatus.REMOVED == rule.getStatus()) {
                throw new IllegalArgumentException("Rule with REMOVED status cannot be updated: " + change.getRuleKey());
            }
            RuleDto ruleDto = rule;
            return ruleDto;
        }
    }

    private void apply(RuleUpdate update, RuleDto rule, UserSession userSession) {
        if (update.isChangeName()) {
            RuleUpdater.updateName(update, rule);
        }
        if (update.isChangeDescription()) {
            RuleUpdater.updateDescription(update, rule);
        }
        if (update.isChangeSeverity()) {
            RuleUpdater.updateSeverity(update, rule);
        }
        if (update.isChangeStatus()) {
            RuleUpdater.updateStatus(update, rule);
        }
        if (update.isChangeMarkdownNote()) {
            this.updateMarkdownNote(update, rule, userSession);
        }
        if (update.isChangeTags()) {
            RuleUpdater.updateTags(update, rule);
        }
        if (update.isChangeDebtRemediationFunction()) {
            RuleUpdater.updateDebtRemediationFunction(update, rule);
        }
    }

    private static void updateName(RuleUpdate update, RuleDto rule) {
        String name = update.getName();
        if (Strings.isNullOrEmpty((String)name)) {
            throw new IllegalArgumentException("The name is missing");
        }
        rule.setName(name);
    }

    private static void updateDescription(RuleUpdate update, RuleDto rule) {
        String description = update.getMarkdownDescription();
        if (Strings.isNullOrEmpty((String)description)) {
            throw new IllegalArgumentException("The description is missing");
        }
        rule.setDescription(description);
        rule.setDescriptionFormat(RuleDto.Format.MARKDOWN);
    }

    private static void updateSeverity(RuleUpdate update, RuleDto rule) {
        String severity = update.getSeverity();
        if (Strings.isNullOrEmpty((String)severity) || !Severity.ALL.contains(severity)) {
            throw new IllegalArgumentException("The severity is invalid");
        }
        rule.setSeverity(severity);
    }

    private static void updateStatus(RuleUpdate update, RuleDto rule) {
        RuleStatus status = update.getStatus();
        if (status == null) {
            throw new IllegalArgumentException("The status is missing");
        }
        rule.setStatus(status);
    }

    private static void updateTags(RuleUpdate update, RuleDto rule) {
        Set<String> tags = update.getTags();
        if (tags == null || tags.isEmpty()) {
            rule.setTags(Collections.emptySet());
        } else {
            RuleTagHelper.applyTags(rule, tags);
        }
    }

    private static void updateDebtRemediationFunction(RuleUpdate update, RuleDto rule) {
        DebtRemediationFunction function = update.getDebtRemediationFunction();
        if (function == null) {
            rule.setRemediationFunction(null);
            rule.setRemediationGapMultiplier(null);
            rule.setRemediationBaseEffort(null);
        } else if (RuleUpdater.isSameAsDefaultFunction(function, rule)) {
            rule.setRemediationFunction(null);
            rule.setRemediationGapMultiplier(null);
            rule.setRemediationBaseEffort(null);
        } else {
            rule.setRemediationFunction(function.type().name());
            rule.setRemediationGapMultiplier(function.gapMultiplier());
            rule.setRemediationBaseEffort(function.baseEffort());
        }
    }

    private void updateMarkdownNote(RuleUpdate update, RuleDto rule, UserSession userSession) {
        if (StringUtils.isBlank((String)update.getMarkdownNote())) {
            rule.setNoteData(null);
            rule.setNoteCreatedAt(null);
            rule.setNoteUpdatedAt(null);
            rule.setNoteUserUuid(null);
        } else {
            long now = this.system.now();
            rule.setNoteData(update.getMarkdownNote());
            rule.setNoteCreatedAt(Long.valueOf(rule.getNoteCreatedAt() != null ? rule.getNoteCreatedAt() : now));
            rule.setNoteUpdatedAt(Long.valueOf(now));
            rule.setNoteUserUuid(userSession.getUuid());
        }
    }

    private static boolean isSameAsDefaultFunction(DebtRemediationFunction fn, RuleDto rule) {
        return new EqualsBuilder().append((Object)fn.type().name(), (Object)rule.getDefRemediationFunction()).append((Object)fn.gapMultiplier(), (Object)rule.getDefRemediationGapMultiplier()).append((Object)fn.baseEffort(), (Object)rule.getDefRemediationBaseEffort()).isEquals();
    }

    private void updateParameters(DbSession dbSession, OrganizationDto organization, RuleUpdate update, RuleDto rule) {
        if (update.isChangeParameters() && update.isCustomRule()) {
            RuleDto customRule = rule;
            Integer templateId = customRule.getTemplateId();
            Preconditions.checkNotNull((Object)templateId, (String)"Rule '%s' has no persisted template!", (Object[])new Object[]{customRule});
            Optional templateRule = this.dbClient.ruleDao().selectDefinitionById((long)templateId.intValue(), dbSession);
            if (!templateRule.isPresent()) {
                throw new IllegalStateException(String.format("Template %s of rule %s does not exist", customRule.getTemplateId(), customRule.getKey()));
            }
            ArrayList paramKeys = Lists.newArrayList();
            Multimap<ActiveRuleDto, ActiveRuleParamDto> activeRuleParamsByActiveRule = this.getActiveRuleParamsByActiveRule(dbSession, organization, customRule);
            this.deleteOrUpdateParameters(dbSession, update, customRule, paramKeys, activeRuleParamsByActiveRule);
        }
    }

    private Multimap<ActiveRuleDto, ActiveRuleParamDto> getActiveRuleParamsByActiveRule(DbSession dbSession, OrganizationDto organization, RuleDto customRule) {
        List activeRuleDtos = this.dbClient.activeRuleDao().selectByRuleId(dbSession, organization, customRule.getId().intValue());
        ImmutableMap activeRuleById = FluentIterable.from((Iterable)activeRuleDtos).uniqueIndex(ActiveRuleDto::getId);
        List activeRuleIds = Lists.transform((List)activeRuleDtos, ActiveRuleDto::getId);
        List activeRuleParamDtos = this.dbClient.activeRuleDao().selectParamsByActiveRuleIds(dbSession, activeRuleIds);
        return FluentIterable.from((Iterable)activeRuleParamDtos).index((Function)new ActiveRuleParamToActiveRule((Map)activeRuleById));
    }

    private void deleteOrUpdateParameters(DbSession dbSession, RuleUpdate update, RuleDto customRule, List<String> paramKeys, Multimap<ActiveRuleDto, ActiveRuleParamDto> activeRuleParamsByActiveRule) {
        for (RuleParamDto ruleParamDto : this.dbClient.ruleDao().selectRuleParamsByRuleKey(dbSession, update.getRuleKey())) {
            String key = ruleParamDto.getName();
            String value = Strings.emptyToNull((String)update.parameter(key));
            ruleParamDto.setDefaultValue(value);
            this.dbClient.ruleDao().updateRuleParam(dbSession, customRule.getDefinition(), ruleParamDto);
            if (value != null) {
                this.updateOrInsertActiveRuleParams(dbSession, ruleParamDto, activeRuleParamsByActiveRule);
            } else {
                this.deleteActiveRuleParams(dbSession, key, activeRuleParamsByActiveRule.values());
            }
            paramKeys.add(key);
        }
    }

    private void updateOrInsertActiveRuleParams(DbSession dbSession, RuleParamDto ruleParamDto, Multimap<ActiveRuleDto, ActiveRuleParamDto> activeRuleParamsByActiveRule) {
        activeRuleParamsByActiveRule.keySet().forEach(new UpdateOrInsertActiveRuleParams(dbSession, this.dbClient, ruleParamDto, activeRuleParamsByActiveRule));
    }

    private void deleteActiveRuleParams(DbSession dbSession, String key, Collection<ActiveRuleParamDto> activeRuleParamDtos) {
        activeRuleParamDtos.forEach(new DeleteActiveRuleParams(dbSession, this.dbClient, key));
    }

    private void update(DbSession session, RuleDto rule) {
        rule.setUpdatedAt(this.system.now());
        this.dbClient.ruleDao().update(session, rule.getDefinition());
        this.dbClient.ruleDao().insertOrUpdate(session, rule.getMetadata());
    }

    private static class DeleteActiveRuleParams
    implements Consumer<ActiveRuleParamDto> {
        private final DbSession dbSession;
        private final DbClient dbClient;
        private final String key;

        public DeleteActiveRuleParams(DbSession dbSession, DbClient dbClient, String key) {
            this.dbSession = dbSession;
            this.dbClient = dbClient;
            this.key = key;
        }

        @Override
        public void accept(@Nonnull ActiveRuleParamDto activeRuleParamDto) {
            if (activeRuleParamDto.getKey().equals(this.key)) {
                this.dbClient.activeRuleDao().deleteParamById(this.dbSession, activeRuleParamDto.getId().intValue());
            }
        }
    }

    private static class UpdateOrInsertActiveRuleParams
    implements Consumer<ActiveRuleDto> {
        private final DbSession dbSession;
        private final DbClient dbClient;
        private final RuleParamDto ruleParamDto;
        private final Multimap<ActiveRuleDto, ActiveRuleParamDto> activeRuleParams;

        private UpdateOrInsertActiveRuleParams(DbSession dbSession, DbClient dbClient, RuleParamDto ruleParamDto, Multimap<ActiveRuleDto, ActiveRuleParamDto> activeRuleParams) {
            this.dbSession = dbSession;
            this.dbClient = dbClient;
            this.ruleParamDto = ruleParamDto;
            this.activeRuleParams = activeRuleParams;
        }

        @Override
        public void accept(@Nonnull ActiveRuleDto activeRuleDto) {
            ImmutableMap activeRuleParamByKey = FluentIterable.from((Iterable)this.activeRuleParams.get((Object)activeRuleDto)).uniqueIndex(ActiveRuleParamDto::getKey);
            ActiveRuleParamDto activeRuleParamDto = (ActiveRuleParamDto)activeRuleParamByKey.get(this.ruleParamDto.getName());
            if (activeRuleParamDto != null) {
                this.dbClient.activeRuleDao().updateParam(this.dbSession, activeRuleParamDto.setValue(this.ruleParamDto.getDefaultValue()));
            } else {
                this.dbClient.activeRuleDao().insertParam(this.dbSession, activeRuleDto, ActiveRuleParamDto.createFor((RuleParamDto)this.ruleParamDto).setValue(this.ruleParamDto.getDefaultValue()));
            }
        }
    }

    private static class ActiveRuleParamToActiveRule
    implements Function<ActiveRuleParamDto, ActiveRuleDto> {
        private final Map<Integer, OrgActiveRuleDto> activeRuleById;

        private ActiveRuleParamToActiveRule(Map<Integer, OrgActiveRuleDto> activeRuleById) {
            this.activeRuleById = activeRuleById;
        }

        public OrgActiveRuleDto apply(@Nonnull ActiveRuleParamDto input) {
            return this.activeRuleById.get(input.getActiveRuleId());
        }
    }
}

