/* packet-ns-rpc.c
 * Routines for netscaler remote procedure call protocol dissection
 * Copyright 2006, Ravi Kondamuru <Ravi.Kondamuru@citrix.com>
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

#include "config.h"

#include <epan/packet.h>

void proto_register_ns_rpc(void);
void proto_reg_handoff_ns_rpc(void);

static int proto_ns_rpc = -1;
static gint ett_nsrpc = -1;

static int hf_nsrpc_dlen = -1;
static int hf_nsrpc_cmd = -1;
static int hf_nsrpc_errcode = -1;

static int hf_ns_remoteIOCTL = -1;


static const value_string ns_rpc_cmd_vals[] = {
	{ 0x01, "LOGIN" },
	{ 0x02, "REMOTE IOCTL" },
	{ 0x03, "PERF DATA" },
	{ 0x04, "LOG DATA" },
	{ 0x05, "KEEP-ALIVE" },
	{ 0x06, "SESSION DATA" },
	{ 0x07, "SYSLOG DATA" },
	{ 0x08, "GSLB INIT SVR" },
	{ 0x09, "GSLB INIT DONE" },
	{ 0x0A, "DH NEGOTIATE" },
	{ 0x0B, "INSECURE TUNNEL" },
	{ 0x0C, "INSECURE CMP TUNNEL" },
	{ 0x0D, "SECURE TUNNEL" },
	{ 0x0E, "SECURE CMP TUNNEL" },
	{ 0x0F, "TRACE DATA" },
	{ 0x1F, "AUDIT LOG DATA" },

	{ (0x01 | 0x80), "ACK-LOGIN" },
	{ (0x02 | 0x80), "ACK-REMOTEIOCTL" },
	{ (0x03 | 0x80), "ACK-SENDPERFDATA" },
	{ (0x04 | 0x80), "ACK-SENDLOGDATA" },
	{ (0x05 | 0x80), "ACK-KEEPALIVE" },
	{ (0x06 | 0x80), "ACK-SENDSESSIONDATA" },
	{ (0x07 | 0x80), "ACK-SENDSYSLOGDATA" },
	{ (0x08 | 0x80), "ACK-GSLBINIT_SVR" },
	{ (0x09 | 0x80), "ACK-GSLBINIT_DONE" },
	{ (0x0A | 0x80), "ACK-DH_NEGOTIATE" },
	{ (0x0B | 0x80), "ACK-INSECURETUNNEL" },
	{ (0x0C | 0x80), "ACK-INSECURECMPTUNNEL" },
	{ (0x0D | 0x80), "ACK-SECURETUNNEL" },
	{ (0x0E | 0x80), "ACK-SECURECMPTUNNEL" },
	{ (0x0F | 0x80), "ACK-SENDTRACEDATA" },
	{ (0x1F | 0x80), "ACK-SENDAUDITLOGDATA" },

	{ 0,              NULL }
};

static const value_string ns_rpc_errcode_vals[] = {
	{ 0x000, "SUCCESS" },

	{ 0x101, "PERM ERR" },
	{ 0x10C, "NO MEM" },
	{ 0x135, "CONN ABORTED" },
	{ 0x150, "AUTH ERR" },
	{ 0x153, "NOCODE" },
	{ 0x154, "NOT SUPERUSER" },
	{ 0x155, "LARGE DATA"},
	{ 0x156, "SMALL DATA" },
	{ 0x160, "LOGIN REQD" },
	{ 0x161, "INVALID ARG" },
	{ 0x162, "INVALID USER/ PASSWD" },
	{ 0x163, "INVALID USER/ PASSWD" },

	{ 0x413, "INTERNAL ERR" },

	{ 0x6FF, "DH REQ" },

	{ 0, NULL }
};


static const value_string ns_rpc_rioctl_vals[] = {

	{ 0x40404E0A, "NSAPI_GETTIME" },
	{ 0x40404E9E, "NSAPI_GET_ACCESS_AGENT_STATE" },
	{ 0x40404EB8, "NSAPI_SYNC_FILE" },
	{ 0x40406EB8, "NSAPI_INIT_DYNMEMPOOLS" },
	{ 0x40804E12, "NSAPI_GETOSPFSTATS" },
	{ 0x40804E5F, "NSAPI_RESETOSPFSTATS" },
	{ 0x40804E9A, "NSAPI_GETDEBUGVARINFO" },
	{ 0x40806E12, "NSAPI_GETLOGININFO" },
	{ 0x41006E30, "NSAPI_GETUNMACSTATS" },
	{ 0x41006E36, "NSAPI_GETICMPSTATS" },
	{ 0x41006E38, "NSAPI_GETUNIPSTATS" },
	{ 0x41006E3A, "NSAPI_GETARPSTATS" },
	{ 0x41006E4A, "NSAPI_MONITORTSTATS" },
	{ 0x42004E04, "NSAPI_GETSYSINFO" },
	{ 0x42004E1F, "NSAPI_GETCONF" },
	{ 0x42004EE2, "NSAPI_ASKPE" },
	{ 0x42004EF9, "NSAPI_GET_SP_FILE_INFO" },
	{ 0x4200502C, "NSAPI_GETHOSTNAME" },
	{ 0x42006E3C, "NSAPI_GETIPSTATS" },
	{ 0x42006E69, "NSAPI_AUTH_QUERY" },
	{ 0x42006ED3, "NSAPI_GETCMPSTATS" },
	{ 0x44004D25, "NSAPI_GETSSLSTATS" },
	{ 0x44004D27, "NSAPI_GETSSLERRSTATS" },
	{ 0x44004D29, "NSAPI_GETSSL1ERRSTATS" },
	{ 0x44006E2E, "NSAPI_GETPCBHASHDISTSERVER" },
	{ 0x44006E2F, "NSAPI_GETPCBHASHDISTCLIENT" },
	{ 0x44006E34, "NSAPI_GETUDPSTATS" },
	{ 0x44006E3E, "NSAPI_GETHTTPSTATS" },
	{ 0x44006E42, "NSAPI_GETSYSTSTATS" },
	{ 0x44006E44, "NSAPI_GETSYSCSTATS" },
	{ 0x44006E45, "NSAPI_GETSYSFSTATS" },
	{ 0x44006E46, "NSAPI_GETCSWTSTATS" },
	{ 0x44006E48, "NSAPI_GETDBGTSTATS" },
	{ 0x44006E4A, "NSAPI_GETDBGCSTATS" },
	{ 0x44006E4B, "NSAPI_GETTCPCSTATS" },
	{ 0x44006E55, "NSAPI_GETPCBTSTATS" },
	{ 0x44006E56, "NSAPI_GETPCBHASHDIST" },
	{ 0x44006E6C, "NSAPI_GETCFG8PARAMS" },
	{ 0x44006EBC, "NSAPI_GETDNSHASHDIST_SOA" },
	{ 0x44006EBD, "NSAPI_GETDNSHASHDIST_CNAME" },
	{ 0x44006EC5, "NSAPI_GETCFGPARAMS" },
	{ 0x44006EC8, "NSAPI_GETDNSHASHDIST" },
	{ 0x44006EC9, "NSAPI_GETDNSHASHDIST_A" },
	{ 0x44006ECA, "NSAPI_GETDNSHASHDIST_NS" },
	{ 0x44006ECB, "NSAPI_GETDNSHASHDIST_MX" },
	{ 0x44006EE3, "NSAPI_GETTCPCLNTRTTSTATS" },
	{ 0x44006EE4, "NSAPI_GETTCPSRVRRTTSTATS" },
	{ 0x44006EE5, "NSAPI_GETTCPCLNTPKTRTTSTATS" },
	{ 0x44006EE6, "NSAPI_GETTCPSRVRPKTRTTSTATS" },
	{ 0x44406E40, "NSAPI_GETTCPTSTATS" },
	{ 0x44806E4C, "NSAPI_GETTCPTSTATS1" },
	{ 0x47804EA4, "NSAPI_GET_MASTER_IP" },
	{ 0x60104E0E, "NSAPI_FLUSHALLARP" },
	{ 0x60105000, "NSAPI_FLUSHALLNSB" },
	{ 0x60105013, "NSAPI_FLUSHALLNATPCB" },
	{ 0x60105018, "NSAPI_FLUSHALLPCB" },
	{ 0x6010501C, "NSAPI_FLUSHALLSI" },
	{ 0x60105024, "NSAPI_FLUSHALLBWPCB" },
	{ 0x60106E2A, "NSAPI_FLUSHMEM" },
	{ 0x60405009, "NSAPI_STARTHWCLK" },
	{ 0x60804E44, "NSAPI_ROUTE_FLUSHALL" },
	{ 0x61006E31, "NSAPI_RESETUNMACSTATS" },
	{ 0x61006E37, "NSAPI_RESETICMPSTATS" },
	{ 0x61006E39, "NSAPI_RESETUNIPSTATS" },
	{ 0x61006E3B, "NSAPI_RESETARPSTATS" },
	{ 0x61006E4B, "NSAPI_RESETMONITORTSTATS" },
	{ 0x62006E3D, "NSAPI_RESETIPSTATS" },
	{ 0x62006ED4, "NSAPI_RESETCMPSTATS" },
	{ 0x64004D26, "NSAPI_RESETSSLSTATS" },
	{ 0x64004D28, "NSAPI_RESETSSLERRSTATS" },
	{ 0x64004D2A, "NSAPI_RESETSSL1ERRSTATS" },
	{ 0x64005003, "NSAPI_RESETPCBTSTATS" },
	{ 0x64006E07, "NSAPI_RESETLBSTATS" },
	{ 0x64006E35, "NSAPI_RESETUDPSTATS" },
	{ 0x64006E3F, "NSAPI_RESETHTTPSTATS" },
	{ 0x64006E43, "NSAPI_RESETSYSTSTATS" },
	{ 0x64006E47, "NSAPI_RESETCSWTSTATS" },
	{ 0x64006E49, "NSAPI_RESETDBGTSTATS" },
	{ 0x64006EE4, "NSAPI_RESETTCPRTTSTATS" },
	{ 0x64406E41, "NSAPI_RESETTCPTSTATS" },
	{ 0x64806E4D, "NSAPI_RESETTCPTSTATS1" },
	{ 0x80404E1E, "NSAPI_SENDREPLY" },
	{ 0x80806E13, "NSAPI_SENDLOGINAUTH" },
	{ 0x80806E1E, "NSAPI_SENDENCRYPTEDAUTH" },
	{ 0xC0106D82, "NSAPI_POST_STARTUP" },
	{ 0xC0106E27, "NSAPI_GETCLTMSS_STATS" },
	{ 0xC0106E28, "NSAPI_GETSVRMSS_STATS" },
	{ 0xC0106ECC, "NSAPI_INIT_AAA" },
	{ 0xC0204D4B, "NSAPI_NSRDFILE_CRLCHUNK" },
	{ 0xC0204D58, "NSAPI_GETBRIDGEENTRY" },
	{ 0xC0204E08, "NSAPI_GETCONSMSG" },
	{ 0xC0204E09, "NSAPI_GETPKTRXSIZESTATS" },
	{ 0xC0204E0A, "NSAPI_GETPKTTXSIZESTATS" },
	{ 0xC0204E1C, "NSAPI_GETNSBTRACEDATA" },
	{ 0xC0204E69, "NSAPI_GETFIRSTRACEDATA" },
	{ 0xC0204E6A, "NSAPI_GETRACEDATA" },
	{ 0xC0204E6B, "NSAPI_WAITRACEDATA" },
	{ 0xC0204ED8, "NSAPI_GETMSSINFO" },
	{ 0xC0204EDA, "NSAPI_GETWININFO" },
	{ 0xC0205001, "NSAPI_ADD2PR_ALLCODENAME" },
	{ 0xC0205002, "NSAPI_ADD2PR_ALLDEVNAME" },
	{ 0xC0205007, "NSAPI_ADDPERFDATAHEADER" },
	{ 0xC020502A, "NSAPI_DR_UPDATE" },
	{ 0xC020502B, "NSAPI_ADD_RET_PERFDATAHEADER" },
	{ 0xC0206D3D, "NSAPI_GET_AAAPARAM" },
	{ 0xC0206E03, "NSAPI_GETPROFCAPTURENOWAIT" },
	{ 0xC0206E04, "NSAPI_GETPROFCAPTURE" },
	{ 0xC0206E07, "NSAPI_GETPROFBUFSIZE" },
	{ 0xC0206E14, "NSAPI_GETNEXTLCT" },
	{ 0xC0206E15, "NSAPI_WAITNEXTLCT" },
	{ 0xC0206E16, "NSAPI_GETMINLCT" },
	{ 0xC0206E19, "NSAPI_GETTCPBUFFERINGINFO" },
	{ 0xC0206E1B, "NSAPI_GET_SPTABLE" },
	{ 0xC0206E1C, "NSAPI_GETNEXT_SPTABLE" },
	{ 0xC0206E91, "NSAPI_DYNAMIC_ROUTING" },
	{ 0xC0206EDD, "NSAPI_GETFIRSTCMPTRACEDATA" },
	{ 0xC0206EDE, "NSAPI_GETCMPTRACEDATA" },
	{ 0xC0206EDF, "NSAPI_WAITCMPTRACEDATA" },
	{ 0xC0206EF7, "NSAPI_GETNEXTBRIDGEENTRY" },
	{ 0xC0404D1F, "NSAPI_GETNEXT_CACHE_FWPXY" },
	{ 0xC0404D3A, "NSAPI_GETMODES" },
	{ 0xC0404D63, "NSAPI_GET_SIPPARAMS" },
	{ 0xC0404D93, "NSAPI_DYNAMIC_ROUTING_RIP" },
	{ 0xC0404D96, "NSAPI_GET_RIP" },
	{ 0xC0404E0B, "NSAPI_GETONEARP" },
	{ 0xC0404E0C, "NSAPI_GETNEXTARP" },
	{ 0xC0404E1D, "NSAPI_SAVECONF" },
	{ 0xC0404E3F, "NSAPI_GET_SLOWSTART" },
	{ 0xC0404E66, "NSAPI_GETNEXTSITENAME" },
	{ 0xC0404E74, "NSAPI_GETIP" },
	{ 0xC0404E7E, "NSAPI_GETLICENSE" },
	{ 0xC0404E81, "NSAPI_GETFEATURES" },
	{ 0xC0404E8F, "NSAPI_SENDARP" },
	{ 0xC0404EA2, "NSAPI_SET_NS_IP" },
	{ 0xC0404EA3, "NSAPI_GET_NS_IP" },
	{ 0xC0404ECC, "NSAPI_GET_SVR_CONFIG" },
	{ 0xC0404EDC, "NSAPI_GETMISCLTINFO" },
	{ 0xC0404EF2, "NSAPI_GETNEXT_ACL" },
	{ 0xC0404EFC, "NSAPI_GET_RESPONSE_FILTER" },
	{ 0xC0406E02, "NSAPI_GETNEXTMIP" },
	{ 0xC0406E03, "NSAPI_GETFREEPORTS" },
	{ 0xC0406E05, "NSAPI_GETMODEMINFO" },
	{ 0xC0406E06, "NSAPI_GETNEXTMODEMINFO" },
	{ 0xC0406E0A, "NSAPI_ENABLEMODEMTEST" },
	{ 0xC0406E0B, "NSAPI_DISABLEMODEMTEST" },
	{ 0xC0406E0E, "NSAPI_GETLOGDATA" },
	{ 0xC0406E11, "NSAPI_PMCREAD" },
	{ 0xC0406E14, "NSAPI_GETLOGBUF_CNTRSTRUCTS" },
	{ 0xC0406E23, "NSAPI_GETCODECOVINFO" },
	{ 0xC0406E26, "NSAPI_GETCODECOV" },
	{ 0xC0406E2C, "NSAPI_GETNEXTEVENT" },
	{ 0xC0406E2D, "NSAPI_WAITNEXTEVENT" },
	{ 0xC0406E62, "NSAPI_GET_CACHEPARAMS" },
	{ 0xC0406E65, "NSAPI_GET_DNSPARAM" },
	{ 0xC0406EDA, "NSAPI_GETCMPINFO" },
	{ 0xC0406EF8, "NSAPI_NSSYNC" },
	{ 0xC0406EF9, "NSAPI_GETNEXTIP" },
	{ 0xC0406EFE, "NSAPI_GETNEXTRNAT" },
	{ 0xC0804D1C, "NSAPI_GET_RPCNODE" },
	{ 0xC0804D73, "NSAPI_GETNEXT_WRAPKEY" },
	{ 0xC0804D7C, "NSAPI_GETNEXT_LB_WLM" },
	{ 0xC0804D7D, "NSAPI_GETNEXT_LB_WLMBINDING" },
	{ 0xC0804D92, "NSAPI_DYNAMIC_ROUTING_OSPF" },
	{ 0xC0804D94, "NSAPI_DYNAMIC_ROUTING_BGP" },
	{ 0xC0804D95, "NSAPI_GET_OSPF" },
	{ 0xC0804D99, "NSAPI_GET_BGP" },
	{ 0xC0804D9D, "NSAPI_GETNEXT_BGP" },
	{ 0xC0804DA0, "NSAPI_CLEAR_BGP" },
	{ 0xC0804DA2, "NSAPI_GETNEXT_ROUTEMAP" },
	{ 0xC0804E06, "NSAPI_GETMEMINFO" },
	{ 0xC0804E18, "NSAPI_GETNEXTSESSION" },
	{ 0xC0804E40, "NSAPI_GETNEXT_ROUTE" },
	{ 0xC0804E43, "NSAPI_GET_ROUTE" },
	{ 0xC0804E4B, "NSAPI_GETCOMMUNITY" },
	{ 0xC0804E4C, "NSAPI_GETNEXTCOMMUNITY" },
	{ 0xC0804E4F, "NSAPI_GETMANAGER" },
	{ 0xC0804E50, "NSAPI_GETNEXTMANAGER" },
	{ 0xC0804E53, "NSAPI_GETTRAPDEST" },
	{ 0xC0804E54, "NSAPI_GETNEXTTRAPDEST" },
	{ 0xC0804ED3, "NSAPI_GETCLIENT_PARAMS" },
	{ 0xC080502F, "NSAPI_IF_NF_CHANGE" },
	{ 0xC0806D12, "NSAPI_GETNEXTSYSGROUP" },
	{ 0xC0806D13, "NSAPI_GETALLBINDINFOSYSGROUP" },
	{ 0xC0806D1B, "NSAPI_GETNEXT_NAMESERVER" },
	{ 0xC0806E10, "NSAPI_GETONEBWPCB" },
	{ 0xC0806E15, "NSAPI_GETMEMTESTINFO" },
	{ 0xC0806E1F, "NSAPI_GETNEXTBWPCB" },
	{ 0xC0806E20, "NSAPI_GETNEXTBWPCBIP" },
	{ 0xC0806E21, "NSAPI_GETNEXTBWPCBLESSIDLE" },
	{ 0xC0806E22, "NSAPI_GETNEXTBWPCBMOREIDLE" },
	{ 0xC0806E24, "NSAPI_GETNEXTBWPCBLESSDATA" },
	{ 0xC0806E25, "NSAPI_GETNEXTBWPCBMOREDATA" },
	{ 0xC0806E32, "NSAPI_GETCODENAME" },
	{ 0xC0806E33, "NSAPI_NEXTCODENAME" },
	{ 0xC0806E35, "NSAPI_GETNEXT_ALL_MT_INFO" },
	{ 0xC0806E68, "NSAPI_SYNCDAEMON" },
	{ 0xC0806E72, "NSAPI_GET_MONITOR_BIND" },
	{ 0xC0806E8A, "NSAPI_GETNEXT_PRISVCSCPOLICY" },
	{ 0xC0806E8D, "NSAPI_GETNEXT_SCSTATS" },
	{ 0xC0806E8E, "NSAPI_GETNEXT_LB_ROUTE" },
	{ 0xC0806E9B, "NSAPI_GETNEXT_LINKCERTKEY" },
	{ 0xC0806EA9, "NSAPI_CREATE_DHPARAM" },
	{ 0xC0806EAA, "NSAPI_CREATE_DSAKEY" },
	{ 0xC0806EAB, "NSAPI_CREATE_RSAKEY" },
	{ 0xC0806EBA, "NSAPI_READFILEPATH" },
	{ 0xC0806ECF, "NSAPI_DOS_GETNEXTPOLICY" },
	{ 0xC0806ED9, "NSAPI_FSYNC_ERRSTATS" },
	{ 0xC0806EDB, "NSAPI_GETNEXT_GSLBSITE" },
	{ 0xC0806EF6, "NSAPI_AUTOSYNC_FILES" },
	{ 0xC1004D6A, "NSAPI_GETNEXT_PATCLASS" },
	{ 0xC1004D6C, "NSAPI_GETBIND_PATCLASS" },
	{ 0xC1004D74, "NSAPI_GET_FIPSINIT" },
	{ 0xC1004E3C, "NSAPI_GETNEXT_VSERVICE" },
	{ 0xC1004E60, "NSAPI_GETONENSB" },
	{ 0xC1004E61, "NSAPI_GETNEXTNSB" },
	{ 0xC1004E62, "NSAPI_GETNEXTNSBACTIVE" },
	{ 0xC1004E63, "NSAPI_GETNEXTNSBID" },
	{ 0xC1004E65, "NSAPI_GETNEXTLDNSENTRY" },
	{ 0xC1004E6C, "NSAPI_GETRACEFILTER" },
	{ 0xC1004E70, "NSAPI_GETNEXT_ALL_VSERVICE" },
	{ 0xC1004E75, "NSAPI_GETMEMNATPCB" },
	{ 0xC1004E76, "NSAPI_GETONENATPCB" },
	{ 0xC1004E77, "NSAPI_GETNEXTNATPCB" },
	{ 0xC1004E78, "NSAPI_GETNEXTNATPCBIP" },
	{ 0xC1004E79, "NSAPI_GETNEXTNATPCBIDLE" },
	{ 0xC1004E90, "NSAPI_GETNICSTATS" },
	{ 0xC1004E95, "NSAPI_GETRXTXTIMEINFO" },
	{ 0xC1004E99, "NSAPI_GETNEXT_ALARM_THRESHOLD" },
	{ 0xC1004EBA, "NSAPI_GETNEXT_EXTACL" },
	{ 0xC1004EBC, "NSAPI_GETONEALERT" },
	{ 0xC1004EBD, "NSAPI_GETNEXTALERT" },
	{ 0xC1004EC3, "NSAPI_CONVERT_PKCS8" },
	{ 0xC1004EC4, "NSAPI_GETNEXT_EXTACLS" },
	{ 0xC1006D03, "NSAPI_GETNEXT_ALLSUFFIX" },
	{ 0xC1006D0C, "NSAPI_GETNEXTSYSUSER" },
	{ 0xC1006D0F, "NSAPI_GETALLBINDINFOSYSUSER" },
	{ 0xC1006D24, "NSAPI_GETNEXT_SESSION" },
	{ 0xC1006D25, "NSAPI_KILL_SESSION" },
	{ 0xC1006E11, "NSAPI_PMCREAD_P4" },
	{ 0xC1006EA1, "NSAPI_GETNEXT_CIPHER" },
	{ 0xC1006EAC, "NSAPI_GETNEXT_ALL_CIPHERGRP" },
	{ 0xC1006EE2, "NSAPI_GET_OID" },
	{ 0xC2004D2F, "NSAPI_DME_OP" },
	{ 0xC2004D70, "NSAPI_GETNEXT_FIPSKEY" },
	{ 0xC2004DF5, "NSAPI_GETNEXT_VRID" },
	{ 0xC2004DF6, "NSAPI_GETNEXT_VRIDIFACE" },
	{ 0xC2004E05, "NSAPI_GETNICINFO" },
	{ 0xC2004E11, "NSAPI_GETONEPCB" },
	{ 0xC2004E13, "NSAPI_GETNEXTPCB" },
	{ 0xC2004E14, "NSAPI_GETNEXTPCBIP" },
	{ 0xC2004E15, "NSAPI_GETNEXTPCBIDLE" },
	{ 0xC2004E73, "NSAPI_GET_LOCATIONPARAMS" },
	{ 0xC2004E93, "NSAPI_GETERRINJECTINFO" },
	{ 0xC2004EB5, "NSAPI_CREATE_CRL" },
	{ 0xC2004ECB, "NSAPI_GETSONE_URL" },
	{ 0xC2004ED6, "NSAPI_GET_CLASS_INFO" },
	{ 0xC2004ED7, "NSAPI_GET_CLASS_URL_INFO" },
	{ 0xC2004EDF, "NSAPI_START_SP_FILE" },
	{ 0xC2004EEE, "NSAPI_DONE_SP_FILE" },
	{ 0xC2004EF7, "NSAPI_GET_GSLBPARAMS" },
	{ 0xC2004EFE, "NSAPI_QUERY_SP_FILE_INFO" },
	{ 0xC2005032, "NSAPI_GET_LACHANNEL" },
	{ 0xC20050F1, "NSAPI_GETNEXT_FIS" },
	{ 0xC20050F2, "NSAPI_GETNEXT_FISIFACE" },
	{ 0xC20050F7, "NSAPI_GETNEXT_CI" },
	{ 0xC2006D27, "NSAPI_GETNEXTAAAUSER" },
	{ 0xC2006D2B, "NSAPI_GETNEXTAAAGROUP" },
	{ 0xC2006D59, "NSAPI_GETBIND2AAAGROUP" },
	{ 0xC2006D5A, "NSAPI_GETBIND2AAAUSER" },
	{ 0xC2006E00, "NSAPI_GET_NEXT_GSLBBINDING" },
	{ 0xC2006E32, "NSAPI_GETDEVNAME" },
	{ 0xC2006E8B, "NSAPI_GETNEXT_GSLBACTION" },
	{ 0xC2006E92, "NSAPI_ADD_GSLBBINDING" },
	{ 0xC2006E93, "NSAPI_GETNEXT_GSLBBINDING" },
	{ 0xC2006EA8, "NSAPI_CREATE_CERT" },
	{ 0xC2006EB3, "NSAPI_CONVERT_PKCS12" },
	{ 0xC2006EE7, "NSAPI_DEVSTS" },
	{ 0xC4004D4A, "NSAPI_PUSH_CRLCHUNK" },
	{ 0xC4004D60, "NSAPI_GET_NEXT_SVC_GROUP" },
	{ 0xC4004D61, "NSAPI_GET_SVC_GROUP_BIND" },
	{ 0xC4004D80, "NSAPI_GETNEXT_ALLAAAAREC" },
	{ 0xC4004D84, "NSAPI_GETNEXT_ALLPTRREC" },
	{ 0xC4004D89, "NSAPI_GETNEXT_ALLSRVREC" },
	{ 0xC4004DA8, "NSAPI_GETNEXT_ALL_SVCSSLPOLBINDING" },
	{ 0xC4004DA9, "NSAPI_GETNEXT_ALL_VSRVSSLPOLBINDING" },
	{ 0xC4004DAA, "NSAPI_GETNEXT_SVCGRPSSLPARAMS" },
	{ 0xC4004DAB, "NSAPI_GETNEXT_ALL_SVCGRPSSLPARAMS" },
	{ 0xC4004E16, "NSAPI_GETSNMPSTATS" },
	{ 0xC4004E17, "NSAPI_RESETSNMPSTATS" },
	{ 0xC4004E19, "NSAPI_GETONESI" },
	{ 0xC4004E1A, "NSAPI_GETNEXTSI" },
	{ 0xC4004E26, "NSAPI_GETMEMSI" },
	{ 0xC4004E47, "NSAPI_GET_SVC" },
	{ 0xC4004E48, "NSAPI_GET_NEXT_SVC" },
	{ 0xC4004E5B, "NSAPI_GET_FIRST_POLICY" },
	{ 0xC4004E5C, "NSAPI_GET_NEXT_POLICY" },
	{ 0xC4004E5D, "NSAPI_GETNEXT_PERCSWVS_INFO" },
	{ 0xC4004E5E, "NSAPI_GETNEXT_ALLCSWVS_INFO" },
	{ 0xC4004E71, "NSAPI_GET_NEXT_ALL_SVC" },
	{ 0xC4004E7B, "NSAPI_GETSTATE_ONHOLD" },
	{ 0xC4004E8E, "NSAPI_GETXTESTINFO" },
	{ 0xC4004E91, "NSAPI_GETCONNTESTINFO" },
	{ 0xC4004E9C, "NSAPI_GET_NEXT_SVC_BINDING" },
	{ 0xC4004EA6, "NSAPI_GET_MAPENTRY" },
	{ 0xC4004EA7, "NSAPI_GETNEXT_MAPENTRY" },
	{ 0xC4004EB4, "NSAPI_GETNEXT_VSVRSSLPARAMS" },
	{ 0xC4004EDE, "NSAPI_GETHTTPRESPBANDSTATS" },
	{ 0xC4004EDF, "NSAPI_GETRESPTIMEBANDSTATS" },
	{ 0xC4004EE0, "NSAPI_GETHTTPREQBANDSTATS" },
	{ 0xC4004EE3, "NSAPI_SENDSNMPTRAP" },
	{ 0xC4004EE4, "NSAPI_GETNEW_GSLBBINDING" },
	{ 0xC4004EE7, "NSAPI_GETNEXT_GSLBVSERVER" },
	{ 0xC4004EEC, "NSAPI_GETNEXT_GSLBSERVICE" },
	{ 0xC4004EF4, "NSAPI_GETNEXT_GSLBLOCATION_CUSTOM" },
	{ 0xC4004EF5, "NSAPI_ADD_GSLBLOCATION_STATIC" },
	{ 0xC4005028, "NSAPI_GETSYSTEMMIB" },
	{ 0xC4005038, "NSAPI_GET_LACP" },
	{ 0xC4006D18, "NSAPI_GETNEXTSYSCMDPLCY" },
	{ 0xC4006D8E, "NSAPI_GET_AUDITMESG" },
	{ 0xC4006E61, "NSAPI_GETONE_MONITOR" },
	{ 0xC4006E66, "NSAPI_GETDNSSTATS" },
	{ 0xC4006E67, "NSAPI_RESETDNSSTATS" },
	{ 0xC4006E71, "NSAPI_GET_MONITOR" },
	{ 0xC4006E7A, "NSAPI_GETNEXT_ALLNSREC" },
	{ 0xC4006E7D, "NSAPI_GETNEXT_ALLAREC" },
	{ 0xC4006E80, "NSAPI_GET_ALLSOAREC" },
	{ 0xC4006E84, "NSAPI_GETNEXT_ALLMXREC" },
	{ 0xC4006E87, "NSAPI_GET_ALLCNAMEREC" },
	{ 0xC4006E98, "NSAPI_GETNEXT_CERTKEY" },
	{ 0xC4006EA6, "NSAPI_GETNEXT_SVCSSLPARAMS" },
	{ 0xC4006EA7, "NSAPI_CREATE_CERTREQ" },
	{ 0xC4006EAD, "NSAPI_GETNEXT_ALL_SSLPARAMS" },
	{ 0xC4006EAE, "NSAPI_GETNEXT_ALL_CERTBINDING" },
	{ 0xC4006EAF, "NSAPI_GETNEXT_ALL_CIPHERBINDING" },
	{ 0xC4006EB0, "NSAPI_GETNEXT_ALL_VSVRSSLPARAMS" },
	{ 0xC4006EB1, "NSAPI_GETNEXT_ALL_SVCSSLPARAMS" },
	{ 0xC4006EB2, "NSAPI_GETNEXT_CIPHERBINDING" },
	{ 0xC4006EB4, "NSAPI_REFRESH_CRL" },
	{ 0xC4006EB5, "NSAPI_REPLACE_CRL" },
	{ 0xC4006EC0, "NSAPI_GETNEXT_ACTION" },
	{ 0xC4006EC7, "NSAPI_GETNEXT_CRL" },
	{ 0xC4006ED6, "NSAPI_GET_NEXT_SVCPOL" },
	{ 0xC4006EE1, "NSAPI_GETNSBUSEBYID" },
	{ 0xC4006EED, "NSAPI_GET_VLANSTS" },
	{ 0xC4006EEE, "NSAPI_GETNEXT_VLANSTS" },
	{ 0xC4006EEF, "NSAPI_GETNEXT_IFACE" },
	{ 0xC4006EF1, "NSAPI_GETMPCPUCONTEXT" },
	{ 0xC4006EFE, "NSAPI_GETNEXT_NSREC" },
	{ 0xC4006EFF, "NSAPI_GETNEXT_AREC" },
	{ 0xC6004E45, "NSAPI_GET_BE_SERV" },
	{ 0xC6004E46, "NSAPI_GET_NEXTBE_SERV" },
	{ 0xC6404E86, "NSAPI_PLCY_GETNEXT_EXPR" },
	{ 0xC6404E87, "NSAPI_PLCY_GETNEXT_RULE" },
	{ 0xC6406E50, "NSAPI_PQ_GETNEXTVSERVERPQPOLICY" },
	{ 0xC6406E51, "NSAPI_PQ_GETNEXTPQPOLICY" },
	{ 0xC6804D04, "NSAPI_GET_GENERIC_POLICY" },
	{ 0xC6804D0F, "NSAPI_GET_CMP_POLICY" },
	{ 0xC6804D11, "NSAPI_GET_FILTER_POLICY" },
	{ 0xC6804D17, "NSAPI_GET_TUNNEL_POLICY" },
	{ 0xC6804D1A, "NSAPI_GET_AUTHOR_POLICY" },
	{ 0xC6804D2E, "NSAPI_GET_REWRITE_POLICY" },
	{ 0xC6804D36, "NSAPI_UNSET_REWRITE_POLICY" },
	{ 0xC6804D37, "NSAPI_SET_REWRITE_POLICY" },
	{ 0xC6804D67, "NSAPI_SET_RESPONDER_POLICY" },
	{ 0xC6804D69, "NSAPI_GET_RESPONDER_POLICY" },
	{ 0xC6804D8A, "NSAPI_GETNEXT_SSLPOLICY" },
	{ 0xC6806D4E, "NSAPI_GET_VPNTRAFFIC_POLICY" },
	{ 0xC6806D55, "NSAPI_GET_VPNSESSION_POLICY" },
	{ 0xC6806D79, "NSAPI_GET_RADIUS_POLICY" },
	{ 0xC6806D7B, "NSAPI_GET_LDAP_POLICY" },
	{ 0xC6806D7D, "NSAPI_GET_TACACS_POLICY" },
	{ 0xC6806D7F, "NSAPI_GET_NT4_POLICY" },
	{ 0xC6806D81, "NSAPI_GET_LOCAL_POLICY" },
	{ 0xC6806D85, "NSAPI_GET_SYSLOG_POLICY" },
	{ 0xC6806D86, "NSAPI_GET_NSLOG_POLICY" },
	{ 0xC6806D91, "NSAPI_GET_CERT_POLICY" },
	{ 0xC6C04E88, "NSAPI_PLCY_GETNEXT_BIND" },
	{ 0xC6C04E8C, "NSAPI_PLCY_GETNEXT_DMN" },
	{ 0xC6C04EAE, "NSAPI_GET_CS_CRDPOLICY" },
	{ 0xC6C04EB1, "NSAPI_GET_CS_POLICY" },
	{ 0xC6C04EB2, "NSAPI_GET_CRD_POLICY" },
	{ 0xC7004D0A, "NSAPI_GETBIND_NETSCALER" },
	{ 0xC7004D0B, "NSAPI_GETBINDCMP_NETSCALER" },
	{ 0xC7004D0C, "NSAPI_GETBINDFILTER_NETSCALER" },
	{ 0xC7004D12, "NSAPI_GETBIND_SSLGLOBAL" },
	{ 0xC7004D18, "NSAPI_GETBINDTUNNEL_NETSCALER" },
	{ 0xC7004D33, "NSAPI_GETBIND_REWRITEGLOBAL" },
	{ 0xC7004D70, "NSAPI_GETBIND_RESPONDERGLOBAL" },
	{ 0xC7006D48, "NSAPI_GETBIND_VPNGLOBAL" },
	{ 0xC7006D4B, "NSAPI_GETBIND_SYSTEMGLOBAL" },
	{ 0xC7804E07, "NSAPI_FORCEFAILOVER" },
	{ 0xC7804EA1, "NSAPI_GET_NEXT_NODE" },
	{ 0xC78050F8, "NSAPI_GETNEXT_NODEMONITOR" },
	{ 0xC7806E89, "NSAPI_GETNEXT_SCPOLICY" },
	{ 0xC7C0505A, "NSAPI_TECHINFO_ADD" },
	{ 0xC7C0505B, "NSAPI_TECHINFO_GET" },
	{ 0xC7C0505C, "NSAPI_TECHINFO_GETNEXT" },
	{ 0xC7C0505D, "NSAPI_TECHINFO_DEL" },
	{ 0xD0226D33, "NSAPI_GET_AAAPLCY" },
	{ 0xD0244E25, "NSAPI_CSPOL_BIND" },
	{ 0xD0244E2B, "NSAPI_GETNEXT_GROUP" },
	{ 0xD0244E2C, "NSAPI_GETLB_VSERVER" },
	{ 0xD0244E2D, "NSAPI_GETNEXT_VIP_INGROUP" },
	{ 0xD0244E2E, "NSAPI_GETNEXT_ALL_GROUP" },
	{ 0xD0244E31, "NSAPI_GETCS_VSERVER" },
	{ 0xD0244E3D, "NSAPI_CRPOL_BIND" },
	{ 0xD0244E49, "NSAPI_GET_VSVR" },
	{ 0xD0244E4A, "NSAPI_GET_NEXT_VSVR" },
	{ 0xD0244EAB, "NSAPI_GETCR_VSERVER" },
	{ 0xD0246D3F, "NSAPI_GETSSLVPNCB_VSERVER" },
	{ 0xD0246D5B, "NSAPI_GETBINDVPN_VSERVER" },
	{ 0xD0246E32, "NSAPI_GETNEXT_ALLCSWALLVS_INFO" },
	{ 0xD0246E8B, "NSAPI_GETNEXT_VIPSCPOL" },
	{ 0xD0246ED7, "NSAPI_PQ_GETNEXTPOLSTATS" },
	{ 0xD0316EC3, "NSAPI_GETNEXT_CPE" },
	{ 0xD0316ECE, "NSAPI_GETNEXT_GSLBPOLICY" },
	{ 0xD0334E77, "NSAPI_GETNEXT_ACTIVE_CACHEPOLICY" },
	{ 0xD0336E59, "NSAPI_GETNEXT_CACHEPOLICY" },
	{ 0xD0414D52, "NSAPI_GETNEXT_CACHE_SELECTOR" },
	{ 0xD0806EE0, "NSAPI_GETNEXT_CACHEURL" },
	{ 0xD0896E5F, "NSAPI_GETNEXT_CACHEGROUP" },
	{ 0xD0914D6C, "NSAPI_GET_RESPONDER_ACTION" },
	{ 0xD0914DB5, "NSAPI_GETNEXT_SSL_ACTION" },
	{ 0xD0916D3B, "NSAPI_GET_SYSLOGPARAMS" },
	{ 0xD0916D3E, "NSAPI_GET_NSLOGPARAMS" },
	{ 0xD0916E02, "NSAPI_GET_RADPARAM" },
	{ 0xD0916E04, "NSAPI_GET_LDAPPARAM" },
	{ 0xD0916E06, "NSAPI_GET_TACACSPARAM" },
	{ 0xD0916E08, "NSAPI_GET_NT4PARAM" },
	{ 0xD0916E0A, "NSAPI_GET_CERTPARAM" },
	{ 0xD0916E0D, "NSAPI_GET_VPNPARAM" },
	{ 0xD0916E10, "NSAPI_GETNEXT_VPNAPPLICATION" },
	{ 0xD0916E13, "NSAPI_GET_GENERIC_ACTION" },
	{ 0xD0916E15, "NSAPI_GET_CMP_ACTION" },
	{ 0xD0916E18, "NSAPI_GET_FILTER_ACTION" },
	{ 0xD0916E1A, "NSAPI_GET_VPNTRAFFIC_ACTION" },
	{ 0xD0916E1C, "NSAPI_GET_RADIUS_ACTION" },
	{ 0xD0916E1F, "NSAPI_GET_LDAP_ACTION" },
	{ 0xD0916E21, "NSAPI_GET_TACACS_ACTION" },
	{ 0xD0916E23, "NSAPI_GET_NT4_ACTION" },
	{ 0xD0916E24, "NSAPI_GET_VPNSESSION_ACTION" },
	{ 0xD0916E26, "NSAPI_GET_CERT_ACTION" },
	{ 0xD0916E28, "NSAPI_GET_REWRITE_ACTION" },
	{ 0xD0916E35, "NSAPI_GET_URL_ACTION" },
	{ 0xD0916E38, "NSAPI_GET_SYSLOG_ACTION" },
	{ 0xD0916E39, "NSAPI_GET_NSLOG_ACTION" },
	{ 0xD0916E40, "NSAPI_SEND_AUDITCONFIG" },
	{ 0xD0916E41, "NSAPI_UNSET_REWRITE_PARAM" },
	{ 0xD0916E42, "NSAPI_GET_REWRITE_PARAM" },
	{ 0xD0916E43, "NSAPI_SET_REWRITE_PARAM" },
	{ 0xD0916E77, "NSAPI_UNSET_RESPONDER_PARAM" },
	{ 0xD0916E78, "NSAPI_GET_RESPONDER_PARAM" },
	{ 0xD0916E79, "NSAPI_SET_RESPONDER_PARAM" },
	{ 0xD2C16E60, "NSAPI_GET_CACHECELL" },
	{ 0xE0084ED1, "NSAPI_SETSNMP" },
	{ 0xE0104E0F, "NSAPI_ENABLE_ARP" },
	{ 0xE0104E10, "NSAPI_DISABLE_ARP" },
	{ 0xE0104EA5, "NSAPI_CLEAR_CONFIG" },
	{ 0xE01050E3, "NSAPI_TI_DO_OP" },
	{ 0xE0106E29, "NSAPI_RSTMSS_STATS" },
	{ 0xE0204D57, "NSAPI_SETBRIDGEENTRY" },
	{ 0xE0204E0B, "NSAPI_RSTTTXRXSIZESTATS" },
	{ 0xE0204E1B, "NSAPI_SETNSBTRACEBUFS" },
	{ 0xE0204ED9, "NSAPI_SETMSSINFO" },
	{ 0xE0204EDB, "NSAPI_SETWININFO" },
	{ 0xE0205005, "NSAPI_SETPROFBUFS" },
	{ 0xE0205006, "NSAPI_REMPROFBUFS" },
	{ 0xE0205008, "NSAPI_SETPROFRATE" },
	{ 0xE020500A, "NSAPI_SETPROFMODE" },
	{ 0xE0205067, "NSAPI_SETRACEBUFS" },
	{ 0xE0205068, "NSAPI_REMTRACEBUFS" },
	{ 0xE020506C, "NSAPI_ENABLEALLNICTRACE" },
	{ 0xE020506D, "NSAPI_DISABLEALLNICTRACE" },
	{ 0xE020506E, "NSAPI_ENABLEONENICTRACE" },
	{ 0xE020506F, "NSAPI_DISABLEONENICTRACE" },
	{ 0xE02050DB, "NSAPI_SETCMPTRACEBUFS" },
	{ 0xE02050DC, "NSAPI_REMCMPTRACEBUFS" },
	{ 0xE02050FD, "NSAPI_EXEC_DB_COMMAND" },
	{ 0xE0206D3C, "NSAPI_SET_AAAPARAM" },
	{ 0xE0206E16, "NSAPI_SETMINLCT" },
	{ 0xE0206E1A, "NSAPI_SETTCPBUFFERINGINFO" },
	{ 0xE0206E1D, "NSAPI_SET_SPTABLE" },
	{ 0xE0404D20, "NSAPI_ADD_CACHE_FWPXY" },
	{ 0xE0404D21, "NSAPI_REM_CACHE_FWPXY" },
	{ 0xE0404D38, "NSAPI_ENABLEMODES" },
	{ 0xE0404D39, "NSAPI_DISABLEMODES" },
	{ 0xE0404D62, "NSAPI_SET_SIPPARAMS" },
	{ 0xE0404D98, "NSAPI_SET_RIP" },
	{ 0xE0404D9C, "NSAPI_UNSET_RIP" },
	{ 0xE0404E0D, "NSAPI_FLUSHONEARP" },
	{ 0xE0404E3E, "NSAPI_SET_SLOWSTART" },
	{ 0xE0404E64, "NSAPI_ADD_SVR" },
	{ 0xE0404E67, "NSAPI_REMOVE_SVR" },
	{ 0xE0404E68, "NSAPI_ENABLE_SVR" },
	{ 0xE0404E72, "NSAPI_SETIP" },
	{ 0xE0404E7C, "NSAPI_ENABLELICENSE" },
	{ 0xE0404E7D, "NSAPI_DISABLELICENSE" },
	{ 0xE0404E7F, "NSAPI_ENABLEFEATURES" },
	{ 0xE0404E80, "NSAPI_DISABLEFEATURES" },
	{ 0xE0404E9D, "NSAPI_START_ACCESS_AGENT" },
	{ 0xE0404EA0, "NSAPI_ENABLEIP" },
	{ 0xE0404EB9, "NSAPI_DISABLEIP" },
	{ 0xE0404EBB, "NSAPI_ADDARP" },
	{ 0xE0404EC7, "NSAPI_CLEAR_ACL" },
	{ 0xE0404ECD, "NSAPI_SET_SVR_CONFIG" },
	{ 0xE0404EDA, "NSAPI_SETCMPINFO" },
	{ 0xE0404EDD, "NSAPI_SETMISCLTINFO" },
	{ 0xE0404EF0, "NSAPI_ADD_ACL" },
	{ 0xE0404EF1, "NSAPI_REMOVE_ACL" },
	{ 0xE0404EFA, "NSAPI_ADD_RESPONSE_FILTER" },
	{ 0xE0404EFB, "NSAPI_REMOVE_RESPONSE_FILTER" },
	{ 0xE040501D, "NSAPI_FLUSH_PROXYDNSRECORDS" },
	{ 0xE0405024, "NSAPI_SETCODECOV" },
	{ 0xE0405025, "NSAPI_REMOVECODECOV" },
	{ 0xE040502E, "NSAPI_ADDEVENT" },
	{ 0xE04050B8, "NSAPI_REBOOT" },
	{ 0xE0406E00, "NSAPI_ADDMIP" },
	{ 0xE0406E01, "NSAPI_DELMIP" },
	{ 0xE0406E05, "NSAPI_SETMODEMINFO" },
	{ 0xE0406E08, "NSAPI_ADDMODEM" },
	{ 0xE0406E09, "NSAPI_REMOVEMODEM" },
	{ 0xE0406E0C, "NSAPI_SETLOGBUFS" },
	{ 0xE0406E0D, "NSAPI_REMLOGBUFS" },
	{ 0xE0406E0F, "NSAPI_SETLOGOPT" },
	{ 0xE0406E2B, "NSAPI_SETEVENTTABLESIZE" },
	{ 0xE0406E63, "NSAPI_SET_CACHEPARAMS" },
	{ 0xE0406E64, "NSAPI_SET_DNSPARAM" },
	{ 0xE0406EB9, "NSAPI_SAVECONFIG" },
	{ 0xE0406EFA, "NSAPI_DELETEIP" },
	{ 0xE0406EFB, "NSAPI_ADDIP" },
	{ 0xE0406EFC, "NSAPI_SETRNAT" },
	{ 0xE0406EFD, "NSAPI_CLEARRNAT" },
	{ 0xE0804D05, "NSAPI_SET_TRAPDEST" },
	{ 0xE0804D07, "NSAPI_UNSET_TRAPDEST" },
	{ 0xE0804D19, "NSAPI_ADD_RPCNODE" },
	{ 0xE0804D1A, "NSAPI_SET_RPCNODE" },
	{ 0xE0804D1B, "NSAPI_DEL_RPCNODE" },
	{ 0xE0804D71, "NSAPI_ADD_WRAPKEY" },
	{ 0xE0804D72, "NSAPI_CREATE_WRAPKEY" },
	{ 0xE0804D76, "NSAPI_DELETE_WRAPKEY" },
	{ 0xE0804D77, "NSAPI_ADD_LB_WLM" },
	{ 0xE0804D78, "NSAPI_SET_LB_WLM" },
	{ 0xE0804D79, "NSAPI_BIND_LB_WLM" },
	{ 0xE0804D7A, "NSAPI_UNBIND_LB_WLM" },
	{ 0xE0804D7B, "NSAPI_DELETE_LB_WLM" },
	{ 0xE0804D97, "NSAPI_SET_OSPF" },
	{ 0xE0804D9A, "NSAPI_ROUTE_SET" },
	{ 0xE0804D9B, "NSAPI_UNSET_OSPF" },
	{ 0xE0804D9E, "NSAPI_SET_BGP" },
	{ 0xE0804D9F, "NSAPI_UNSET_BGP" },
	{ 0xE0804DA1, "NSAPI_ADD_BGP" },
	{ 0xE0804DA3, "NSAPI_ADD_ROUTEMAP" },
	{ 0xE0804DA4, "NSAPI_UNSET_ROUTEMAP" },
	{ 0xE0804DA5, "NSAPI_SET_ROUTEMAP" },
	{ 0xE0804DA6, "NSAPI_UNSET_ROUTE" },
	{ 0xE0804DA7, "NSAPI_CLEAR_ROUTE" },
	{ 0xE0804E41, "NSAPI_ROUTE_DELETE" },
	{ 0xE0804E42, "NSAPI_ROUTE_ADD" },
	{ 0xE0804E4D, "NSAPI_SETCOMMUNITY" },
	{ 0xE0804E4E, "NSAPI_DELETECOMMUNITY" },
	{ 0xE0804E51, "NSAPI_SETMANAGER" },
	{ 0xE0804E52, "NSAPI_DELETEMANAGER" },
	{ 0xE0804E55, "NSAPI_ADDTRAPDEST" },
	{ 0xE0804E56, "NSAPI_DELETETRAPDEST" },
	{ 0xE0804E9B, "NSAPI_SETDEBUGVARINFO" },
	{ 0xE0804ED2, "NSAPI_SETCLIENT_PARAMS" },
	{ 0xE0805010, "NSAPI_MSROP" },
	{ 0xE0805023, "NSAPI_FLUSHONEBWPCB" },
	{ 0xE0806D10, "NSAPI_ADDSYSGROUP" },
	{ 0xE0806D11, "NSAPI_RMSYSGROUP" },
	{ 0xE0806D14, "NSAPI_BIND2SYSGROUP" },
	{ 0xE0806D15, "NSAPI_UNBINDFROMSYSGROUP" },
	{ 0xE0806D19, "NSAPI_ADD_NAMESERVER" },
	{ 0xE0806D1A, "NSAPI_RM_NAMESERVER" },
	{ 0xE0806E16, "NSAPI_SETMEMTESTINFO" },
	{ 0xE0806E17, "NSAPI_STARTMEMTEST" },
	{ 0xE0806E18, "NSAPI_ENDMEMTEST" },
	{ 0xE0806E53, "NSAPI_PQ_BINDPQPOLICY" },
	{ 0xE0806E54, "NSAPI_PQ_UNBINDPQPOLICY" },
	{ 0xE0806E73, "NSAPI_BIND_MONITOR" },
	{ 0xE0806E74, "NSAPI_UNBIND_MONITOR" },
	{ 0xE0806E75, "NSAPI_ENABLE_MONITOR" },
	{ 0xE0806E76, "NSAPI_DISABLE_MONITOR" },
	{ 0xE0806E77, "NSAPI_NEW_MONITOR" },
	{ 0xE0806E8F, "NSAPI_LB_ROUTE_DELETE" },
	{ 0xE0806E90, "NSAPI_LB_ROUTE_ADD" },
	{ 0xE0806E99, "NSAPI_ADD_LINKCERTKEY" },
	{ 0xE0806E9A, "NSAPI_DELETE_LINKCERTKEY" },
	{ 0xE0806E9D, "NSAPI_ADD_GSLBSITE" },
	{ 0xE0806E9E, "NSAPI_SET_GSLBSITE" },
	{ 0xE0806EB7, "NSAPI_REM_GSLBSITE" },
	{ 0xE0806EBB, "NSAPI_PUTFILE" },
	{ 0xE0806ECD, "NSAPI_DOS_ADDPOLICY" },
	{ 0xE0806ECE, "NSAPI_DOS_REMOVEPOLICY" },
	{ 0xE0806ED0, "NSAPI_DOS_SETPOLICY" },
	{ 0xE1004D3B, "NSAPI_RENUMBER_EXTACL_PRIORITIES" },
	{ 0xE1004D67, "NSAPI_ADD_PATCLASS" },
	{ 0xE1004D68, "NSAPI_BIND_PATCLASS" },
	{ 0xE1004D69, "NSAPI_UNBIND_PATCLASS" },
	{ 0xE1004D6B, "NSAPI_REMOVE_PATCLASS" },
	{ 0xE1004E3A, "NSAPI_ADD_VIRSERVICE" },
	{ 0xE1004E3B, "NSAPI_DELETE_VIRSERVICE" },
	{ 0xE1004E58, "NSAPI_UNSET_ALARM_THRESHOLD" },
	{ 0xE1004E6D, "NSAPI_ENABLE_SNMP_ALARM" },
	{ 0xE1004E6E, "NSAPI_DISABLE_SNMP_ALARM" },
	{ 0xE1004E96, "NSAPI_SETRXTXTIMEINFO" },
	{ 0xE1004E97, "NSAPI_RESETRXTXTIMEINFO" },
	{ 0xE1004E98, "NSAPI_SET_ALARM_THRESHOLD" },
	{ 0xE1004EB3, "NSAPI_ADD_EXTACL" },
	{ 0xE1004EB6, "NSAPI_REMOVE_EXTACL" },
	{ 0xE1004EBC, "NSAPI_ADDALERT" },
	{ 0xE1004EBD, "NSAPI_REMOVEALERT" },
	{ 0xE1004EC2, "NSAPI_CLEAR_EXTACL" },
	{ 0xE1004EC3, "NSAPI_ENABLE_EXTACL" },
	{ 0xE1004EC5, "NSAPI_SET_EXTACL" },
	{ 0xE1004ED0, "NSAPI_DISABLE_EXTACL" },
	{ 0xE1004EF8, "NSAPI_APPLY_EXTACL" },
	{ 0xE1005004, "NSAPI_RESETNIC" },
	{ 0xE100500B, "NSAPI_ADDNSB" },
	{ 0xE100500C, "NSAPI_ADDNSB_ALIGNED128B" },
	{ 0xE100500D, "NSAPI_ADDNSB_ALIGNED4MB" },
	{ 0xE1005011, "NSAPI_FLUSHMEMNATPCB" },
	{ 0xE1005012, "NSAPI_FLUSHONENATPCB" },
	{ 0xE1005019, "NSAPI_RESET_FIPSINIT" },
	{ 0xE1005022, "NSAPI_SET_FIPSINIT" },
	{ 0xE1005070, "NSAPI_SETRACEFILTER" },
	{ 0xE1006D01, "NSAPI_ADD_DNSSUFFIX" },
	{ 0xE1006D02, "NSAPI_RMV_DNSSUFFIX" },
	{ 0xE1006D0A, "NSAPI_ADDSYSUSER" },
	{ 0xE1006D0B, "NSAPI_RMSYSUSER" },
	{ 0xE1006D0D, "NSAPI_BIND2SYSUSER" },
	{ 0xE1006D0E, "NSAPI_UNBINDFROMSYSUSER" },
	{ 0xE1006D1F, "NSAPI_SETSYSUSER" },
	{ 0xE1006E9F, "NSAPI_ADD_CIPHERGRP" },
	{ 0xE1006EA0, "NSAPI_DELETE_CIPHERGRP" },
	{ 0xE2004D6E, "NSAPI_ADD_FIPSKEY" },
	{ 0xE2004D6F, "NSAPI_DELETE_FIPSKEY" },
	{ 0xE2004DF1, "NSAPI_ADD_VRID" },
	{ 0xE2004DF2, "NSAPI_DEL_VRID" },
	{ 0xE2004DF3, "NSAPI_BIND_VRID" },
	{ 0xE2004DF4, "NSAPI_UNBIND_VRID" },
	{ 0xE2004E57, "NSAPI_UNSETCONF" },
	{ 0xE2004E6F, "NSAPI_SET_LOCATIONPARAMS" },
	{ 0xE2004E70, "NSAPI_SET_CLASS_RELATED" },
	{ 0xE2004E94, "NSAPI_SETERRINJECTINFO" },
	{ 0xE2004E9F, "NSAPI_SETCONF" },
	{ 0xE2004EBE, "NSAPI_ENABLE_URL_LOG" },
	{ 0xE2004EBF, "NSAPI_DISABLE_URL_LOG" },
	{ 0xE2004EC1, "NSAPI_SET_MAX_URLS" },
	{ 0xE2004EC8, "NSAPI_SETONE_URL" },
	{ 0xE2004EC9, "NSAPI_DELETEONE_URL" },
	{ 0xE2004ECA, "NSAPI_DELETEALL_URL" },
	{ 0xE2004ECE, "NSAPI_RESETALL_URL" },
	{ 0xE2004ED4, "NSAPI_SET_CLASS_INFO" },
	{ 0xE2004ED5, "NSAPI_SET_CLASS_URL_INFO" },
	{ 0xE2004EDC, "NSAPI_SETONE_URL_CONTENT" },
	{ 0xE2004EED, "NSAPI_CLEAR_SP_DATA" },
	{ 0xE2004EF6, "NSAPI_SET_GSLBPARAMS" },
	{ 0xE2004EFD, "NSAPI_ADD_SP_FILE_INFO" },
	{ 0xE2004EFF, "NSAPI_FLUSH_SP_INFO" },
	{ 0xE200500E, "NSAPI_SIGN_FIPSKEY" },
	{ 0xE200500F, "NSAPI_EXPPUB_FIPSKEY" },
	{ 0xE2005014, "NSAPI_CREATE_FIPSKEY" },
	{ 0xE2005015, "NSAPI_IMPORT_FIPSKEY" },
	{ 0xE2005016, "NSAPI_FLUSHONEPCB" },
	{ 0xE2005017, "NSAPI_EXPORT_FIPSKEY" },
	{ 0xE200501E, "NSAPI_INIT_FIPSSIMTARGET" },
	{ 0xE200501F, "NSAPI_ENABLE_FIPSSIMTARGET" },
	{ 0xE2005020, "NSAPI_ENABLE_FIPSSIMSOURCE" },
	{ 0xE2005021, "NSAPI_INIT_FIPSSIMSOURCE" },
	{ 0xE200502D, "NSAPI_SETHOSTNAME" },
	{ 0xE2005033, "NSAPI_ADD_LACHANNEL" },
	{ 0xE2005034, "NSAPI_DEL_LACHANNEL" },
	{ 0xE2005035, "NSAPI_SET_LACHANNEL" },
	{ 0xE2005036, "NSAPI_BIND_LACHANNEL" },
	{ 0xE2005037, "NSAPI_UNBIND_LACHANNEL" },
	{ 0xE20050E6, "NSAPI_DEVCTL" },
	{ 0xE20050E7, "NSAPI_RESETINTERFACE" },
	{ 0xE20050E8, "NSAPI_DISABLEINTERFACE" },
	{ 0xE20050E9, "NSAPI_ENABLEINTERFACE" },
	{ 0xE20050EA, "NSAPI_CLEARINTERFACE" },
	{ 0xE20050F3, "NSAPI_ADD_FIS" },
	{ 0xE20050F4, "NSAPI_DEL_FIS" },
	{ 0xE20050F5, "NSAPI_BIND_FIS" },
	{ 0xE20050F6, "NSAPI_UNBIND_FIS" },
	{ 0xE2006D28, "NSAPI_ADDAAAUSER" },
	{ 0xE2006D29, "NSAPI_RMAAAUSER" },
	{ 0xE2006D2A, "NSAPI_SETAAAUSER" },
	{ 0xE2006D2C, "NSAPI_ADDAAAGROUP" },
	{ 0xE2006D2D, "NSAPI_RMAAAGROUP" },
	{ 0xE2006D2E, "NSAPI_BIND2AAAUSER" },
	{ 0xE2006D2F, "NSAPI_UNBINDFROMAAAUSER" },
	{ 0xE2006D30, "NSAPI_BIND2AAAGROUP" },
	{ 0xE2006D31, "NSAPI_UNBINDFROMAAAGROUP" },
	{ 0xE2006E88, "NSAPI_ADD_GSLBACTION" },
	{ 0xE2006E89, "NSAPI_REM_GSLBACTION" },
	{ 0xE2006E8A, "NSAPI_SET_GSLBACTION" },
	{ 0xE2006E94, "NSAPI_DEL_GSLBBINDING" },
	{ 0xE4004D59, "NSAPI_ADD_SERVICE_GROUP" },
	{ 0xE4004D5A, "NSAPI_DELETE_SERVICE_GROUP" },
	{ 0xE4004D5B, "NSAPI_SETIN_SERVICE_GROUP" },
	{ 0xE4004D5C, "NSAPI_SETOUTOF_SERVICE_GROUP" },
	{ 0xE4004D5D, "NSAPI_SET_SVC_GROUP_PARAM" },
	{ 0xE4004D5E, "NSAPI_BIND_SVC_GROUP" },
	{ 0xE4004D5F, "NSAPI_UNBIND_SVC_GROUP" },
	{ 0xE4004D64, "NSAPI_ADD_GSLB_SERVICE" },
	{ 0xE4004D65, "NSAPI_DELETE_GSLB_SERVICE" },
	{ 0xE4004D66, "NSAPI_SET_GSLB_SVC_PARAM" },
	{ 0xE4004D7E, "NSAPI_ADD_AAAAREC" },
	{ 0xE4004D7F, "NSAPI_DEL_AAAAREC" },
	{ 0xE4004D82, "NSAPI_ADD_PTRREC" },
	{ 0xE4004D83, "NSAPI_DEL_PTRREC" },
	{ 0xE4004D86, "NSAPI_ADD_SRVREC" },
	{ 0xE4004D87, "NSAPI_SET_SRVREC" },
	{ 0xE4004D88, "NSAPI_DEL_SRVREC" },
	{ 0xE4004D8C, "NSAPI_BIND_SSLVSRVR" },
	{ 0xE4004D8D, "NSAPI_UNBIND_SSLVSRVR" },
	{ 0xE4004D8E, "NSAPI_BIND_SSLSRVC" },
	{ 0xE4004D8F, "NSAPI_UNBIND_SSLSRVC" },
	{ 0xE4004DAC, "NSAPI_SET_SSLSVCGRPPARAMS" },
	{ 0xE4004E33, "NSAPI_ADD_SERVICE" },
	{ 0xE4004E34, "NSAPI_DELETE_SERVICE" },
	{ 0xE4004E35, "NSAPI_SETIN_SERVICE" },
	{ 0xE4004E36, "NSAPI_SETOUTOF_SERVICE" },
	{ 0xE4004E37, "NSAPI_SETIN_SERVER" },
	{ 0xE4004E38, "NSAPI_SETOUTOF_SERVER" },
	{ 0xE4004E39, "NSAPI_SET_SVC_PARAM" },
	{ 0xE4004E59, "NSAPI_INSERT_POLICY" },
	{ 0xE4004E5A, "NSAPI_REMOVE_POLICY" },
	{ 0xE4004E7A, "NSAPI_SETSTATE_ONHOLD" },
	{ 0xE4004EA8, "NSAPI_ADD_MAPENTRY" },
	{ 0xE4004EA9, "NSAPI_DEL_MAPENTRY" },
	{ 0xE4004EC6, "NSAPI_UNSET_GSLB_VSERVER" },
	{ 0xE4004EDE, "NSAPI_SETHTTPRESPBANDSTATS" },
	{ 0xE4004EDF, "NSAPI_SETRESPTIMEBANDSTATS" },
	{ 0xE4004EE1, "NSAPI_SETHTTPREQBANDSTATS" },
	{ 0xE4004EE5, "NSAPI_UNBIND_GSLBVSERVER" },
	{ 0xE4004EE6, "NSAPI_BIND_GSLBVSERVER" },
	{ 0xE4004EE8, "NSAPI_REM_GSLBVSERVER" },
	{ 0xE4004EE9, "NSAPI_SET_GSLBVSERVER" },
	{ 0xE4004EEA, "NSAPI_ADD_GSLBVSERVER" },
	{ 0xE4004EEF, "NSAPI_ADD_GSLBLOCATION_CUSTOM" },
	{ 0xE4004EF3, "NSAPI_DEL_GSLBLOCATION_CUSTOM" },
	{ 0xE400501A, "NSAPI_FLUSHMEMSI" },
	{ 0xE400501B, "NSAPI_FLUSHONESI" },
	{ 0xE4005029, "NSAPI_SETSYSTEMMIB" },
	{ 0xE4005039, "NSAPI_SET_LACP" },
	{ 0xE400508F, "NSAPI_SETXTESTINFO" },
	{ 0xE4005092, "NSAPI_SETCONNTESTINFO" },
	{ 0xE4006D16, "NSAPI_ADDSYSCMDPLCY" },
	{ 0xE4006D17, "NSAPI_RMSYSCMDPLCY" },
	{ 0xE4006D20, "NSAPI_SETSYSCMDPLCY" },
	{ 0xE4006D39, "NSAPI_UNSET_SVC_PARAM" },
	{ 0xE4006E68, "NSAPI_SET_SSLVSPARAMS" },
	{ 0xE4006E6A, "NSAPI_UPDATE_CERTKEY" },
	{ 0xE4006E6C, "NSAPI_SETCFG8PARAMS" },
	{ 0xE4006E6E, "NSAPI_ADD_MONITOR" },
	{ 0xE4006E6F, "NSAPI_SET_MONITOR" },
	{ 0xE4006E70, "NSAPI_DEL_MONITOR" },
	{ 0xE4006E78, "NSAPI_ADD_NSREC" },
	{ 0xE4006E79, "NSAPI_DEL_NSREC" },
	{ 0xE4006E7B, "NSAPI_ADD_AREC" },
	{ 0xE4006E7C, "NSAPI_DEL_AREC" },
	{ 0xE4006E7E, "NSAPI_ADD_SOAREC" },
	{ 0xE4006E7F, "NSAPI_SET_SOAREC" },
	{ 0xE4006E81, "NSAPI_ADD_MXREC" },
	{ 0xE4006E82, "NSAPI_SET_MXREC" },
	{ 0xE4006E83, "NSAPI_DEL_MXREC" },
	{ 0xE4006E85, "NSAPI_ADD_CNAMEREC" },
	{ 0xE4006E86, "NSAPI_DEL_CNAMEREC" },
	{ 0xE4006E95, "NSAPI_DEL_SOAREC" },
	{ 0xE4006E96, "NSAPI_ADD_CERTKEY" },
	{ 0xE4006E97, "NSAPI_DELETE_CERTKEY" },
	{ 0xE4006E9C, "NSAPI_ADD_CRL" },
	{ 0xE4006EA2, "NSAPI_ADD_VIRCERTKEY" },
	{ 0xE4006EA3, "NSAPI_DELETE_VIRCERTKEY" },
	{ 0xE4006EA4, "NSAPI_ADD_VIRCIPHER" },
	{ 0xE4006EA5, "NSAPI_SET_SSLSVCPARAMS" },
	{ 0xE4006EB6, "NSAPI_SET_CRL" },
	{ 0xE4006EBE, "NSAPI_ADD_ACTION" },
	{ 0xE4006EBF, "NSAPI_REM_ACTION" },
	{ 0xE4006EC5, "NSAPI_SETCFGPARAMS" },
	{ 0xE4006EC6, "NSAPI_DELETE_CRL" },
	{ 0xE4006ED1, "NSAPI_BIND_SVCPOL" },
	{ 0xE4006ED2, "NSAPI_UNBIND_SVCPOL" },
	{ 0xE4006EE8, "NSAPI_ADD_VLAN" },
	{ 0xE4006EE9, "NSAPI_DEL_VLAN" },
	{ 0xE4006EEA, "NSAPI_SET_VLAN" },
	{ 0xE4006EEB, "NSAPI_BIND_IFACE" },
	{ 0xE4006EEC, "NSAPI_UNBIND_IFACE" },
	{ 0xE4006EF0, "NSAPI_CLEAR_VLAN" },
	{ 0xE4006EF2, "NSAPI_ENABLEMPCPU" },
	{ 0xE4006EF3, "NSAPI_DISABLEMPCPU" },
	{ 0xE4006EF4, "NSAPI_STOPMPCPU" },
	{ 0xE4006EF5, "NSAPI_SETPREFETCHSIZEMPCPU" },
	{ 0xE6004D45, "NSAPI_SET_SERVER" },
	{ 0xE6004E20, "NSAPI_ADD_SERVER" },
	{ 0xE6004E21, "NSAPI_DELETE_SERVER" },
	{ 0xE6404E82, "NSAPI_PLCY_DEFINEEXPR" },
	{ 0xE6404E83, "NSAPI_PLCY_DEFINERULE" },
	{ 0xE6404E89, "NSAPI_PLCY_DEL_EXPR" },
	{ 0xE6404E8A, "NSAPI_PLCY_DEL_RULE" },
	{ 0xE6404EB7, "NSAPI_PLCY_ADD_DEF" },
	{ 0xE6404ECF, "NSAPI_PLCY_SET_EXPR" },
	{ 0xE6406E4E, "NSAPI_PQ_ADDPQPOLICY" },
	{ 0xE6406E4F, "NSAPI_PQ_REMOVEPQPOLICY" },
	{ 0xE6406E52, "NSAPI_PQ_SETPQPOLICY" },
	{ 0xE6804D02, "NSAPI_ADD_GENERIC_POLICY" },
	{ 0xE6804D0E, "NSAPI_ADD_CMP_POLICY" },
	{ 0xE6804D10, "NSAPI_ADD_FILTER_POLICY" },
	{ 0xE6804D16, "NSAPI_ADD_TUNNEL_POLICY" },
	{ 0xE6804D18, "NSAPI_ADD_AUTHOR_POLICY" },
	{ 0xE6804D19, "NSAPI_REMOVE_AUTHOR_POLICY" },
	{ 0xE6804D2D, "NSAPI_ADD_REWRITE_POLICY" },
	{ 0xE6804D3C, "NSAPI_ADD_AUDIT_SYSLOG_POLICY" },
	{ 0xE6804D3D, "NSAPI_REMOVE_AUDIT_SYSLOG_POLICY" },
	{ 0xE6804D3E, "NSAPI_REMOVE_CMP_POLICY" },
	{ 0xE6804D3F, "NSAPI_REMOVE_FILTER_POLICY" },
	{ 0xE6804D40, "NSAPI_REMOVE_REWRITE_POLICY" },
	{ 0xE6804D41, "NSAPI_REMOVE_TUNNEL_POLICY" },
	{ 0xE6804D42, "NSAPI_SET_CMP_POLICY" },
	{ 0xE6804D43, "NSAPI_SET_FILTER_POLICY" },
	{ 0xE6804D44, "NSAPI_SET_TUNNEL_POLICY" },
	{ 0xE6804D4A, "NSAPI_ADD_AUDIT_NSLOG_POLICY" },
	{ 0xE6804D56, "NSAPI_REMOVE_AUDIT_NSLOG_POLICY" },
	{ 0xE6804D68, "NSAPI_ADD_RESPONDER_POLICY" },
	{ 0xE6804D6A, "NSAPI_REMOVE_RESPONDER_POLICY" },
	{ 0xE6804D8B, "NSAPI_RM_SSLPOLICY" },
	{ 0xE6804D91, "NSAPI_ADD_SSLPOLICY" },
	{ 0xE6806D4D, "NSAPI_ADD_VPNTRAFFIC_POLICY" },
	{ 0xE6806D4F, "NSAPI_REMOVE_GENERIC_POLICY" },
	{ 0xE6806D50, "NSAPI_SET_GENERIC_POLICY" },
	{ 0xE6806D54, "NSAPI_ADD_VPNSESSION_POLICY" },
	{ 0xE6806D78, "NSAPI_ADD_RADIUS_POLICY" },
	{ 0xE6806D7A, "NSAPI_ADD_LDAP_POLICY" },
	{ 0xE6806D7C, "NSAPI_ADD_TACACS_POLICY" },
	{ 0xE6806D7E, "NSAPI_ADD_NT4_POLICY" },
	{ 0xE6806D80, "NSAPI_ADD_LOCAL_POLICY" },
	{ 0xE6806D87, "NSAPI_SET_AUDIT_POLICY" },
	{ 0xE6806D90, "NSAPI_ADD_CERT_POLICY" },
	{ 0xE6806D97, "NSAPI_REMOVE_AUTH_CERT_POLICY" },
	{ 0xE6806D98, "NSAPI_REMOVE_AUTH_LDAP_POLICY" },
	{ 0xE6806D99, "NSAPI_REMOVE_AUTH_LOCAL_POLICY" },
	{ 0xE6806D9A, "NSAPI_REMOVE_AUTH_NT4_POLICY" },
	{ 0xE6806D9B, "NSAPI_REMOVE_AUTH_RADIUS_POLICY" },
	{ 0xE6806D9C, "NSAPI_REMOVE_AUTH_TACAS_POLICY" },
	{ 0xE6806D9D, "NSAPI_REMOVE_VPN_SESSION_POLICY" },
	{ 0xE6806D9E, "NSAPI_REMOVE_VPN_TRAFFIC_POLICY" },
	{ 0xE6806D9F, "NSAPI_SET_AUTH_CERT_POLICY" },
	{ 0xE6806DA0, "NSAPI_SET_AUTH_LDAP_POLICY" },
	{ 0xE6806DA1, "NSAPI_SET_AUTH_LOCAL_POLICY" },
	{ 0xE6806DA2, "NSAPI_SET_AUTH_NT4_POLICY" },
	{ 0xE6806DA3, "NSAPI_SET_AUTH_RADIUS_POLICY" },
	{ 0xE6806DA4, "NSAPI_SET_AUTH_TACAS_POLICY" },
	{ 0xE6806DA5, "NSAPI_SET_AUTHOR_POLICY" },
	{ 0xE6806DA6, "NSAPI_SET_VPN_SESSION_POLICY" },
	{ 0xE6806DA7, "NSAPI_SET_VPN_TRAFFIC_POLICY" },
	{ 0xE6C04D03, "NSAPI_ADD_CR_POLICY" },
	{ 0xE6C04D06, "NSAPI_REMOVE_CR_POLICY" },
	{ 0xE6C04D46, "NSAPI_SET_CR_POLICY" },
	{ 0xE6C04E84, "NSAPI_PLCY_BINDRULE" },
	{ 0xE6C04E85, "NSAPI_PLCY_ADDDMN" },
	{ 0xE6C04E8B, "NSAPI_PLCY_DEL_BIND" },
	{ 0xE6C04E8D, "NSAPI_PLCY_DEL_DMN" },
	{ 0xE6C04EAD, "NSAPI_ADD_CS_POLICY" },
	{ 0xE6C04EAF, "NSAPI_REMOVE_CS_POLICY" },
	{ 0xE6C04EEB, "NSAPI_SET_CS_CRDPOLICY" },
	{ 0xE7004D08, "NSAPI_BIND_NETSCALER" },
	{ 0xE7004D09, "NSAPI_UNBIND_NETSCALER" },
	{ 0xE7004D13, "NSAPI_UNBIND_SSLGLOBAL" },
	{ 0xE7004D31, "NSAPI_BIND_REWRITEGLOBAL" },
	{ 0xE7004D32, "NSAPI_UNBIND_REWRITEGLOBAL" },
	{ 0xE7004D4C, "NSAPI_BIND_CMP_NETSCALER" },
	{ 0xE7004D4D, "NSAPI_BIND_FILTER_NETSCALER" },
	{ 0xE7004D4E, "NSAPI_BIND_TUNNEL_NETSCALER" },
	{ 0xE7004D4F, "NSAPI_UNBIND_CMP_NETSCALER" },
	{ 0xE7004D50, "NSAPI_UNBIND_FILTER_NETSCALER" },
	{ 0xE7004D51, "NSAPI_UNBIND_TUNNEL_NETSCALER" },
	{ 0xE7004D6E, "NSAPI_BIND_RESPONDERGLOBAL" },
	{ 0xE7004D6F, "NSAPI_UNBIND_RESPONDERGLOBAL" },
	{ 0xE7006D46, "NSAPI_BIND_VPNGLOBAL" },
	{ 0xE7006D47, "NSAPI_UNBIND_VPNGLOBAL" },
	{ 0xE7006D49, "NSAPI_BIND_SYSTEMGLOBAL" },
	{ 0xE7006D4A, "NSAPI_UNBIND_SYSTEMGLOBAL" },
	{ 0xE7006D52, "NSAPI_BIND_SSLGLOBAL" },
	{ 0xE7804E92, "NSAPI_FORCESYNC" },
	{ 0xE7805012, "NSAPI_SETHAPRIORITY" },
	{ 0xE78050A0, "NSAPI_ADD_NODE" },
	{ 0xE78050B9, "NSAPI_DELETE_NODE" },
	{ 0xE78050F9, "NSAPI_BIND_NODEMONITOR" },
	{ 0xE78050FA, "NSAPI_UNBIND_NODEMONITOR" },
	{ 0xE7806E87, "NSAPI_ADD_SCPOLICY" },
	{ 0xE7806E88, "NSAPI_DEL_SCPOLICY" },
	{ 0xE7806E8C, "NSAPI_SET_SCPOLICY" },
	{ 0xF0224EC0, "NSAPI_SET_AAAPLCY" },
	{ 0xF0226D32, "NSAPI_ADD_AAAPLCY" },
	{ 0xF0226D58, "NSAPI_REM_AAAPLCY" },
	{ 0xF0244D01, "NSAPI_BINDCR_VSERVER" },
	{ 0xF0244D0D, "NSAPI_UNBINDCR_VSERVER" },
	{ 0xF0244D1D, "NSAPI_SET_CSVSERVER" },
	{ 0xF0244D1E, "NSAPI_SET_VPNVSERVER" },
	{ 0xF0244D22, "NSAPI_SET_CRVSERVER" },
	{ 0xF0244D23, "NSAPI_UNSET_CRVSERVER" },
	{ 0xF0244D24, "NSAPI_UNSET_VPNVSERVER" },
	{ 0xF0244D2B, "NSAPI_SETOUTOF_LB_VSERVER" },
	{ 0xF0244D2C, "NSAPI_SETOUTOF_GSLB_VSERVER" },
	{ 0xF0244D48, "NSAPI_UNSET_CSVSERVER" },
	{ 0xF0244E22, "NSAPI_ADD_VSERVER" },
	{ 0xF0244E23, "NSAPI_SETOUTOF_VSERVER" },
	{ 0xF0244E24, "NSAPI_SETIN_VSERVER" },
	{ 0xF0244E27, "NSAPI_ADD_GROUP" },
	{ 0xF0244E28, "NSAPI_REMOVE_VSERVER_FROMGROUP" },
	{ 0xF0244E29, "NSAPI_REMOVE_GROUP" },
	{ 0xF0244E2A, "NSAPI_ADDLB_VSERVER" },
	{ 0xF0244E2F, "NSAPI_DELETE_VSERVER" },
	{ 0xF0244E30, "NSAPI_ADDCS_VSERVER" },
	{ 0xF0244E32, "NSAPI_SET_VIRSERVICE" },
	{ 0xF0244E85, "NSAPI_UNSET_VIRSERVICE" },
	{ 0xF0244EAA, "NSAPI_ADDCR_VSERVER" },
	{ 0xF0244EAC, "NSAPI_BINDCS_VSERVER" },
	{ 0xF0244EAE, "NSAPI_UNSET_LB_VSERVER" },
	{ 0xF0244EB0, "NSAPI_UNBINDCS_VSERVER" },
	{ 0xF0246D21, "NSAPI_BINDVPNVSERVER" },
	{ 0xF0246D22, "NSAPI_UNBINDVPNVSERVER" },
	{ 0xF0246D3E, "NSAPI_ADDSSLVPNCB_VSERVER" },
	{ 0xF0246D40, "NSAPI_SETSSLVPNCB_VSERVER" },
	{ 0xF0246D6C, "NSAPI_SETIN_VPN_VSERVER" },
	{ 0xF0246D6D, "NSAPI_SETIN_CS_VSERVER" },
	{ 0xF0246D6E, "NSAPI_SETIN_CR_VSERVER" },
	{ 0xF0246D6F, "NSAPI_SETOUTOF_VPN_VSERVER" },
	{ 0xF0246D70, "NSAPI_SETOUTOF_CS_VSERVER" },
	{ 0xF0246D71, "NSAPI_SETOUTOF_CR_VSERVER" },
	{ 0xF0246D72, "NSAPI_DELETE_VPN_VSERVER" },
	{ 0xF0246D73, "NSAPI_DELETE_CS_VSERVER" },
	{ 0xF0246D74, "NSAPI_DELETE_CR_VSERVER" },
	{ 0xF0246E6B, "NSAPI_DELETE_LB_VSERVER" },
	{ 0xF0246E6D, "NSAPI_SET_VSERVER" },
	{ 0xF0246ED5, "NSAPI_SETIN_GSLB_VSERVER" },
	{ 0xF0246ED8, "NSAPI_SET_GROUP" },
	{ 0xF0246EDC, "NSAPI_SETIN_LB_VSERVER" },
	{ 0xF0316EC1, "NSAPI_ADD_CPE" },
	{ 0xF0316EC2, "NSAPI_REM_CPE" },
	{ 0xF0316ECC, "NSAPI_ADD_GSLBPOLICY" },
	{ 0xF0316ECD, "NSAPI_REM_GSLBPOLICY" },
	{ 0xF0316ECF, "NSAPI_SET_GSLBPOLICY" },
	{ 0xF0334E78, "NSAPI_BIND_CACHEPOLICY" },
	{ 0xF0334E79, "NSAPI_UNBIND_CACHEPOLICY" },
	{ 0xF0336E57, "NSAPI_ADD_CACHEPOLICY" },
	{ 0xF0336E58, "NSAPI_REM_CACHEPOLICY" },
	{ 0xF0414D53, "NSAPI_ADD_CACHE_SELECTOR" },
	{ 0xF0414D54, "NSAPI_SET_CACHE_SELECTOR" },
	{ 0xF0414D55, "NSAPI_REM_CACHE_SELECTOR" },
	{ 0xF0896E5A, "NSAPI_SET_CACHEGROUP" },
	{ 0xF0896E5B, "NSAPI_EXPIRE_CACHEGROUP" },
	{ 0xF0896E5C, "NSAPI_FLUSH_CACHEGROUP" },
	{ 0xF0896E5D, "NSAPI_ADD_CACHEGROUP" },
	{ 0xF0896E5E, "NSAPI_REM_CACHEGROUP" },
	{ 0xF0914D6B, "NSAPI_ADD_RESPONDER_ACTION" },
	{ 0xF0914DB4, "NSAPI_ADD_SSL_ACTION" },
	{ 0xF0914DB6, "NSAPI_REMOVE_SSL_ACTION" },
	{ 0xF0916D00, "NSAPI_SET_VPNTRAFFIC_ACTION" },
	{ 0xF0916D04, "NSAPI_SET_VPNSESSION_ACTION" },
	{ 0xF0916D05, "NSAPI_SET_AUTH_CERT_ACTION" },
	{ 0xF0916D06, "NSAPI_SET_AUTH_LDAP_ACTION" },
	{ 0xF0916D07, "NSAPI_SET_AUTH_RADIUS_ACTION" },
	{ 0xF0916D08, "NSAPI_SET_AUTH_NT4_ACTION" },
	{ 0xF0916D09, "NSAPI_SET_AUTH_TACACS_ACTION" },
	{ 0xF0916D34, "NSAPI_ADD_AUDIT_SYSLOG_ACTION" },
	{ 0xF0916D35, "NSAPI_REMOVE_AUDIT_SYSLOG_ACTION" },
	{ 0xF0916D3A, "NSAPI_SET_SYSLOGPARAMS" },
	{ 0xF0916D3C, "NSAPI_UNSET_SYSLOGPARAMS" },
	{ 0xF0916D3D, "NSAPI_SET_NSLOGPARAMS" },
	{ 0xF0916D3F, "NSAPI_UNSET_NSLOGPARAMS" },
	{ 0xF0916DB4, "NSAPI_SET_AUDIT_NSLOG_ACTION" },
	{ 0xF0916DB5, "NSAPI_SET_AUDIT_SYSLOG_ACTION" },
	{ 0xF0916DB6, "NSAPI_ADD_AUDIT_NSLOG_ACTION" },
	{ 0xF0916DB7, "NSAPI_REMOVE_AUDIT_NSLOG_ACTION" },
	{ 0xF0916DB8, "NSAPI_UNSET_AUDIT_SYSLOG_ACTION" },
	{ 0xF0916DB9, "NSAPI_UNSET_AUDIT_NSLOG_ACTION" },
	{ 0xF0916E01, "NSAPI_SET_RADPARAM" },
	{ 0xF0916E03, "NSAPI_SET_LDAPPARAM" },
	{ 0xF0916E05, "NSAPI_SET_TACACSPARAM" },
	{ 0xF0916E07, "NSAPI_SET_NT4PARAM" },
	{ 0xF0916E09, "NSAPI_SET_CERTPARAM" },
	{ 0xF0916E0B, "NSAPI_SET_VPNPARAM" },
	{ 0xF0916E0C, "NSAPI_UNSET_VPNPARAM" },
	{ 0xF0916E0E, "NSAPI_VPNAPPLICATION_ADD" },
	{ 0xF0916E0F, "NSAPI_VPNAPPLICATION_DELETE" },
	{ 0xF0916E11, "NSAPI_REMOVE_GENERIC_ACTION" },
	{ 0xF0916E12, "NSAPI_ADD_GENERIC_ACTION" },
	{ 0xF0916E14, "NSAPI_ADD_CMP_ACTION" },
	{ 0xF0916E16, "NSAPI_ADD_FILTER_ACTION" },
	{ 0xF0916E17, "NSAPI_SET_FILTER_ACTION" },
	{ 0xF0916E19, "NSAPI_ADD_VPNTRAFFIC_ACTION" },
	{ 0xF0916E1B, "NSAPI_ADD_RADIUS_ACTION" },
	{ 0xF0916E1D, "NSAPI_ADD_VPNSESSION_ACTION" },
	{ 0xF0916E1E, "NSAPI_ADD_LDAP_ACTION" },
	{ 0xF0916E20, "NSAPI_ADD_TACACS_ACTION" },
	{ 0xF0916E22, "NSAPI_ADD_NT4_ACTION" },
	{ 0xF0916E25, "NSAPI_ADD_CERT_ACTION" },
	{ 0xF0916E27, "NSAPI_ADD_REWRITE_ACTION" },
	{ 0xF0916E2B, "NSAPI_REMOVE_AUTH_CERT_ACTION" },
	{ 0xF0916E2C, "NSAPI_REMOVE_AUTH_LDAP_ACTION" },
	{ 0xF0916E2D, "NSAPI_REMOVE_AUTH_RADIUS_ACTION" },
	{ 0xF0916E2E, "NSAPI_REMOVE_AUTH_NT4_ACTION" },
	{ 0xF0916E2F, "NSAPI_REMOVE_AUTH_TACAS_ACTION" },
	{ 0xF0916E30, "NSAPI_REMOVE_VPN_SESSION_ACTION" },
	{ 0xF0916E31, "NSAPI_REMOVE_VPN_TRAFFIC_ACTION" },
	{ 0xF0916E32, "NSAPI_REMOVE_FILTER_ACTION" },
	{ 0xF0916E33, "NSAPI_REMOVE_CMP_ACTION" },
	{ 0xF0916E34, "NSAPI_REMOVE_REWRITE_ACTION" },
	{ 0xF0916E36, "NSAPI_ADD_URL_ACTION" },
	{ 0xF0916E37, "NSAPI_RMV_URL_ACTION" },
	{ 0xF0916E6D, "NSAPI_REMOVE_RESPONDER_ACTION" },
	{ 0xF2C15026, "NSAPI_FLUSH_CACHECELL" },
	{ 0xF2C15027, "NSAPI_EXPIRE_CACHECELL" },

	{ 0, NULL }
};
static value_string_ext ns_rpc_rioctl_vals_ext = VALUE_STRING_EXT_INIT(ns_rpc_rioctl_vals);


static int
dissect_ns_rpc(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data _U_)
{
	guint16 rpc_errcode;
	guint32 datalen, rpc_cmd, remote_ioctl;
	int offset = 0;
	proto_item *ti;
	proto_tree *ns_rpc_tree;

	col_set_str(pinfo->cinfo, COL_PROTOCOL, "NS-RPC");
	col_clear(pinfo->cinfo, COL_INFO);

	ti = proto_tree_add_item(tree, proto_ns_rpc, tvb, 0, -1, ENC_NA);
	ns_rpc_tree = proto_item_add_subtree(ti, ett_nsrpc);

	proto_tree_add_item_ret_uint(ns_rpc_tree, hf_nsrpc_dlen, tvb, offset, 4, ENC_LITTLE_ENDIAN, &datalen);
	offset += 8;	/* skip length, signature, and 2 more bytes */
	proto_tree_add_item_ret_uint(ns_rpc_tree, hf_nsrpc_cmd, tvb, offset, 1, ENC_LITTLE_ENDIAN, &rpc_cmd);
	offset += 2;
	if (rpc_cmd & 0x80)
	{
		col_add_fstr(pinfo->cinfo, COL_INFO, "Resp: %s", val_to_str((rpc_cmd&(~0x80)), ns_rpc_cmd_vals, "0x%02X"));
		proto_tree_add_item(ns_rpc_tree, hf_nsrpc_errcode, tvb, offset, 2, ENC_LITTLE_ENDIAN);
	}
	else
	{
		col_add_fstr(pinfo->cinfo, COL_INFO, "Req: %s", val_to_str(rpc_cmd, ns_rpc_cmd_vals, "0x%02X"));
	}
	rpc_errcode = tvb_get_letohs(tvb, offset);
	offset += 2;
	offset += 4;

	if ((rpc_cmd&(~0x80)) == 0x02) /* remote ioctl */
	{
		proto_tree_add_item_ret_uint(ns_rpc_tree, hf_ns_remoteIOCTL, tvb, offset, 4, ENC_LITTLE_ENDIAN, &remote_ioctl);
		col_append_fstr(pinfo->cinfo, COL_INFO, ": %s", val_to_str_ext(remote_ioctl, &ns_rpc_rioctl_vals_ext, "0x%04X"));
	}

	if (rpc_cmd & 0x80) /* Is this a command response */
	{
		if ((rpc_cmd&(~0x80)) == 0x02) /* remote ioctl */
			col_append_fstr(pinfo->cinfo, COL_INFO, "; ErrCode: %s", val_to_str(tvb_get_letohl(tvb, (offset+12)), ns_rpc_errcode_vals, "0x%04X"));
		else
			col_append_fstr(pinfo->cinfo, COL_INFO, "; Status: %s", val_to_str(rpc_errcode, ns_rpc_errcode_vals, "0x%04X"));
	}

	return tvb_captured_length(tvb);
}

static gboolean
dissect_ns_rpc_heur(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, void* data)
{
	static const guint8 ns_rpc_sig[2] = { 0xA5, 0xA5 };

	/* Check the signature */
	if (tvb_memeql(tvb, 4, ns_rpc_sig, sizeof ns_rpc_sig) != 0)
		return FALSE;

	dissect_ns_rpc(tvb, pinfo, tree, data);
	return TRUE;
}

void
proto_register_ns_rpc(void)
{
	static hf_register_info hf_nsrpc[] = {
		{ &hf_nsrpc_dlen,
		  { "Data Len", "nstrace.rpc.dlen", FT_UINT32, BASE_DEC,NULL, 0x0,
			NULL, HFILL }},

		{ &hf_nsrpc_cmd,
		  { "Command", "nstrace.rpc.cmd", FT_UINT8, BASE_HEX,  VALS(ns_rpc_cmd_vals), 0x0,
			NULL, HFILL }},

		{ &hf_nsrpc_errcode,
		  { "Status Code", "nstrace.rpc.errcode", FT_UINT16, BASE_HEX, VALS(ns_rpc_errcode_vals), 0x0,
			NULL, HFILL }},

		{ &hf_ns_remoteIOCTL,
			{ "Remote IOCTL", "nstrace.mep.mfu.remoteIOCTL", FT_UINT32, BASE_HEX|BASE_EXT_STRING, &ns_rpc_rioctl_vals_ext, 0x0,
			NULL, HFILL }},
	};

	static gint *ett[] = {
		&ett_nsrpc,
	};

	proto_ns_rpc = proto_register_protocol("NetScaler RPC Protocol", "NetScaler RPC", "nstrace.rpc");
	proto_register_field_array(proto_ns_rpc, hf_nsrpc, array_length(hf_nsrpc));
	proto_register_subtree_array(ett, array_length(ett));

	register_dissector("nsrpc", dissect_ns_rpc, proto_ns_rpc);
}


void proto_reg_handoff_ns_rpc(void)
{
	/* Heuristic not strong enough to be enabled by default */
	heur_dissector_add("tcp", dissect_ns_rpc_heur, "netscaler-nsrpc", "nsrpc", proto_ns_rpc, HEURISTIC_DISABLE);
}

/*
 * Editor modelines  -  https://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 8
 * tab-width: 8
 * indent-tabs-mode: t
 * End:
 *
 * vi: set shiftwidth=8 tabstop=8 noexpandtab:
 * :indentSize=8:tabSize=8:noTabs=false:
 */
