// Boost.Geometry (aka GGL, Generic Geometry Library)
// Unit Test

// Copyright (c) 2012-2019 Barend Gehrels, Amsterdam, the Netherlands.

// This file was modified by Oracle on 2016.
// Modifications copyright (c) 2016, Oracle and/or its affiliates.
// Contributed and/or modified by Adam Wulkiewicz, on behalf of Oracle

// Use, modification and distribution is subject to the Boost Software License,
// Version 1.0. (See accompanying file LICENSE_1_0.txt or copy at
// http://www.boost.org/LICENSE_1_0.txt)

#include "test_buffer.hpp"

static std::string const simplex = "MULTILINESTRING((0 0,4 5),(5 4,10 0))";
static std::string const two_bends = "MULTILINESTRING((0 0,4 5,7 4,10 6),(1 5,5 9,8 6))";
static std::string const turn_inside = "MULTILINESTRING((0 0,4 5,7 4,10 6),(1 5,5 9,8 6),(0 4,-2 6))";

static std::string const bend_near_start1 = "MULTILINESTRING((10 0,11 0,15 2),(9 0,8 0,4 2))";
static std::string const bend_near_start2 = "MULTILINESTRING((10 0,11 0,12 1.5,15 3),(9 0,8 0,7 1.5,4 3))";

static std::string const degenerate0 = "MULTILINESTRING()";
static std::string const degenerate1 = "MULTILINESTRING((5 5))";
static std::string const degenerate2 = "MULTILINESTRING((5 5),(9 9))";
static std::string const degenerate3 = "MULTILINESTRING((5 5),(9 9),(4 10))";
static std::string const degenerate4 = "MULTILINESTRING((5 5,5 5),(9 9,9 9,10 10,9 9,9 9,9 9),(4 10,4 10,3 11,4 12,3 11,4 10,4 10))";

static std::string const crossing = "MULTILINESTRING((0 0,10 10),(10 0,0 10))";

static std::string const mikado1 = "MULTILINESTRING((-2 0,-17 -11,3.7142857142857144125969171000179 -2.4285714285714283811046243499732),(11.406143344709896325639419956133 0.75426621160409546007485914742574,12 1,11.403846153846153299582510953769 0.75),(4.25 -2.25,-19 -12,-1 0))";
static std::string const mikado2 = "MULTILINESTRING((-6.1176470588235289937983907293528 -12.696078431372548322997317882255,-6.116279069767442067018237139564 -12.697674418604652402109422837384),(-1.8000000000000007105427357601002 -1.6000000000000000888178419700125,-9.7619047619047627506461140001193 -1.238095238095238137532305700006),(-10.537366548042705005627794889733 -1.2028469750889678735461529868189,-10.567164179104477028658948256634 -1.201492537313432862333684170153),(1.9041095890410959512450972397346 3.4931506849315070439843111671507,-7 1,-7 -4),(-5.540540540540540348501963308081 -5.459459459459459651498036691919,-1.521739130434782261147574899951 -1.2173913043478261641894278000109),(1.2040816326530612290213184678578 2.530612244897959328682190971449,-6.288135593220339103481819620356 -4.711864406779660896518180379644),(-8.4018691588785046064913331065327 -6.755140186915888023122533923015,-8.4131455399061039202024403493851 -6.7660406885758996864410619309638),(-2 5,-12 12,-11.088000000000000966338120633736 0.14400000000000012789769243681803),(-10.720812182741116913575751823373 -0.80456852791878175068518430634867,-10.696969696969697238841945363674 -0.78787878787878784514475682954071),(-10.411764705882353254651206952985 -0.58823529411764707841570043456159,0.11913357400722013323957071406767 6.7833935018050537379963316197973),(-1.5283018867924527128820955113042 -1.2264150943396234794136034906842,1.1922525107604009519945975625888 6.1190817790530829256567812990397),(0.33490011750881265584212087560445 6.6498237367802577324482626863755,-11.548480463096961301516785169952 1.3335745296671497328588884556666))";
static std::string const mikado3 = "MULTILINESTRING((1 18,4.0559006211180124168436122999992 7.8136645962732922399140989000443),(6.7243816254416959310447055031545 -1.0812720848056533995418249105569,7 -2,7 -8,14 3.6666666666666669627261399000417),(15.297872340425531234586742357351 5.8297872340425538340014099958353,16 7,15.214285714285713524418497399893 5.8445378151260509724806979647838),(13.685863874345550073030608473346 3.5968586387434555717845796607435,-1 -18,-3.7900797165633304253162805252941 -11.117803365810452476125647081062),(-11.540540540540540348501963308081 8,-16 19,8 14),(1 -10,6.5999999999999996447286321199499 -1.200000000000000177635683940025),(11.5 6.5,15 12),(19 10,11.564231738035264385189293534495 6.4886649874055422060337150469422),(-13.438785504407443127661281323526 -5.3183153770812925387190261972137,-17 -7,-12.970074812967581578959652688354 -7.7556109725685784539450651209336),(-2.3532338308457703135445626685396 -9.7462686567164187323442092747428,-1 -10,12.285714285714286475581502600107 3.2857142857142864755815026001073),(14.90000000000000035527136788005 5.9000000000000003552713678800501,15 6,14.893004115226338157640384451952 5.9012345679012341292946075554937),(11.987804878048780921062643756159 3.2195121951219514144781896902714,-11 -18),(-12.210826210826210669324609625619 -11.703703703703702387883822666481,-15 -15,-11.463576158940396609864365018439 -15.589403973509934786534358863719),(-8.9189189189189193029960733838379 -16.013513513513512265262761502527,-3 -17,-7.0297239915074314353660156484693 -14.210191082802548834251865628175),(-12.450511945392491952588898129761 -10.457337883959045399251408525743,-16 -8,-12.923076923076923350208744523115 -8),(-0.52380952380952372493538859998807 -8,18 -8),(2 -19,-2.2961165048543685784920853620861 -9.6917475728155331182733789319173),(6.0463576158940393057150686217938 -1.7284768211920527036795647291001,7 -3,6.4482758620689653028534848999698 -1.3448275862068967967388744000345),(-1.3333333333333339254522798000835 8,4 16,2.9090909090909091716525836091023 8),(0.64705882352941168633719826175366 -6.8823529411764710062016092706472,-3 -16))";
static std::string const mikado4 = "MULTILINESTRING((-15 2,-15 -17,-6 11,-1.9358288770053475591481628725887 10.572192513368984023713892383967),(2.1545064377682408007785852532834 10.14163090128755406738036981551,6.87603305785123986026974307606 9.6446280991735537924114396446384),(8.4810810810810810522752944962122 9.475675675675674369813350494951,13 9),(-15 0,-8 9,-2.9850746268656713766631582984701 4.4865671641791049495395782287233),(-1.8235294117647056211239942058455 3.4411764705882355031008046353236,-1.1428571428571423496123315999284 2.8285714285714291804652020800859),(1.2307692307692308375521861307789 0.69230769230769229061195346730528,1.2857142857142858094476878250134 0.64285714285714290472384391250671,2 0,1.9459459459459460539676456392044 0.51351351351351348650808859019889),(1.908127208480565384363103476062 0.87279151943462895957281943992712,1.9078014184397162900097555393586 0.87588652482269502286271745106205),(1.4685990338164249813246442499803 5.0483091787439615671928550000302,0.63551401869158885560295857430901 12.962616822429906093816498469096,0 19,2.9565217391304345895264304999728 8.6521739130434784925682834000327),(0 19,3.4942528735632185643567027000245 6.770114942528735468840750399977),(4.75 2.375,5.2427184466019420838733822165523 0.65048543689320226235395239200443),(5.5384615384615383248956277384423 -0.38461538461538458122390693461057,5.7358490566037731994697423942853 -1.0754716981132084185901476303115),(5.9777777777777778567269706400111 -1.9222222222222207221875578397885,6.867052023121386739035187929403 -5.0346820809248553629799971531611,10 -16,-14 -19,-12 -12),(0 10,1.9476439790575916788384347455576 5.4554973821989527493769855936989),(-4 1,-4.2790697674418600726653494348284 0.16279069767441856075862460784265))";

static std::string const mysql_2015_04_10a = "MULTILINESTRING((-58 19, 61 88),(1.922421e+307 1.520384e+308, 15 42, 89 -93,-89 -22),(-63 -5, -262141 -536870908, -3 87, 77 -69))";
static std::string const mysql_2015_04_10b = "MULTILINESTRING((-58 19, 61 88),                                                     (-63 -5, -262141 -536870908, -3 87, 77 -69))";

static std::string const mysql_2015_09_08a = "MULTILINESTRING((7 -4, -3 -5), (72057594037927936 15, 72057594037927940 70368744177660, 32771 36028797018963964, 8589934589 2305843009213693953, 7 2, 9.300367e+307 9.649737e+307, -4092 -274877906946, 5 10, -3 4))";
static std::string const mysql_2015_09_08b = "MULTILINESTRING((-9 -10, 0 -1, 5 -10, -6 7, -7 7, 5.041061e+307 9.926906e+307, 6.870356e+307 1.064454e+307, 35184372088830 288230376151711743, 183673728842483250000000000000000000000.000000 244323751784861950000000000000000000000.000000), (-23530 -7131, -6 1, 1 1, 2 -6, 32766 -4194302, -4 -6), (134217725 0, 50336782742294697000000000000000000000.000000 36696596077212901000000000000000000000.000000, 7434 16486, 3.025467e+307 8.926790e+307), (2147483646 67108868, 71328904281592545000000000000000000000.000000 225041650340452780000000000000000000000.000000, -7 4, 1.667154e+307 3.990414e+307))"; 

static std::string const mysql_23023665_1 = "MULTILINESTRING((-5 15, 7 15, 19 -10, -11 -2),(2 13, 2 -9))";

static std::string const touching1 = "MULTILINESTRING((-5 15, 9 15),(2 13, 2 0),(-4 14.75,-4 13),(8 15.25,8 13),(1.5 1,-1 1),(2.25 4,5 4))";
static std::string const touching2 = "MULTILINESTRING((-5 15, 9 15),(2 13, 2 0),(-11 -2,9 14))";

static std::string const ticket_13444 = "MULTILINESTRING((-14900.3540680666540 6736752.97813679560,-14903.4321076464080 6736753.66526194570,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.2786583450830 6736754.3021534560,-14906.2786583450830 6736754.3021534560,-14922.5830341660040 6736758.55049021640,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.4931871261580 6736765.77697354280,-14943.4931871261580 6736765.77697354280,-14977.8347379614440 6736783.40904572790,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-15011.938643947820 6736805.44739563390,-15011.4855624884060 6736805.15855005480,-15044.495275804080 6736826.22277528610,-15044.636350994740 6736826.3142731870,-15044.636350994740 6736826.3142731870,-15044.636350994740 6736826.3142731870,-15044.636350994740 6736826.3142731870,-15044.636350994740 6736826.3142731870,-15044.636350994740 6736826.3142731870,-15044.636350994740 6736826.3142731870,-15059.3568953647720 6736834.30329997840,-15068.0541424639030 6736839.02352138140,-15069.4928800905110 6736839.75371170980,-15070.9534140734160 6736840.41931523480,-15087.9426162696510 6736846.96412046530,-15089.5494797231820 6736847.51131521720,-15091.1563320447620 6736848.05671593270,-15094.2349059580680 6736848.97707971370,-15335.9054238793630 6736921.18738844430000000000),"
        "(-14900.3540680666540 6736752.97813679560,-14903.4321076464080 6736753.66526194570,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.2786583450830 6736754.3021534560,-14906.2786583450830 6736754.3021534560,-14922.5830341660040 6736758.55049021640,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.4931871261580 6736765.77697354280,-14943.4931871261580 6736765.77697354280,-14977.8347379614440 6736783.40904572790,-14978.1533343440550 6736783.59921684120,-14983.6129319775540 6736787.37752308510,-14985.1947819415070 6736788.51675631760,-14986.7766319054570 6736789.65598971210,-14987.9123578781710 6736790.53687736760,-14992.4552617690170 6736794.05684083610,-14998.267508417350 6736798.72501083650,-15010.441029442540 6736809.15394911820,-15011.6583859978380 6736810.19630545840,-15022.6145949955260 6736819.58290076720,-15059.7358046474820 6736849.75929930060,-15061.0431630110740 6736850.7029863410,-15062.3505213746680 6736851.644879410,-15070.4409770615170 6736856.86027784460,-15087.1738995260860 6736866.37430707270,-15088.6108003810950 6736867.09552943430,-15095.9473782124990 6736870.45405826350,-15104.0074102825180 6736873.63317977920,-15116.207314027020 6736878.44671658990,-15123.8358828827350 6736880.87232399170,-15134.5288992083770 6736884.24879940230,-15180.7309406607860 6736898.17991495970000000000),"
        "(-14900.3540680666540 6736752.97813679560,-14903.4321076464080 6736753.66526194570,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.2786583450830 6736754.3021534560,-14906.2786583450830 6736754.3021534560,-14922.5830341660040 6736758.55049021640,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.4931871261580 6736765.77697354280,-14943.4931871261580 6736765.77697354280,-14977.8347379614440 6736783.40904572790,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-15011.938643947820 6736805.44739563390,-15011.4855624884060 6736805.15855005480,-15044.495275804080 6736826.22277528610,-15044.636350994740 6736826.3142731870,-15045.8532288762310 6736827.08752021570,-15074.3737944831540 6736845.21847973950,-15077.090034585930 6736846.94438557510,-15078.4820514222590 6736847.81092678480,-15078.071961550180 6736847.56872581780,-15087.3648458486150 6736853.06579258570,-15090.2635942565180 6736854.6786728650,-15097.9514630049710 6736858.31707360780,-15103.8265161100060 6736861.09610959050,-15130.264249516680 6736873.60626847480,-15134.7519500720480 6736875.32141580710,-15146.8630765226570 6736879.66849063620,-15194.4412940478560 6736894.61146595520,-15195.9796069589630 6736895.07075342350000000000),"
        "(-14900.3540680666540 6736752.97813679560,-14903.4321076464080 6736753.66526194570,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.2786583450830 6736754.3021534560,-14906.2786583450830 6736754.3021534560,-14922.5830341660040 6736758.55049021640,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.4931871261580 6736765.77697354280,-14943.4931871261580 6736765.77697354280,-14977.8347379614440 6736783.40904572790,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-15011.938643947820 6736805.44739563390,-15011.4855624884060 6736805.15855005480,-15044.495275804080 6736826.22277528610,-15044.636350994740 6736826.3142731870,-15045.8532288762310 6736827.08752021570,-15074.3737944831540 6736845.21847973950,-15077.090034585930 6736846.94438557510,-15078.4820514222590 6736847.81092678480,-15078.071961550180 6736847.56872581780,-15078.071961550180 6736847.56872581780,-15078.071961550180 6736847.56872581780,-15078.071961550180 6736847.56872581780,-15078.071961550180 6736847.56872581780,-15078.071961550180 6736847.56872581780,-15078.071961550180 6736847.56872581780,-15082.7943681199050 6736849.88488502430,-15097.5442785716020 6736857.11683175990,-15106.6291957973660 6736860.3533587540,-15108.2275099140060 6736860.85390822030,-15125.8089540650780 6736866.36354256610,-15127.375820425570 6736866.840769290,-15215.2540981133260 6736893.00754821770,-15218.3352769027190 6736893.92791708470,-15437.0992937759950 6736959.30128452830,-15461.7487574869660 6736966.66788661480,-15515.650723588680 6736982.80231144560,-15574.1728604397850 6737000.31823665460,-15594.0923367237140 6737006.20472987550,-15710.5446690337120 6737040.61236372680,-15712.0936018242430 6737041.03398283110,-15734.2281022037190 6737046.24412034730,-15742.1332777301140 6737048.10462824910,-15743.8434568030350 6737048.30018792950,-15748.6193127583390 6737048.65262600970000000000),"
        "(-14900.3540680666540 6736752.97813679560,-14903.4321076464080 6736753.66526194570,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.2786583450830 6736754.3021534560,-14906.2786583450830 6736754.3021534560,-14922.5830341660040 6736758.55049021640,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14943.4931871261580 6736765.77697354280,-14943.4931871261580 6736765.77697354280,-14977.8347379614440 6736783.40904572790,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-14978.1533343440550 6736783.59921684120,-15011.938643947820 6736805.44739563390,-15011.4855624884060 6736805.15855005480,-15011.4855624884060 6736805.15855005480,-15011.4855624884060 6736805.15855005480,-15011.4855624884060 6736805.15855005480,-15011.4855624884060 6736805.15855005480,-15011.4855624884060 6736805.15855005480,-15011.4855624884060 6736805.15855005480,-15024.686885240 6736812.53218883280,-15026.1816388343670 6736813.3144049160,-15030.6658773535710 6736815.65925955030,-15036.7550423671660 6736818.56027804410,-15041.3219133443780 6736820.73649111390,-15048.9609128363750 6736824.08244226970,-15050.4611878774870 6736824.67628146990,-15084.9584078886780 6736837.58108171260,-15086.4811249391280 6736838.09598240720000000000),"
        "(-14900.3540680666540 6736752.97813679560,-14903.4321076464080 6736753.66526194570,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.2786583450830 6736754.3021534560,-14906.2786583450830 6736754.3021534560,-14922.5830341660040 6736758.55049021640,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14980.5459018954430 6736780.45959997360,-14980.5459018954430 6736780.45959997360,-14980.5459018954430 6736780.45959997360,-14980.5459018954430 6736780.45959997360,-14980.5459018954430 6736780.45959997360,-14980.5459018954430 6736780.45959997360,-14980.0377729478370 6736780.26763486860,-14980.0377729478370 6736780.26763486860,-14991.6555536792150 6736785.84897732640,-15011.9866671761430 6736795.61588732150,-15012.4519381197970 6736795.86346895620,-15020.770743478080 6736800.37376175450,-15022.2480644401890 6736801.17391738670,-15026.6800384584770 6736803.57797288240,-15038.4986506831680 6736809.98460484760,-15041.4533037393390 6736811.58671214340,-15058.2989705841660 6736819.98118544460,-15081.4365374991940 6736828.96412467860,-15082.9566942013570 6736829.48440706080,-15099.6784290571030 6736835.20392734280,-15104.2978317020080 6736836.58357383120,-15111.9968435314880 6736838.88358319360,-15113.5366592557220 6736839.34286751130,-15175.1290655861190 6736857.7411727840,-15252.1195734991230 6736880.73776734710,-15421.5006278668590 6736931.32583960150,-15423.0404547230450 6736931.78512913550,-15470.7754991468520 6736946.04105874990,-15498.4926942527330 6736954.32084811290,-15500.0325433728140 6736954.78013895270,-15575.3855081806230 6736977.37871549190,-15636.8689327295510 6736995.84730348080,-15638.406032258210 6736996.30839077380,-15730.631692283210 6737024.00955883970,-15733.7340329077980 6737024.91558956820,-15735.3732012775510 6737025.31388429920,-15737.0123696473070 6737025.7121790480,-15740.0858017133340 6737026.31913025120000000000),"
        "(-14900.3540680666540 6736752.97813679560,-14903.4321076464080 6736753.66526194570,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.2786583450830 6736754.3021534560,-14906.2786583450830 6736754.3021534560,-14922.5830341660040 6736758.55049021640,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14980.5459018954430 6736780.45959997360,-14980.5459018954430 6736780.45959997360,-14980.5459018954430 6736780.45959997360,-14980.5459018954430 6736780.45959997360,-14980.5459018954430 6736780.45959997360,-14980.5459018954430 6736780.45959997360,-14980.0377729478370 6736780.26763486860,-14980.0377729478370 6736780.26763486860,-14991.6555536792150 6736785.84897732640,-15011.9866671761430 6736795.61588732150,-15012.4519381197970 6736795.86346895620,-15012.4519381197970 6736795.86346895620,-15012.4519381197970 6736795.86346895620,-15012.4519381197970 6736795.86346895620,-15012.4519381197970 6736795.86346895620,-15012.4519381197970 6736795.86346895620,-15012.4519381197970 6736795.86346895620,-15021.625933202080 6736800.2499708710,-15023.1507875867560 6736800.96759922520,-15027.7253396088290 6736803.12227872670,-15042.7733302609810 6736809.10550915170,-15074.0109486029650 6736819.38555251160,-15075.5483598261980 6736819.84483573210,-15089.3850385714010 6736823.99094439390,-15135.5072973447730 6736837.81251791210,-15197.0045366425070 6736856.23952505740,-15275.413533166740 6736879.73487190530000000000),"
        "(-14900.3540680666540 6736752.97813679560,-14903.4321076464080 6736753.66526194570,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.2786583450830 6736754.3021534560,-14906.2786583450830 6736754.3021534560,-14922.5830341660040 6736758.55049021640,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14980.5459018954430 6736780.45959997360,-14980.5459018954430 6736780.45959997360,-15030.7446692891440 6736797.61627536640,-15031.1498499716740 6736797.74544842350,-15038.6298407030480 6736800.08671043440,-15040.2125700909780 6736800.58187397570,-15044.9607693867150 6736802.06915886140,-15048.1262281625750 6736803.05948622340,-15059.2053672739280 6736806.52922108490,-15060.808434732820 6736807.02797317230,-15070.4268283542390 6736810.02228038110,-15090.7124010474890 6736816.33562555160,-15093.7912866553690 6736817.2649561130,-15109.1857035628240 6736821.91519868650,-15121.5012459943460 6736825.63611227830,-15139.9829308673380 6736831.19236878680,-15292.4567809757460 6736877.02759306690,-15297.0794564136850 6736878.40904073880,-15352.5515616689190 6736894.99530161920000000000),"
        "(-14900.3540680666540 6736752.97813679560,-14903.4321076464080 6736753.66526194570,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.2786583450830 6736754.3021534560,-14906.2786583450830 6736754.3021534560,-14922.5830341660040 6736758.55049021640,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14980.5459018954430 6736780.45959997360,-14980.5459018954430 6736780.45959997360,-15030.7446692891440 6736797.61627536640,-15031.1498499716740 6736797.74544842350,-15031.1498499716740 6736797.74544842350,-15031.1498499716740 6736797.74544842350,-15031.1498499716740 6736797.74544842350,-15031.1498499716740 6736797.74544842350,-15031.1498499716740 6736797.74544842350,-15031.1498499716740 6736797.74544842350,-15063.4394263695590 6736806.12734892960,-15077.96973686180 6736809.90028341770,-15079.4709581185830 6736810.20886396620,-15112.4978702955780 6736817.01378568450,-15112.4978702955780 6736817.01378568450,-15112.4978702955780 6736817.01378568450,-15112.4978702955780 6736817.01378568450,-15112.4978702955780 6736817.01378568450,-15112.4978702955780 6736817.01378568450,-15111.7280180932050 6736816.8612893540,-15111.7280180932050 6736816.8612893540,-15136.404287537560 6736822.99882040450,-15160.8745380003330 6736830.05133549120,-15177.1880420194980 6736834.75361365080,-15266.4520692562510 6736861.47645545290,-15269.4815178782760 6736862.38605624720,-15320.982200112470 6736877.84569711610,-15416.0519172795560 6736906.38250955010,-15422.1735541886120 6736908.2142799450,-15480.3290491648930 6736925.62239861490,-15523.3135120558580 6736938.44483037390,-15529.4541957297860 6736940.27660798650,-15676.8305371123480 6736984.23760042710,-15684.5052033691940 6736986.53406332340,-15705.9942844730870 6736992.96236877050,-15715.0598543713510 6736996.18818513960,-15722.6144959532430 6736998.8775624940,-15729.8982749501720 6737002.20385167190000000000),"
        "(-14900.3540680666540 6736752.97813679560,-14903.4321076464080 6736753.66526194570,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.2786583450830 6736754.3021534560,-14906.2786583450830 6736754.3021534560,-14922.5830341660040 6736758.55049021640,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14980.5459018954430 6736780.45959997360,-14980.5459018954430 6736780.45959997360,-15030.7446692891440 6736797.61627536640,-15031.1498499716740 6736797.74544842350,-15031.1498499716740 6736797.74544842350,-15031.1498499716740 6736797.74544842350,-15031.1498499716740 6736797.74544842350,-15031.1498499716740 6736797.74544842350,-15031.1498499716740 6736797.74544842350,-15031.1498499716740 6736797.74544842350,-15063.4394263695590 6736806.12734892960,-15077.96973686180 6736809.90028341770,-15079.4709581185830 6736810.20886396620,-15112.4978702955780 6736817.01378568450,-15112.4978702955780 6736817.01378568450,-15123.5406969890390 6736819.20973316490,-15126.6957919203090 6736819.83765943070,-15137.7386408776670 6736822.03360766730,-15142.4712888405410 6736822.97549742090,-15186.6427180658250 6736831.75929733830,-15188.2202710974380 6736832.07326094250,-15189.7978241290440 6736832.38722456990,-15190.0044887636720 6736832.42848835510,-15222.1556839025690 6736840.72251470480,-15234.5402332864270 6736844.34835063950,-15269.996782405240 6736854.92625629810,-15379.4496123019090 6736887.58221805560,-15387.1575630864950 6736889.88224184050,-15474.9321578005460 6736916.19621079690,-15481.0917769198550 6736918.0423361780,-15541.148430687430 6736936.04611730850,-15680.1650234030040 6736978.02997740640,-15694.0778224160020 6736982.24254881030,-15695.6236827885860 6736982.71081178360,-15697.0432178030370 6736983.18086888920,-15719.1585489661970 6736991.21490258910,-15720.7032404841320 6736991.83387157130,-15723.554088115150 6736993.14357420990,-15726.4049394568230 6736994.45417464520000000000),"
        "(-14900.3540680666540 6736752.97813679560,-14903.4321076464080 6736753.66526194570,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.5703376748190 6736754.36135745610,-14906.2786583450830 6736754.3021534560,-14906.2786583450830 6736754.3021534560,-14922.5830341660040 6736758.55049021640,-14943.1461152178110 6736765.65497725270,-14943.1461152178110 6736765.65497725270,-14980.5459018954430 6736780.45959997360,-14980.5459018954430 6736780.45959997360,-15030.7446692891440 6736797.61627536640,-15031.1498499716740 6736797.74544842350,-15031.1498499716740 6736797.74544842350,-15031.1498499716740 6736797.74544842350,-15031.1498499716740 6736797.74544842350,-15031.1498499716740 6736797.74544842350,-15031.1498499716740 6736797.74544842350,-15031.1498499716740 6736797.74544842350,-15063.4394263695590 6736806.12734892960,-15077.96973686180 6736809.90028341770,-15079.4709581185830 6736810.20886396620,-15112.4978702955780 6736817.01378568450,-15112.4978702955780 6736817.01378568450,-15123.5406969890390 6736819.20973316490,-15126.6957919203090 6736819.83765943070,-15137.7386408776670 6736822.03360766730,-15142.4712888405410 6736822.97549742090,-15186.6427180658250 6736831.75929733830,-15188.2202710974380 6736832.07326094250,-15189.7978241290440 6736832.38722456990,-15190.0044887636720 6736832.42848835510,-15190.0044887636720 6736832.42848835510,-15190.0044887636720 6736832.42848835510,-15190.0044887636720 6736832.42848835510,-15190.0044887636720 6736832.42848835510,-15190.0044887636720 6736832.42848835510,-15190.0044887636720 6736832.42848835510,-15221.983183219660 6736838.87281871770,-15268.3061621668480 6736848.24688855470,-15269.855061561530 6736848.63441014480,-15300.8331273788140 6736856.38484578950,-15302.3963647239070 6736856.82798435070,-15308.5770232269750 6736858.67050808390,-15393.5610720781510 6736884.00839143530,-15395.0947874942920 6736884.46947239150,-15454.9098758556520 6736902.42873865920000000000),"
        "(-15035.8556810692230 6736780.3429856580,-15067.3888192633710 6736786.679630680,-15089.4620493951220 6736791.11456749960,-15098.9220131172680 6736793.01628063250,-15102.0753269366810 6736793.64958708270,-15102.0753269366810 6736793.64958708270,-15105.2380806489130 6736794.28289358780,-15106.8190066610910 6736794.59864982220,-15166.8942062558330 6736806.61713062690,-15195.3829901481420 6736812.31510593650,-15290.2728818098240 6736831.29463120830,-15329.8103366688590 6736839.2029293180,-15347.0557184092210 6736843.2109036660,-15351.6484711680370 6736844.44164282270,-15368.3596751999520 6736850.43566813880,-15371.3980739090370 6736851.52467590480,-15377.4497910459360 6736854.09380189050,-15380.4756551803580 6736855.37657110580,-15413.5284714028580 6736872.262497440,-15414.9655503690590 6736872.99627892490,-15443.2170790306680 6736887.49968991610,-15454.8128073709250 6736892.57876041620,-15456.2628661897430 6736893.2013100730,-15459.1629949593350 6736894.44461543950,-15471.4299019053380 6736898.62305579240,-15486.7635383708280 6736903.84745482360,-15524.6976920000360 6736915.32428000490,-15532.5896984306750 6736917.71222237870,-15543.2866888621390 6736920.94877348280,-15543.2866888621390 6736920.94877348280,-15543.2866888621390 6736920.94877348280,-15543.2866888621390 6736920.94877348280,-15543.2866888621390 6736920.94877348280,-15543.2866888621390 6736920.94877348280,-15543.1144664779570 6736920.89674465540,-15543.1144664779570 6736920.89674465540,-15569.2930703739980 6736931.11772403310000000000),"
        "(-14999.0450639284860 6736757.94409741740,-14999.0450639284860 6736757.94409741740,-14999.0450639284860 6736757.94409741740,-14999.0450639284860 6736757.94409741740,-14998.3237136282440 6736757.80416062470,-14998.3237136282440 6736757.80416062470,-15074.4307232707360 6736785.59960193090,-15101.5656395162080 6736793.51682595350,-15101.5656395162080 6736793.51682595350,-15102.0753269366810 6736793.64958708270,-15102.0753269366810 6736793.64958708270,-15102.0753269366810 6736793.64958708270,-15102.0753269366810 6736793.64958708270,-15102.0753269366810 6736793.64958708270,-15105.2380806489130 6736794.28289358780,-15106.8190066610910 6736794.59864982220,-15166.8942062558330 6736806.61713062690,-15195.3829901481420 6736812.31510593650,-15290.2728818098240 6736831.29463120830,-15329.8103366688590 6736839.2029293180,-15347.0557184092210 6736843.2109036660,-15351.6484711680370 6736844.44164282270,-15368.3596751999520 6736850.43566813880,-15371.3980739090370 6736851.52467590480,-15377.4497910459360 6736854.09380189050,-15380.4756551803580 6736855.37657110580,-15413.5284714028580 6736872.262497440,-15414.9655503690590 6736872.99627892490,-15443.2170790306680 6736887.49968991610,-15454.8128073709250 6736892.57876041620,-15456.2628661897430 6736893.2013100730,-15459.1629949593350 6736894.44461543950,-15471.4299019053380 6736898.62305579240,-15486.7635383708280 6736903.84745482360,-15524.6976920000360 6736915.32428000490,-15532.5896984306750 6736917.71222237870,-15543.2866888621390 6736920.94877348280,-15543.2866888621390 6736920.94877348280,-15670.4529210342480 6736958.21046822050,-15675.0913147842820 6736959.22952028270,-15689.4798710369670 6736961.8112390470,-15695.8747861639310 6736962.95767313520,-15697.484822222950 6736963.16040719490,-15703.9249664590210 6736963.96775526740000000000),"
        "(-15024.331987571450 6736740.12011628040,-15043.3966526176450 6736741.54639214930,-15098.6064446248090 6736747.23176345690,-15109.6695982572860 6736748.86256334840,-15114.2194484843550 6736749.66271390670,-15209.0076497912120 6736768.50036193520,-15329.0727003271190 6736792.35965132710,-15330.651878623290 6736792.66464307440,-15351.2999521063080 6736795.90293848980,-15398.9493695778070 6736803.37524280050,-15419.5974430608280 6736806.61354248320,-15424.1870343461050 6736807.59490003720,-15445.5403613309970 6736812.9842953617))";

template <bool Clockwise, typename P>
void test_all()
{
    typedef bg::model::linestring<P> linestring;
    typedef bg::model::multi_linestring<linestring> multi_linestring_type;
    typedef bg::model::polygon<P, Clockwise> polygon;
    typedef typename bg::coordinate_type<P>::type coor_type;

    bg::strategy::buffer::join_miter join_miter;
    bg::strategy::buffer::join_round join_round(100);
    bg::strategy::buffer::join_round_by_divide join_round_by_divide(4);
    bg::strategy::buffer::end_flat end_flat;
    bg::strategy::buffer::end_round end_round(100);

    bg::strategy::buffer::end_round end_round32(32);
    bg::strategy::buffer::join_round join_round32(32);

    // Round joins / round ends
    test_one<multi_linestring_type, polygon>("simplex", simplex, join_round, end_round, 49.0217, 1.5);
    test_one<multi_linestring_type, polygon>("two_bends", two_bends, join_round, end_round, 74.73, 1.5);
    test_one<multi_linestring_type, polygon>("turn_inside", turn_inside, join_round, end_round, 86.3313, 1.5);
    test_one<multi_linestring_type, polygon>("two_bends_asym", two_bends, join_round, end_round, 58.3395, 1.5, ut_settings(), 0.75);

    // Round joins / flat ends:
    test_one<multi_linestring_type, polygon>("simplex", simplex, join_round, end_flat, 38.2623, 1.5);
    test_one<multi_linestring_type, polygon>("two_bends", two_bends, join_round, end_flat, 64.6217, 1.5);

    test_one<multi_linestring_type, polygon>("bend_near_start1", bend_near_start1, join_round, end_flat, 202.5910, 9.0);
    test_one<multi_linestring_type, polygon>("bend_near_start2", bend_near_start2, join_round, end_flat, 231.4988, 9.0);

    // TODO this should be fixed test_one<multi_linestring_type, polygon>("turn_inside", turn_inside, join_round, end_flat, 99, 1.5);
    test_one<multi_linestring_type, polygon>("two_bends_asym", two_bends, join_round, end_flat, 52.3793, 1.5, ut_settings(), 0.75);

    // This one is far from done:
    // test_one<multi_linestring_type, polygon>("turn_inside_asym_neg", turn_inside, join_round, end_flat, 99, +1.5, ut_settings(), -1.0);

    // Miter / divide joins, various ends
    test_one<multi_linestring_type, polygon>("two_bends", two_bends, join_round_by_divide, end_flat, 64.6217, 1.5);
    test_one<multi_linestring_type, polygon>("two_bends", two_bends, join_miter, end_flat, 65.1834, 1.5);
    test_one<multi_linestring_type, polygon>("two_bends", two_bends, join_miter, end_round, 75.2917, 1.5);

    test_one<multi_linestring_type, polygon>("degenerate0", degenerate0, join_round, end_round, 0.0, 3.0);
    test_one<multi_linestring_type, polygon>("degenerate1", degenerate1, join_round, end_round, 28.2503, 3.0);
    test_one<multi_linestring_type, polygon>("degenerate2", degenerate2, join_round, end_round, 56.0457, 3.0);
    test_one<multi_linestring_type, polygon>("degenerate3", degenerate3, join_round, end_round, 80.4531, 3.0);
    test_one<multi_linestring_type, polygon>("degenerate4", degenerate4, join_round, end_round, 104.3142, 3.0);

    test_one<multi_linestring_type, polygon>("crossing", crossing, join_round32, end_flat, 2628.4272, 50.0);
    test_one<multi_linestring_type, polygon>("crossing", crossing, join_round32, end_round32,  9893.764, 50.0);

    // Cases formly causing a segmentation fault because a generated side was skipped
    // (The expected area for large distances is about R*R*PI where R is distance)
    // Note that for large distances the flat ends (not tested here) still give weird effects
    {
        // The results can differ between compilers and platforms

        ut_settings settings(40.0);

        ut_settings settings10 = settings;
        settings10.tolerance *= 10.0;

        test_one<multi_linestring_type, polygon>("mikado1_large", mikado1, join_round32, end_round32, 5455052125.0, 41751.0, settings);
        test_one<multi_linestring_type, polygon>("mikado1_small", mikado1, join_round32, end_round32, 1057.37, 10.0);
        test_one<multi_linestring_type, polygon>("mikado1_small", mikado1, join_round32, end_flat, 874.590, 10.0);

        test_one<multi_linestring_type, polygon>("mikado2_large", mikado2, join_round32, end_round32, 19878812253.0, 79610.0, settings10);
        test_one<multi_linestring_type, polygon>("mikado2_small", mikado2, join_round32, end_round32, 1082.470, 10.0);
        test_one<multi_linestring_type, polygon>("mikado2_small", mikado2, join_round32, end_flat, 711.678, 10.0);

        // BSD         29151950588
        // msvc        29151950611
        // clang/linux 29151950612
        // mingw       29151950711
        test_one<multi_linestring_type, polygon>("mikado3_large", mikado3, join_round32, end_round32, 29151950650.0, 96375.0, settings10);
        test_one<multi_linestring_type, polygon>("mikado3_small", mikado3, join_round32, end_round32, 2533.285, 10.0);
        test_one<multi_linestring_type, polygon>("mikado3_small", mikado3, join_round32, end_flat, 2136.236, 10.0);

        test_one<multi_linestring_type, polygon>("mikado4_large", mikado4, join_round32, end_round32, 11212832169.0, 59772.0, settings);
        test_one<multi_linestring_type, polygon>("mikado4_small", mikado4, join_round32, end_round32, 2103.686, 10.0);
        test_one<multi_linestring_type, polygon>("mikado4_small", mikado4, join_round32, end_flat, 1930.785, 10.0);
    }

    if (! BOOST_GEOMETRY_CONDITION((boost::is_same<coor_type, float>::value)))
    {
        // Coordinates in one linestring vary so much that
        // length = geometry::math::sqrt(dx * dx + dy * dy); returns a value of inf for length
        // That geometry is skipped for the buffer
        // SQL Server reports area 2117753600 (for b)
        // PostGIS    reports      2087335072 (for b)
        // BG (2017)  reports      1063005187 (for a/b) which apparently misses parts
        // BG (2015)  reported      794569660 (for a/b)
        // BG (earlier) reported    927681870 (for a/b)
        test_one<multi_linestring_type, polygon>("mysql_2015_04_10a",
            mysql_2015_04_10a, join_round32, end_round32,
            ut_settings::ignore_area(), 0.98, ut_settings::assertions_only());
        test_one<multi_linestring_type, polygon>("mysql_2015_04_10b",
            mysql_2015_04_10b, join_round32, end_round32,
            ut_settings::ignore_area(), 0.98, ut_settings::assertions_only());

        // Two other cases with <inf> for length calculation
        test_one<multi_linestring_type, polygon>("mysql_2015_09_08a",
            mysql_2015_09_08a, join_round32, end_round32,
            ut_settings::ignore_area(), 4051744443.0, ut_settings::assertions_only());
        test_one<multi_linestring_type, polygon>("mysql_2015_09_08b",
            mysql_2015_09_08b, join_round32, end_round32,
            ut_settings::ignore_area(), 2061380362.0, ut_settings::assertions_only());
    }

    // Generates first no interior, then one touching point (no interior),
    // then one real interior ring, then one complete polygon
    test_one<multi_linestring_type, polygon>("mysql_23023665_1",
            mysql_23023665_1, join_round32, end_round32, 1, 1, 186.5504, 1.0);
    test_one<multi_linestring_type, polygon>("touching1_1",
            touching1, join_round32, end_round32, 2, 0, 78.70773, 1.0
#if defined(BOOST_GEOMETRY_USE_RESCALING)
            , ut_settings::ignore_validity() // false positive, due to rescaling. As we remove it, it is gone
#endif
                                             );
    test_one<multi_linestring_type, polygon>("touching2_1",
            touching2, join_round32, end_round32, 1, 1, 107.8991, 1.0);
    test_one<multi_linestring_type, polygon>("mysql_23023665_1_09",
            mysql_23023665_1, join_round32, end_round32, 1, 0, 167.8062, 0.9);
    test_one<multi_linestring_type, polygon>("mysql_23023665_1_11",
            mysql_23023665_1, join_round32, end_round32, 1, 1, 205.1473, 1.1);
    test_one<multi_linestring_type, polygon>("mysql_23023665_1_20",
            mysql_23023665_1, join_round32, end_round32, 1, 1, 368.8422, 2.0);
    test_one<multi_linestring_type, polygon>("mysql_23023665_1_59",
            mysql_23023665_1, join_round32, end_round32, 1, 1, 1020.7214, 5.9821); // very small triangle
    test_one<multi_linestring_type, polygon>("mysql_23023665_1_60",
            mysql_23023665_1, join_round32, end_round32, 1, 0, 1023.3061, 6.0); // no interior anymore

    test_one<multi_linestring_type, polygon>("mysql_23023665_1",
            mysql_23023665_1, join_round32, end_flat, 1, 0, 180.3075, 1.0);
    test_one<multi_linestring_type, polygon>("mysql_23023665_1_09",
            mysql_23023665_1, join_round32, end_flat, 1, 0, 162.7494, 0.9);
    test_one<multi_linestring_type, polygon>("mysql_23023665_1_11",
            mysql_23023665_1, join_round32, end_flat, 1, 0, 197.7607, 1.1);
    test_one<multi_linestring_type, polygon>("mysql_23023665_1_20",
            mysql_23023665_1, join_round32, end_flat, 1, 1, 350.1135, 2.0);

#if ! defined(BOOST_GEOMETRY_USE_RESCALING) || defined(BOOST_GEOMETRY_TEST_FAILURES)
    {
        // Cases failing with rescaling
        ut_settings settings(10.0, false);
        test_one<multi_linestring_type, polygon>("ticket_13444_1",
                ticket_13444, join_round32, end_round32, 3, 0, 11801.7832, 1.0, settings);
        test_one<multi_linestring_type, polygon>("ticket_13444_3",
                ticket_13444, join_round32, end_round32, 3, 1, 34132.0882, 3.0, settings);
        test_one<multi_linestring_type, polygon>("ticket_13444_5",
                ticket_13444, join_round32, end_round32, 2, 1, 50525.1110, 5.0, settings);
    }
#endif

}


int test_main(int, char* [])
{
    BoostGeometryWriteTestConfiguration();

    test_all<true, bg::model::point<default_test_type, 2, bg::cs::cartesian> >();

#if ! defined(BOOST_GEOMETRY_TEST_ONLY_ONE_ORDER)
    test_all<false, bg::model::point<default_test_type, 2, bg::cs::cartesian> >();
#endif

#if defined(BOOST_GEOMETRY_TEST_FAILURES)
    BoostGeometryWriteExpectedFailures(9, 6, 9, 3);
#endif
    return 0;
}
