//===-- Hard-to-round inputs for hypotf ------------------------------C++--===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_LIBC_TEST_SRC_MATH_HYPOTTEST_HARD_TO_ROUND_H
#define LLVM_LIBC_TEST_SRC_MATH_HYPOTTEST_HARD_TO_ROUND_H

#include "utils/MPFRWrapper/MPFRUtils.h"

namespace mpfr = __llvm_libc::testing::mpfr;

constexpr int N_HARD_TO_ROUND = 1216;
constexpr mpfr::BinaryInput<float> HYPOTF_HARD_TO_ROUND[N_HARD_TO_ROUND] = {
    {0x1.ffffecp-1f, 0x1.000002p+27},
    {0x1.900004p+34, 0x1.400002p+23}, /* 45 identical bits */
    {0x1.05555p+34, 0x1.bffffep+23},  /* 44 identical bits */
    {0x1.e5fffap+34, 0x1.affffep+23}, /* 45 identical bits */
    {0x1.260002p+34, 0x1.500002p+23}, /* 45 identical bits */
    {0x1.fffffap+34, 0x1.fffffep+23}, /* 45 identical bits */
    {0x1.8ffffap+34, 0x1.3ffffep+23}, /* 45 identical bits */
    {0x1.87fffcp+35, 0x1.bffffep+23}, /* 47 identical bits */
    {0x1.b8e50ap-52, -0x1.db1e78p-64},
    {0x1.03b54cp-33, 0x1.6ca6bep-45},
    {0x1.e2eff6p+97, -0x1.044cb2p+108},
    {-0x1.6b05c4p-127, 0x1.6b3146p-126},
    {-0x1.6b05c4p-127, 0x1.6b3146p-126},
    {0x1.26b188p-127, -0x1.a4f2fp-128},
    {0x1.e2eff6p+97, -0x1.044cb2p+108},
    // only inexact hard-to-round case with ulp(x) = 2^12*ulp(y):
    {0x1.87fffcp+35, 0x1.bffffep+23},
    // 1200 exact cases with ulp(x) = 2^12*ulp(y):
    {0x1.f0008p+35, 0x1.f7fp+23},
    {0x1.6401dp+35, 0x1.aafp+23},
    {0x1.c0029p+35, 0x1.defp+23},
    {0x1.100274p+35, 0x1.753p+23},
    {0x1.38030cp+35, 0x1.8fbp+23},
    {0x1.940334p+35, 0x1.c6dp+23},
    {0x1.cc041cp+35, 0x1.e55p+23},
    {0x1.0c03f8p+35, 0x1.727p+23},
    {0x1.e404a4p+35, 0x1.f1dp+23},
    {0x1.b004ecp+35, 0x1.d65p+23},
    {0x1.a804ep+35, 0x1.d1fp+23},
    {0x1.5804ccp+35, 0x1.a3bp+23},
    {0x1.98051p+35, 0x1.c91p+23},
    {0x1.480524p+35, 0x1.99dp+23},
    {0x1.900668p+35, 0x1.c49p+23},
    {0x1.28061cp+35, 0x1.855p+23},
    {0x1.7406d8p+35, 0x1.b47p+23},
    {0x1.dc07acp+35, 0x1.edbp+23},
    {0x1.2c078p+35, 0x1.87fp+23},
    {0x1.4407bcp+35, 0x1.975p+23},
    {0x1.4c08ccp+35, 0x1.9c5p+23},
    {0x1.14088p+35, 0x1.77fp+23},
    {0x1.0008ap+35, 0x1.6a1p+23},
    {0x1.bc0b14p+35, 0x1.dcdp+23},
    {0x1.9c0bfcp+35, 0x1.cb5p+23},
    {0x1.240b9cp+35, 0x1.82bp+23},
    {0x1.d00da8p+35, 0x1.e77p+23},
    {0x1.800e68p+35, 0x1.bb7p+23},
    {0x1.8c0eacp+35, 0x1.c25p+23},
    {0x1.080d0cp+35, 0x1.6fbp+23},
    {0x1.600ea8p+35, 0x1.a89p+23},
    {0x1.300fc8p+35, 0x1.8a9p+23},
    {0x1.ec12ap+35, 0x1.f5fp+23},
    {0x1.401094p+35, 0x1.94dp+23},
    {0x1.5012b4p+35, 0x1.9edp+23},
    {0x1.781634p+35, 0x1.b6dp+23},
    {0x1.b8181cp+35, 0x1.dabp+23},
    {0x1.a017f8p+35, 0x1.cd9p+23},
    {0x1.18161cp+35, 0x1.7abp+23},
    {0x1.d41bb8p+35, 0x1.e99p+23},
    {0x1.2018p+35, 0x1.801p+23},
    {0x1.881cp+35, 0x1.c01p+23},
    {0x1.ac1fc4p+35, 0x1.d43p+23},
    {0x1.fc21ep+35, 0x1.fe1p+23},
    {0x1.e02304p+35, 0x1.efdp+23},
    {0x1.341ef4p+35, 0x1.8d3p+23},
    {0x1.3c1facp+35, 0x1.925p+23},
    {0x1.041dbp+35, 0x1.6cfp+23},
    {0x1.5c2124p+35, 0x1.a63p+23},
    {0x1.f827cp+35, 0x1.fc1p+23},
    {0x1.5422dcp+35, 0x1.a15p+23},
    {0x1.e828cp+35, 0x1.f3fp+23},
    {0x1.b429a8p+35, 0x1.d89p+23},
    {0x1.a42904p+35, 0x1.cfdp+23},
    {0x1.d82e4cp+35, 0x1.ebbp+23},
    {0x1.7c2b34p+35, 0x1.b93p+23},
    {0x1.f431ap+35, 0x1.fa1p+23},
    {0x1.842e64p+35, 0x1.bddp+23},
    {0x1.6c2f04p+35, 0x1.afdp+23},
    {0x1.1c2b48p+35, 0x1.7d7p+23},
    {0x1.683048p+35, 0x1.ad7p+23},
    {0x1.703364p+35, 0x1.b23p+23},
    {0x1.10311cp+35, 0x1.755p+23},
    {0x1.c43ab4p+35, 0x1.e13p+23},
    {0x1.c83b7cp+35, 0x1.e35p+23},
    {0x1.383504p+35, 0x1.8fdp+23},
    {0x1.0c3248p+35, 0x1.729p+23},
    {0x1.64373p+35, 0x1.ab1p+23},
    {0x1.f03f8p+35, 0x1.f81p+23},
    {0x1.943c1p+35, 0x1.c6fp+23},
    {0x1.c03e7p+35, 0x1.df1p+23},
    {0x1.583944p+35, 0x1.a3dp+23},
    {0x1.48386p+35, 0x1.99fp+23},
    {0x1.2836c8p+35, 0x1.857p+23},
    {0x1.cc40c8p+35, 0x1.e57p+23},
    {0x1.a83f2p+35, 0x1.d21p+23},
    {0x1.b03fb8p+35, 0x1.d67p+23},
    {0x1.983e34p+35, 0x1.c93p+23},
    {0x1.2c388p+35, 0x1.881p+23},
    {0x1.e442ep+35, 0x1.f1fp+23},
    {0x1.0035e4p+35, 0x1.6a3p+23},
    {0x1.903efcp+35, 0x1.c4bp+23},
    {0x1.743d68p+35, 0x1.b49p+23},
    {0x1.443aa8p+35, 0x1.977p+23},
    {0x1.14378p+35, 0x1.781p+23},
    {0x1.4c3c58p+35, 0x1.9c7p+23},
    {0x1.dc4564p+35, 0x1.eddp+23},
    {0x1.243bf4p+35, 0x1.82dp+23},
    {0x1.bc46bp+35, 0x1.dcfp+23},
    {0x1.083b04p+35, 0x1.6fdp+23},
    {0x1.9c4568p+35, 0x1.cb7p+23},
    {0x1.d04a98p+35, 0x1.e79p+23},
    {0x1.6043bcp+35, 0x1.a8bp+23},
    {0x1.8045d8p+35, 0x1.bb9p+23},
    {0x1.8c46f8p+35, 0x1.c27p+23},
    {0x1.30411cp+35, 0x1.8abp+23},
    {0x1.40433p+35, 0x1.94fp+23},
    {0x1.50469p+35, 0x1.9efp+23},
    {0x1.ec516p+35, 0x1.f61p+23},
    {0x1.184574p+35, 0x1.7adp+23},
    {0x1.784d1p+35, 0x1.b6fp+23},
    {0x1.b85374p+35, 0x1.dadp+23},
    {0x1.204804p+35, 0x1.803p+23},
    {0x1.a051acp+35, 0x1.cdbp+23},
    {0x1.d458ecp+35, 0x1.e9bp+23},
    {0x1.885404p+35, 0x1.c03p+23},
    {0x1.044b5p+35, 0x1.6d1p+23},
    {0x1.34509cp+35, 0x1.8d5p+23},
    {0x1.ac5a4cp+35, 0x1.d45p+23},
    {0x1.3c51f8p+35, 0x1.927p+23},
    {0x1.fc61a4p+35, 0x1.fe3p+23},
    {0x1.5c55ecp+35, 0x1.a65p+23},
    {0x1.e061p+35, 0x1.effp+23},
    {0x1.545708p+35, 0x1.a17p+23},
    {0x1.f86744p+35, 0x1.fc3p+23},
    {0x1.e8674p+35, 0x1.f41p+23},
    {0x1.b464bcp+35, 0x1.d8bp+23},
    {0x1.a463p+35, 0x1.cffp+23},
    {0x1.7c625cp+35, 0x1.b95p+23},
    {0x1.d86bc4p+35, 0x1.ebdp+23},
    {0x1.1c5af8p+35, 0x1.7d9p+23},
    {0x1.84662p+35, 0x1.bdfp+23},
    {0x1.f470e4p+35, 0x1.fa3p+23},
    {0x1.6c65p+35, 0x1.affp+23},
    {0x1.6865f8p+35, 0x1.ad9p+23},
    {0x1.7069acp+35, 0x1.b25p+23},
    {0x1.105fc8p+35, 0x1.757p+23},
    {0x1.0c609cp+35, 0x1.72bp+23},
    {0x1.3867p+35, 0x1.8ffp+23},
    {0x1.646c94p+35, 0x1.ab3p+23},
    {0x1.c476dcp+35, 0x1.e15p+23},
    {0x1.c877e8p+35, 0x1.e37p+23},
    {0x1.286778p+35, 0x1.859p+23},
    {0x1.486bap+35, 0x1.9a1p+23},
    {0x1.586dcp+35, 0x1.a3fp+23},
    {0x1.00632cp+35, 0x1.6a5p+23},
    {0x1.2c6984p+35, 0x1.883p+23},
    {0x1.146684p+35, 0x1.783p+23},
    {0x1.9474fp+35, 0x1.c71p+23},
    {0x1.f07e84p+35, 0x1.f83p+23},
    {0x1.c07a54p+35, 0x1.df3p+23},
    {0x1.446d98p+35, 0x1.979p+23},
    {0x1.98775cp+35, 0x1.c95p+23},
    {0x1.a87964p+35, 0x1.d23p+23},
    {0x1.4c6fe8p+35, 0x1.9c9p+23},
    {0x1.b07a88p+35, 0x1.d69p+23},
    {0x1.7473fcp+35, 0x1.b4bp+23},
    {0x1.cc7d78p+35, 0x1.e59p+23},
    {0x1.907794p+35, 0x1.c4dp+23},
    {0x1.246c5p+35, 0x1.82fp+23},
    {0x1.e4812p+35, 0x1.f21p+23},
    {0x1.0869p+35, 0x1.6ffp+23},
    {0x1.dc832p+35, 0x1.edfp+23},
    {0x1.307274p+35, 0x1.8adp+23},
    {0x1.bc825p+35, 0x1.dd1p+23},
    {0x1.9c7ed8p+35, 0x1.cb9p+23},
    {0x1.6078d4p+35, 0x1.a8dp+23},
    {0x1.4075dp+35, 0x1.951p+23},
    {0x1.807d4cp+35, 0x1.bbbp+23},
    {0x1.8c7f48p+35, 0x1.c29p+23},
    {0x1.d0878cp+35, 0x1.e7bp+23},
    {0x1.507a7p+35, 0x1.9f1p+23},
    {0x1.1874dp+35, 0x1.7afp+23},
    {0x1.ec9024p+35, 0x1.f63p+23},
    {0x1.20780cp+35, 0x1.805p+23},
    {0x1.7883fp+35, 0x1.b71p+23},
    {0x1.b88edp+35, 0x1.dafp+23},
    {0x1.a08b64p+35, 0x1.cddp+23},
    {0x1.0478f4p+35, 0x1.6d3p+23},
    {0x1.888c0cp+35, 0x1.c05p+23},
    {0x1.d49624p+35, 0x1.e9dp+23},
    {0x1.348248p+35, 0x1.8d7p+23},
    {0x1.3c8448p+35, 0x1.929p+23},
    {0x1.ac94d8p+35, 0x1.d47p+23},
    {0x1.5c8ab8p+35, 0x1.a67p+23},
    {0x1.548b38p+35, 0x1.a19p+23},
    {0x1.fca16cp+35, 0x1.fe5p+23},
    {0x1.e09fp+35, 0x1.f01p+23},
    {0x1.f8a6ccp+35, 0x1.fc5p+23},
    {0x1.b49fd4p+35, 0x1.d8dp+23},
    {0x1.a49dp+35, 0x1.d01p+23},
    {0x1.e8a5c4p+35, 0x1.f43p+23},
    {0x1.1c8aacp+35, 0x1.7dbp+23},
    {0x1.7c9988p+35, 0x1.b97p+23},
    {0x1.849dep+35, 0x1.be1p+23},
    {0x1.d8a94p+35, 0x1.ebfp+23},
    {0x1.6c9bp+35, 0x1.b01p+23},
    {0x1.689bacp+35, 0x1.adbp+23},
    {0x1.108e78p+35, 0x1.759p+23},
    {0x1.f4b02cp+35, 0x1.fa5p+23},
    {0x1.0c8ef4p+35, 0x1.72dp+23},
    {0x1.709ff8p+35, 0x1.b27p+23},
    {0x1.3899p+35, 0x1.901p+23},
    {0x1.28982cp+35, 0x1.85bp+23},
    {0x1.009078p+35, 0x1.6a7p+23},
    {0x1.64a1fcp+35, 0x1.ab5p+23},
    {0x1.489ee4p+35, 0x1.9a3p+23},
    {0x1.2c9a8cp+35, 0x1.885p+23},
    {0x1.14958cp+35, 0x1.785p+23},
    {0x1.58a24p+35, 0x1.a41p+23},
    {0x1.c4b308p+35, 0x1.e17p+23},
    {0x1.c8b458p+35, 0x1.e39p+23},
    {0x1.44a08cp+35, 0x1.97bp+23},
    {0x1.94add4p+35, 0x1.c73p+23},
    {0x1.4ca37cp+35, 0x1.9cbp+23},
    {0x1.249cbp+35, 0x1.831p+23},
    {0x1.0897p+35, 0x1.701p+23},
    {0x1.c0b63cp+35, 0x1.df5p+23},
    {0x1.74aa94p+35, 0x1.b4dp+23},
    {0x1.98b088p+35, 0x1.c97p+23},
    {0x1.f0bd8cp+35, 0x1.f85p+23},
    {0x1.a8b3acp+35, 0x1.d25p+23},
    {0x1.b0b55cp+35, 0x1.d6bp+23},
    {0x1.90b03p+35, 0x1.c4fp+23},
    {0x1.ccba2cp+35, 0x1.e5bp+23},
    {0x1.e4bf64p+35, 0x1.f23p+23},
    {0x1.30a3dp+35, 0x1.8afp+23},
    {0x1.dcc0ep+35, 0x1.ee1p+23},
    {0x1.40a874p+35, 0x1.953p+23},
    {0x1.60adfp+35, 0x1.a8fp+23},
    {0x1.9cb84cp+35, 0x1.cbbp+23},
    {0x1.bcbdf4p+35, 0x1.dd3p+23},
    {0x1.80b4c4p+35, 0x1.bbdp+23},
    {0x1.8cb79cp+35, 0x1.c2bp+23},
    {0x1.50ae54p+35, 0x1.9f3p+23},
    {0x1.18a43p+35, 0x1.7b1p+23},
    {0x1.d0c484p+35, 0x1.e7dp+23},
    {0x1.20a818p+35, 0x1.807p+23},
    {0x1.78bad4p+35, 0x1.b73p+23},
    {0x1.ecceecp+35, 0x1.f65p+23},
    {0x1.04a69cp+35, 0x1.6d5p+23},
    {0x1.a0c52p+35, 0x1.cdfp+23},
    {0x1.b8ca3p+35, 0x1.db1p+23},
    {0x1.34b3f8p+35, 0x1.8d9p+23},
    {0x1.88c418p+35, 0x1.c07p+23},
    {0x1.3cb69cp+35, 0x1.92bp+23},
    {0x1.d4d36p+35, 0x1.e9fp+23},
    {0x1.5cbf88p+35, 0x1.a69p+23},
    {0x1.accf68p+35, 0x1.d49p+23},
    {0x1.54bf6cp+35, 0x1.a1bp+23},
    {0x1.e0dd04p+35, 0x1.f03p+23},
    {0x1.fce138p+35, 0x1.fe7p+23},
    {0x1.1cba64p+35, 0x1.7ddp+23},
    {0x1.b4dafp+35, 0x1.d8fp+23},
    {0x1.a4d704p+35, 0x1.d03p+23},
    {0x1.7cd0b8p+35, 0x1.b99p+23},
    {0x1.f8e658p+35, 0x1.fc7p+23},
    {0x1.e8e44cp+35, 0x1.f45p+23},
    {0x1.10bd2cp+35, 0x1.75bp+23},
    {0x1.84d5a4p+35, 0x1.be3p+23},
    {0x1.6cd104p+35, 0x1.b03p+23},
    {0x1.0cbd5p+35, 0x1.72fp+23},
    {0x1.68d164p+35, 0x1.addp+23},
    {0x1.d8e6cp+35, 0x1.ec1p+23},
    {0x1.38cb04p+35, 0x1.903p+23},
    {0x1.70d648p+35, 0x1.b29p+23},
    {0x1.00bdc8p+35, 0x1.6a9p+23},
    {0x1.f4ef78p+35, 0x1.fa7p+23},
    {0x1.28c8e4p+35, 0x1.85dp+23},
    {0x1.14c498p+35, 0x1.787p+23},
    {0x1.2ccb98p+35, 0x1.887p+23},
    {0x1.48d22cp+35, 0x1.9a5p+23},
    {0x1.64d768p+35, 0x1.ab7p+23},
    {0x1.58d6c4p+35, 0x1.a43p+23},
    {0x1.44d384p+35, 0x1.97dp+23},
    {0x1.08c504p+35, 0x1.703p+23},
    {0x1.24cd14p+35, 0x1.833p+23},
    {0x1.4cd714p+35, 0x1.9cdp+23},
    {0x1.c4ef38p+35, 0x1.e19p+23},
    {0x1.c8f0ccp+35, 0x1.e3bp+23},
    {0x1.94e6bcp+35, 0x1.c75p+23},
    {0x1.74e13p+35, 0x1.b4fp+23},
    {0x1.c0f228p+35, 0x1.df7p+23},
    {0x1.98e9b8p+35, 0x1.c99p+23},
    {0x1.90e8dp+35, 0x1.c51p+23},
    {0x1.a8edf8p+35, 0x1.d27p+23},
    {0x1.30d53p+35, 0x1.8b1p+23},
    {0x1.b0f034p+35, 0x1.d6dp+23},
    {0x1.f0fc98p+35, 0x1.f87p+23},
    {0x1.ccf6e4p+35, 0x1.e5dp+23},
    {0x1.40db1cp+35, 0x1.955p+23},
    {0x1.60e31p+35, 0x1.a91p+23},
    {0x1.e4fdacp+35, 0x1.f25p+23},
    {0x1.18d394p+35, 0x1.7b3p+23},
    {0x1.dcfea4p+35, 0x1.ee3p+23},
    {0x1.9cf1c4p+35, 0x1.cbdp+23},
    {0x1.80ec4p+35, 0x1.bbfp+23},
    {0x1.50e23cp+35, 0x1.9f5p+23},
    {0x1.bcf99cp+35, 0x1.dd5p+23},
    {0x1.8ceff4p+35, 0x1.c2dp+23},
    {0x1.20d828p+35, 0x1.809p+23},
    {0x1.d1018p+35, 0x1.e7fp+23},
    {0x1.04d448p+35, 0x1.6d7p+23},
    {0x1.78f1bcp+35, 0x1.b75p+23},
    {0x1.34e5acp+35, 0x1.8dbp+23},
    {0x1.ed0db8p+35, 0x1.f67p+23},
    {0x1.3ce8f4p+35, 0x1.92dp+23},
    {0x1.a0feep+35, 0x1.ce1p+23},
    {0x1.b90594p+35, 0x1.db3p+23},
    {0x1.88fc28p+35, 0x1.c09p+23},
    {0x1.5cf45cp+35, 0x1.a6bp+23},
    {0x1.54f3a4p+35, 0x1.a1dp+23},
    {0x1.d510ap+35, 0x1.ea1p+23},
    {0x1.ad09fcp+35, 0x1.d4bp+23},
    {0x1.1cea2p+35, 0x1.7dfp+23},
    {0x1.e11b0cp+35, 0x1.f05p+23},
    {0x1.fd2108p+35, 0x1.fe9p+23},
    {0x1.7d07ecp+35, 0x1.b9bp+23},
    {0x1.b5161p+35, 0x1.d91p+23},
    {0x1.a5110cp+35, 0x1.d05p+23},
    {0x1.10ebe4p+35, 0x1.75dp+23},
    {0x1.0cebbp+35, 0x1.731p+23},
    {0x1.e922d8p+35, 0x1.f47p+23},
    {0x1.f925e8p+35, 0x1.fc9p+23},
    {0x1.850d6cp+35, 0x1.be5p+23},
    {0x1.6d070cp+35, 0x1.b05p+23},
    {0x1.69072p+35, 0x1.adfp+23},
    {0x1.00eb1cp+35, 0x1.6abp+23},
    {0x1.38fd0cp+35, 0x1.905p+23},
    {0x1.d92444p+35, 0x1.ec3p+23},
    {0x1.28f9ap+35, 0x1.85fp+23},
    {0x1.14f3a8p+35, 0x1.789p+23},
    {0x1.710c9cp+35, 0x1.b2bp+23},
    {0x1.2cfca8p+35, 0x1.889p+23},
    {0x1.490578p+35, 0x1.9a7p+23},
    {0x1.f52ec8p+35, 0x1.fa9p+23},
    {0x1.650cd8p+35, 0x1.ab9p+23},
    {0x1.590b4cp+35, 0x1.a45p+23},
    {0x1.08f30cp+35, 0x1.705p+23},
    {0x1.45068p+35, 0x1.97fp+23},
    {0x1.24fd7cp+35, 0x1.835p+23},
    {0x1.4d0abp+35, 0x1.9cfp+23},
    {0x1.c52b6cp+35, 0x1.e1bp+23},
    {0x1.951fa8p+35, 0x1.c77p+23},
    {0x1.7517dp+35, 0x1.b51p+23},
    {0x1.310694p+35, 0x1.8b3p+23},
    {0x1.c92d44p+35, 0x1.e3dp+23},
    {0x1.9922ecp+35, 0x1.c9bp+23},
    {0x1.912174p+35, 0x1.c53p+23},
    {0x1.c12e18p+35, 0x1.df9p+23},
    {0x1.a92848p+35, 0x1.d29p+23},
    {0x1.b12b1p+35, 0x1.d6fp+23},
    {0x1.410dc8p+35, 0x1.957p+23},
    {0x1.1902fcp+35, 0x1.7b5p+23},
    {0x1.cd33ap+35, 0x1.e5fp+23},
    {0x1.f13ba8p+35, 0x1.f89p+23},
    {0x1.611834p+35, 0x1.a93p+23},
    {0x1.511628p+35, 0x1.9f7p+23},
    {0x1.21083cp+35, 0x1.80bp+23},
    {0x1.e53bf8p+35, 0x1.f27p+23},
    {0x1.8123cp+35, 0x1.bc1p+23},
    {0x1.9d2b4p+35, 0x1.cbfp+23},
    {0x1.dd3c6cp+35, 0x1.ee5p+23},
    {0x1.8d285p+35, 0x1.c2fp+23},
    {0x1.bd3548p+35, 0x1.dd7p+23},
    {0x1.0501f8p+35, 0x1.6d9p+23},
    {0x1.d13e8p+35, 0x1.e81p+23},
    {0x1.7928a8p+35, 0x1.b77p+23},
    {0x1.351764p+35, 0x1.8ddp+23},
    {0x1.3d1b5p+35, 0x1.92fp+23},
    {0x1.ed4c88p+35, 0x1.f69p+23},
    {0x1.a138a4p+35, 0x1.ce3p+23},
    {0x1.b940fcp+35, 0x1.db5p+23},
    {0x1.89343cp+35, 0x1.c0bp+23},
    {0x1.5d2934p+35, 0x1.a6dp+23},
    {0x1.5527ep+35, 0x1.a1fp+23},
    {0x1.d54de4p+35, 0x1.ea3p+23},
    {0x1.ad4494p+35, 0x1.d4dp+23},
    {0x1.1d19ep+35, 0x1.7e1p+23},
    {0x1.111aap+35, 0x1.75fp+23},
    {0x1.e15918p+35, 0x1.f07p+23},
    {0x1.0d1a14p+35, 0x1.733p+23},
    {0x1.7d3f24p+35, 0x1.b9dp+23},
    {0x1.fd60dcp+35, 0x1.febp+23},
    {0x1.a54b18p+35, 0x1.d07p+23},
    {0x1.b55134p+35, 0x1.d93p+23},
    {0x1.011874p+35, 0x1.6adp+23},
    {0x1.6d3d18p+35, 0x1.b07p+23},
    {0x1.854538p+35, 0x1.be7p+23},
    {0x1.693cep+35, 0x1.ae1p+23},
    {0x1.392f18p+35, 0x1.907p+23},
    {0x1.292a6p+35, 0x1.861p+23},
    {0x1.e96168p+35, 0x1.f49p+23},
    {0x1.f9657cp+35, 0x1.fcbp+23},
    {0x1.1522bcp+35, 0x1.78bp+23},
    {0x1.2d2dbcp+35, 0x1.88bp+23},
    {0x1.7142f4p+35, 0x1.b2dp+23},
    {0x1.4938c8p+35, 0x1.9a9p+23},
    {0x1.d961ccp+35, 0x1.ec5p+23},
    {0x1.092118p+35, 0x1.707p+23},
    {0x1.252de8p+35, 0x1.837p+23},
    {0x1.65424cp+35, 0x1.abbp+23},
    {0x1.593fd8p+35, 0x1.a47p+23},
    {0x1.45398p+35, 0x1.981p+23},
    {0x1.4d3e5p+35, 0x1.9d1p+23},
    {0x1.f56e1cp+35, 0x1.fabp+23},
    {0x1.3137fcp+35, 0x1.8b5p+23},
    {0x1.754e74p+35, 0x1.b53p+23},
    {0x1.955898p+35, 0x1.c79p+23},
    {0x1.c567a4p+35, 0x1.e1dp+23},
    {0x1.c969cp+35, 0x1.e3fp+23},
    {0x1.193268p+35, 0x1.7b7p+23},
    {0x1.414078p+35, 0x1.959p+23},
    {0x1.915a1cp+35, 0x1.c55p+23},
    {0x1.995c24p+35, 0x1.c9dp+23},
    {0x1.a9629cp+35, 0x1.d2bp+23},
    {0x1.c16a0cp+35, 0x1.dfbp+23},
    {0x1.b165fp+35, 0x1.d71p+23},
    {0x1.614d5cp+35, 0x1.a95p+23},
    {0x1.213854p+35, 0x1.80dp+23},
    {0x1.cd706p+35, 0x1.e61p+23},
    {0x1.514a18p+35, 0x1.9f9p+23},
    {0x1.f17abcp+35, 0x1.f8bp+23},
    {0x1.815b44p+35, 0x1.bc3p+23},
    {0x1.052facp+35, 0x1.6dbp+23},
    {0x1.9d64cp+35, 0x1.cc1p+23},
    {0x1.8d60bp+35, 0x1.c31p+23},
    {0x1.e57a48p+35, 0x1.f29p+23},
    {0x1.bd70f8p+35, 0x1.dd9p+23},
    {0x1.dd7a38p+35, 0x1.ee7p+23},
    {0x1.35492p+35, 0x1.8dfp+23},
    {0x1.d17b84p+35, 0x1.e83p+23},
    {0x1.795f98p+35, 0x1.b79p+23},
    {0x1.3d4dbp+35, 0x1.931p+23},
    {0x1.a1726cp+35, 0x1.ce5p+23},
    {0x1.b97c68p+35, 0x1.db7p+23},
    {0x1.896c54p+35, 0x1.c0dp+23},
    {0x1.5d5e1p+35, 0x1.a6fp+23},
    {0x1.ed8b5cp+35, 0x1.f6bp+23},
    {0x1.555c2p+35, 0x1.a21p+23},
    {0x1.1d49a4p+35, 0x1.7e3p+23},
    {0x1.ad7f3p+35, 0x1.d4fp+23},
    {0x1.d58b2cp+35, 0x1.ea5p+23},
    {0x1.11496p+35, 0x1.761p+23},
    {0x1.0d487cp+35, 0x1.735p+23},
    {0x1.0145dp+35, 0x1.6afp+23},
    {0x1.7d766p+35, 0x1.b9fp+23},
    {0x1.e19728p+35, 0x1.f09p+23},
    {0x1.a58528p+35, 0x1.d09p+23},
    {0x1.b58c5cp+35, 0x1.d95p+23},
    {0x1.396128p+35, 0x1.909p+23},
    {0x1.295b24p+35, 0x1.863p+23},
    {0x1.1551d4p+35, 0x1.78dp+23},
    {0x1.6d7328p+35, 0x1.b09p+23},
    {0x1.fda0b4p+35, 0x1.fedp+23},
    {0x1.6972a4p+35, 0x1.ae3p+23},
    {0x1.857d08p+35, 0x1.be9p+23},
    {0x1.2d5ed4p+35, 0x1.88dp+23},
    {0x1.094f28p+35, 0x1.709p+23},
    {0x1.496c1cp+35, 0x1.9abp+23},
    {0x1.e99ffcp+35, 0x1.f4bp+23},
    {0x1.f9a514p+35, 0x1.fcdp+23},
    {0x1.71795p+35, 0x1.b2fp+23},
    {0x1.255e58p+35, 0x1.839p+23},
    {0x1.597468p+35, 0x1.a49p+23},
    {0x1.456c84p+35, 0x1.983p+23},
    {0x1.6577c4p+35, 0x1.abdp+23},
    {0x1.d99f58p+35, 0x1.ec7p+23},
    {0x1.4d71f4p+35, 0x1.9d3p+23},
    {0x1.316968p+35, 0x1.8b7p+23},
    {0x1.f5ad74p+35, 0x1.fadp+23},
    {0x1.1961d8p+35, 0x1.7b9p+23},
    {0x1.75851cp+35, 0x1.b55p+23},
    {0x1.95918cp+35, 0x1.c7bp+23},
    {0x1.41732cp+35, 0x1.95bp+23},
    {0x1.c5a3ep+35, 0x1.e1fp+23},
    {0x1.9192c8p+35, 0x1.c57p+23},
    {0x1.c9a64p+35, 0x1.e41p+23},
    {0x1.21687p+35, 0x1.80fp+23},
    {0x1.99956p+35, 0x1.c9fp+23},
    {0x1.618288p+35, 0x1.a97p+23},
    {0x1.a99cf4p+35, 0x1.d2dp+23},
    {0x1.c1a604p+35, 0x1.dfdp+23},
    {0x1.b1a0d4p+35, 0x1.d73p+23},
    {0x1.517e0cp+35, 0x1.9fbp+23},
    {0x1.055d64p+35, 0x1.6ddp+23},
    {0x1.cdad24p+35, 0x1.e63p+23},
    {0x1.8192ccp+35, 0x1.bc5p+23},
    {0x1.f1b9d4p+35, 0x1.f8dp+23},
    {0x1.8d9914p+35, 0x1.c33p+23},
    {0x1.9d9e44p+35, 0x1.cc3p+23},
    {0x1.bdacacp+35, 0x1.ddbp+23},
    {0x1.e5b89cp+35, 0x1.f2bp+23},
    {0x1.357aep+35, 0x1.8e1p+23},
    {0x1.ddb808p+35, 0x1.ee9p+23},
    {0x1.3d8014p+35, 0x1.933p+23},
    {0x1.79968cp+35, 0x1.b7bp+23},
    {0x1.d1b88cp+35, 0x1.e85p+23},
    {0x1.5d92fp+35, 0x1.a71p+23},
    {0x1.a1ac38p+35, 0x1.ce7p+23},
    {0x1.559064p+35, 0x1.a23p+23},
    {0x1.89a47p+35, 0x1.c0fp+23},
    {0x1.b9b7d8p+35, 0x1.db9p+23},
    {0x1.1d796cp+35, 0x1.7e5p+23},
    {0x1.edca34p+35, 0x1.f6dp+23},
    {0x1.117824p+35, 0x1.763p+23},
    {0x1.adb9dp+35, 0x1.d51p+23},
    {0x1.0d76e8p+35, 0x1.737p+23},
    {0x1.d5c878p+35, 0x1.ea7p+23},
    {0x1.01733p+35, 0x1.6b1p+23},
    {0x1.1580fp+35, 0x1.78fp+23},
    {0x1.7dadap+35, 0x1.ba1p+23},
    {0x1.298becp+35, 0x1.865p+23},
    {0x1.39933cp+35, 0x1.90bp+23},
    {0x1.2d8ffp+35, 0x1.88fp+23},
    {0x1.e1d53cp+35, 0x1.f0bp+23},
    {0x1.a5bf3cp+35, 0x1.d0bp+23},
    {0x1.b5c788p+35, 0x1.d97p+23},
    {0x1.6da93cp+35, 0x1.b0bp+23},
    {0x1.69a86cp+35, 0x1.ae5p+23},
    {0x1.097d3cp+35, 0x1.70bp+23},
    {0x1.85b4dcp+35, 0x1.bebp+23},
    {0x1.499f74p+35, 0x1.9adp+23},
    {0x1.fde09p+35, 0x1.fefp+23},
    {0x1.258eccp+35, 0x1.83bp+23},
    {0x1.71afbp+35, 0x1.b31p+23},
    {0x1.459f8cp+35, 0x1.985p+23},
    {0x1.59a8fcp+35, 0x1.a4bp+23},
    {0x1.e9de94p+35, 0x1.f4dp+23},
    {0x1.f9e4bp+35, 0x1.fcfp+23},
    {0x1.65ad4p+35, 0x1.abfp+23},
    {0x1.4da59cp+35, 0x1.9d5p+23},
    {0x1.319ad8p+35, 0x1.8b9p+23},
    {0x1.d9dce8p+35, 0x1.ec9p+23},
    {0x1.19914cp+35, 0x1.7bbp+23},
    {0x1.41a5e4p+35, 0x1.95dp+23},
    {0x1.75bbc8p+35, 0x1.b57p+23},
    {0x1.f5ecdp+35, 0x1.fafp+23},
    {0x1.21989p+35, 0x1.811p+23},
    {0x1.95ca84p+35, 0x1.c7dp+23},
    {0x1.61b7b8p+35, 0x1.a99p+23},
    {0x1.c5e02p+35, 0x1.e21p+23},
    {0x1.91cb78p+35, 0x1.c59p+23},
    {0x1.058b2p+35, 0x1.6dfp+23},
    {0x1.99ceap+35, 0x1.ca1p+23},
    {0x1.51b204p+35, 0x1.9fdp+23},
    {0x1.c9e2c4p+35, 0x1.e43p+23},
    {0x1.a9d75p+35, 0x1.d2fp+23},
    {0x1.b1dbbcp+35, 0x1.d75p+23},
    {0x1.c1e2p+35, 0x1.dffp+23},
    {0x1.cde9ecp+35, 0x1.e65p+23},
    {0x1.81ca58p+35, 0x1.bc7p+23},
    {0x1.8dd17cp+35, 0x1.c35p+23},
    {0x1.35aca4p+35, 0x1.8e3p+23},
    {0x1.9dd7ccp+35, 0x1.cc5p+23},
    {0x1.f1f8fp+35, 0x1.f8fp+23},
    {0x1.bde864p+35, 0x1.dddp+23},
    {0x1.3db27cp+35, 0x1.935p+23},
    {0x1.e5f6f4p+35, 0x1.f2dp+23},
    {0x1.ddf5dcp+35, 0x1.eebp+23},
    {0x1.79cd84p+35, 0x1.b7dp+23},
    {0x1.d1f598p+35, 0x1.e87p+23},
    {0x1.5dc7d4p+35, 0x1.a73p+23},
    {0x1.55c4acp+35, 0x1.a25p+23},
    {0x1.1da938p+35, 0x1.7e7p+23},
    {0x1.a1e608p+35, 0x1.ce9p+23},
    {0x1.89dc9p+35, 0x1.c11p+23},
    {0x1.b9f34cp+35, 0x1.dbbp+23},
    {0x1.11a6ecp+35, 0x1.765p+23},
    {0x1.ee091p+35, 0x1.f6fp+23},
    {0x1.0da558p+35, 0x1.739p+23},
    {0x1.01a094p+35, 0x1.6b3p+23},
    {0x1.adf474p+35, 0x1.d53p+23},
    {0x1.d605c8p+35, 0x1.ea9p+23},
    {0x1.15b01p+35, 0x1.791p+23},
    {0x1.29bcb8p+35, 0x1.867p+23},
    {0x1.39c554p+35, 0x1.90dp+23},
    {0x1.2dc11p+35, 0x1.891p+23},
    {0x1.7de4e4p+35, 0x1.ba3p+23},
    {0x1.09ab54p+35, 0x1.70dp+23},
    {0x1.69de38p+35, 0x1.ae7p+23},
    {0x1.6ddf54p+35, 0x1.b0dp+23},
    {0x1.a5f954p+35, 0x1.d0dp+23},
    {0x1.b602b8p+35, 0x1.d99p+23},
    {0x1.49d2dp+35, 0x1.9afp+23},
    {0x1.85ecb4p+35, 0x1.bedp+23},
    {0x1.25bf44p+35, 0x1.83dp+23},
    {0x1.e21354p+35, 0x1.f0dp+23},
    {0x1.45d298p+35, 0x1.987p+23},
    {0x1.71e614p+35, 0x1.b33p+23},
    {0x1.59dd94p+35, 0x1.a4dp+23},
    {0x1.fe207p+35, 0x1.ff1p+23},
    {0x1.4dd948p+35, 0x1.9d7p+23},
    {0x1.65e2cp+35, 0x1.ac1p+23},
    {0x1.31cc4cp+35, 0x1.8bbp+23},
    {0x1.ea1d3p+35, 0x1.f4fp+23},
    {0x1.19c0c4p+35, 0x1.7bdp+23},
    {0x1.fa245p+35, 0x1.fd1p+23},
    {0x1.da1a7cp+35, 0x1.ecbp+23},
    {0x1.41d8ap+35, 0x1.95fp+23},
    {0x1.21c8b4p+35, 0x1.813p+23},
    {0x1.75f278p+35, 0x1.b59p+23},
    {0x1.05b8ep+35, 0x1.6e1p+23},
    {0x1.96038p+35, 0x1.c7fp+23},
    {0x1.61ececp+35, 0x1.a9bp+23},
    {0x1.f62c3p+35, 0x1.fb1p+23},
    {0x1.51e6p+35, 0x1.9ffp+23},
    {0x1.92042cp+35, 0x1.c5bp+23},
    {0x1.c61c64p+35, 0x1.e23p+23},
    {0x1.9a07e4p+35, 0x1.ca3p+23},
    {0x1.aa11bp+35, 0x1.d31p+23},
    {0x1.ca1f4cp+35, 0x1.e45p+23},
    {0x1.b216a8p+35, 0x1.d77p+23},
    {0x1.c21ep+35, 0x1.e01p+23},
    {0x1.35de6cp+35, 0x1.8e5p+23},
    {0x1.8201e8p+35, 0x1.bc9p+23},
    {0x1.ce26b8p+35, 0x1.e67p+23},
    {0x1.8e09e8p+35, 0x1.c37p+23},
    {0x1.3de4e8p+35, 0x1.937p+23},
    {0x1.9e1158p+35, 0x1.cc7p+23},
    {0x1.f2381p+35, 0x1.f91p+23},
    {0x1.be242p+35, 0x1.ddfp+23},
    {0x1.7a048p+35, 0x1.b7fp+23},
    {0x1.e6355p+35, 0x1.f2fp+23},
    {0x1.de33b4p+35, 0x1.eedp+23},
    {0x1.1dd908p+35, 0x1.7e9p+23},
    {0x1.d232a8p+35, 0x1.e89p+23},
    {0x1.55f8f8p+35, 0x1.a27p+23},
    {0x1.5dfcbcp+35, 0x1.a75p+23},
    {0x1.11d5b8p+35, 0x1.767p+23},
    {0x1.8a14b4p+35, 0x1.c13p+23},
    {0x1.a21fdcp+35, 0x1.cebp+23},
    {0x1.0dd3ccp+35, 0x1.73bp+23},
    {0x1.ba2ec4p+35, 0x1.dbdp+23},
    {0x1.01cdfcp+35, 0x1.6b5p+23},
    {0x1.ee47fp+35, 0x1.f71p+23},
    {0x1.15df34p+35, 0x1.793p+23},
    {0x1.ae2f1cp+35, 0x1.d55p+23},
    {0x1.29ed88p+35, 0x1.869p+23},
    {0x1.39f77p+35, 0x1.90fp+23},
    {0x1.d6431cp+35, 0x1.eabp+23},
    {0x1.09d97p+35, 0x1.70fp+23},
    {0x1.2df234p+35, 0x1.893p+23},
    {0x1.7e1c2cp+35, 0x1.ba5p+23},
    {0x1.25efcp+35, 0x1.83fp+23},
    {0x1.6a1408p+35, 0x1.ae9p+23},
    {0x1.4a063p+35, 0x1.9b1p+23},
    {0x1.6e157p+35, 0x1.b0fp+23},
    {0x1.a6337p+35, 0x1.d0fp+23},
    {0x1.86249p+35, 0x1.befp+23},
    {0x1.b63decp+35, 0x1.d9bp+23},
    {0x1.4605a8p+35, 0x1.989p+23},
    {0x1.e2517p+35, 0x1.f0fp+23},
    {0x1.5a123p+35, 0x1.a4fp+23},
    {0x1.721c7cp+35, 0x1.b35p+23},
    {0x1.31fdc4p+35, 0x1.8bdp+23},
    {0x1.4e0cf8p+35, 0x1.9d9p+23},
    {0x1.661844p+35, 0x1.ac3p+23},
    {0x1.19f04p+35, 0x1.7bfp+23},
    {0x1.fe6054p+35, 0x1.ff3p+23},
    {0x1.ea5bdp+35, 0x1.f51p+23},
    {0x1.21f8dcp+35, 0x1.815p+23},
    {0x1.fa63f4p+35, 0x1.fd3p+23},
    {0x1.420b6p+35, 0x1.961p+23},
    {0x1.05e6a4p+35, 0x1.6e3p+23},
    {0x1.da5814p+35, 0x1.ecdp+23},
    {0x1.76292cp+35, 0x1.b5bp+23},
    {0x1.622224p+35, 0x1.a9dp+23},
    {0x1.963c8p+35, 0x1.c81p+23},
    {0x1.521ap+35, 0x1.a01p+23},
    {0x1.923ce4p+35, 0x1.c5dp+23},
    {0x1.f66b94p+35, 0x1.fb3p+23},
    {0x1.9a412cp+35, 0x1.ca5p+23},
    {0x1.c658acp+35, 0x1.e25p+23},
    {0x1.aa4c14p+35, 0x1.d33p+23},
    {0x1.361038p+35, 0x1.8e7p+23},
    {0x1.ca5bd8p+35, 0x1.e47p+23},
    {0x1.b25198p+35, 0x1.d79p+23},
    {0x1.c25a04p+35, 0x1.e03p+23},
    {0x1.82397cp+35, 0x1.bcbp+23},
    {0x1.3e1758p+35, 0x1.939p+23},
    {0x1.8e4258p+35, 0x1.c39p+23},
    {0x1.ce6388p+35, 0x1.e69p+23},
    {0x1.9e4ae8p+35, 0x1.cc9p+23},
    {0x1.7a3b8p+35, 0x1.b81p+23},
    {0x1.be5fep+35, 0x1.de1p+23},
    {0x1.f27734p+35, 0x1.f93p+23},
    {0x1.1e08dcp+35, 0x1.7ebp+23},
    {0x1.de719p+35, 0x1.eefp+23},
    {0x1.e673bp+35, 0x1.f31p+23},
    {0x1.120488p+35, 0x1.769p+23},
    {0x1.562d48p+35, 0x1.a29p+23},
    {0x1.5e31a8p+35, 0x1.a77p+23},
    {0x1.d26fbcp+35, 0x1.e8bp+23},
    {0x1.0e0244p+35, 0x1.73dp+23},
    {0x1.01fb68p+35, 0x1.6b7p+23},
    {0x1.8a4cdcp+35, 0x1.c15p+23},
    {0x1.a259b4p+35, 0x1.cedp+23},
    {0x1.ba6a4p+35, 0x1.dbfp+23},
    {0x1.160e5cp+35, 0x1.795p+23},
    {0x1.2a1e5cp+35, 0x1.86bp+23},
    {0x1.ee86d4p+35, 0x1.f73p+23},
    {0x1.0a079p+35, 0x1.711p+23},
    {0x1.3a299p+35, 0x1.911p+23},
    {0x1.2e235cp+35, 0x1.895p+23},
    {0x1.ae69c8p+35, 0x1.d57p+23},
    {0x1.d68074p+35, 0x1.eadp+23},
    {0x1.26204p+35, 0x1.841p+23},
    {0x1.7e5378p+35, 0x1.ba7p+23},
    {0x1.4a3994p+35, 0x1.9b3p+23},
    {0x1.6a49dcp+35, 0x1.aebp+23},
    {0x1.6e4b9p+35, 0x1.b11p+23},
    {0x1.865c7p+35, 0x1.bf1p+23},
    {0x1.4638bcp+35, 0x1.98bp+23},
    {0x1.a66d9p+35, 0x1.d11p+23},
    {0x1.b67924p+35, 0x1.d9dp+23},
    {0x1.322f4p+35, 0x1.8bfp+23},
    {0x1.5a46dp+35, 0x1.a51p+23},
    {0x1.1a1fcp+35, 0x1.7c1p+23},
    {0x1.4e40acp+35, 0x1.9dbp+23},
    {0x1.7252e8p+35, 0x1.b37p+23},
    {0x1.e28f9p+35, 0x1.f11p+23},
    {0x1.664dccp+35, 0x1.ac5p+23},
    {0x1.222908p+35, 0x1.817p+23},
    {0x1.06146cp+35, 0x1.6e5p+23},
    {0x1.423e24p+35, 0x1.963p+23},
    {0x1.fea03cp+35, 0x1.ff5p+23},
    {0x1.ea9a74p+35, 0x1.f53p+23},
    {0x1.faa39cp+35, 0x1.fd5p+23},
    {0x1.765fe4p+35, 0x1.b5dp+23},
    {0x1.da95bp+35, 0x1.ecfp+23},
    {0x1.524e04p+35, 0x1.a03p+23},
    {0x1.62576p+35, 0x1.a9fp+23},
    {0x1.967584p+35, 0x1.c83p+23},
    {0x1.9275ap+35, 0x1.c5fp+23},
    {0x1.364208p+35, 0x1.8e9p+23},
    {0x1.9a7a78p+35, 0x1.ca7p+23},
    {0x1.c694f8p+35, 0x1.e27p+23},
    {0x1.f6aafcp+35, 0x1.fb5p+23},
    {0x1.aa867cp+35, 0x1.d35p+23},
    {0x1.3e49ccp+35, 0x1.93bp+23},
    {0x1.b28c8cp+35, 0x1.d7bp+23},
    {0x1.ca9868p+35, 0x1.e49p+23},
    {0x1.c2960cp+35, 0x1.e05p+23},
    {0x1.827114p+35, 0x1.bcdp+23},
    {0x1.8e7accp+35, 0x1.c3bp+23},
    {0x1.9e847cp+35, 0x1.ccbp+23},
    {0x1.cea05cp+35, 0x1.e6bp+23},
    {0x1.1e38b4p+35, 0x1.7edp+23},
    {0x1.7a7284p+35, 0x1.b83p+23},
    {0x1.be9ba4p+35, 0x1.de3p+23},
    {0x1.12335cp+35, 0x1.76bp+23},
    {0x1.f2b65cp+35, 0x1.f95p+23},
    {0x1.0e30cp+35, 0x1.73fp+23},
    {0x1.56619cp+35, 0x1.a2bp+23},
    {0x1.5e6698p+35, 0x1.a79p+23},
    {0x1.deaf7p+35, 0x1.ef1p+23},
    {0x1.0228d8p+35, 0x1.6b9p+23},
    {0x1.e6b214p+35, 0x1.f33p+23},
    {0x1.d2acd4p+35, 0x1.e8dp+23},
    {0x1.8a8508p+35, 0x1.c17p+23},
    {0x1.163d88p+35, 0x1.797p+23},
    {0x1.a2939p+35, 0x1.cefp+23},
    {0x1.2a4f34p+35, 0x1.86dp+23},
    {0x1.baa5cp+35, 0x1.dc1p+23},
    {0x1.0a35b4p+35, 0x1.713p+23},
    {0x1.3a5bb4p+35, 0x1.913p+23},
    {0x1.2e5488p+35, 0x1.897p+23},
    {0x1.aea478p+35, 0x1.d59p+23},
    {0x1.eec5bcp+35, 0x1.f75p+23},
    {0x1.2650c4p+35, 0x1.843p+23},
    {0x1.4a6cfcp+35, 0x1.9b5p+23},
    {0x1.7e8ac8p+35, 0x1.ba9p+23},
    {0x1.d6bddp+35, 0x1.eafp+23},
    {0x1.6a7fb4p+35, 0x1.aedp+23},
    {0x1.6e81b4p+35, 0x1.b13p+23},
    {0x1.466bd4p+35, 0x1.98dp+23},
    {0x1.3260cp+35, 0x1.8c1p+23},
    {0x1.1a4f44p+35, 0x1.7c3p+23},
    {0x1.869454p+35, 0x1.bf3p+23},
    {0x1.5a7b74p+35, 0x1.a53p+23},
    {0x1.a6a7b4p+35, 0x1.d13p+23},
    {0x1.4e7464p+35, 0x1.9ddp+23},
    {0x1.b6b46p+35, 0x1.d9fp+23},
    {0x1.728958p+35, 0x1.b39p+23},
    {0x1.668358p+35, 0x1.ac7p+23},
    {0x1.225938p+35, 0x1.819p+23},
    {0x1.064238p+35, 0x1.6e7p+23},
    {0x1.e2cdb4p+35, 0x1.f13p+23},
    {0x1.4270ecp+35, 0x1.965p+23},
    {0x1.fee028p+35, 0x1.ff7p+23},
    {0x1.7696ap+35, 0x1.b5fp+23},
    {0x1.ead91cp+35, 0x1.f55p+23},
    {0x1.52820cp+35, 0x1.a05p+23},
    {0x1.628cap+35, 0x1.aa1p+23},
    {0x1.fae348p+35, 0x1.fd7p+23},
    {0x1.dad35p+35, 0x1.ed1p+23},
    {0x1.96ae8cp+35, 0x1.c85p+23},
    {0x1.3673dcp+35, 0x1.8ebp+23},
    {0x1.92ae6p+35, 0x1.c61p+23},
    {0x1.3e7c44p+35, 0x1.93dp+23},
    {0x1.9ab3c8p+35, 0x1.ca9p+23},
    {0x1.aac0e8p+35, 0x1.d37p+23},
    {0x1.c6d148p+35, 0x1.e29p+23},
    {0x1.b2c784p+35, 0x1.d7dp+23},
    {0x1.f6ea68p+35, 0x1.fb7p+23},
    {0x1.82a8bp+35, 0x1.bcfp+23},
    {0x1.cad4fcp+35, 0x1.e4bp+23},
    {0x1.c2d218p+35, 0x1.e07p+23},
    {0x1.1e689p+35, 0x1.7efp+23},
    {0x1.8eb344p+35, 0x1.c3dp+23},
    {0x1.126234p+35, 0x1.76dp+23},
    {0x1.9ebe14p+35, 0x1.ccdp+23},
    {0x1.cedd34p+35, 0x1.e6dp+23},
    {0x1.7aa98cp+35, 0x1.b85p+23},
    {0x1.0e5f4p+35, 0x1.741p+23},
    {0x1.02564cp+35, 0x1.6bbp+23},
    {0x1.5695f4p+35, 0x1.a2dp+23},
    {0x1.bed76cp+35, 0x1.de5p+23},
    {0x1.5e9b8cp+35, 0x1.a7bp+23},
    {0x1.f2f588p+35, 0x1.f97p+23},
    {0x1.deed54p+35, 0x1.ef3p+23},
    {0x1.e6f07cp+35, 0x1.f35p+23},
    {0x1.166cb8p+35, 0x1.799p+23},
    {0x1.d2e9fp+35, 0x1.e8fp+23},
    {0x1.8abd38p+35, 0x1.c19p+23},
    {0x1.2a801p+35, 0x1.86fp+23},
    {0x1.0a63dcp+35, 0x1.715p+23},
    {0x1.a2cd7p+35, 0x1.cf1p+23},
    {0x1.2e85b8p+35, 0x1.899p+23},
    {0x1.3a8ddcp+35, 0x1.915p+23},
    {0x1.bae144p+35, 0x1.dc3p+23},
    {0x1.26814cp+35, 0x1.845p+23},
    {0x1.4aa068p+35, 0x1.9b7p+23},
    {0x1.aedf2cp+35, 0x1.d5bp+23},
    {0x1.ef04a8p+35, 0x1.f77p+23},
    {0x1.7ec21cp+35, 0x1.babp+23},
    {0x1.6ab59p+35, 0x1.aefp+23},
    {0x1.329244p+35, 0x1.8c3p+23},
    {0x1.6eb7dcp+35, 0x1.b15p+23},
    {0x1.469efp+35, 0x1.98fp+23},
    {0x1.1a7eccp+35, 0x1.7c5p+23},
    {0x1.d6fb3p+35, 0x1.eb1p+23},
    {0x1.5ab01cp+35, 0x1.a55p+23},
    {0x1.4ea82p+35, 0x1.9dfp+23},
    {0x1.86cc3cp+35, 0x1.bf5p+23},
    {0x1.a6e1dcp+35, 0x1.d15p+23},
    {0x1.22896cp+35, 0x1.81bp+23},
    {0x1.067008p+35, 0x1.6e9p+23},
    {0x1.b6efap+35, 0x1.da1p+23},
    {0x1.72bfccp+35, 0x1.b3bp+23},
    {0x1.66b8e8p+35, 0x1.ac9p+23},
    {0x1.42a3b8p+35, 0x1.967p+23},
    {0x1.e30bdcp+35, 0x1.f15p+23},
    {0x1.52b618p+35, 0x1.a07p+23},
    {0x1.76cd6p+35, 0x1.b61p+23},
    {0x1.62c1e4p+35, 0x1.aa3p+23},
    {0x1.ff2018p+35, 0x1.ff9p+23},
    {0x1.eb17c8p+35, 0x1.f57p+23},
    {0x1.36a5b4p+35, 0x1.8edp+23},
    {0x1.96e798p+35, 0x1.c87p+23},
    {0x1.fb22f8p+35, 0x1.fd9p+23},
    {0x1.db10f4p+35, 0x1.ed3p+23},
    {0x1.3eaecp+35, 0x1.93fp+23},
    {0x1.92e724p+35, 0x1.c63p+23},
    {0x1.9aed1cp+35, 0x1.cabp+23},
    {0x1.aafb58p+35, 0x1.d39p+23},
    {0x1.1e987p+35, 0x1.7f1p+23},
    {0x1.c70d9cp+35, 0x1.e2bp+23},
    {0x1.82e05p+35, 0x1.bd1p+23},
    {0x1.b3028p+35, 0x1.d7fp+23},
    {0x1.12911p+35, 0x1.76fp+23},
    {0x1.cb1194p+35, 0x1.e4dp+23},
    {0x1.f729d8p+35, 0x1.fb9p+23},
    {0x1.c30e28p+35, 0x1.e09p+23},
    {0x1.8eebcp+35, 0x1.c3fp+23},
    {0x1.0e8dc4p+35, 0x1.743p+23},
    {0x1.0283c4p+35, 0x1.6bdp+23},
    {0x1.7ae098p+35, 0x1.b87p+23},
    {0x1.9ef7bp+35, 0x1.ccfp+23},
    {0x1.cf1a1p+35, 0x1.e6fp+23},
    {0x1.56ca5p+35, 0x1.a2fp+23},
    {0x1.5ed084p+35, 0x1.a7dp+23},
    {0x1.bf1338p+35, 0x1.de7p+23},
    {0x1.169becp+35, 0x1.79bp+23},
    {0x1.0a9208p+35, 0x1.717p+23},
    {0x1.2ab0fp+35, 0x1.871p+23},
    {0x1.f334b8p+35, 0x1.f99p+23},
    {0x1.df2b3cp+35, 0x1.ef5p+23},
    {0x1.e72ee8p+35, 0x1.f37p+23},
    {0x1.8af56cp+35, 0x1.c1bp+23},
    {0x1.2eb6ecp+35, 0x1.89bp+23},
    {0x1.d3271p+35, 0x1.e91p+23},
    {0x1.3ac008p+35, 0x1.917p+23},
    {0x1.a30754p+35, 0x1.cf3p+23},
    {0x1.26b1d8p+35, 0x1.847p+23},
    {0x1.bb1cccp+35, 0x1.dc5p+23},
    {0x1.4ad3d8p+35, 0x1.9b9p+23},
    {0x1.af19e4p+35, 0x1.d5dp+23},
    {0x1.1aae58p+35, 0x1.7c7p+23},
    {0x1.32c3ccp+35, 0x1.8c5p+23},
    {0x1.6aeb7p+35, 0x1.af1p+23},
    {0x1.46d21p+35, 0x1.991p+23},
    {0x1.7ef974p+35, 0x1.badp+23},
    {0x1.ef4398p+35, 0x1.f79p+23},
    {0x1.6eee08p+35, 0x1.b17p+23},
    {0x1.5ae4c8p+35, 0x1.a57p+23},
    {0x1.4edbep+35, 0x1.9e1p+23},
    {0x1.d73894p+35, 0x1.eb3p+23},
    {0x1.069ddcp+35, 0x1.6ebp+23},
    {0x1.22b9a4p+35, 0x1.81dp+23},
    {0x1.870428p+35, 0x1.bf7p+23},
    {0x1.72f644p+35, 0x1.b3dp+23},
    {0x1.a71c08p+35, 0x1.d17p+23},
    {0x1.66ee7cp+35, 0x1.acbp+23},
    {0x1.b72ae4p+35, 0x1.da3p+23},
    {0x1.42d688p+35, 0x1.969p+23},
    {0x1.e34a08p+35, 0x1.f17p+23},
    {0x1.52ea28p+35, 0x1.a09p+23},
    {0x1.62f72cp+35, 0x1.aa5p+23},
    {0x1.770424p+35, 0x1.b63p+23},
    {0x1.36d79p+35, 0x1.8efp+23},
    {0x1.ff600cp+35, 0x1.ffbp+23},
    {0x1.3ee14p+35, 0x1.941p+23},
    {0x1.eb5678p+35, 0x1.f59p+23},
    {0x1.9720a8p+35, 0x1.c89p+23},
    {0x1.931fecp+35, 0x1.c65p+23},
    {0x1.db4e9cp+35, 0x1.ed5p+23},
    {0x1.fb62acp+35, 0x1.fdbp+23},
    {0x1.1ec854p+35, 0x1.7f3p+23},
    {0x1.12bffp+35, 0x1.771p+23},
    {0x1.9b2674p+35, 0x1.cadp+23},
    {0x1.0ebc4cp+35, 0x1.745p+23},
    {0x1.8317f4p+35, 0x1.bd3p+23},
    {0x1.ab35ccp+35, 0x1.d3bp+23},
    {0x1.b33d8p+35, 0x1.d81p+23},
    {0x1.02b14p+35, 0x1.6bfp+23},
    {0x1.c749f4p+35, 0x1.e2dp+23},
    {0x1.8f244p+35, 0x1.c41p+23},
    {0x1.c34a3cp+35, 0x1.e0bp+23},
    {0x1.cb4e3p+35, 0x1.e4fp+23},
    {0x1.f7694cp+35, 0x1.fbbp+23},
    {0x1.7b17a8p+35, 0x1.b89p+23},
    {0x1.56febp+35, 0x1.a31p+23},
    {0x1.9f315p+35, 0x1.cd1p+23},
    {0x1.5f058p+35, 0x1.a7fp+23},
    {0x1.16cb24p+35, 0x1.79dp+23},
    {0x1.cf56fp+35, 0x1.e71p+23},
    {0x1.0ac038p+35, 0x1.719p+23},
    {0x1.bf4f08p+35, 0x1.de9p+23},
    {0x1.2ae1d4p+35, 0x1.873p+23},
    {0x1.2ee824p+35, 0x1.89dp+23},
    {0x1.3af238p+35, 0x1.919p+23},
    {0x1.f373ecp+35, 0x1.f9bp+23},
    {0x1.8b2da4p+35, 0x1.c1dp+23},
    {0x1.df6928p+35, 0x1.ef7p+23},
    {0x1.26e268p+35, 0x1.849p+23},
    {0x1.e76d58p+35, 0x1.f39p+23},
    {0x1.d36434p+35, 0x1.e93p+23},
    {0x1.a3413cp+35, 0x1.cf5p+23},
    {0x1.4b074cp+35, 0x1.9bbp+23},
    {0x1.bb5858p+35, 0x1.dc7p+23},
    {0x1.1adde8p+35, 0x1.7c9p+23},
    {0x1.32f558p+35, 0x1.8c7p+23},
    {0x1.470534p+35, 0x1.993p+23},
    {0x1.af54ap+35, 0x1.d5fp+23},
    {0x1.6b2154p+35, 0x1.af3p+23},
    {0x1.7f30dp+35, 0x1.bafp+23},
    {0x1.06cbb4p+35, 0x1.6edp+23},
    {0x1.6f2438p+35, 0x1.b19p+23},
    {0x1.22e9ep+35, 0x1.81fp+23},
    {0x1.5b1978p+35, 0x1.a59p+23},
    {0x1.4f0fa4p+35, 0x1.9e3p+23},
    {0x1.ef828cp+35, 0x1.f7bp+23},
    {0x1.873c18p+35, 0x1.bf9p+23},
    {0x1.d775fcp+35, 0x1.eb5p+23},
    {0x1.732ccp+35, 0x1.b3fp+23},
    {0x1.672414p+35, 0x1.acdp+23},
    {0x1.43095cp+35, 0x1.96bp+23},
    {0x1.a75638p+35, 0x1.d19p+23},
    {0x1.b7662cp+35, 0x1.da5p+23},
    {0x1.531e3cp+35, 0x1.a0bp+23},
    {0x1.37097p+35, 0x1.8f1p+23},
    {0x1.e38838p+35, 0x1.f19p+23},
    {0x1.632c78p+35, 0x1.aa7p+23},
    {0x1.773aecp+35, 0x1.b65p+23},
    {0x1.3f13c4p+35, 0x1.943p+23},
    {0x1.9759bcp+35, 0x1.c8bp+23},
    {0x1.1ef83cp+35, 0x1.7f5p+23},
    {0x1.12eed4p+35, 0x1.773p+23},
    {0x1.eb952cp+35, 0x1.f5bp+23},
    {0x1.ffa004p+35, 0x1.ffdp+23},
    {0x1.9358b8p+35, 0x1.c67p+23},
    {0x1.0eead8p+35, 0x1.747p+23},
    {0x1.db8c48p+35, 0x1.ed7p+23},
    {0x1.02decp+35, 0x1.6c1p+23},
    {0x1.fba264p+35, 0x1.fddp+23},
    {0x1.9b5fdp+35, 0x1.cafp+23},
    {0x1.834f9cp+35, 0x1.bd5p+23},
    {0x1.ab7044p+35, 0x1.d3dp+23},
    {0x1.b37884p+35, 0x1.d83p+23},
    {0x1.c7865p+35, 0x1.e2fp+23},
    {0x1.8f5cc4p+35, 0x1.c43p+23},
    {0x1.c38654p+35, 0x1.e0dp+23},
    {0x1.7b4ebcp+35, 0x1.b8bp+23},
    {0x1.16fa6p+35, 0x1.79fp+23},
    {0x1.cb8adp+35, 0x1.e51p+23},
    {0x1.573314p+35, 0x1.a33p+23},
    {0x1.5f3a8p+35, 0x1.a81p+23},
    {0x1.0aee6cp+35, 0x1.71bp+23},
    {0x1.f7a8c4p+35, 0x1.fbdp+23},
    {0x1.9f6af4p+35, 0x1.cd3p+23},
    {0x1.2b12bcp+35, 0x1.875p+23},
    {0x1.cf93d4p+35, 0x1.e73p+23},
    {0x1.2f196p+35, 0x1.89fp+23},
    {0x1.bf8adcp+35, 0x1.debp+23},
    {0x1.3b246cp+35, 0x1.91bp+23},
    {0x1.2712fcp+35, 0x1.84bp+23},
    {0x1.8b65ep+35, 0x1.c1fp+23},
    {0x1.dfa718p+35, 0x1.ef9p+23},
    {0x1.f3b324p+35, 0x1.f9dp+23},
    {0x1.a37b28p+35, 0x1.cf7p+23},
    {0x1.d3a15cp+35, 0x1.e95p+23},
    {0x1.e7abccp+35, 0x1.f3bp+23},
    {0x1.4b3ac4p+35, 0x1.9bdp+23},
    {0x1.1b0d7cp+35, 0x1.7cbp+23},
    {0x1.3326e8p+35, 0x1.8c9p+23},
    {0x1.bb93e8p+35, 0x1.dc9p+23},
    {0x1.47385cp+35, 0x1.995p+23},
    {0x1.06f99p+35, 0x1.6efp+23},
    {0x1.6b573cp+35, 0x1.af5p+23},
    {0x1.231a2p+35, 0x1.821p+23},
    {0x1.af8f6p+35, 0x1.d61p+23},
    {0x1.7f683p+35, 0x1.bb1p+23},
    {0x1.6f5a6cp+35, 0x1.b1bp+23},
    {0x1.4f436cp+35, 0x1.9e5p+23},
    {0x1.5b4e2cp+35, 0x1.a5bp+23},
    {0x1.efc184p+35, 0x1.f7dp+23},
    {0x1.87740cp+35, 0x1.bfbp+23},
    {0x1.73634p+35, 0x1.b41p+23},
    {0x1.433c34p+35, 0x1.96dp+23},
    {0x1.6759bp+35, 0x1.acfp+23},
    {0x1.d7b368p+35, 0x1.eb7p+23},
    {0x1.a7906cp+35, 0x1.d1bp+23},
    {0x1.b7a178p+35, 0x1.da7p+23},
    {0x1.535254p+35, 0x1.a0dp+23},
    {0x1.373b54p+35, 0x1.8f3p+23},
    {0x1.6361c8p+35, 0x1.aa9p+23},
    {0x1.7771b8p+35, 0x1.b67p+23},
    {0x1.e3c66cp+35, 0x1.f1bp+23},
    {0x1.3f464cp+35, 0x1.945p+23},
    {0x1.1f2828p+35, 0x1.7f7p+23},
    {0x1.131dbcp+35, 0x1.775p+23},
    {0x1.0f1968p+35, 0x1.749p+23},
    {0x1.9792d4p+35, 0x1.c8dp+23},
    {0x1.030c44p+35, 0x1.6c3p+23},
    {0x1.939188p+35, 0x1.c69p+23},
    {0x1.ebd3e4p+35, 0x1.f5dp+23},
    {0x1.ffep+35, 0x1.fffp+23},
    {0x1.9b993p+35, 0x1.cb1p+23},
    {0x1.dbc9f8p+35, 0x1.ed9p+23},
    {0x1.838748p+35, 0x1.bd7p+23},
    {0x1.fbe22p+35, 0x1.fdfp+23},
    {0x1.abaacp+35, 0x1.d3fp+23},
    {0x1.1729ap+35, 0x1.7a1p+23},
    {0x1.b3b38cp+35, 0x1.d85p+23},
    {0x1.8f954cp+35, 0x1.c45p+23},
    {0x1.0b1ca4p+35, 0x1.71dp+23},
    {0x1.c7c2bp+35, 0x1.e31p+23},
    {0x1.57677cp+35, 0x1.a35p+23},
    {0x1.7b85d4p+35, 0x1.b8dp+23},
    {0x1.5f6f84p+35, 0x1.a83p+23},
    {0x1.c3c27p+35, 0x1.e0fp+23},
    {0x1.2b43a8p+35, 0x1.877p+23},
    {0x1.cbc774p+35, 0x1.e53p+23},
    {0x1.9fa49cp+35, 0x1.cd5p+23},
    {0x1.2f4aap+35, 0x1.8a1p+23},
    {0x1.f7e84p+35, 0x1.fbfp+23},
    {0x1.3b56a4p+35, 0x1.91dp+23},
    {0x1.cfd0bcp+35, 0x1.e75p+23},
    {0x1.274394p+35, 0x1.84dp+23},
    {0x1.bfc6b4p+35, 0x1.dedp+23},
    {0x1.8b9e2p+35, 0x1.c21p+23},
    {0x1.1b3d14p+35, 0x1.7cdp+23},
    {0x1.4b6e4p+35, 0x1.9bfp+23},
    {0x1.dfe50cp+35, 0x1.efbp+23},
    {0x1.a3b518p+35, 0x1.cf9p+23},
    {0x1.33587cp+35, 0x1.8cbp+23},
    {0x1.f3f26p+35, 0x1.f9fp+23},
    {0x1.d3de88p+35, 0x1.e97p+23},
    {0x1.e7ea44p+35, 0x1.f3dp+23},
    {0x1.07277p+35, 0x1.6f1p+23},
    {0x1.476b88p+35, 0x1.997p+23},
    {0x1.bbcf7cp+35, 0x1.dcbp+23},
    {0x1.234a64p+35, 0x1.823p+23},
    {0x1.6b8d28p+35, 0x1.af7p+23},
    {0x1.4f7738p+35, 0x1.9e7p+23},
    {0x1.5b82e4p+35, 0x1.a5dp+23},
    {0x1.6f90a4p+35, 0x1.b1dp+23},
    {0x1.7f9f94p+35, 0x1.bb3p+23},
    {0x1.afca24p+35, 0x1.d63p+23},
    {0x1.436f1p+35, 0x1.96fp+23},
    {0x1.87ac04p+35, 0x1.bfdp+23},
    {0x1.a7caa4p+35, 0x1.d1dp+23},
    {0x1.7399c4p+35, 0x1.b43p+23},
    {0x1.678f5p+35, 0x1.ad1p+23},
    {0x1.d7f0d8p+35, 0x1.eb9p+23},
    {0x1.b7dcc8p+35, 0x1.da9p+23},
    {0x1.376d3cp+35, 0x1.8f5p+23},
    {0x1.53867p+35, 0x1.a0fp+23},
    {0x1.63971cp+35, 0x1.aabp+23},
    {0x1.3f78d8p+35, 0x1.947p+23},
    {0x1.77a888p+35, 0x1.b69p+23},
    {0x1.134ca8p+35, 0x1.777p+23},
    {0x1.1f5818p+35, 0x1.7f9p+23},
    {0x1.b3ee98p+35, 0x1.d87p+23},
    {0x1.0f47fcp+35, 0x1.74bp+23},
    {0x1.0339ccp+35, 0x1.6c5p+23},
    {0x1.9bd294p+35, 0x1.cb3p+23},
    {0x1.97cbfp+35, 0x1.c8fp+23},
    {0x1.abe54p+35, 0x1.d41p+23},
    {0x1.93ca5cp+35, 0x1.c6bp+23},
    {0x1.83bef8p+35, 0x1.bd9p+23},
    {0x1.8fcdd8p+35, 0x1.c47p+23},
    {0x1.1758e4p+35, 0x1.7a3p+23},
    {0x1.0b4aep+35, 0x1.71fp+23},
    {0x1.c7ff14p+35, 0x1.e33p+23},
    {0x1.1b6cbp+35, 0x1.7cfp+23},
    {0x1.579be8p+35, 0x1.a37p+23},
    {0x1.2b7498p+35, 0x1.879p+23},
    {0x1.7bbcfp+35, 0x1.b8fp+23},
    {0x1.5fa48cp+35, 0x1.a85p+23},
    {0x1.a3ef0cp+35, 0x1.cfbp+23},
    {0x1.2f7be4p+35, 0x1.8a3p+23},
    {0x1.9fde48p+35, 0x1.cd7p+23},
    {0x1.c3fe9p+35, 0x1.e11p+23},
    {0x1.5bb7ap+35, 0x1.a5fp+23},
    {0x1.8bd664p+35, 0x1.c23p+23},
    {0x1.3b88ep+35, 0x1.91fp+23},
    {0x1.4ba1cp+35, 0x1.9c1p+23},
    {0x1.27743p+35, 0x1.84fp+23},
    {0x1.075554p+35, 0x1.6f3p+23},
    {0x1.87e4p+35, 0x1.bffp+23},
    {0x1.237aacp+35, 0x1.825p+23},
    {0x1.4fab08p+35, 0x1.9e9p+23},
    {0x1.338a14p+35, 0x1.8cdp+23},
    {0x1.6bc318p+35, 0x1.af9p+23},
    {0x1.73d04cp+35, 0x1.b45p+23},
    {0x1.0f7694p+35, 0x1.74dp+23},
    {0x1.479eb8p+35, 0x1.999p+23},
    {0x1.67c4f4p+35, 0x1.ad3p+23},
    {0x1.6fc6ep+35, 0x1.b1fp+23},
    {0x1.7fd6fcp+35, 0x1.bb5p+23},
    {0x1.137b98p+35, 0x1.779p+23},
    {0x1.53ba9p+35, 0x1.a11p+23},
    {0x1.83f6acp+35, 0x1.bdbp+23},
    {0x1.379f28p+35, 0x1.8f7p+23},
    {0x1.43a1fp+35, 0x1.971p+23},
    {0x1.1b9c5p+35, 0x1.7d1p+23},
    {0x1.0b792p+35, 0x1.721p+23},
    {0x1.2ba58cp+35, 0x1.87bp+23},
    {0x1.77df5cp+35, 0x1.b6bp+23},
    {0x1.63cc74p+35, 0x1.aadp+23},
    {0x1.5bec6p+35, 0x1.a61p+23},
    {0x1.5fd998p+35, 0x1.a87p+23},
    {0x1.4bd544p+35, 0x1.9c3p+23},
    {0x1.07833cp+35, 0x1.6f5p+23},
    {0x1.3fab68p+35, 0x1.949p+23},
    {0x1.2fad2cp+35, 0x1.8a5p+23},
    {0x1.1f880cp+35, 0x1.7fbp+23},
    {0x1.23aaf8p+35, 0x1.827p+23},
    {0x1.036758p+35, 0x1.6c7p+23},
    {0x1.7bf41p+35, 0x1.b91p+23},
    {0x1.4fdedcp+35, 0x1.9ebp+23},
    {0x1.3bbb2p+35, 0x1.921p+23},
    {0x1.6bf90cp+35, 0x1.afbp+23},
    {0x1.0fa53p+35, 0x1.74fp+23},
    {0x1.57d058p+35, 0x1.a39p+23},
    {0x1.17882cp+35, 0x1.7a5p+23},
    {0x1.27a4dp+35, 0x1.851p+23},
    {0x1.33bbbp+35, 0x1.8cfp+23},
    {0x1.67fa9cp+35, 0x1.ad5p+23},
    {0x1.53eeb4p+35, 0x1.a13p+23},
    {0x1.37d118p+35, 0x1.8f9p+23},
    {0x1.13aa8cp+35, 0x1.77bp+23},
    {0x1.1bcbf4p+35, 0x1.7d3p+23},
    {0x1.0ba764p+35, 0x1.723p+23},
    {0x1.2bd684p+35, 0x1.87dp+23},
    {0x1.47d1ecp+35, 0x1.99bp+23},
    {0x1.43d4d4p+35, 0x1.973p+23},
    {0x1.6ffd2p+35, 0x1.b21p+23},
    {0x1.3fddfcp+35, 0x1.94bp+23},
    {0x1.07b128p+35, 0x1.6f7p+23},
    {0x1.2fde78p+35, 0x1.8a7p+23},
    {0x1.23db48p+35, 0x1.829p+23},
    {0x1.0394e8p+35, 0x1.6c9p+23},
    {0x1.1fb804p+35, 0x1.7fdp+23},
    {0x1.3bed64p+35, 0x1.923p+23},
    {0x1.0fd3dp+35, 0x1.751p+23},
    {0x1.17b778p+35, 0x1.7a7p+23},
    {0x1.27d574p+35, 0x1.853p+23},
    {0x1.33ed5p+35, 0x1.8d1p+23},
    {0x1.13d984p+35, 0x1.77dp+23},
    {0x1.1bfb9cp+35, 0x1.7d5p+23},
    {0x1.0bd5acp+35, 0x1.725p+23},
    {0x1.07df18p+35, 0x1.6f9p+23},
    {0x1.03c27cp+35, 0x1.6cbp+23},
    {0x1.1fe8p+35, 0x1.7ffp+23},
    {0x1.17e6c8p+35, 0x1.7a9p+23},
    {0x1.03f014p+35, 0x1.6cdp+23},
};

#endif // LLVM_LIBC_TEST_SRC_MATH_HYPOTTEST_HARD_TO_ROUND_H
