#! /bin/bash
#
# This file is part of the LibreOffice project.
#
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.
#

# Assemble Flatpak app files and metadata under /app/, copying from the
# installation tree generated by 'make distro-pack-install' (at
# $PREFIXDIR):

set -e

cp -r "${PREFIXDIR?}"/lib/libreoffice /app/

## libreoffice-*.desktop -> org.libreoffice.LibreOffice-*.desktop:
mkdir -p /app/share/applications
for i in "${PREFIXDIR?}"/share/applications/libreoffice-*.desktop
do
 sed -e 's,^Exec=libreoffice,Exec=/app/libreoffice/program/soffice,' \
  -e 's/^Icon=libreoffice-/Icon=org.libreoffice.LibreOffice-/' "$i" \
  >/app/share/applications/org.libreoffice.LibreOffice-"${i#"${PREFIXDIR?}"/share/applications/libreoffice-}"
done
mv /app/share/applications/org.libreoffice.LibreOffice-startcenter.desktop \
 /app/share/applications/org.libreoffice.LibreOffice.desktop

# Flatpak .desktop exports take precedence over system ones due to
# the order of XDG_DATA_DIRS - re-associating text/plain seems a bit much
sed -i "s/text\/plain;//" /app/share/applications/org.libreoffice.LibreOffice-writer.desktop

## icons/hicolor/*/apps/libreoffice-* ->
## icons/hicolor/*/apps/org.libreoffice.LibreOffice-*:
mkdir -p /app/share/icons
for i in "${PREFIXDIR?}"/share/icons/hicolor/*/apps/libreoffice-*
do
 mkdir -p \
  "$(dirname /app/share/icons/hicolor/"${i#"${PREFIXDIR?}"/share/icons/hicolor/}")"
 cp -a "$i" \
  "$(dirname /app/share/icons/hicolor/"${i#"${PREFIXDIR?}"/share/icons/hicolor/}")"/"$(basename "$i")"
 cp -a "$i" \
  "$(dirname /app/share/icons/hicolor/"${i#"${PREFIXDIR?}"/share/icons/hicolor/}")"/org.libreoffice.LibreOffice-"${i##*/apps/libreoffice-}"
done

mkdir -p /app/share/runtime/locale
for i in $(ls /app/libreoffice/program/resource)
do
  lang="${i%[_@]*}"
  mkdir -p /app/share/runtime/locale/"${lang}"/resource
  mv /app/libreoffice/program/resource/"${i}" /app/share/runtime/locale/"${lang}"/resource
  ln -s ../../../share/runtime/locale/"${lang}"/resource/"${i}" /app/libreoffice/program/resource
done

for i in /app/libreoffice/share/registry/Langpack-*.xcd /app/libreoffice/share/registry/res/{fcfg_langpack,registry}_*.xcd
do
  basename="$(basename "${i}" .xcd)"
  lang="${basename#Langpack-}"
  lang="${lang#fcfg_langpack_}"
  lang="${lang#registry_}"

  # ship the base app with at least one Langpack/fcfg_langpack
  if [ "${lang}" = "en-US" ]
  then
    continue
  fi

  lang="${lang%-*}"
  mkdir -p /app/share/runtime/locale/"${lang}"/registry
  mv "${i}" /app/share/runtime/locale/"${lang}"/registry
  ln -rs /app/share/runtime/locale/"${lang}"/registry/"${basename}".xcd "${i}"
done

## org.libreoffice.LibreOffice.appdata.xml is manually derived from the various
## inst/share/appdata/libreoffice-*.appdata.xml (at least recent GNOME Software
## doesn't show more than five screenshots anyway, so restrict to one each from
## the five libreoffice-*.appdata.xml: Writer, Calc, Impress, Draw, Base):
mkdir -p /app/share/appdata
cat <<EOF >/app/share/appdata/org.libreoffice.LibreOffice.appdata.xml
<?xml version="1.0" encoding="UTF-8"?>
<component type="desktop">
 <id>org.libreoffice.LibreOffice.desktop</id>
 <metadata_license>CC0-1.0</metadata_license>
 <project_license>MPL-2.0</project_license>
 <name>LibreOffice</name>
 <summary>The LibreOffice productivity suite</summary>
 <description>
  <p>LibreOffice is a powerful office suite.  Its clean interface and
  feature-rich tools help you unleash your creativity and enhance your
  productivity.  LibreOffice includes several applications that make it the most
  powerful Free and Open Source office suite on the market: Writer (word
  processing), Calc (spreadsheets), Impress (presentations), Draw (vector
  graphics and flowcharts), Base (databases), and Math (formula editing).</p>
  <p>LibreOffice supports opening and saving into a wide variety of formats, so
  you can easily share documents with users of other popular office suites
  without worrying about compatibility.</p>
 </description>
 <url type="homepage">http://www.libreoffice.org/discover/libreoffice/</url>
 <url type="bugtracker">https://bugs.documentfoundation.org/</url>
 <url type="donation">https://donate.libreoffice.org/</url>
 <url type="faq">https://wiki.documentfoundation.org/Faq</url>
 <url type="help">http://www.libreoffice.org/get-help/documentation/</url>
 <url type="translate">https://wiki.documentfoundation.org/Translating_LibreOffice</url>
 <screenshots>
  <screenshot type="default">
   <image>http://www.libreoffice.org/assets/Uploads/Discover/Screenshots/_resampled/ResizedImage400300-Writer-02-Table-of-Contents-Getting-Started-Guide-4-2.png</image>
   <caption><!-- Describe this screenshot in less than ~10 words --></caption>
  </screenshot>
  <screenshot>
   <image>http://www.libreoffice.org/assets/Uploads/Discover/Screenshots/_resampled/ResizedImage400300-Calc-02-exemple-ods.png</image>
   <caption><!-- Describe this screenshot in less than ~10 words --></caption>
  </screenshot>
  <screenshot>
   <image>http://www.libreoffice.org/assets/Uploads/Discover/Screenshots/_resampled/ResizedImage400300-Impress-01-Blank.png</image>
   <caption><!-- Describe this screenshot in less than ~10 words --></caption>
  </screenshot>
  <screenshot>
   <image>http://www.libreoffice.org/assets/Uploads/Discover/Screenshots/_resampled/ResizedImage400300-Draw-02-Tiger-fdo82219.png</image>
   <caption><!-- Describe this screenshot in less than ~10 words --></caption>
  </screenshot>
  <screenshot>
   <image>http://www.libreoffice.org/assets/Uploads/Screenshots/Base/Screenshot-78.png</image>
   <caption><!-- Describe this screenshot in less than ~10 words --></caption>
  </screenshot>
 </screenshots>
 <developer_name>The Document Foundation</developer_name>
 <update_contact>libreoffice_at_lists.freedesktop.org</update_contact>
 <kudos>
  <kudo>HiDpiIcon</kudo>
  <kudo>HighContrast</kudo>
  <kudo>ModernToolkit</kudo>
  <kudo>UserDocs</kudo>
 </kudos>
 <releases>
  <release
    version="${LIBO_VERSION_MAJOR?}.${LIBO_VERSION_MINOR?}.${LIBO_VERSION_MICRO?}.${LIBO_VERSION_PATCH?}"
    date="$(date +%Y-%m-%d)"/>
 </releases>
</component>
EOF

# append the appdata for the different components
for i in "${PREFIXDIR?}"/share/appdata/libreoffice-*.appdata.xml
do
  sed "1 d; s/<id>libreoffice/<id>org.libreoffice.LibreOffice/" "$i" \
    >>/app/share/appdata/org.libreoffice.LibreOffice.appdata.xml
done

## see <https://github.com/flatpak/flatpak/blob/master/app/
## flatpak-builtins-build-finish.c> for further places where build-finish would
## look for data:
## cp ... /app/share/dbus-1/services/
## cp ... /app/share/gnome-shell/search-providers/
