/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.server.authentication;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import io.jsonwebtoken.Claims;
import java.util.Collections;
import java.util.Date;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Nullable;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.time.DateUtils;
import org.sonar.api.config.Configuration;
import org.sonar.api.server.ServerSide;
import org.sonar.api.utils.System2;
import org.sonar.db.DbClient;
import org.sonar.db.DbSession;
import org.sonar.db.user.UserDto;
import org.sonar.server.authentication.Cookies;
import org.sonar.server.authentication.JwtCsrfVerifier;
import org.sonar.server.authentication.JwtSerializer;

@ServerSide
public class JwtHttpHandler {
    private static final String SESSION_TIMEOUT_IN_MINUTES_PROPERTY = "sonar.web.sessionTimeoutInMinutes";
    private static final int SESSION_TIMEOUT_DEFAULT_VALUE_IN_MINUTES = 4320;
    private static final int MAX_SESSION_TIMEOUT_IN_MINUTES = 129600;
    private static final String JWT_COOKIE = "JWT-SESSION";
    private static final String LAST_REFRESH_TIME_PARAM = "lastRefreshTime";
    private static final String CSRF_JWT_PARAM = "xsrfToken";
    private static final int SESSION_DISCONNECT_IN_SECONDS = 7776000;
    private static final int SESSION_REFRESH_IN_SECONDS = 300;
    private final System2 system2;
    private final DbClient dbClient;
    private final JwtSerializer jwtSerializer;
    private final int sessionTimeoutInSeconds;
    private final JwtCsrfVerifier jwtCsrfVerifier;

    public JwtHttpHandler(System2 system2, DbClient dbClient, Configuration config, JwtSerializer jwtSerializer, JwtCsrfVerifier jwtCsrfVerifier) {
        this.jwtSerializer = jwtSerializer;
        this.dbClient = dbClient;
        this.system2 = system2;
        this.sessionTimeoutInSeconds = JwtHttpHandler.getSessionTimeoutInSeconds(config);
        this.jwtCsrfVerifier = jwtCsrfVerifier;
    }

    public void generateToken(UserDto user, Map<String, Object> properties, HttpServletRequest request, HttpServletResponse response) {
        String csrfState = this.jwtCsrfVerifier.generateState(request, response, this.sessionTimeoutInSeconds);
        String token = this.jwtSerializer.encode(new JwtSerializer.JwtSession(user.getUuid(), this.sessionTimeoutInSeconds, (Map<String, Object>)ImmutableMap.builder().putAll(properties).put((Object)LAST_REFRESH_TIME_PARAM, (Object)this.system2.now()).put((Object)CSRF_JWT_PARAM, (Object)csrfState).build()));
        response.addCookie(JwtHttpHandler.createCookie(request, JWT_COOKIE, token, this.sessionTimeoutInSeconds));
    }

    public void generateToken(UserDto user, HttpServletRequest request, HttpServletResponse response) {
        this.generateToken(user, Collections.emptyMap(), request, response);
    }

    public Optional<UserDto> validateToken(HttpServletRequest request, HttpServletResponse response) {
        Optional<Token> token = this.getToken(request, response);
        if (token.isPresent()) {
            return Optional.of(token.get().getUserDto());
        }
        return Optional.empty();
    }

    public Optional<Token> getToken(HttpServletRequest request, HttpServletResponse response) {
        Optional<String> encodedToken = JwtHttpHandler.getTokenFromCookie(request);
        if (!encodedToken.isPresent()) {
            return Optional.empty();
        }
        return this.validateToken(encodedToken.get(), request, response);
    }

    private static Optional<String> getTokenFromCookie(HttpServletRequest request) {
        Optional<Cookie> jwtCookie = Cookies.findCookie(JWT_COOKIE, request);
        if (!jwtCookie.isPresent()) {
            return Optional.empty();
        }
        Cookie cookie = jwtCookie.get();
        String token = cookie.getValue();
        if (StringUtils.isEmpty((String)token)) {
            return Optional.empty();
        }
        return Optional.of(token);
    }

    private Optional<Token> validateToken(String tokenEncoded, HttpServletRequest request, HttpServletResponse response) {
        Claims token;
        Optional<Claims> claims = this.jwtSerializer.decode(tokenEncoded);
        if (!claims.isPresent()) {
            return Optional.empty();
        }
        Date now = new Date(this.system2.now());
        if (now.after(DateUtils.addSeconds((Date)(token = claims.get()).getIssuedAt(), (int)7776000))) {
            return Optional.empty();
        }
        this.jwtCsrfVerifier.verifyState(request, (String)token.get((Object)CSRF_JWT_PARAM), token.getSubject());
        if (now.after(DateUtils.addSeconds((Date)JwtHttpHandler.getLastRefreshDate(token), (int)300))) {
            this.refreshToken(token, request, response);
        }
        Optional<UserDto> user = this.selectUserFromUuid(token.getSubject());
        return user.map(userDto -> new Token((UserDto)userDto, (Map)claims.get()));
    }

    private static Date getLastRefreshDate(Claims token) {
        Long lastFreshTime = (Long)token.get((Object)LAST_REFRESH_TIME_PARAM);
        Objects.requireNonNull(lastFreshTime, "last refresh time is missing in token");
        return new Date(lastFreshTime);
    }

    private void refreshToken(Claims token, HttpServletRequest request, HttpServletResponse response) {
        String refreshToken = this.jwtSerializer.refresh(token, this.sessionTimeoutInSeconds);
        response.addCookie(JwtHttpHandler.createCookie(request, JWT_COOKIE, refreshToken, this.sessionTimeoutInSeconds));
        this.jwtCsrfVerifier.refreshState(request, response, (String)token.get((Object)CSRF_JWT_PARAM), this.sessionTimeoutInSeconds);
    }

    public void removeToken(HttpServletRequest request, HttpServletResponse response) {
        response.addCookie(JwtHttpHandler.createCookie(request, JWT_COOKIE, null, 0));
        this.jwtCsrfVerifier.removeState(request, response);
    }

    private static Cookie createCookie(HttpServletRequest request, String name, @Nullable String value, int expirationInSeconds) {
        return Cookies.newCookieBuilder(request).setName(name).setValue(value).setHttpOnly(true).setExpiry(expirationInSeconds).build();
    }

    private Optional<UserDto> selectUserFromUuid(String userUuid) {
        try (DbSession dbSession = this.dbClient.openSession(false);){
            UserDto user = this.dbClient.userDao().selectByUuid(dbSession, userUuid);
            Optional<Object> optional = Optional.ofNullable(user != null && user.isActive() ? user : null);
            return optional;
        }
    }

    private static int getSessionTimeoutInSeconds(Configuration config) {
        int minutes = config.getInt(SESSION_TIMEOUT_IN_MINUTES_PROPERTY).orElse(4320);
        Preconditions.checkArgument((minutes > 0 ? 1 : 0) != 0, (String)"Property %s must be strictly positive. Got %s", (Object[])new Object[]{SESSION_TIMEOUT_IN_MINUTES_PROPERTY, minutes});
        Preconditions.checkArgument((minutes <= 129600 ? 1 : 0) != 0, (String)"Property %s must not be greater than 3 months (%s minutes). Got %s minutes", (Object[])new Object[]{SESSION_TIMEOUT_IN_MINUTES_PROPERTY, 129600, minutes});
        return minutes * 60;
    }

    public static class Token {
        private final UserDto userDto;
        private final Map<String, Object> properties;

        public Token(UserDto userDto, Map<String, Object> properties) {
            this.userDto = userDto;
            this.properties = properties;
        }

        public UserDto getUserDto() {
            return this.userDto;
        }

        public Map<String, Object> getProperties() {
            return this.properties;
        }
    }
}

