/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.updatecenter.common;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonar.updatecenter.common.Artifact;
import org.sonar.updatecenter.common.Component;
import org.sonar.updatecenter.common.FormatUtils;
import org.sonar.updatecenter.common.Plugin;
import org.sonar.updatecenter.common.PluginReferential;
import org.sonar.updatecenter.common.Release;
import org.sonar.updatecenter.common.Scanner;
import org.sonar.updatecenter.common.Sonar;
import org.sonar.updatecenter.common.UpdateCenter;
import org.sonar.updatecenter.common.Version;
import org.sonar.updatecenter.common.exception.SonarVersionRangeException;

public final class UpdateCenterDeserializer {
    public static final String DATE_SUFFIX = ".date";
    public static final String DESCRIPTION_SUFFIX = ".description";
    public static final String MAVEN_GROUPID_SUFFIX = ".mavenGroupId";
    public static final String MAVEN_ARTIFACTID_SUFFIX = ".mavenArtifactId";
    public static final String CHANGELOG_URL_SUFFIX = ".changelogUrl";
    public static final String DOWNLOAD_URL_SUFFIX = ".downloadUrl";
    public static final String DOWNLOAD_DEVELOPER_URL_SUFFIX = ".downloadDeveloperUrl";
    public static final String DOWNLOAD_ENTERPRISE_URL_SUFFIX = ".downloadEnterpriseUrl";
    public static final String DOWNLOAD_DATACENTER_URL_SUFFIX = ".downloadDatacenterUrl";
    public static final String DISPLAY_VERSION_SUFFIX = ".displayVersion";
    public static final String SONAR_PREFIX = "sonar.";
    public static final String DEFAULTS_PREFIX = "defaults";
    public static final String PLUGINS = "plugins";
    public static final String SCANNERS = "scanners";
    private static final String PUBLIC_VERSIONS = "publicVersions";
    private static final String PRIVATE_VERSIONS = "privateVersions";
    private static final String ARCHIVED_VERSIONS = "archivedVersions";
    private static final String DEV_VERSION = "devVersion";
    private static final String LATEST_KEYWORD = "LATEST";
    private static final String FLAVORS_PREFIX = "flavors";
    private static final Logger LOGGER = LoggerFactory.getLogger(UpdateCenterDeserializer.class);
    private Mode mode;
    private boolean ignoreError;
    private boolean includeArchives;

    public UpdateCenterDeserializer(Mode mode, boolean ignoreError) {
        this(mode, ignoreError, false);
    }

    public UpdateCenterDeserializer(Mode mode, boolean ignoreError, boolean includeArchives) {
        this.mode = mode;
        this.ignoreError = ignoreError;
        this.includeArchives = includeArchives;
    }

    public static String getDownloadUrlSuffix(Release.Edition edition) {
        switch (edition) {
            case DEVELOPER: {
                return DOWNLOAD_DEVELOPER_URL_SUFFIX;
            }
            case ENTERPRISE: {
                return DOWNLOAD_ENTERPRISE_URL_SUFFIX;
            }
            case DATACENTER: {
                return DOWNLOAD_DATACENTER_URL_SUFFIX;
            }
        }
        return DOWNLOAD_URL_SUFFIX;
    }

    public UpdateCenter fromManyFiles(File mainFile) throws IOException {
        try (InputStream in = Files.newInputStream(mainFile.toPath(), new OpenOption[0]);){
            Properties props = new Properties();
            props.load(in);
            UpdateCenterDeserializer.loadProperties(mainFile, props, PLUGINS);
            UpdateCenterDeserializer.loadProperties(mainFile, props, SCANNERS);
            UpdateCenter pluginReferential = this.fromProperties(props);
            pluginReferential.setDate(new Date(mainFile.lastModified()));
            UpdateCenter updateCenter = pluginReferential;
            return updateCenter;
        }
    }

    private static void loadProperties(File file, Properties props, String listKey) throws IOException {
        String[] keys2;
        for (String key : keys2 = UpdateCenterDeserializer.getArray(props, listKey)) {
            File propFile = new File(file.getParent(), key + ".properties");
            try (InputStream fileInputStream = Files.newInputStream(propFile.toPath(), new OpenOption[0]);){
                Properties p = new Properties();
                p.load(fileInputStream);
                for (Map.Entry<Object, Object> prop : p.entrySet()) {
                    props.put(key + "." + prop.getKey(), prop.getValue());
                }
            }
        }
    }

    public UpdateCenter fromProperties(Properties p) {
        Sonar sonar = new Sonar();
        Date date = FormatUtils.toDate(p.getProperty("date"), true);
        ArrayList<Plugin> plugins = new ArrayList<Plugin>();
        ArrayList<Scanner> scanners = new ArrayList<Scanner>();
        this.parseSonar(p, sonar);
        this.parsePlugins(p, sonar, plugins);
        this.parseScanners(p, sonar, scanners);
        this.validatePublicPluginSQVersionOverlap(plugins);
        this.validateLATESTonLatestPluginVersion(plugins);
        PluginReferential pluginReferential = PluginReferential.create(plugins);
        for (Plugin plugin : pluginReferential.getPlugins()) {
            for (Release release : plugin.getAllReleases()) {
                String[] requiredReleases;
                for (String requiresPluginKey : requiredReleases = StringUtils.split(StringUtils.defaultIfEmpty(this.get(p, plugin.getKey(), release.getVersion().getName() + ".requirePlugins", false), ""), ",")) {
                    String[] split = requiresPluginKey.split(":");
                    String requiredPluginReleaseKey = split[0];
                    String requiredMinimumReleaseVersion = split[1];
                    pluginReferential.addOutgoingDependency(release, requiredPluginReleaseKey, requiredMinimumReleaseVersion);
                }
            }
        }
        return UpdateCenter.create(pluginReferential, scanners, sonar).setDate(date);
    }

    private void reportError(String message) {
        if (!this.ignoreError) {
            throw new IllegalStateException(message);
        }
        LOGGER.error(message);
    }

    private void validatePublicPluginSQVersionOverlap(List<Plugin> plugins) {
        for (Plugin plugin : plugins) {
            this.validatePublicPluginSQVersionOverlap(plugin);
        }
    }

    private void validatePublicPluginSQVersionOverlap(Plugin plugin) {
        HashMap<Version, Release> sonarVersion = new HashMap<Version, Release>();
        for (Release r : plugin.getPublicReleases()) {
            for (Version v : r.getRequiredSonarVersions()) {
                if (sonarVersion.containsKey(v)) {
                    this.reportError("SQ version " + v + " is declared compatible with two public versions of " + UpdateCenterDeserializer.pluginName(plugin) + " plugin: " + r.getVersion() + " and " + ((Release)sonarVersion.get(v)).getVersion());
                }
                sonarVersion.put(v, r);
            }
        }
    }

    private void validateLATESTonLatestPluginVersion(List<Plugin> plugins) {
        for (Plugin plugin : plugins) {
            TreeSet<Release> publicAndArchivedReleases = new TreeSet<Release>(plugin.getPublicReleases());
            publicAndArchivedReleases.addAll(plugin.getArchivedReleases());
            for (Release r : publicAndArchivedReleases) {
                Version[] versionsWLatest = r.getSonarVersionFromString(LATEST_KEYWORD);
                if (r.equals(publicAndArchivedReleases.last()) || versionsWLatest.length <= 0) continue;
                this.reportError("Only the latest release of plugin " + UpdateCenterDeserializer.pluginName(plugin) + " may depend on " + LATEST_KEYWORD + " SonarQube");
            }
        }
    }

    private static String pluginName(Component component) {
        return StringUtils.isNotBlank(component.getName()) ? component.getName() : component.getKey();
    }

    private void parseScanners(Properties p, Sonar sonar, List<Scanner> scanners) {
        String[] scannerKeys;
        for (String pluginKey : scannerKeys = UpdateCenterDeserializer.getArray(p, SCANNERS)) {
            Scanner scanner = Scanner.factory(pluginKey);
            this.parseComponent(p, sonar, pluginKey, scanner);
            if (scanner.getAllReleases().isEmpty()) continue;
            scanners.add(scanner);
        }
    }

    private void parseComponent(Properties p, Sonar sonar, String key, Component c) {
        c.setName(this.get(p, key, "name", false));
        c.setDescription(this.get(p, key, "description", false));
        c.setCategory(this.get(p, key, "category", true));
        c.setHomepageUrl(this.get(p, key, "homepageUrl", false));
        c.setLicense(this.get(p, key, "license", false));
        c.setOrganization(this.get(p, key, "organization", false));
        c.setOrganizationUrl(this.get(p, key, "organizationUrl", false));
        c.setTermsConditionsUrl(this.get(p, key, "termsConditionsUrl", false));
        c.setIssueTrackerUrl(this.get(p, key, "issueTrackerUrl", false));
        c.setSourcesUrl(this.get(p, key, "scm", false));
        c.setDevelopers(Arrays.asList(UpdateCenterDeserializer.getArray(p, key, "developers")));
        HashMap<String, Map.Entry<String, Integer>> flavorsLabel = new HashMap<String, Map.Entry<String, Integer>>();
        this.parseFlavors(p, key, flavorsLabel);
        this.parseReleases(p, sonar, key, c, PUBLIC_VERSIONS, flavorsLabel, true, false);
        if (this.mode == Mode.DEV) {
            this.parseReleases(p, sonar, key, c, PRIVATE_VERSIONS, flavorsLabel, false, false);
            this.parseDevVersions(p, sonar, key, c, flavorsLabel);
        }
        if (this.includeArchives) {
            this.parseReleases(p, sonar, key, c, PRIVATE_VERSIONS, flavorsLabel, false, false);
            this.parseReleases(p, sonar, key, c, ARCHIVED_VERSIONS, flavorsLabel, false, false);
        } else {
            this.parseReleases(p, sonar, key, c, ARCHIVED_VERSIONS, flavorsLabel, false, true);
        }
    }

    private void parsePlugins(Properties p, Sonar sonar, List<Plugin> plugins) {
        String[] pluginKeys;
        for (String pluginKey : pluginKeys = UpdateCenterDeserializer.getArray(p, PLUGINS)) {
            Plugin plugin = Plugin.factory(pluginKey);
            this.parseComponent(p, sonar, pluginKey, plugin);
            if (UpdateCenterDeserializer.isPluginCompatibleWithAnySqRelease(plugin, sonar)) {
                plugins.add(plugin);
                continue;
            }
            LOGGER.warn("The plugin {} is not compatible with any public SQ versions.", (Object)pluginKey);
        }
    }

    private static boolean isPluginCompatibleWithAnySqRelease(Plugin plugin, Sonar sonar) {
        return sonar.getMajorReleases().stream().map(sonarMajorRelease -> plugin.getLastCompatible(sonarMajorRelease.getVersion())).anyMatch(Objects::nonNull);
    }

    private void parseReleases(Properties p, Sonar sonar, String pluginKey, Component component, String key, HashMap<String, Map.Entry<String, Integer>> flavosLabel, boolean isPublicRelease, boolean isArchivedRelease) {
        String[] pluginPublicReleases;
        for (String pluginVersion : pluginPublicReleases = UpdateCenterDeserializer.getArray(p, pluginKey, key)) {
            Release releaseToAdd = this.parseRelease(p, sonar, pluginKey, component, isPublicRelease, isArchivedRelease, pluginVersion, flavosLabel);
            Optional<Release> alreadyExistingRelease = component.getAllReleases().stream().filter(r -> r.getArtifact().equals(releaseToAdd.getArtifact())).filter(r -> r.getVersion().equals(releaseToAdd.getVersion())).findFirst();
            if (alreadyExistingRelease.isPresent()) {
                this.reportDuplicateReleaseDeclaration(alreadyExistingRelease.get(), releaseToAdd);
                continue;
            }
            component.addRelease(releaseToAdd);
        }
    }

    private void parseFlavors(Properties p, String pluginKey, HashMap<String, Map.Entry<String, Integer>> flavosLabel) {
        String[] flavors = UpdateCenterDeserializer.getArray(p, pluginKey, FLAVORS_PREFIX);
        for (int i = 0; i < flavors.length; ++i) {
            flavosLabel.put(flavors[i], new AbstractMap.SimpleEntry<String, Integer>(this.get(p, pluginKey, "flavors." + flavors[i] + ".label", true), i));
        }
    }

    private Release parseRelease(Properties p, Sonar sonar, String pluginKey, Component component, boolean isPublicRelease, boolean isArchivedRelease, String pluginVersion, HashMap<String, Map.Entry<String, Integer>> flavosLabel) {
        Release release = new Release((Artifact)component, pluginVersion);
        try {
            release.setPublic(isPublicRelease);
            release.setArchived(isArchivedRelease);
            this.parseDownloadUrl(p, pluginKey, pluginVersion, isPublicRelease, flavosLabel, release);
            release.setChangelogUrl(this.getOrDefault(p, pluginKey, pluginVersion, CHANGELOG_URL_SUFFIX, false));
            release.setDisplayVersion(this.getOrDefault(p, pluginKey, pluginVersion, DISPLAY_VERSION_SUFFIX, false));
            release.setDate(FormatUtils.toDate(this.getOrDefault(p, pluginKey, pluginVersion, DATE_SUFFIX, isPublicRelease), false));
            release.setDescription(this.getOrDefault(p, pluginKey, pluginVersion, DESCRIPTION_SUFFIX, isPublicRelease));
            if (component.needArtifact()) {
                release.setGroupId(this.getOrDefault(p, pluginKey, pluginVersion, MAVEN_GROUPID_SUFFIX, true));
                release.setArtifactId(this.getOrDefault(p, pluginKey, pluginVersion, MAVEN_ARTIFACTID_SUFFIX, true));
            }
            if (component.needSqVersion()) {
                Version[] requiredSonarVersions = this.getRequiredSonarVersions(p, pluginKey, pluginVersion, sonar, isArchivedRelease);
                if (!isArchivedRelease && requiredSonarVersions.length == 0) {
                    this.reportError("Plugin " + UpdateCenterDeserializer.pluginName(component) + " version " + pluginVersion + " should declare compatible SQ versions");
                }
                for (Version requiredSonarVersion : requiredSonarVersions) {
                    release.addRequiredSonarVersions(requiredSonarVersion);
                }
            }
        }
        catch (IllegalArgumentException ex) {
            throw new IllegalArgumentException("issue while processing plugin " + pluginKey, ex);
        }
        return release;
    }

    private void reportDuplicateReleaseDeclaration(Release alreadyExistingRelease, Release releaseToAdd) {
        if (alreadyExistingRelease.isArchived() != releaseToAdd.isArchived()) {
            String message = "Plugin " + releaseToAdd.getKey() + ": " + releaseToAdd.getVersion() + " cannot be both public and archived.";
            this.reportError(message);
        } else {
            String message = "Duplicate version for plugin " + releaseToAdd.getKey() + ": " + releaseToAdd.getVersion();
            this.reportError(message);
        }
    }

    private void parseDownloadUrl(Properties p, String pluginKey, String pluginVersion, boolean isPublicRelease, HashMap<String, Map.Entry<String, Integer>> flavorLabel, Release release) {
        for (Map.Entry<String, Map.Entry<String, Integer>> flavor : flavorLabel.entrySet()) {
            String url = this.get(p, pluginKey, pluginVersion + DOWNLOAD_URL_SUFFIX + "." + flavor.getKey(), false);
            if (url == null) continue;
            release.addScannerDownloadUrlAndLabel(flavor.getKey(), flavor.getValue().getKey(), url, flavor.getValue().getValue());
        }
        String url = this.getOrDefault(p, pluginKey, pluginVersion, DOWNLOAD_URL_SUFFIX, false);
        if (url != null) {
            release.setDownloadUrl(url);
        }
        if (!release.hasDownloadUrl() && isPublicRelease) {
            this.reportError("Download url is missing");
        }
    }

    private void parseDevVersions(Properties p, Sonar sonar, String pluginKey, Component component, HashMap<String, Map.Entry<String, Integer>> flavosLabel) {
        String devVersion = this.get(p, pluginKey, DEV_VERSION, false);
        if (StringUtils.isNotBlank(devVersion)) {
            Release release = this.parseRelease(p, sonar, pluginKey, component, false, false, devVersion, flavosLabel);
            component.setDevRelease(release);
        }
    }

    private void parseSonar(Properties p, Sonar sonar) {
        this.parseSonarVersions(p, sonar);
        if (this.mode == Mode.DEV) {
            this.parseSonarDevVersions(p, sonar);
        }
        this.parseSonarLtsVersion(p, sonar);
    }

    private void parseSonarDevVersions(Properties p, Sonar sonar) {
        String devVersion = this.get(p, DEV_VERSION, true);
        Release release = this.parseSonarVersion(p, sonar, false, devVersion);
        sonar.setDevRelease(release);
    }

    private void parseSonarLtsVersion(Properties p, Sonar sonar) {
        String ltsVersion = this.get(p, "ltsVersion", true);
        sonar.setLtsRelease(ltsVersion);
        if (!sonar.getReleases().contains(sonar.getLtsRelease())) {
            this.reportError("ltsVersion seems wrong as it is not listed in SonarQube versions");
        }
    }

    private void parseSonarVersions(Properties p, Sonar sonar) {
        this.parseSonarVersions(p, sonar, PUBLIC_VERSIONS, true);
        if (this.mode == Mode.DEV || this.includeArchives) {
            this.parseSonarVersions(p, sonar, PRIVATE_VERSIONS, false);
        }
    }

    private void parseSonarVersions(Properties p, Sonar sonar, String key, boolean isPublicRelease) {
        for (String sonarVersion : UpdateCenterDeserializer.getArray(p, key)) {
            Release release = this.parseSonarVersion(p, sonar, isPublicRelease, sonarVersion);
            if (!sonar.getAllReleases().contains(release)) {
                sonar.addRelease(release);
                continue;
            }
            this.reportError("Duplicate version for SonarQube: " + sonarVersion);
        }
    }

    private Release parseSonarVersion(Properties p, Sonar sonar, boolean isPublicRelease, String sonarVersion) {
        Release release = new Release((Artifact)sonar, sonarVersion);
        release.setPublic(isPublicRelease);
        release.setChangelogUrl(this.getOrDefault(p, sonarVersion, CHANGELOG_URL_SUFFIX, isPublicRelease));
        release.setDisplayVersion(this.getOrDefault(p, sonarVersion, DISPLAY_VERSION_SUFFIX, false));
        release.setDescription(this.getOrDefault(p, sonarVersion, DESCRIPTION_SUFFIX, isPublicRelease));
        for (Release.Edition edition : Release.Edition.values()) {
            String downloadUrl = this.getOrDefault(p, sonarVersion, UpdateCenterDeserializer.getDownloadUrlSuffix(edition), edition == Release.Edition.COMMUNITY && isPublicRelease);
            if (downloadUrl == null) continue;
            release.setDownloadUrl(downloadUrl, edition);
        }
        release.setDate(FormatUtils.toDate(this.getOrDefault(p, sonarVersion, DATE_SUFFIX, isPublicRelease), false));
        return release;
    }

    private Version[] getRequiredSonarVersions(Properties p, String pluginKey, String pluginVersion, Sonar sonar, boolean isArchived) {
        String sqVersions = this.get(p, pluginKey, pluginVersion + ".sqVersions", !isArchived);
        List<String> patterns = UpdateCenterDeserializer.split(StringUtils.defaultIfEmpty(sqVersions, ""));
        LinkedList<Version> result = new LinkedList<Version>();
        for (String pattern : patterns) {
            if (pattern == null) continue;
            Matcher multipleEltMatcher = Pattern.compile("\\[(.*),(.*)\\]").matcher(pattern);
            Matcher simpleEltMatcher = Pattern.compile("\\[(.*)\\]").matcher(pattern);
            if (multipleEltMatcher.matches()) {
                Version low = UpdateCenterDeserializer.resolveLowVersion(multipleEltMatcher.group(1), pattern, pluginKey);
                Version high = UpdateCenterDeserializer.resolveKeywordAndStar(multipleEltMatcher.group(2), sonar, pluginKey);
                UpdateCenterDeserializer.resolveRangeOfRequiredSQVersion(sonar, result, low, high);
                continue;
            }
            if (simpleEltMatcher.matches()) {
                result.add(UpdateCenterDeserializer.resolveKeywordAndStar(simpleEltMatcher.group(1), sonar, pluginKey));
                continue;
            }
            result.add(UpdateCenterDeserializer.resolveKeywordAndStar(pattern, sonar, pluginKey));
        }
        return result.toArray(new Version[result.size()]);
    }

    private static void resolveRangeOfRequiredSQVersion(Sonar sonar, List<Version> result, Version low, Version high) {
        sonar.getAllReleases().stream().filter(Objects::nonNull).map(Release::getVersion).filter(Objects::nonNull).filter(version -> version.compareTo(low) >= 0 && version.compareTo(high) <= 0).forEach(version -> {
            String fromString = version.equals(low) ? low.getFromString() : (version.equals(high) ? high.getFromString() : "");
            result.add(Version.create(version, fromString));
        });
    }

    private static List<String> split(String requiredSonarVersions) {
        ArrayList<String> splitted = new ArrayList<String>();
        int skipCommas = 0;
        String s = "";
        for (char c : requiredSonarVersions.toCharArray()) {
            if (c == ',' && skipCommas == 0) {
                splitted.add(s);
                s = "";
                continue;
            }
            if (c == '[') {
                ++skipCommas;
            }
            if (c == ']') {
                --skipCommas;
            }
            s = s + Character.toString(c);
        }
        if (StringUtils.isNotBlank(s)) {
            splitted.add(s);
        }
        return splitted;
    }

    private static Version resolveLowVersion(String versionStr, String range, String pluginKey) {
        if (LATEST_KEYWORD.equals(versionStr)) {
            throw new SonarVersionRangeException(String.format("Cannot use LATEST keyword at the start of a range in '%s' (in plugin '%s'). Use 'sqVersions=LATEST' instead.", range, pluginKey));
        }
        if (versionStr.endsWith("*")) {
            throw new SonarVersionRangeException(String.format("Cannot use a wildcard version at the start of a range in '%s' (in plugin '%s'). If you want to mark this range as compatible with any MAJOR.MINOR.* version, use the MAJOR.MINOR version instead (e.g.: 'sqVersions=[6.7,6.7.*]', 'sqVersions=[6.7,LATEST]').", range, pluginKey));
        }
        return Version.create(versionStr);
    }

    private static Version resolveKeywordAndStar(String versionStr, Sonar sonar, String pluginKey) {
        if (LATEST_KEYWORD.equals(versionStr)) {
            return Version.create(sonar.getAllReleases().last().getVersion(), LATEST_KEYWORD);
        }
        if (versionStr.endsWith("*")) {
            return UpdateCenterDeserializer.resolveWithWildcard(versionStr, sonar, pluginKey);
        }
        return Version.create(versionStr);
    }

    private static Version resolveWithWildcard(String versionStr, Sonar sonar, String pluginKey) {
        String prefix = versionStr.substring(0, versionStr.length() - 1);
        String prefixWithoutDot = prefix.endsWith(".") ? prefix.substring(0, prefix.length() - 1) : prefix;
        Release found = null;
        for (Release r : sonar.getAllReleases()) {
            if (!r.getVersion().toString().equals(prefixWithoutDot) && !r.getVersion().toString().startsWith(prefix)) continue;
            found = r;
        }
        if (found != null) {
            return Version.create(found.getVersion(), "*");
        }
        throw new IllegalStateException(String.format("Unable to resolve version '%s' (in plugin '%s')", versionStr, pluginKey));
    }

    private String getOrDefault(Properties props, String sqVersion, String suffix, boolean required) {
        String defaultKey;
        String key = sqVersion + suffix;
        String value = UpdateCenterDeserializer.getOrDefault(props, key, defaultKey = DEFAULTS_PREFIX + suffix);
        if (StringUtils.isBlank(value) && required) {
            this.reportUndefined(key);
        }
        return value;
    }

    private void reportUndefined(String key) {
        this.reportError(key + " should be defined");
    }

    private String get(Properties props, String key, boolean required) {
        String value = UpdateCenterDeserializer.get(props, key);
        if (StringUtils.isBlank(value) && required) {
            this.reportUndefined(key);
        }
        return value;
    }

    private static String get(Properties props, String key) {
        return StringUtils.defaultIfEmpty(props.getProperty(key), null);
    }

    private static String getOrDefault(Properties props, String key, String defaultKey) {
        if (props.containsKey(key)) {
            return props.getProperty(key);
        }
        return StringUtils.defaultIfEmpty(props.getProperty(defaultKey), null);
    }

    private String getOrDefault(Properties props, String pluginKey, String version, String suffix, boolean required) {
        String defaultKey;
        String key = pluginKey + "." + version + suffix;
        String value = UpdateCenterDeserializer.getOrDefault(props, key, defaultKey = pluginKey + "." + DEFAULTS_PREFIX + suffix);
        if (StringUtils.isBlank(value) && required) {
            this.reportUndefined(key);
        }
        return value;
    }

    private String get(Properties p, String pluginKey, String field, boolean required) {
        String key = pluginKey + "." + field;
        String value = UpdateCenterDeserializer.get(p, key);
        if (StringUtils.isBlank(value) && required) {
            this.reportUndefined(key);
        }
        return value;
    }

    private static String[] getArray(Properties props, String key) {
        return StringUtils.split(StringUtils.defaultIfEmpty(props.getProperty(key), ""), ",");
    }

    private static String[] getArray(Properties p, String pluginKey, String field) {
        return UpdateCenterDeserializer.getArray(p, pluginKey + "." + field);
    }

    public static enum Mode {
        PROD,
        DEV;

    }
}

