"""Helper function and aspect to collect first-party packages.

These are used in node rules to link the node_modules before launching a program.
This supports path re-mapping, to support short module names.
See pathMapping doc: https://github.com/Microsoft/TypeScript/issues/5039

This reads the module_root and module_name attributes from rules in
the transitive closure, rolling these up to provide a mapping to the
linker, which uses the mappings to link a node_modules directory for
runtimes to locate all the first-party packages.
"""

# Can't load from //:providers.bzl directly as that introduces a circular dep
load("//internal/providers:linkable_package_info.bzl", "LinkablePackageInfo")
load("//internal/providers:npm_package_info.bzl", "NpmPackageInfo")

def _debug(vars, *args):
    if "VERBOSE_LOGS" in vars.keys():
        print("[link_node_modules.bzl]", *args)

# Arbitrary name; must be chosen to globally avoid conflicts with any other aspect
_ASPECT_RESULT_NAME = "link_node_modules__aspect_result"

# Traverse 'srcs' in addition so that we can go across a genrule
_MODULE_MAPPINGS_DEPS_NAMES = ["data", "deps", "srcs"]

def add_arg(args, arg):
    """Add an argument

    Args:
        args: either a list or a ctx.actions.Args object
        arg: string arg to append on the end
    """
    if (type(args) == type([])):
        args.append(arg)
    else:
        args.add(arg)

def _link_mapping(label, mappings, k, v):
    if k in mappings and mappings[k] != v:
        # Allow all other mappings to win over legacy "_tslibrary"
        if mappings[k][0] == "_tslibrary":
            return True
        elif v[0] == "_tslibrary":
            return False
        if v[1] == mappings[k][1]:
            # Allow "execroot" to win over "runfiles"
            # For example,
            #    mappings[k] = ["runfiles", "angular/packages/compiler"]
            #    v           = ["execroot", "angular/packages/compiler"]
            if mappings[k][0] == "runfiles":
                return True
            elif v[0] == "runfiles":
                return False
        fail(("conflicting mapping at '%s': '%s' maps to both %s and %s" % (label, k, mappings[k], v)), "deps")
    else:
        return True

def write_node_modules_manifest(ctx, extra_data = [], mnemonic = None, link_workspace_root = False):
    """Writes a manifest file read by the linker, containing info about resolving runtime dependencies

    Args:
        ctx: starlark rule execution context
        extra_data: labels to search for npm packages that need to be linked (ctx.attr.deps and ctx.attr.data will always be searched)
        mnemonic: optional action mnemonic, used to differentiate module mapping files from the same rule context
        link_workspace_root: Link the workspace root to the bin_dir to support absolute requires like 'my_wksp/path/to/file'.
            If source files need to be required then they can be copied to the bin_dir with copy_to_bin.
    """

    mappings = {ctx.workspace_name: ["execroot", ctx.bin_dir.path]} if link_workspace_root else {}
    node_modules_root = ""

    # Look through data/deps attributes to find...
    for dep in extra_data + getattr(ctx.attr, "data", []) + getattr(ctx.attr, "deps", []):
        # ...the root directory for the third-party node_modules; we'll symlink the local "node_modules" to it
        if NpmPackageInfo in dep:
            possible_root = "/".join([dep[NpmPackageInfo].workspace, "node_modules"])
            if not node_modules_root:
                node_modules_root = possible_root
            elif node_modules_root != possible_root:
                fail("All npm dependencies need to come from a single workspace. Found '%s' and '%s'." % (node_modules_root, possible_root))

        # ...first-party packages to be linked into the node_modules tree
        for k, v in getattr(dep, _ASPECT_RESULT_NAME, {}).items():
            if _link_mapping(dep.label, mappings, k, v):
                # Special case for ts_library module_name for legacy behavior and for AMD name
                # work-around. Do not propagate tslibrary root type to runtime as it is not
                # needed at runtime. See comment below in _get_module_mappings for more info.
                if v[0] == "_tslibrary":
                    v = ["execroot", v[1]]
                _debug(ctx.var, "Linking %s: %s" % (k, v))
                mappings[k] = v

    # Write the result to a file, and use the magic node option --bazel_node_modules_manifest
    # The launcher.sh will peel off this argument and pass it to the linker rather than the program.
    prefix = ctx.label.name
    if mnemonic != None:
        prefix += "_%s" % mnemonic
    modules_manifest = ctx.actions.declare_file("_%s.module_mappings.json" % prefix)
    content = {
        "bin": ctx.bin_dir.path,
        "modules": mappings,
        "root": node_modules_root,
        "workspace": ctx.workspace_name,
    }
    ctx.actions.write(modules_manifest, str(content))
    return modules_manifest

def _get_module_mappings(target, ctx):
    """Returns the module_mappings from the given attrs.

    Collects a {module_name - module_root} hash from all transitive dependencies,
    checking for collisions. If a module has a non-empty `module_root` attribute,
    all sources underneath it are treated as if they were rooted at a folder
    `module_name`.

    Args:
      target: target
      ctx: ctx

    Returns:
      The module mappings
    """
    mappings = {}

    # Propogate transitive mappings
    for name in _MODULE_MAPPINGS_DEPS_NAMES:
        for dep in getattr(ctx.rule.attr, name, []):
            for k, v in getattr(dep, _ASPECT_RESULT_NAME, {}).items():
                # A package which was reachable transitively via a *_binary or *_test
                # rule is assumed to be in the runfiles of that binary,
                # so we switch the linker target root.
                # In theory we ought to be able to tell that the files really are
                # propagated through the runfiles output of the target we are visiting
                # but it seems that this info is only available in Bazel Java internals.
                # TODO: revisit whether this is the correct condition after downstream testing
                if ctx.rule.kind.endswith("_binary") or ctx.rule.kind.endswith("_test"):
                    v = ["runfiles", v[1]]
                if _link_mapping(target.label, mappings, k, v):
                    mappings[k] = v
                    _debug(ctx.var, "target %s propagating module mapping %s: %s" % (dep.label, k, v))

    # Look for LinkablePackageInfo mapping in this node
    if not LinkablePackageInfo in target:
        # No mappings contributed here, short-circuit with the transitive ones we collected
        _debug(ctx.var, "No LinkablePackageInfo for", target.label)
        return mappings

    linkable_package_info = target[LinkablePackageInfo]

    mn = linkable_package_info.package_name
    mr = ["execroot", linkable_package_info.path]

    # Special case for ts_library module_name for legacy behavior and for AMD name work-around
    # Tag the mapping as "_tslibrary" so it can be overridden by any other mapping if found.
    #
    # In short, ts_library module_name attribute results in both setting the AMD name (which
    # desired and necessary in devmode which outputs UMD) and in making a linkable mapping. Because
    # of this, you can get in the situation where a ts_library module_name and a downstream pkg_name
    # package_name conflict and result in duplicate mappings. This work-around will make this
    # situation work however it is not a recommended pattern since a ts_library can be a dep of a
    # pkg_npm but not vice-verse at the moment since ts_library cannot handle directory artifacts as
    # deps.
    if hasattr(linkable_package_info, "_tslibrary") and linkable_package_info._tslibrary:
        mr[0] = "_tslibrary"

    if _link_mapping(target.label, mappings, mn, mr):
        _debug(ctx.var, "target %s adding module mapping %s: %s" % (target.label, mn, mr))
        mappings[mn] = mr

    return mappings

def _module_mappings_aspect_impl(target, ctx):
    # Use a dictionary to construct the result struct
    # so that we can reference the _ASPECT_RESULT_NAME variable
    return struct(**{
        _ASPECT_RESULT_NAME: _get_module_mappings(target, ctx),
    })

module_mappings_aspect = aspect(
    _module_mappings_aspect_impl,
    attr_aspects = _MODULE_MAPPINGS_DEPS_NAMES,
)
