﻿//------------------------------------------------------------------------------
//
// Copyright (c) Microsoft Corporation.
// All rights reserved.
//
// This code is licensed under the MIT License.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files(the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and / or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions :
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Security.Cryptography.X509Certificates;
using Microsoft.IdentityModel.Protocols.WsFederation;
using Microsoft.IdentityModel.Tokens;
using Microsoft.IdentityModel.Xml;

namespace Microsoft.IdentityModel.TestUtils
{
    public static class ReferenceMetadata
    {
        public static List<SecurityKey> MetadataSigningKeys
        {
            get => new List<SecurityKey>
            {
                new X509SecurityKey(new X509Certificate2(Convert.FromBase64String(X509CertificateData1))),
                new X509SecurityKey(new X509Certificate2(Convert.FromBase64String(X509CertificateData2))),
                new X509SecurityKey(new X509Certificate2(Convert.FromBase64String(X509CertificateData3)))
            };
        }

        public static SecurityKey MetadataSigningKey
        {
            get => new X509SecurityKey(new X509Certificate2(Convert.FromBase64String(X509CertificateData1)));
        }

        public static SecurityKey AdfsV2MetadataSigningKey
        {
            get => new X509SecurityKey(new X509Certificate2(Convert.FromBase64String(X509CertificateDataAdfsV2)));
        }

        public static SecurityKey AdfsV3MetadataSigningKey
        {
            get => new X509SecurityKey(new X509Certificate2(Convert.FromBase64String(X509CertificateDataAdfsV3)));
        }

        public static SecurityKey AdfsV4MetadataSigningKey
        {
            get => new X509SecurityKey(new X509Certificate2(Convert.FromBase64String(X509CertificateDataAdfsV4)));
        }

        public static X509Certificate2 X509Certificate1
        {
            get => new X509Certificate2(Convert.FromBase64String(X509CertificateData1));
        }

        public static X509Certificate2 X509Certificate2
        {
            get => new X509Certificate2(Convert.FromBase64String(X509CertificateData2));
        }

        public static X509Certificate2 X509Certificate3
        {
            get => new X509Certificate2(Convert.FromBase64String(X509CertificateData3));
        }

        public static X509Certificate2 X509CertificateAdfsV2
        {
            get => new X509Certificate2(Convert.FromBase64String(X509CertificateDataAdfsV2));
        }

        public static X509Certificate2 X509CertificateAdfsV3
        {
            get => new X509Certificate2(Convert.FromBase64String(X509CertificateDataAdfsV3));
        }

        public static X509Certificate2 X509CertificateAdfsV4
        {
            get => new X509Certificate2(Convert.FromBase64String(X509CertificateDataAdfsV4));
        }

        public static string AADCommonMetadata { get => @"<?xml version=""1.0"" encoding=""utf-8""?><EntityDescriptor ID=""_0ded55d8-a72f-4e13-ab9e-f40be80b1476"" entityID=""https://sts.windows.net/{tenantid}/"" xmlns=""urn:oasis:names:tc:SAML:2.0:metadata""><Signature xmlns=""http://www.w3.org/2000/09/xmldsig#""><SignedInfo><CanonicalizationMethod Algorithm=""http://www.w3.org/2001/10/xml-exc-c14n#"" /><SignatureMethod Algorithm=""http://www.w3.org/2001/04/xmldsig-more#rsa-sha256"" /><Reference URI=""#_0ded55d8-a72f-4e13-ab9e-f40be80b1476""><Transforms><Transform Algorithm=""http://www.w3.org/2000/09/xmldsig#enveloped-signature"" /><Transform Algorithm=""http://www.w3.org/2001/10/xml-exc-c14n#"" /></Transforms><DigestMethod Algorithm=""http://www.w3.org/2001/04/xmlenc#sha256"" /><DigestValue>lnicj3SNizOF6QI1rWw8JrouoiXMslqtFB5ztWW6qvs=</DigestValue></Reference></SignedInfo><SignatureValue>KD9uWOD/9pvF1NlNCpYoXymUPS1l9uIBgBDe0uOQgQv+tUI/1jJX4UpjADDHCOx6HCl5ZgZSXNmOC2lLSJEwmv21BZzI+PAOxF5hdH99cS/lMC/hxgyWdLVeGnr1I4WbPxGqVmjFNuBdBMaourO4z/5f3D2JZQmgnlu8H+4gv2SpjeZz/YhIN6ZrNfmHwsKZashMGtSmE5uHro+uO5yO17Gr9YfUbtokLRIq5Dk9kqnxG8YZF1C1nC9O0PMdlHb4ubwgO20Cvz5sU2iswn9m68btS5TLF5OVhETzyKir1QA+H1tCgGRqIWd4Geyoucdct1r4zAJGCNIekdKnY3NXwg==</SignatureValue><KeyInfo><X509Data><X509Certificate>MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD</X509Certificate></X509Data></KeyInfo></Signature><RoleDescriptor xsi:type=""fed:SecurityTokenServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/wsfed/federation/200706"" xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706""><KeyDescriptor use=""signing""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD</X509Certificate></X509Data></KeyInfo></KeyDescriptor><KeyDescriptor use=""signing""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIDBTCCAe2gAwIBAgIQXxLnqm1cOoVGe62j7W7wZzANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDMyNjAwMDAwMFoXDTE5MDMyNzAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKJGarCm4IF0/Gz5Xx/zyZwD2rdJJZtO2Ukk1Oz+Br1sLVY8I5vj5esB+lotmLEblA9N/w188vmTvykaEzUl49NA4s86x44SW6WtdQbGJ0IjpQJUalUMyy91vIBkK/7K3nBXeVBsRk7tm528leoQ05/aZ+1ycJBIU+1oGYThv8MOjyHAlXJmCaGXwXTisZ+hHjcwlMk/+ZEutHflKLIpPUNEi7j4Xw+zp9UKo5pzWIr/iJ4HjvCkFofW90AMF2xp8dMhpbVcfJGS/Ii3J66LuNLCH/HtSZ42FO+tnRL/nNzzFWUhGT92Q5VFVngfWJ3PAg1zz8I1wowLD2fiB2udGXcCAwEAAaMhMB8wHQYDVR0OBBYEFFXPbFXjmMR3BluF+2MeSXd1NQ3rMA0GCSqGSIb3DQEBCwUAA4IBAQAsd3wGVilJxDtbY1K2oAsWLdNJgmCaYdrtdlAsjGlarSQSzBH0Ybf78fcPX//DYaLXlvaEGKVKp0jPq+RnJ17oP/RJpJTwVXPGRIlZopLIgnKpWlS/PS0uKAdNvLmz1zbGSILdcF+Qf41OozD4QNsS1c9YbDO4vpC9v8x3PVjfJvJwPonzNoOsLXA+8IONSXwCApsnmrwepKu8sifsFYSwgrwxRPGTEAjkdzRJ0yMqiY/VoJ7lqJ/FBJqqAjGPGq/yI9rVoG+mbO1amrIDWHHTKgfbKk0bXGtVUbsayyHR5jSgadmkLBh5AaN/HcgDK/jINrnpiQ+/2ewH/8qLaQ3B</X509Certificate></X509Data></KeyInfo></KeyDescriptor><KeyDescriptor use=""signing""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIDKDCCAhCgAwIBAgIQBHJvVNxP1oZO4HYKh+rypDANBgkqhkiG9w0BAQsFADAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXMwHhcNMTYxMTE2MDgwMDAwWhcNMTgxMTE2MDgwMDAwWjAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXMwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQChn5BCs24Hh6L0BNitPrV5s+2/DBhaeytOmnghJKnqeJlhv3ZczShRM2Cp38LW8Y3wn7L3AJtolaSkF/joKN1l6GupzM+HOEdq7xZxFehxIHW7+25mG/WigBnhsBzLv1SR4uIbrQeS5M0kkLwJ9pOnVH3uzMGG6TRXPnK3ivlKl97AiUEKdlRjCQNLXvYf1ZqlC77c/ZCOHSX4kvIKR2uG+LNlSTRq2rn8AgMpFT4DSlEZz4RmFQvQupQzPpzozaz/gadBpJy/jgDmJlQMPXkHp7wClvbIBGiGRaY6eZFxNV96zwSR/GPNkTObdw2S8/SiAgvIhIcqWTPLY6aVTqJfAgMBAAGjWDBWMFQGA1UdAQRNMEuAEDUj0BrjP0RTbmoRPTRMY3WhJTAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXOCEARyb1TcT9aGTuB2Cofq8qQwDQYJKoZIhvcNAQELBQADggEBAGnLhDHVz2gLDiu9L34V3ro/6xZDiSWhGyHcGqky7UlzQH3pT5so8iF5P0WzYqVtogPsyC2LPJYSTt2vmQugD4xlu/wbvMFLcV0hmNoTKCF1QTVtEQiAiy0Aq+eoF7Al5fV1S3Sune0uQHimuUFHCmUuF190MLcHcdWnPAmzIc8fv7quRUUsExXmxSX2ktUYQXzqFyIOSnDCuWFm6tpfK5JXS8fW5bpqTlrysXXz/OW/8NFGq/alfjrya4ojrOYLpunGriEtNPwK7hxj1AlCYEWaRHRXaUIW1ByoSff/6Y6+ZhXPUe0cDlNRt/qIz5aflwO7+W8baTS4O8m/icu7ItE=</X509Certificate></X509Data></KeyInfo></KeyDescriptor><fed:ClaimTypesOffered><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/name"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Name</auth:DisplayName><auth:Description>The mutable display name of the user.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/nameidentifier"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Subject</auth:DisplayName><auth:Description>An immutable, globally unique, non-reusable identifier of the user that is unique to the application for which a token is issued.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/givenname"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Given Name</auth:DisplayName><auth:Description>First name of the user.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/surname"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Surname</auth:DisplayName><auth:Description>Last name of the user.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/identity/claims/displayname"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Display Name</auth:DisplayName><auth:Description>Display name of the user.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/identity/claims/nickname"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Nick Name</auth:DisplayName><auth:Description>Nick name of the user.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/authenticationinstant"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Authentication Instant</auth:DisplayName><auth:Description>The time (UTC) when the user is authenticated to Windows Azure Active Directory.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/authenticationmethod"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Authentication Method</auth:DisplayName><auth:Description>The method that Windows Azure Active Directory uses to authenticate users.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/identity/claims/objectidentifier"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>ObjectIdentifier</auth:DisplayName><auth:Description>Primary identifier for the user in the directory. Immutable, globally unique, non-reusable.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/identity/claims/tenantid"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>TenantId</auth:DisplayName><auth:Description>Identifier for the user's tenant.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/identity/claims/identityprovider"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>IdentityProvider</auth:DisplayName><auth:Description>Identity provider for the user.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/emailaddress"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Email</auth:DisplayName><auth:Description>Email address of the user.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/groups"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Groups</auth:DisplayName><auth:Description>Groups of the user.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/identity/claims/accesstoken"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>External Access Token</auth:DisplayName><auth:Description>Access token issued by external identity provider.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/expiration"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>External Access Token Expiration</auth:DisplayName><auth:Description>UTC expiration time of access token issued by external identity provider.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/identity/claims/openid2_id"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>External OpenID 2.0 Identifier</auth:DisplayName><auth:Description>OpenID 2.0 identifier issued by external identity provider.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/claims/groups.link"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>GroupsOverageClaim</auth:DisplayName><auth:Description>Issued when number of user's group claims exceeds return limit.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/role"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Role Claim</auth:DisplayName><auth:Description>Roles that the user or Service Principal is attached to</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/wids"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>RoleTemplate Id Claim</auth:DisplayName><auth:Description>Role template id of the Built-in Directory Roles that the user is a member of</auth:Description></auth:ClaimType></fed:ClaimTypesOffered><fed:SecurityTokenServiceEndpoint><wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing""><wsa:Address>https://login.microsoftonline.com/common/wsfed</wsa:Address></wsa:EndpointReference></fed:SecurityTokenServiceEndpoint><fed:PassiveRequestorEndpoint><wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing""><wsa:Address>https://login.microsoftonline.com/common/wsfed</wsa:Address></wsa:EndpointReference></fed:PassiveRequestorEndpoint></RoleDescriptor><RoleDescriptor xsi:type=""fed:ApplicationServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/wsfed/federation/200706"" xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706""><KeyDescriptor use=""signing""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD</X509Certificate></X509Data></KeyInfo></KeyDescriptor><KeyDescriptor use=""signing""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIDBTCCAe2gAwIBAgIQXxLnqm1cOoVGe62j7W7wZzANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDMyNjAwMDAwMFoXDTE5MDMyNzAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKJGarCm4IF0/Gz5Xx/zyZwD2rdJJZtO2Ukk1Oz+Br1sLVY8I5vj5esB+lotmLEblA9N/w188vmTvykaEzUl49NA4s86x44SW6WtdQbGJ0IjpQJUalUMyy91vIBkK/7K3nBXeVBsRk7tm528leoQ05/aZ+1ycJBIU+1oGYThv8MOjyHAlXJmCaGXwXTisZ+hHjcwlMk/+ZEutHflKLIpPUNEi7j4Xw+zp9UKo5pzWIr/iJ4HjvCkFofW90AMF2xp8dMhpbVcfJGS/Ii3J66LuNLCH/HtSZ42FO+tnRL/nNzzFWUhGT92Q5VFVngfWJ3PAg1zz8I1wowLD2fiB2udGXcCAwEAAaMhMB8wHQYDVR0OBBYEFFXPbFXjmMR3BluF+2MeSXd1NQ3rMA0GCSqGSIb3DQEBCwUAA4IBAQAsd3wGVilJxDtbY1K2oAsWLdNJgmCaYdrtdlAsjGlarSQSzBH0Ybf78fcPX//DYaLXlvaEGKVKp0jPq+RnJ17oP/RJpJTwVXPGRIlZopLIgnKpWlS/PS0uKAdNvLmz1zbGSILdcF+Qf41OozD4QNsS1c9YbDO4vpC9v8x3PVjfJvJwPonzNoOsLXA+8IONSXwCApsnmrwepKu8sifsFYSwgrwxRPGTEAjkdzRJ0yMqiY/VoJ7lqJ/FBJqqAjGPGq/yI9rVoG+mbO1amrIDWHHTKgfbKk0bXGtVUbsayyHR5jSgadmkLBh5AaN/HcgDK/jINrnpiQ+/2ewH/8qLaQ3B</X509Certificate></X509Data></KeyInfo></KeyDescriptor><KeyDescriptor use=""signing""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIDKDCCAhCgAwIBAgIQBHJvVNxP1oZO4HYKh+rypDANBgkqhkiG9w0BAQsFADAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXMwHhcNMTYxMTE2MDgwMDAwWhcNMTgxMTE2MDgwMDAwWjAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXMwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQChn5BCs24Hh6L0BNitPrV5s+2/DBhaeytOmnghJKnqeJlhv3ZczShRM2Cp38LW8Y3wn7L3AJtolaSkF/joKN1l6GupzM+HOEdq7xZxFehxIHW7+25mG/WigBnhsBzLv1SR4uIbrQeS5M0kkLwJ9pOnVH3uzMGG6TRXPnK3ivlKl97AiUEKdlRjCQNLXvYf1ZqlC77c/ZCOHSX4kvIKR2uG+LNlSTRq2rn8AgMpFT4DSlEZz4RmFQvQupQzPpzozaz/gadBpJy/jgDmJlQMPXkHp7wClvbIBGiGRaY6eZFxNV96zwSR/GPNkTObdw2S8/SiAgvIhIcqWTPLY6aVTqJfAgMBAAGjWDBWMFQGA1UdAQRNMEuAEDUj0BrjP0RTbmoRPTRMY3WhJTAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXOCEARyb1TcT9aGTuB2Cofq8qQwDQYJKoZIhvcNAQELBQADggEBAGnLhDHVz2gLDiu9L34V3ro/6xZDiSWhGyHcGqky7UlzQH3pT5so8iF5P0WzYqVtogPsyC2LPJYSTt2vmQugD4xlu/wbvMFLcV0hmNoTKCF1QTVtEQiAiy0Aq+eoF7Al5fV1S3Sune0uQHimuUFHCmUuF190MLcHcdWnPAmzIc8fv7quRUUsExXmxSX2ktUYQXzqFyIOSnDCuWFm6tpfK5JXS8fW5bpqTlrysXXz/OW/8NFGq/alfjrya4ojrOYLpunGriEtNPwK7hxj1AlCYEWaRHRXaUIW1ByoSff/6Y6+ZhXPUe0cDlNRt/qIz5aflwO7+W8baTS4O8m/icu7ItE=</X509Certificate></X509Data></KeyInfo></KeyDescriptor><fed:TargetScopes><wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing""><wsa:Address>https://sts.windows.net/%7Btenantid%7D/</wsa:Address></wsa:EndpointReference></fed:TargetScopes><fed:ApplicationServiceEndpoint><wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing""><wsa:Address>https://login.microsoftonline.com/common/wsfed</wsa:Address></wsa:EndpointReference></fed:ApplicationServiceEndpoint><fed:PassiveRequestorEndpoint><wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing""><wsa:Address>https://login.microsoftonline.com/common/wsfed</wsa:Address></wsa:EndpointReference></fed:PassiveRequestorEndpoint></RoleDescriptor><IDPSSODescriptor protocolSupportEnumeration=""urn:oasis:names:tc:SAML:2.0:protocol""><KeyDescriptor use=""signing""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD</X509Certificate></X509Data></KeyInfo></KeyDescriptor><KeyDescriptor use=""signing""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIDBTCCAe2gAwIBAgIQXxLnqm1cOoVGe62j7W7wZzANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDMyNjAwMDAwMFoXDTE5MDMyNzAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKJGarCm4IF0/Gz5Xx/zyZwD2rdJJZtO2Ukk1Oz+Br1sLVY8I5vj5esB+lotmLEblA9N/w188vmTvykaEzUl49NA4s86x44SW6WtdQbGJ0IjpQJUalUMyy91vIBkK/7K3nBXeVBsRk7tm528leoQ05/aZ+1ycJBIU+1oGYThv8MOjyHAlXJmCaGXwXTisZ+hHjcwlMk/+ZEutHflKLIpPUNEi7j4Xw+zp9UKo5pzWIr/iJ4HjvCkFofW90AMF2xp8dMhpbVcfJGS/Ii3J66LuNLCH/HtSZ42FO+tnRL/nNzzFWUhGT92Q5VFVngfWJ3PAg1zz8I1wowLD2fiB2udGXcCAwEAAaMhMB8wHQYDVR0OBBYEFFXPbFXjmMR3BluF+2MeSXd1NQ3rMA0GCSqGSIb3DQEBCwUAA4IBAQAsd3wGVilJxDtbY1K2oAsWLdNJgmCaYdrtdlAsjGlarSQSzBH0Ybf78fcPX//DYaLXlvaEGKVKp0jPq+RnJ17oP/RJpJTwVXPGRIlZopLIgnKpWlS/PS0uKAdNvLmz1zbGSILdcF+Qf41OozD4QNsS1c9YbDO4vpC9v8x3PVjfJvJwPonzNoOsLXA+8IONSXwCApsnmrwepKu8sifsFYSwgrwxRPGTEAjkdzRJ0yMqiY/VoJ7lqJ/FBJqqAjGPGq/yI9rVoG+mbO1amrIDWHHTKgfbKk0bXGtVUbsayyHR5jSgadmkLBh5AaN/HcgDK/jINrnpiQ+/2ewH/8qLaQ3B</X509Certificate></X509Data></KeyInfo></KeyDescriptor><KeyDescriptor use=""signing""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIDKDCCAhCgAwIBAgIQBHJvVNxP1oZO4HYKh+rypDANBgkqhkiG9w0BAQsFADAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXMwHhcNMTYxMTE2MDgwMDAwWhcNMTgxMTE2MDgwMDAwWjAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXMwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQChn5BCs24Hh6L0BNitPrV5s+2/DBhaeytOmnghJKnqeJlhv3ZczShRM2Cp38LW8Y3wn7L3AJtolaSkF/joKN1l6GupzM+HOEdq7xZxFehxIHW7+25mG/WigBnhsBzLv1SR4uIbrQeS5M0kkLwJ9pOnVH3uzMGG6TRXPnK3ivlKl97AiUEKdlRjCQNLXvYf1ZqlC77c/ZCOHSX4kvIKR2uG+LNlSTRq2rn8AgMpFT4DSlEZz4RmFQvQupQzPpzozaz/gadBpJy/jgDmJlQMPXkHp7wClvbIBGiGRaY6eZFxNV96zwSR/GPNkTObdw2S8/SiAgvIhIcqWTPLY6aVTqJfAgMBAAGjWDBWMFQGA1UdAQRNMEuAEDUj0BrjP0RTbmoRPTRMY3WhJTAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXOCEARyb1TcT9aGTuB2Cofq8qQwDQYJKoZIhvcNAQELBQADggEBAGnLhDHVz2gLDiu9L34V3ro/6xZDiSWhGyHcGqky7UlzQH3pT5so8iF5P0WzYqVtogPsyC2LPJYSTt2vmQugD4xlu/wbvMFLcV0hmNoTKCF1QTVtEQiAiy0Aq+eoF7Al5fV1S3Sune0uQHimuUFHCmUuF190MLcHcdWnPAmzIc8fv7quRUUsExXmxSX2ktUYQXzqFyIOSnDCuWFm6tpfK5JXS8fW5bpqTlrysXXz/OW/8NFGq/alfjrya4ojrOYLpunGriEtNPwK7hxj1AlCYEWaRHRXaUIW1ByoSff/6Y6+ZhXPUe0cDlNRt/qIz5aflwO7+W8baTS4O8m/icu7ItE=</X509Certificate></X509Data></KeyInfo></KeyDescriptor><SingleLogoutService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect"" Location=""https://login.microsoftonline.com/common/saml2"" /><SingleSignOnService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect"" Location=""https://login.microsoftonline.com/common/saml2"" /><SingleSignOnService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-POST"" Location=""https://login.microsoftonline.com/common/saml2"" /></IDPSSODescriptor></EntityDescriptor>"; }

        public static string AADCommonMetadataKeyInfoX509Data { get => "MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD"; }

        public static string AADCommonMetadataSignature { get => @"<Signature xmlns=""http://www.w3.org/2000/09/xmldsig#""><SignedInfo><CanonicalizationMethod Algorithm=""http://www.w3.org/2001/10/xml-exc-c14n#"" /><SignatureMethod Algorithm=""http://www.w3.org/2001/04/xmldsig-more#rsa-sha256"" /><Reference URI=""#_0ded55d8-a72f-4e13-ab9e-f40be80b1476""><Transforms><Transform Algorithm=""http://www.w3.org/2000/09/xmldsig#enveloped-signature"" /><Transform Algorithm=""http://www.w3.org/2001/10/xml-exc-c14n#"" /></Transforms><DigestMethod Algorithm=""http://www.w3.org/2001/04/xmlenc#sha256"" /><DigestValue>lnicj3SNizOF6QI1rWw8JrouoiXMslqtFB5ztWW6qvs=</DigestValue></Reference></SignedInfo><SignatureValue>KD9uWOD/9pvF1NlNCpYoXymUPS1l9uIBgBDe0uOQgQv+tUI/1jJX4UpjADDHCOx6HCl5ZgZSXNmOC2lLSJEwmv21BZzI+PAOxF5hdH99cS/lMC/hxgyWdLVeGnr1I4WbPxGqVmjFNuBdBMaourO4z/5f3D2JZQmgnlu8H+4gv2SpjeZz/YhIN6ZrNfmHwsKZashMGtSmE5uHro+uO5yO17Gr9YfUbtokLRIq5Dk9kqnxG8YZF1C1nC9O0PMdlHb4ubwgO20Cvz5sU2iswn9m68btS5TLF5OVhETzyKir1QA+H1tCgGRqIWd4Geyoucdct1r4zAJGCNIekdKnY3NXwg==</SignatureValue><KeyInfo><X509Data><X509Certificate>MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD</X509Certificate></X509Data></KeyInfo></Signature>"; }

        public static string AADCommonMetadataSignatureValue { get => "KD9uWOD/9pvF1NlNCpYoXymUPS1l9uIBgBDe0uOQgQv+tUI/1jJX4UpjADDHCOx6HCl5ZgZSXNmOC2lLSJEwmv21BZzI+PAOxF5hdH99cS/lMC/hxgyWdLVeGnr1I4WbPxGqVmjFNuBdBMaourO4z/5f3D2JZQmgnlu8H+4gv2SpjeZz/YhIN6ZrNfmHwsKZashMGtSmE5uHro+uO5yO17Gr9YfUbtokLRIq5Dk9kqnxG8YZF1C1nC9O0PMdlHb4ubwgO20Cvz5sU2iswn9m68btS5TLF5OVhETzyKir1QA+H1tCgGRqIWd4Geyoucdct1r4zAJGCNIekdKnY3NXwg=="; }

        public static CanonicalizingTransfrom AADCommonMetadataReferenceC14nTransform { get => new ExclusiveCanonicalizationTransform(); }

        public static EnvelopedSignatureTransform AADCommonMetadataReferenceEnvelopedTransform { get => new EnvelopedSignatureTransform(); }

        public static string AADCommonMetadataReferenceDigestMethod { get => "http://www.w3.org/2001/04/xmlenc#sha256"; }

        public static string AADCommonMetadataReferenceDigestValue { get => "lnicj3SNizOF6QI1rWw8JrouoiXMslqtFB5ztWW6qvs="; }

        public static string AADCommonMetadataReferenceURI { get => "#_0ded55d8-a72f-4e13-ab9e-f40be80b1476"; }

        public static string AdfsV2Metadata { get => @"<EntityDescriptor ID=""_43144f19-220f-44a7-b8d9-12207fc27b25"" entityID=""http://fs.msidlab7.com/adfs/services/trust"" xmlns=""urn:oasis:names:tc:SAML:2.0:metadata""><Signature xmlns=""http://www.w3.org/2000/09/xmldsig#""><SignedInfo><CanonicalizationMethod Algorithm=""http://www.w3.org/2001/10/xml-exc-c14n#""/><SignatureMethod Algorithm=""http://www.w3.org/2001/04/xmldsig-more#rsa-sha256""/><Reference URI=""#_43144f19-220f-44a7-b8d9-12207fc27b25""><Transforms><Transform Algorithm=""http://www.w3.org/2000/09/xmldsig#enveloped-signature""/><Transform Algorithm=""http://www.w3.org/2001/10/xml-exc-c14n#""/></Transforms><DigestMethod Algorithm=""http://www.w3.org/2001/04/xmlenc#sha256""/><DigestValue>DfDxwwbmXG1Y/6WVzloBWxosQn/NyS8PNO2XxZlpqeE=</DigestValue></Reference></SignedInfo><SignatureValue>WV5tNHkh/mTd6Rca9Ezutau61aEldUjUsQqHWRjSczGVLpqGgoWb90Y/XXGC/NFw1PfPLqxi8l+UiIIRO/MFGZUSBakhZAEcHANBk5kwufAGxrQhbKeMz4N/TEJgPBGhhqWxQ0wVqvMhGORrhRb87fYPM7xx4iFkgCTj9lBbhovTiDndzT3gjt0M6mnH8CVrx13mf8zK+I+lAo/1QRUaAHrIt/QJT3snE29+5kWwqiaFQj+OX1KyBLsGfRc/tE1V35OvJV/aK4+L8SBIKQsUqptphneNh7V2bOjshXWmvSaV115NfFwFlgq+4Aq6N1M1DwIndq3VQKWalrXbVLGGow==</SignatureValue><KeyInfo><X509Data><X509Certificate>MIIC2jCCAcKgAwIBAgIQFuP8MbE4ipBGdYQ2Wmg6FDANBgkqhkiG9w0BAQsFADApMScwJQYDVQQDEx5BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiNy5jb20wHhcNMTYxMjAzMDIzNjEwWhcNMTcxMjAzMDIzNjEwWjApMScwJQYDVQQDEx5BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiNy5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCQXc06bWagsWphlP+yFdK/6YPZNvYC5BSzV2bmbYp76YCzpDAaYTBL6toAgjMBoTWHeeO/K8GpKxjK5RzBCWJL0j6Ao+GLGiX430VEvWCxCVbi4H3GY7T2MUangLIBEbutFbgcKxSdLIG9KEUKZXZSnKxx4W6b/m56dswPwpPJ536KWgjovNKy+/XfAONFkg4Rj6HvFN7ylNSBwWfcFdw+C8aZn4swW8msLMm4ridHxR0jnHgqrJ8RQjBQfcWaCuS18EaJxVrCN13OnDNa1SnQqKdePcczeeL6NRK4luxCTNJ6pJ8GLSxNA5rxbv3uMZ0VcaANLFp0KZDJ7xr3oTEdAgMBAAEwDQYJKoZIhvcNAQELBQADggEBAE3plOFUXZ/O/9pEF4ZiBvjttxXz/ntaKlC9WDZQaysjwAvwwQf6lCWBYrOn9muQrDeAriYOZHxRXxzwdqUgHRUQ6NZxNQUgc+ElH1v+E25ns8vP8yS8U9FhQUU8IfkLYcOt3bY0PcGCJhKsHSJoHrsKqQqWWQN4f48ocrLOdAhS0VLTxpa84i8mGNO9Iaavdd6g0vfC0dgBaBRyWDO0MAywiqx1LsxOXBidxn/Z4bm/2+lcAnRXALowifDPkDvIkx+zjU2YvcYWDiNl3sei3NfS68sYycGMaCmwXUyjQy13FMrOuEo2KqMmWHPMyZbbEyVBsytq5+W1Rx+2ElWAzJg=</X509Certificate></X509Data></KeyInfo></Signature><RoleDescriptor xsi:type=""fed:ApplicationServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/ws-sx/ws-trust/200512 http://schemas.xmlsoap.org/ws/2005/02/trust http://docs.oasis-open.org/wsfed/federation/200706"" ServiceDisplayName=""fs.msidlab7.com"" xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706""><KeyDescriptor use=""encryption""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIC4DCCAcigAwIBAgIQbAsx9yDH6I5FStrFP8cd+DANBgkqhkiG9w0BAQsFADAsMSowKAYDVQQDEyFBREZTIEVuY3J5cHRpb24gLSBmcy5tc2lkbGFiNy5jb20wHhcNMTYxMjAzMDIzNjEwWhcNMTcxMjAzMDIzNjEwWjAsMSowKAYDVQQDEyFBREZTIEVuY3J5cHRpb24gLSBmcy5tc2lkbGFiNy5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCtbm/El8DtU/mt5Cg83uMSPn4ouh6BC9TCPAEuLSDFIHrHFaGNg7TP+JGdvFxwnHM2mClQpF01Q+4eVAL5ds9Q1g+AMisX6iLWw4ai0QuhKUk0oF6bD3xMnIZBv2eW3ZlNK7yzaM8eanHpPS8tY5Wnuxk310EmWgJpa22/AEzfui2PCRL3PyeynOlRvnyIJtOcP79BX8DoQ/sRAW3byGBceUvBgVI/+XczC1n5ugwLed1dCrEw2wZdjhlaUQmTpOXi6AkhkZkeiEsns8oIqfhqHLSAdInNhx3OawveixQztyblrrdiOd7E7VCq1q/n7f19sR2iwF0pYp9Ylac5NDWnAgMBAAEwDQYJKoZIhvcNAQELBQADggEBAHvgl5NB7MQ4/PmjZ5kfKd+p0+Kw1fwTWwGGJSMM+dMk6GoOgP38leqO/cuAZ+SD4BeKRYSVOHMXYuDeBW3KU5I2yjYOG4YgUN6x07F+lGnD4cZqdI/gUignZjFdQujpv5mIJEDAm6YYx8uLVXsMHR6oa4Wi7kQF98S+Lo4aicru9WqZRN9JxeNZoa55vZO7SwuAjElrcR09qbuzLPWNZLtp5RJJVKxoyJ6Qc2EfR/SZ0ifopM8ozmz6t1dZpW3HxCXrH7fVVsRl+oPXymepawIPXqr/kJ7jFPRsblPRKIQluugHkSC3LrQLr23uBMieHFzDerWll00ayZqUFWIrfHA=</X509Certificate></X509Data></KeyInfo></KeyDescriptor><fed:ClaimTypesRequested><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/emailaddress"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>E-Mail Address</auth:DisplayName><auth:Description>The e-mail address of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/givenname"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Given Name</auth:DisplayName><auth:Description>The given name of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/name"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Name</auth:DisplayName><auth:Description>The unique name of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/upn"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>UPN</auth:DisplayName><auth:Description>The user principal name (UPN) of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/CommonName"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Common Name</auth:DisplayName><auth:Description>The common name of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/EmailAddress"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>AD FS 1.x E-Mail Address</auth:DisplayName><auth:Description>The e-mail address of the user when interoperating with AD FS 1.1 or ADFS 1.0</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/Group"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Group</auth:DisplayName><auth:Description>A group that the user is a member of</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/UPN"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>AD FS 1.x UPN</auth:DisplayName><auth:Description>The UPN of the user when interoperating with AD FS 1.1 or ADFS 1.0</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/role"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Role</auth:DisplayName><auth:Description>A role that the user has</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/surname"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Surname</auth:DisplayName><auth:Description>The surname of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/privatepersonalidentifier"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>PPID</auth:DisplayName><auth:Description>The private identifier of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/nameidentifier"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Name ID</auth:DisplayName><auth:Description>The SAML name identifier of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/authenticationinstant"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Authentication time stamp</auth:DisplayName><auth:Description>Used to display the time and date that the user was authenticated</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/authenticationmethod"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Authentication method</auth:DisplayName><auth:Description>The method used to authenticate the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/denyonlysid"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Deny only group SID</auth:DisplayName><auth:Description>The deny-only group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/denyonlyprimarysid"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Deny only primary SID</auth:DisplayName><auth:Description>The deny-only primary SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/denyonlyprimarygroupsid"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Deny only primary group SID</auth:DisplayName><auth:Description>The deny-only primary group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/groupsid"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Group SID</auth:DisplayName><auth:Description>The group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/primarygroupsid"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Primary group SID</auth:DisplayName><auth:Description>The primary group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/primarysid"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Primary SID</auth:DisplayName><auth:Description>The primary SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/windowsaccountname"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Windows account name</auth:DisplayName><auth:Description>The domain account name of the user in the form of &lt;domain&gt;\&lt;user&gt;</auth:Description></auth:ClaimType></fed:ClaimTypesRequested><fed:TargetScopes><wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing""><wsa:Address>https://fs.msidlab7.com/adfs/services/trust/2005/issuedtokenmixedasymmetricbasic256</wsa:Address></wsa:EndpointReference><wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing""><wsa:Address>https://fs.msidlab7.com/adfs/services/trust/2005/issuedtokenmixedsymmetricbasic256</wsa:Address></wsa:EndpointReference><wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing""><wsa:Address>https://fs.msidlab7.com/adfs/services/trust/13/issuedtokenmixedasymmetricbasic256</wsa:Address></wsa:EndpointReference><wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing""><wsa:Address>https://fs.msidlab7.com/adfs/services/trust/13/issuedtokenmixedsymmetricbasic256</wsa:Address></wsa:EndpointReference><wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing""><wsa:Address>https://fs.msidlab7.com/adfs/ls/</wsa:Address></wsa:EndpointReference><wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing""><wsa:Address>http://fs.msidlab7.com/adfs/services/trust</wsa:Address></wsa:EndpointReference></fed:TargetScopes><fed:ApplicationServiceEndpoint><wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing""><wsa:Address>https://fs.msidlab7.com/adfs/services/trust/2005/issuedtokenmixedasymmetricbasic256</wsa:Address></wsa:EndpointReference></fed:ApplicationServiceEndpoint><fed:PassiveRequestorEndpoint><wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing""><wsa:Address>https://fs.msidlab7.com/adfs/ls/</wsa:Address></wsa:EndpointReference></fed:PassiveRequestorEndpoint></RoleDescriptor><RoleDescriptor xsi:type=""fed:SecurityTokenServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/ws-sx/ws-trust/200512 http://schemas.xmlsoap.org/ws/2005/02/trust http://docs.oasis-open.org/wsfed/federation/200706"" ServiceDisplayName=""fs.msidlab7.com"" xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706""><KeyDescriptor use=""signing""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIC2jCCAcKgAwIBAgIQFuP8MbE4ipBGdYQ2Wmg6FDANBgkqhkiG9w0BAQsFADApMScwJQYDVQQDEx5BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiNy5jb20wHhcNMTYxMjAzMDIzNjEwWhcNMTcxMjAzMDIzNjEwWjApMScwJQYDVQQDEx5BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiNy5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCQXc06bWagsWphlP+yFdK/6YPZNvYC5BSzV2bmbYp76YCzpDAaYTBL6toAgjMBoTWHeeO/K8GpKxjK5RzBCWJL0j6Ao+GLGiX430VEvWCxCVbi4H3GY7T2MUangLIBEbutFbgcKxSdLIG9KEUKZXZSnKxx4W6b/m56dswPwpPJ536KWgjovNKy+/XfAONFkg4Rj6HvFN7ylNSBwWfcFdw+C8aZn4swW8msLMm4ridHxR0jnHgqrJ8RQjBQfcWaCuS18EaJxVrCN13OnDNa1SnQqKdePcczeeL6NRK4luxCTNJ6pJ8GLSxNA5rxbv3uMZ0VcaANLFp0KZDJ7xr3oTEdAgMBAAEwDQYJKoZIhvcNAQELBQADggEBAE3plOFUXZ/O/9pEF4ZiBvjttxXz/ntaKlC9WDZQaysjwAvwwQf6lCWBYrOn9muQrDeAriYOZHxRXxzwdqUgHRUQ6NZxNQUgc+ElH1v+E25ns8vP8yS8U9FhQUU8IfkLYcOt3bY0PcGCJhKsHSJoHrsKqQqWWQN4f48ocrLOdAhS0VLTxpa84i8mGNO9Iaavdd6g0vfC0dgBaBRyWDO0MAywiqx1LsxOXBidxn/Z4bm/2+lcAnRXALowifDPkDvIkx+zjU2YvcYWDiNl3sei3NfS68sYycGMaCmwXUyjQy13FMrOuEo2KqMmWHPMyZbbEyVBsytq5+W1Rx+2ElWAzJg=</X509Certificate></X509Data></KeyInfo></KeyDescriptor><fed:TokenTypesOffered><fed:TokenType Uri=""urn:oasis:names:tc:SAML:2.0:assertion""/><fed:TokenType Uri=""urn:oasis:names:tc:SAML:1.0:assertion""/></fed:TokenTypesOffered><fed:ClaimTypesOffered><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/emailaddress"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>E-Mail Address</auth:DisplayName><auth:Description>The e-mail address of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/givenname"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Given Name</auth:DisplayName><auth:Description>The given name of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/name"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Name</auth:DisplayName><auth:Description>The unique name of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/upn"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>UPN</auth:DisplayName><auth:Description>The user principal name (UPN) of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/CommonName"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Common Name</auth:DisplayName><auth:Description>The common name of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/EmailAddress"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>AD FS 1.x E-Mail Address</auth:DisplayName><auth:Description>The e-mail address of the user when interoperating with AD FS 1.1 or ADFS 1.0</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/Group"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Group</auth:DisplayName><auth:Description>A group that the user is a member of</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/UPN"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>AD FS 1.x UPN</auth:DisplayName><auth:Description>The UPN of the user when interoperating with AD FS 1.1 or ADFS 1.0</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/role"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Role</auth:DisplayName><auth:Description>A role that the user has</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/surname"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Surname</auth:DisplayName><auth:Description>The surname of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/privatepersonalidentifier"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>PPID</auth:DisplayName><auth:Description>The private identifier of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/nameidentifier"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Name ID</auth:DisplayName><auth:Description>The SAML name identifier of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/authenticationinstant"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Authentication time stamp</auth:DisplayName><auth:Description>Used to display the time and date that the user was authenticated</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/authenticationmethod"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Authentication method</auth:DisplayName><auth:Description>The method used to authenticate the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/denyonlysid"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Deny only group SID</auth:DisplayName><auth:Description>The deny-only group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/denyonlyprimarysid"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Deny only primary SID</auth:DisplayName><auth:Description>The deny-only primary SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/denyonlyprimarygroupsid"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Deny only primary group SID</auth:DisplayName><auth:Description>The deny-only primary group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/groupsid"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Group SID</auth:DisplayName><auth:Description>The group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/primarygroupsid"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Primary group SID</auth:DisplayName><auth:Description>The primary group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/primarysid"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Primary SID</auth:DisplayName><auth:Description>The primary SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/windowsaccountname"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Windows account name</auth:DisplayName><auth:Description>The domain account name of the user in the form of &lt;domain&gt;\&lt;user&gt;</auth:Description></auth:ClaimType></fed:ClaimTypesOffered><fed:SecurityTokenServiceEndpoint><wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing""><wsa:Address>https://fs.msidlab7.com/adfs/services/trust/2005/certificatemixed</wsa:Address><Metadata xmlns=""http://www.w3.org/2005/08/addressing""><Metadata xmlns=""http://schemas.xmlsoap.org/ws/2004/09/mex"" xmlns:xsd=""http://www.w3.org/2001/XMLSchema"" xmlns:wsx=""http://schemas.xmlsoap.org/ws/2004/09/mex""><wsx:MetadataSection Dialect=""http://schemas.xmlsoap.org/ws/2004/09/mex"" xmlns=""""><wsx:MetadataReference><Address xmlns=""http://www.w3.org/2005/08/addressing"">https://fs.msidlab7.com/adfs/services/trust/mex</Address></wsx:MetadataReference></wsx:MetadataSection></Metadata></Metadata></wsa:EndpointReference></fed:SecurityTokenServiceEndpoint><fed:PassiveRequestorEndpoint><wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing""><wsa:Address>https://fs.msidlab7.com/adfs/ls/</wsa:Address></wsa:EndpointReference></fed:PassiveRequestorEndpoint></RoleDescriptor><SPSSODescriptor WantAssertionsSigned=""true"" protocolSupportEnumeration=""urn:oasis:names:tc:SAML:2.0:protocol""><KeyDescriptor use=""encryption""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIC4DCCAcigAwIBAgIQbAsx9yDH6I5FStrFP8cd+DANBgkqhkiG9w0BAQsFADAsMSowKAYDVQQDEyFBREZTIEVuY3J5cHRpb24gLSBmcy5tc2lkbGFiNy5jb20wHhcNMTYxMjAzMDIzNjEwWhcNMTcxMjAzMDIzNjEwWjAsMSowKAYDVQQDEyFBREZTIEVuY3J5cHRpb24gLSBmcy5tc2lkbGFiNy5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCtbm/El8DtU/mt5Cg83uMSPn4ouh6BC9TCPAEuLSDFIHrHFaGNg7TP+JGdvFxwnHM2mClQpF01Q+4eVAL5ds9Q1g+AMisX6iLWw4ai0QuhKUk0oF6bD3xMnIZBv2eW3ZlNK7yzaM8eanHpPS8tY5Wnuxk310EmWgJpa22/AEzfui2PCRL3PyeynOlRvnyIJtOcP79BX8DoQ/sRAW3byGBceUvBgVI/+XczC1n5ugwLed1dCrEw2wZdjhlaUQmTpOXi6AkhkZkeiEsns8oIqfhqHLSAdInNhx3OawveixQztyblrrdiOd7E7VCq1q/n7f19sR2iwF0pYp9Ylac5NDWnAgMBAAEwDQYJKoZIhvcNAQELBQADggEBAHvgl5NB7MQ4/PmjZ5kfKd+p0+Kw1fwTWwGGJSMM+dMk6GoOgP38leqO/cuAZ+SD4BeKRYSVOHMXYuDeBW3KU5I2yjYOG4YgUN6x07F+lGnD4cZqdI/gUignZjFdQujpv5mIJEDAm6YYx8uLVXsMHR6oa4Wi7kQF98S+Lo4aicru9WqZRN9JxeNZoa55vZO7SwuAjElrcR09qbuzLPWNZLtp5RJJVKxoyJ6Qc2EfR/SZ0ifopM8ozmz6t1dZpW3HxCXrH7fVVsRl+oPXymepawIPXqr/kJ7jFPRsblPRKIQluugHkSC3LrQLr23uBMieHFzDerWll00ayZqUFWIrfHA=</X509Certificate></X509Data></KeyInfo></KeyDescriptor><KeyDescriptor use=""signing""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIC2jCCAcKgAwIBAgIQFuP8MbE4ipBGdYQ2Wmg6FDANBgkqhkiG9w0BAQsFADApMScwJQYDVQQDEx5BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiNy5jb20wHhcNMTYxMjAzMDIzNjEwWhcNMTcxMjAzMDIzNjEwWjApMScwJQYDVQQDEx5BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiNy5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCQXc06bWagsWphlP+yFdK/6YPZNvYC5BSzV2bmbYp76YCzpDAaYTBL6toAgjMBoTWHeeO/K8GpKxjK5RzBCWJL0j6Ao+GLGiX430VEvWCxCVbi4H3GY7T2MUangLIBEbutFbgcKxSdLIG9KEUKZXZSnKxx4W6b/m56dswPwpPJ536KWgjovNKy+/XfAONFkg4Rj6HvFN7ylNSBwWfcFdw+C8aZn4swW8msLMm4ridHxR0jnHgqrJ8RQjBQfcWaCuS18EaJxVrCN13OnDNa1SnQqKdePcczeeL6NRK4luxCTNJ6pJ8GLSxNA5rxbv3uMZ0VcaANLFp0KZDJ7xr3oTEdAgMBAAEwDQYJKoZIhvcNAQELBQADggEBAE3plOFUXZ/O/9pEF4ZiBvjttxXz/ntaKlC9WDZQaysjwAvwwQf6lCWBYrOn9muQrDeAriYOZHxRXxzwdqUgHRUQ6NZxNQUgc+ElH1v+E25ns8vP8yS8U9FhQUU8IfkLYcOt3bY0PcGCJhKsHSJoHrsKqQqWWQN4f48ocrLOdAhS0VLTxpa84i8mGNO9Iaavdd6g0vfC0dgBaBRyWDO0MAywiqx1LsxOXBidxn/Z4bm/2+lcAnRXALowifDPkDvIkx+zjU2YvcYWDiNl3sei3NfS68sYycGMaCmwXUyjQy13FMrOuEo2KqMmWHPMyZbbEyVBsytq5+W1Rx+2ElWAzJg=</X509Certificate></X509Data></KeyInfo></KeyDescriptor><SingleLogoutService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect"" Location=""https://fs.msidlab7.com/adfs/ls/""/><SingleLogoutService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-POST"" Location=""https://fs.msidlab7.com/adfs/ls/""/><NameIDFormat>urn:oasis:names:tc:SAML:1.1:nameid-format:emailAddress</NameIDFormat><NameIDFormat>urn:oasis:names:tc:SAML:2.0:nameid-format:persistent</NameIDFormat><NameIDFormat>urn:oasis:names:tc:SAML:2.0:nameid-format:transient</NameIDFormat><AssertionConsumerService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-POST"" Location=""https://fs.msidlab7.com/adfs/ls/"" index=""0"" isDefault=""true""/><AssertionConsumerService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Artifact"" Location=""https://fs.msidlab7.com/adfs/ls/"" index=""1""/><AssertionConsumerService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect"" Location=""https://fs.msidlab7.com/adfs/ls/"" index=""2""/></SPSSODescriptor><IDPSSODescriptor protocolSupportEnumeration=""urn:oasis:names:tc:SAML:2.0:protocol""><KeyDescriptor use=""encryption""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIC4DCCAcigAwIBAgIQbAsx9yDH6I5FStrFP8cd+DANBgkqhkiG9w0BAQsFADAsMSowKAYDVQQDEyFBREZTIEVuY3J5cHRpb24gLSBmcy5tc2lkbGFiNy5jb20wHhcNMTYxMjAzMDIzNjEwWhcNMTcxMjAzMDIzNjEwWjAsMSowKAYDVQQDEyFBREZTIEVuY3J5cHRpb24gLSBmcy5tc2lkbGFiNy5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCtbm/El8DtU/mt5Cg83uMSPn4ouh6BC9TCPAEuLSDFIHrHFaGNg7TP+JGdvFxwnHM2mClQpF01Q+4eVAL5ds9Q1g+AMisX6iLWw4ai0QuhKUk0oF6bD3xMnIZBv2eW3ZlNK7yzaM8eanHpPS8tY5Wnuxk310EmWgJpa22/AEzfui2PCRL3PyeynOlRvnyIJtOcP79BX8DoQ/sRAW3byGBceUvBgVI/+XczC1n5ugwLed1dCrEw2wZdjhlaUQmTpOXi6AkhkZkeiEsns8oIqfhqHLSAdInNhx3OawveixQztyblrrdiOd7E7VCq1q/n7f19sR2iwF0pYp9Ylac5NDWnAgMBAAEwDQYJKoZIhvcNAQELBQADggEBAHvgl5NB7MQ4/PmjZ5kfKd+p0+Kw1fwTWwGGJSMM+dMk6GoOgP38leqO/cuAZ+SD4BeKRYSVOHMXYuDeBW3KU5I2yjYOG4YgUN6x07F+lGnD4cZqdI/gUignZjFdQujpv5mIJEDAm6YYx8uLVXsMHR6oa4Wi7kQF98S+Lo4aicru9WqZRN9JxeNZoa55vZO7SwuAjElrcR09qbuzLPWNZLtp5RJJVKxoyJ6Qc2EfR/SZ0ifopM8ozmz6t1dZpW3HxCXrH7fVVsRl+oPXymepawIPXqr/kJ7jFPRsblPRKIQluugHkSC3LrQLr23uBMieHFzDerWll00ayZqUFWIrfHA=</X509Certificate></X509Data></KeyInfo></KeyDescriptor><KeyDescriptor use=""signing""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIC2jCCAcKgAwIBAgIQFuP8MbE4ipBGdYQ2Wmg6FDANBgkqhkiG9w0BAQsFADApMScwJQYDVQQDEx5BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiNy5jb20wHhcNMTYxMjAzMDIzNjEwWhcNMTcxMjAzMDIzNjEwWjApMScwJQYDVQQDEx5BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiNy5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCQXc06bWagsWphlP+yFdK/6YPZNvYC5BSzV2bmbYp76YCzpDAaYTBL6toAgjMBoTWHeeO/K8GpKxjK5RzBCWJL0j6Ao+GLGiX430VEvWCxCVbi4H3GY7T2MUangLIBEbutFbgcKxSdLIG9KEUKZXZSnKxx4W6b/m56dswPwpPJ536KWgjovNKy+/XfAONFkg4Rj6HvFN7ylNSBwWfcFdw+C8aZn4swW8msLMm4ridHxR0jnHgqrJ8RQjBQfcWaCuS18EaJxVrCN13OnDNa1SnQqKdePcczeeL6NRK4luxCTNJ6pJ8GLSxNA5rxbv3uMZ0VcaANLFp0KZDJ7xr3oTEdAgMBAAEwDQYJKoZIhvcNAQELBQADggEBAE3plOFUXZ/O/9pEF4ZiBvjttxXz/ntaKlC9WDZQaysjwAvwwQf6lCWBYrOn9muQrDeAriYOZHxRXxzwdqUgHRUQ6NZxNQUgc+ElH1v+E25ns8vP8yS8U9FhQUU8IfkLYcOt3bY0PcGCJhKsHSJoHrsKqQqWWQN4f48ocrLOdAhS0VLTxpa84i8mGNO9Iaavdd6g0vfC0dgBaBRyWDO0MAywiqx1LsxOXBidxn/Z4bm/2+lcAnRXALowifDPkDvIkx+zjU2YvcYWDiNl3sei3NfS68sYycGMaCmwXUyjQy13FMrOuEo2KqMmWHPMyZbbEyVBsytq5+W1Rx+2ElWAzJg=</X509Certificate></X509Data></KeyInfo></KeyDescriptor><SingleLogoutService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect"" Location=""https://fs.msidlab7.com/adfs/ls/""/><SingleLogoutService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-POST"" Location=""https://fs.msidlab7.com/adfs/ls/""/><NameIDFormat>urn:oasis:names:tc:SAML:1.1:nameid-format:emailAddress</NameIDFormat><NameIDFormat>urn:oasis:names:tc:SAML:2.0:nameid-format:persistent</NameIDFormat><NameIDFormat>urn:oasis:names:tc:SAML:2.0:nameid-format:transient</NameIDFormat><SingleSignOnService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect"" Location=""https://fs.msidlab7.com/adfs/ls/""/><SingleSignOnService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-POST"" Location=""https://fs.msidlab7.com/adfs/ls/""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/emailaddress"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""E-Mail Address"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/givenname"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Given Name"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/name"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Name"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/upn"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""UPN"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/claims/CommonName"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Common Name"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/claims/EmailAddress"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""AD FS 1.x E-Mail Address"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/claims/Group"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Group"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/claims/UPN"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""AD FS 1.x UPN"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/role"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Role"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/surname"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Surname"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/privatepersonalidentifier"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""PPID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/nameidentifier"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Name ID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/authenticationinstant"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Authentication time stamp"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/authenticationmethod"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Authentication method"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/denyonlysid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Deny only group SID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/denyonlyprimarysid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Deny only primary SID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/denyonlyprimarygroupsid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Deny only primary group SID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/groupsid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Group SID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/primarygroupsid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Primary group SID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/primarysid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Primary SID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/windowsaccountname"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Windows account name"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/></IDPSSODescriptor><ContactPerson contactType=""support""/></EntityDescriptor>"; }

        public static string AdfsV2SignatureValue { get => @"WV5tNHkh/mTd6Rca9Ezutau61aEldUjUsQqHWRjSczGVLpqGgoWb90Y/XXGC/NFw1PfPLqxi8l+UiIIRO/MFGZUSBakhZAEcHANBk5kwufAGxrQhbKeMz4N/TEJgPBGhhqWxQ0wVqvMhGORrhRb87fYPM7xx4iFkgCTj9lBbhovTiDndzT3gjt0M6mnH8CVrx13mf8zK+I+lAo/1QRUaAHrIt/QJT3snE29+5kWwqiaFQj+OX1KyBLsGfRc/tE1V35OvJV/aK4+L8SBIKQsUqptphneNh7V2bOjshXWmvSaV115NfFwFlgq+4Aq6N1M1DwIndq3VQKWalrXbVLGGow=="; }

        public static string AdfsV2ReferenceDigestValue { get => "DfDxwwbmXG1Y/6WVzloBWxosQn/NyS8PNO2XxZlpqeE="; }

        public static string AdfsV2ReferenceURI { get => "#_43144f19-220f-44a7-b8d9-12207fc27b25"; }

        public static string AdfsV3Metadata { get => @"<EntityDescriptor ID=""_7d30a77e-57fa-4c05-93ab-63a2d4e1a336"" entityID=""http://fs.msidlab2.com/adfs/services/trust"" xmlns=""urn:oasis:names:tc:SAML:2.0:metadata""><ds:Signature xmlns:ds=""http://www.w3.org/2000/09/xmldsig#""><ds:SignedInfo><ds:CanonicalizationMethod Algorithm=""http://www.w3.org/2001/10/xml-exc-c14n#""/><ds:SignatureMethod Algorithm=""http://www.w3.org/2001/04/xmldsig-more#rsa-sha256""/><ds:Reference URI=""#_7d30a77e-57fa-4c05-93ab-63a2d4e1a336""><ds:Transforms><ds:Transform Algorithm=""http://www.w3.org/2000/09/xmldsig#enveloped-signature""/><ds:Transform Algorithm=""http://www.w3.org/2001/10/xml-exc-c14n#""/></ds:Transforms><ds:DigestMethod Algorithm=""http://www.w3.org/2001/04/xmlenc#sha256""/><ds:DigestValue>/WtSzs2am3kxkT+nYdFWs/+udQe78CzNYMquzKUjUqk=</ds:DigestValue></ds:Reference></ds:SignedInfo><ds:SignatureValue>LlgnhZB+Zdtt1bxJpWHhAxzbeDA6Oz8BHMge6MIyfZrSBNLnvg/tGYneOhOl7mu0cg8cCJCLaXWBUfPOWDAebXqKYvApA1yu1ALd2sg03Q/A1ti/kWRhiJempc/vX27upSXppjTfaBbiqeoY6qpFJwzBSl3NfOS9b1Zf6BsxyZ6L8msmcsEv4tj17+DiDPcoGDK4uqyHctQDa1CySWLTQduqZ5IABZ2LipDCsfz7Tg1rGCCbHLhiiP0ge0fact38ZM1XNZ7gtiLCufUUUgG5civp5ifYJEWmIZ/SdwjUeMMHejd36iinRoAAsM/bW/trZVo3PQjCGpHxoqR+FMt/Dw==</ds:SignatureValue><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIC2jCCAcKgAwIBAgIQE4Ec5zS+mZ1CjdwUbnFNejANBgkqhkiG9w0BAQsFADApMScwJQYDVQQDEx5BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiMi5jb20wHhcNMTcwMzEzMTgxMTM0WhcNMTgwMzEzMTgxMTM0WjApMScwJQYDVQQDEx5BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiMi5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCYh9E/B76B0jUBedTJCBXYkXd3NVav5GFOI43Cu3mcnQvT82112PY7TSW/GTOJL1oVhAyLR6nNjPKD0f7J+3gkrXYpvDK8g69GWI5Gy3ObA95VhzpWKwqnRJrblJfYvQtiA+dQTGH+x5tSbHieGni71lqanCwGMP4PnLKTAGtRXEm1Yz/l0k57PTiKBybIlhW5YUm/Nl8d1qlp+eQdM/bhkj9imnANghSAcXU//6zeioG63ad0COltc5Eh4bMUNBGdx15gMyu40T2TDojeYIS+3rkYoRmQHJ9LoHt43DgqPBp8K+bdT/5gVUnRcRv0hpTi5JhPyqWm3h4CAJb2NNr7AgMBAAEwDQYJKoZIhvcNAQELBQADggEBAAcvWybsF8Tt/FRahhgWlONayLorboAD3qxUOfzaBQS32Q6AUPWatQCJX6pRC9tLtQr2PdJ4hDWfcGmMuFUzArFqBfAFFvb5CTlplNKQw53iZhEdvKEhY77d+j4BPpHmIvHDq9dshl0oPWV5ywiWJfBmf/6I7DRffXtBkj3IpuVQWrtOycZHsW1ymsURCQEr/YlqPlOCtoMCfl6wvOR/zLF2/sheVBgrwJTVCljBP4DENwK6he3dioyNM9k9ITa0Z2SpXu2+pHJopRc94H+KJG/LTX0y+Fq6bh54/U5ULE/OzDzKRLqJETqOvWNWaMw/tGUTo6gGr4KAbhel/PTIztQ=</X509Certificate></X509Data></KeyInfo></ds:Signature><RoleDescriptor xsi:type=""fed:ApplicationServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/ws-sx/ws-trust/200512 http://schemas.xmlsoap.org/ws/2005/02/trust http://docs.oasis-open.org/wsfed/federation/200706"" ServiceDisplayName=""fs.msidlab2.com AD FS"" xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706""><KeyDescriptor use=""encryption""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIC4DCCAcigAwIBAgIQEuEDj9AbNoFF6Zly+fbyhTANBgkqhkiG9w0BAQsFADAsMSowKAYDVQQDEyFBREZTIEVuY3J5cHRpb24gLSBmcy5tc2lkbGFiMi5jb20wHhcNMTcwMzEzMTgxMTM0WhcNMTgwMzEzMTgxMTM0WjAsMSowKAYDVQQDEyFBREZTIEVuY3J5cHRpb24gLSBmcy5tc2lkbGFiMi5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDfhryn2qhHUnqlhJ2qgTvYeF3KgbC2UdwRIQXgCmdRjl/YMbFygQNan9xI+wW6woGpTAKn0GKYtB3zAI+BnC+1fAhuYcOwc/zXDPklfB5gi+jM8bNNokTog7gpSPJahdSPJch6RSNhhA9HCjjhmDYLnQ9zud0bdcZIzDjjA4p1kDrqO5aokLRBjcF6A+53wzweyapy9jZpZNbPTVBMyTIo/CmqIDiLEadoRYOSZlZRfhgUovc9jmYm08rtFrznYXg7H9Fqy06V6y3GoYExgyE2TvSWlaPD4yeUs2YbXiH6UmzakzGybSSz6BIawqitHPE/qWXZmZet7CEZ2hlndYqTAgMBAAEwDQYJKoZIhvcNAQELBQADggEBAFjhxKiA9P5BPDaKNxlVLzz+iEZvV2wwpWtT/5nt613TG0M5aSmLbkRRYDqp+Rcf3mlnz6QR4STIkSVnaQY1SetJmFpRq1C+z26r6oe3pho1fu7ICRTazOC7H+iKP9RXCYZ3hQgUGS7uhoumUzBUbr0PgS1cLnHE3Bte7xjVVyYFtPVootQM/BZ4m5xVLf5Wpj2ntoscwPMSDpp1U8pUx/QRiMdV+lETt6kq+Q4t6QTfAl0iBZFDLm+5BmUjl1jUXSCjxfhYpUkJZRUdqwbXFYk7kXOT+zwJHQs3o+/0jxBzLcKRT8RmV5BjDyjdgY7u98yqSIwhCeJA1/AbrQMzRU8=</X509Certificate></X509Data></KeyInfo></KeyDescriptor><fed:ClaimTypesRequested><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/emailaddress"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>E-Mail Address</auth:DisplayName><auth:Description>The e-mail address of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/givenname"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Given Name</auth:DisplayName><auth:Description>The given name of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/name"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Name</auth:DisplayName><auth:Description>The unique name of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/upn"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>UPN</auth:DisplayName><auth:Description>The user principal name (UPN) of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/CommonName"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Common Name</auth:DisplayName><auth:Description>The common name of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/EmailAddress"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>AD FS 1.x E-Mail Address</auth:DisplayName><auth:Description>The e-mail address of the user when interoperating with AD FS 1.1 or AD FS 1.0</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/Group"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Group</auth:DisplayName><auth:Description>A group that the user is a member of</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/UPN"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>AD FS 1.x UPN</auth:DisplayName><auth:Description>The UPN of the user when interoperating with AD FS 1.1 or AD FS 1.0</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/role"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Role</auth:DisplayName><auth:Description>A role that the user has</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/surname"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Surname</auth:DisplayName><auth:Description>The surname of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/privatepersonalidentifier"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>PPID</auth:DisplayName><auth:Description>The private identifier of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/nameidentifier"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Name ID</auth:DisplayName><auth:Description>The SAML name identifier of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/authenticationinstant"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Authentication time stamp</auth:DisplayName><auth:Description>Used to display the time and date that the user was authenticated</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/authenticationmethod"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Authentication method</auth:DisplayName><auth:Description>The method used to authenticate the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/denyonlysid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Deny only group SID</auth:DisplayName><auth:Description>The deny-only group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/denyonlyprimarysid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Deny only primary SID</auth:DisplayName><auth:Description>The deny-only primary SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/denyonlyprimarygroupsid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Deny only primary group SID</auth:DisplayName><auth:Description>The deny-only primary group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/groupsid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Group SID</auth:DisplayName><auth:Description>The group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/primarygroupsid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Primary group SID</auth:DisplayName><auth:Description>The primary group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/primarysid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Primary SID</auth:DisplayName><auth:Description>The primary SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/windowsaccountname"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Windows account name</auth:DisplayName><auth:Description>The domain account name of the user in the form of domain\user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/isregistereduser"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Is Registered User</auth:DisplayName><auth:Description>User is registered to use this device</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/identifier"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Device Identifier</auth:DisplayName><auth:Description>Identifier of the device</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/registrationid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Device Registration Identifier</auth:DisplayName><auth:Description>Identifier for Device Registration</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/displayname"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Device Registration DisplayName</auth:DisplayName><auth:Description>Display name of Device Registration</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/ostype"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Device OS type</auth:DisplayName><auth:Description>OS type of the device</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/osversion"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Device OS Version</auth:DisplayName><auth:Description>OS version of the device</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/ismanaged"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Is Managed Device</auth:DisplayName><auth:Description>Device is managed by a management service</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-forwarded-client-ip"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Forwarded Client IP</auth:DisplayName><auth:Description>IP address of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-client-application"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Client Application</auth:DisplayName><auth:Description>Type of the Client Application</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-client-user-agent"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Client User Agent</auth:DisplayName><auth:Description>Device type the client is using to access the application</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-client-ip"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Client IP</auth:DisplayName><auth:Description>IP address of the client</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-endpoint-absolute-path"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Endpoint Path</auth:DisplayName><auth:Description>Absolute Endpoint path which can be used to determine active versus passive clients</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-proxy"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Proxy</auth:DisplayName><auth:Description>DNS name of the federation server proxy that passed the request</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/relyingpartytrustid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Application Identifier</auth:DisplayName><auth:Description>Identifier for the Relying Party</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/applicationpolicy"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Application policies</auth:DisplayName><auth:Description>Application policies of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/authoritykeyidentifier"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Authority Key Identifier</auth:DisplayName><auth:Description>The Authority Key Identifier extension of the certificate that signed an issued certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/basicconstraints"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Basic Constraint</auth:DisplayName><auth:Description>One of the basic constraints of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/eku"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Enhanced Key Usage</auth:DisplayName><auth:Description>Describes one of the enhanced key usages of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/issuer"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Issuer</auth:DisplayName><auth:Description>The name of the certificate authority that issued the X.509 certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/issuername"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Issuer Name</auth:DisplayName><auth:Description>The distinguished name of the certificate issuer</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/keyusage"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Key Usage</auth:DisplayName><auth:Description>One of the key usages of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/notafter"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Not After</auth:DisplayName><auth:Description>Date in local time after which a certificate is no longer valid</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/notbefore"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Not Before</auth:DisplayName><auth:Description>The date in local time on which a certificate becomes valid</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/certificatepolicy"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Certificate Policies</auth:DisplayName><auth:Description>The policies under which the certificate has been issued</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/rsa"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Public Key</auth:DisplayName><auth:Description>Public Key of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/rawdata"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Certificate Raw Data</auth:DisplayName><auth:Description>The raw data of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/san"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Subject Alternative Name</auth:DisplayName><auth:Description>One of the alternative names of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/serialnumber"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Serial Number</auth:DisplayName><auth:Description>The serial number of a certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/signaturealgorithm"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Signature Algorithm</auth:DisplayName><auth:Description>The algorithm used to create the signature of a certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/subject"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Subject</auth:DisplayName><auth:Description>The subject from the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/subjectkeyidentifier"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Subject Key Identifier</auth:DisplayName><auth:Description>Describes the subject key identifier of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/subjectname"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Subject Name</auth:DisplayName><auth:Description>The subject distinguished name from a certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/certificatetemplateinformation"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>V2 Template Name</auth:DisplayName><auth:Description>The name of the version 2 certificate template used when issuing or renewing a certificate. The extension is Microsoft specific.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/certificatetemplatename"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>V1 Template Name</auth:DisplayName><auth:Description>The name of the version 1 certificate template used when issuing or renewing a certificate. The extension is Microsoft specific.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/thumbprint"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Thumbprint</auth:DisplayName><auth:Description>Thumbprint of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/x509version"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>X.509 Version</auth:DisplayName><auth:Description>The X.509 format version of a certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2012/01/insidecorporatenetwork"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Inside Corporate Network</auth:DisplayName><auth:Description>Used to indicate if a request originated inside corporate network</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2012/01/passwordexpirationtime"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Password Expiration Time</auth:DisplayName><auth:Description>Used to display the time when the password expires</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2012/01/passwordexpirationdays"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Password Expiration Days</auth:DisplayName><auth:Description>Used to display the number of days to password expiry</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2012/01/passwordchangeurl"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Update Password URL</auth:DisplayName><auth:Description>Used to display the web address of update password service</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/claims/authnmethodsreferences"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Authentication Methods References</auth:DisplayName><auth:Description>Used to indicate all authentication methods used to authenticate the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/client-request-id"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Client Request ID</auth:DisplayName><auth:Description>Identifier for a user session</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2013/11/alternateloginid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Alternate Login ID</auth:DisplayName><auth:Description>Alternate login ID of the user</auth:Description></auth:ClaimType></fed:ClaimTypesRequested><fed:TargetScopes><EndpointReference xmlns=""http://www.w3.org/2005/08/addressing""><Address>https://fs.msidlab2.com/adfs/services/trust/2005/issuedtokenmixedasymmetricbasic256</Address></EndpointReference><EndpointReference xmlns=""http://www.w3.org/2005/08/addressing""><Address>https://fs.msidlab2.com/adfs/services/trust/2005/issuedtokenmixedsymmetricbasic256</Address></EndpointReference><EndpointReference xmlns=""http://www.w3.org/2005/08/addressing""><Address>https://fs.msidlab2.com/adfs/services/trust/13/issuedtokenmixedasymmetricbasic256</Address></EndpointReference><EndpointReference xmlns=""http://www.w3.org/2005/08/addressing""><Address>https://fs.msidlab2.com/adfs/services/trust/13/issuedtokenmixedsymmetricbasic256</Address></EndpointReference><EndpointReference xmlns=""http://www.w3.org/2005/08/addressing""><Address>https://fs.msidlab2.com/adfs/ls/</Address></EndpointReference><EndpointReference xmlns=""http://www.w3.org/2005/08/addressing""><Address>http://fs.msidlab2.com/adfs/services/trust</Address></EndpointReference></fed:TargetScopes><fed:ApplicationServiceEndpoint><EndpointReference xmlns=""http://www.w3.org/2005/08/addressing""><Address>https://fs.msidlab2.com/adfs/services/trust/2005/issuedtokenmixedasymmetricbasic256</Address></EndpointReference></fed:ApplicationServiceEndpoint><fed:PassiveRequestorEndpoint><EndpointReference xmlns=""http://www.w3.org/2005/08/addressing""><Address>https://fs.msidlab2.com/adfs/ls/</Address></EndpointReference></fed:PassiveRequestorEndpoint></RoleDescriptor><RoleDescriptor xsi:type=""fed:SecurityTokenServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/ws-sx/ws-trust/200512 http://schemas.xmlsoap.org/ws/2005/02/trust http://docs.oasis-open.org/wsfed/federation/200706"" ServiceDisplayName=""fs.msidlab2.com AD FS"" xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706""><KeyDescriptor use=""signing""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIC2jCCAcKgAwIBAgIQE4Ec5zS+mZ1CjdwUbnFNejANBgkqhkiG9w0BAQsFADApMScwJQYDVQQDEx5BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiMi5jb20wHhcNMTcwMzEzMTgxMTM0WhcNMTgwMzEzMTgxMTM0WjApMScwJQYDVQQDEx5BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiMi5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCYh9E/B76B0jUBedTJCBXYkXd3NVav5GFOI43Cu3mcnQvT82112PY7TSW/GTOJL1oVhAyLR6nNjPKD0f7J+3gkrXYpvDK8g69GWI5Gy3ObA95VhzpWKwqnRJrblJfYvQtiA+dQTGH+x5tSbHieGni71lqanCwGMP4PnLKTAGtRXEm1Yz/l0k57PTiKBybIlhW5YUm/Nl8d1qlp+eQdM/bhkj9imnANghSAcXU//6zeioG63ad0COltc5Eh4bMUNBGdx15gMyu40T2TDojeYIS+3rkYoRmQHJ9LoHt43DgqPBp8K+bdT/5gVUnRcRv0hpTi5JhPyqWm3h4CAJb2NNr7AgMBAAEwDQYJKoZIhvcNAQELBQADggEBAAcvWybsF8Tt/FRahhgWlONayLorboAD3qxUOfzaBQS32Q6AUPWatQCJX6pRC9tLtQr2PdJ4hDWfcGmMuFUzArFqBfAFFvb5CTlplNKQw53iZhEdvKEhY77d+j4BPpHmIvHDq9dshl0oPWV5ywiWJfBmf/6I7DRffXtBkj3IpuVQWrtOycZHsW1ymsURCQEr/YlqPlOCtoMCfl6wvOR/zLF2/sheVBgrwJTVCljBP4DENwK6he3dioyNM9k9ITa0Z2SpXu2+pHJopRc94H+KJG/LTX0y+Fq6bh54/U5ULE/OzDzKRLqJETqOvWNWaMw/tGUTo6gGr4KAbhel/PTIztQ=</X509Certificate></X509Data></KeyInfo></KeyDescriptor><fed:TokenTypesOffered><fed:TokenType Uri=""urn:oasis:names:tc:SAML:2.0:assertion""/><fed:TokenType Uri=""urn:oasis:names:tc:SAML:1.0:assertion""/></fed:TokenTypesOffered><fed:ClaimTypesOffered><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/emailaddress"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>E-Mail Address</auth:DisplayName><auth:Description>The e-mail address of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/givenname"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Given Name</auth:DisplayName><auth:Description>The given name of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/name"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Name</auth:DisplayName><auth:Description>The unique name of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/upn"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>UPN</auth:DisplayName><auth:Description>The user principal name (UPN) of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/CommonName"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Common Name</auth:DisplayName><auth:Description>The common name of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/EmailAddress"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>AD FS 1.x E-Mail Address</auth:DisplayName><auth:Description>The e-mail address of the user when interoperating with AD FS 1.1 or AD FS 1.0</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/Group"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Group</auth:DisplayName><auth:Description>A group that the user is a member of</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/UPN"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>AD FS 1.x UPN</auth:DisplayName><auth:Description>The UPN of the user when interoperating with AD FS 1.1 or AD FS 1.0</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/role"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Role</auth:DisplayName><auth:Description>A role that the user has</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/surname"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Surname</auth:DisplayName><auth:Description>The surname of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/privatepersonalidentifier"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>PPID</auth:DisplayName><auth:Description>The private identifier of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/nameidentifier"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Name ID</auth:DisplayName><auth:Description>The SAML name identifier of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/authenticationinstant"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Authentication time stamp</auth:DisplayName><auth:Description>Used to display the time and date that the user was authenticated</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/authenticationmethod"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Authentication method</auth:DisplayName><auth:Description>The method used to authenticate the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/denyonlysid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Deny only group SID</auth:DisplayName><auth:Description>The deny-only group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/denyonlyprimarysid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Deny only primary SID</auth:DisplayName><auth:Description>The deny-only primary SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/denyonlyprimarygroupsid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Deny only primary group SID</auth:DisplayName><auth:Description>The deny-only primary group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/groupsid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Group SID</auth:DisplayName><auth:Description>The group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/primarygroupsid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Primary group SID</auth:DisplayName><auth:Description>The primary group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/primarysid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Primary SID</auth:DisplayName><auth:Description>The primary SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/windowsaccountname"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Windows account name</auth:DisplayName><auth:Description>The domain account name of the user in the form of domain\user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/isregistereduser"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Is Registered User</auth:DisplayName><auth:Description>User is registered to use this device</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/identifier"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Device Identifier</auth:DisplayName><auth:Description>Identifier of the device</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/registrationid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Device Registration Identifier</auth:DisplayName><auth:Description>Identifier for Device Registration</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/displayname"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Device Registration DisplayName</auth:DisplayName><auth:Description>Display name of Device Registration</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/ostype"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Device OS type</auth:DisplayName><auth:Description>OS type of the device</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/osversion"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Device OS Version</auth:DisplayName><auth:Description>OS version of the device</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/ismanaged"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Is Managed Device</auth:DisplayName><auth:Description>Device is managed by a management service</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-forwarded-client-ip"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Forwarded Client IP</auth:DisplayName><auth:Description>IP address of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-client-application"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Client Application</auth:DisplayName><auth:Description>Type of the Client Application</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-client-user-agent"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Client User Agent</auth:DisplayName><auth:Description>Device type the client is using to access the application</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-client-ip"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Client IP</auth:DisplayName><auth:Description>IP address of the client</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-endpoint-absolute-path"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Endpoint Path</auth:DisplayName><auth:Description>Absolute Endpoint path which can be used to determine active versus passive clients</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-proxy"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Proxy</auth:DisplayName><auth:Description>DNS name of the federation server proxy that passed the request</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/relyingpartytrustid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Application Identifier</auth:DisplayName><auth:Description>Identifier for the Relying Party</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/applicationpolicy"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Application policies</auth:DisplayName><auth:Description>Application policies of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/authoritykeyidentifier"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Authority Key Identifier</auth:DisplayName><auth:Description>The Authority Key Identifier extension of the certificate that signed an issued certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/basicconstraints"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Basic Constraint</auth:DisplayName><auth:Description>One of the basic constraints of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/eku"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Enhanced Key Usage</auth:DisplayName><auth:Description>Describes one of the enhanced key usages of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/issuer"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Issuer</auth:DisplayName><auth:Description>The name of the certificate authority that issued the X.509 certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/issuername"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Issuer Name</auth:DisplayName><auth:Description>The distinguished name of the certificate issuer</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/keyusage"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Key Usage</auth:DisplayName><auth:Description>One of the key usages of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/notafter"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Not After</auth:DisplayName><auth:Description>Date in local time after which a certificate is no longer valid</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/notbefore"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Not Before</auth:DisplayName><auth:Description>The date in local time on which a certificate becomes valid</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/certificatepolicy"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Certificate Policies</auth:DisplayName><auth:Description>The policies under which the certificate has been issued</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/rsa"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Public Key</auth:DisplayName><auth:Description>Public Key of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/rawdata"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Certificate Raw Data</auth:DisplayName><auth:Description>The raw data of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/san"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Subject Alternative Name</auth:DisplayName><auth:Description>One of the alternative names of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/serialnumber"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Serial Number</auth:DisplayName><auth:Description>The serial number of a certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/signaturealgorithm"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Signature Algorithm</auth:DisplayName><auth:Description>The algorithm used to create the signature of a certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/subject"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Subject</auth:DisplayName><auth:Description>The subject from the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/subjectkeyidentifier"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Subject Key Identifier</auth:DisplayName><auth:Description>Describes the subject key identifier of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/subjectname"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Subject Name</auth:DisplayName><auth:Description>The subject distinguished name from a certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/certificatetemplateinformation"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>V2 Template Name</auth:DisplayName><auth:Description>The name of the version 2 certificate template used when issuing or renewing a certificate. The extension is Microsoft specific.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/certificatetemplatename"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>V1 Template Name</auth:DisplayName><auth:Description>The name of the version 1 certificate template used when issuing or renewing a certificate. The extension is Microsoft specific.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/thumbprint"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Thumbprint</auth:DisplayName><auth:Description>Thumbprint of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/x509version"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>X.509 Version</auth:DisplayName><auth:Description>The X.509 format version of a certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2012/01/insidecorporatenetwork"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Inside Corporate Network</auth:DisplayName><auth:Description>Used to indicate if a request originated inside corporate network</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2012/01/passwordexpirationtime"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Password Expiration Time</auth:DisplayName><auth:Description>Used to display the time when the password expires</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2012/01/passwordexpirationdays"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Password Expiration Days</auth:DisplayName><auth:Description>Used to display the number of days to password expiry</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2012/01/passwordchangeurl"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Update Password URL</auth:DisplayName><auth:Description>Used to display the web address of update password service</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/claims/authnmethodsreferences"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Authentication Methods References</auth:DisplayName><auth:Description>Used to indicate all authentication methods used to authenticate the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/client-request-id"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Client Request ID</auth:DisplayName><auth:Description>Identifier for a user session</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2013/11/alternateloginid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Alternate Login ID</auth:DisplayName><auth:Description>Alternate login ID of the user</auth:Description></auth:ClaimType></fed:ClaimTypesOffered><fed:SecurityTokenServiceEndpoint><EndpointReference xmlns=""http://www.w3.org/2005/08/addressing""><Address>https://fs.msidlab2.com/adfs/services/trust/2005/certificatemixed</Address><Metadata><Metadata xmlns=""http://schemas.xmlsoap.org/ws/2004/09/mex"" xmlns:xsd=""http://www.w3.org/2001/XMLSchema"" xmlns:wsx=""http://schemas.xmlsoap.org/ws/2004/09/mex""><wsx:MetadataSection Dialect=""http://schemas.xmlsoap.org/ws/2004/09/mex"" xmlns=""""><wsx:MetadataReference><Address xmlns=""http://www.w3.org/2005/08/addressing"">https://fs.msidlab2.com/adfs/services/trust/mex</Address></wsx:MetadataReference></wsx:MetadataSection></Metadata></Metadata></EndpointReference></fed:SecurityTokenServiceEndpoint><fed:PassiveRequestorEndpoint><EndpointReference xmlns=""http://www.w3.org/2005/08/addressing""><Address>https://fs.msidlab2.com/adfs/ls/</Address></EndpointReference></fed:PassiveRequestorEndpoint></RoleDescriptor><SPSSODescriptor WantAssertionsSigned=""true"" protocolSupportEnumeration=""urn:oasis:names:tc:SAML:2.0:protocol""><KeyDescriptor use=""encryption""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIC4DCCAcigAwIBAgIQEuEDj9AbNoFF6Zly+fbyhTANBgkqhkiG9w0BAQsFADAsMSowKAYDVQQDEyFBREZTIEVuY3J5cHRpb24gLSBmcy5tc2lkbGFiMi5jb20wHhcNMTcwMzEzMTgxMTM0WhcNMTgwMzEzMTgxMTM0WjAsMSowKAYDVQQDEyFBREZTIEVuY3J5cHRpb24gLSBmcy5tc2lkbGFiMi5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDfhryn2qhHUnqlhJ2qgTvYeF3KgbC2UdwRIQXgCmdRjl/YMbFygQNan9xI+wW6woGpTAKn0GKYtB3zAI+BnC+1fAhuYcOwc/zXDPklfB5gi+jM8bNNokTog7gpSPJahdSPJch6RSNhhA9HCjjhmDYLnQ9zud0bdcZIzDjjA4p1kDrqO5aokLRBjcF6A+53wzweyapy9jZpZNbPTVBMyTIo/CmqIDiLEadoRYOSZlZRfhgUovc9jmYm08rtFrznYXg7H9Fqy06V6y3GoYExgyE2TvSWlaPD4yeUs2YbXiH6UmzakzGybSSz6BIawqitHPE/qWXZmZet7CEZ2hlndYqTAgMBAAEwDQYJKoZIhvcNAQELBQADggEBAFjhxKiA9P5BPDaKNxlVLzz+iEZvV2wwpWtT/5nt613TG0M5aSmLbkRRYDqp+Rcf3mlnz6QR4STIkSVnaQY1SetJmFpRq1C+z26r6oe3pho1fu7ICRTazOC7H+iKP9RXCYZ3hQgUGS7uhoumUzBUbr0PgS1cLnHE3Bte7xjVVyYFtPVootQM/BZ4m5xVLf5Wpj2ntoscwPMSDpp1U8pUx/QRiMdV+lETt6kq+Q4t6QTfAl0iBZFDLm+5BmUjl1jUXSCjxfhYpUkJZRUdqwbXFYk7kXOT+zwJHQs3o+/0jxBzLcKRT8RmV5BjDyjdgY7u98yqSIwhCeJA1/AbrQMzRU8=</X509Certificate></X509Data></KeyInfo></KeyDescriptor><KeyDescriptor use=""signing""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIC2jCCAcKgAwIBAgIQE4Ec5zS+mZ1CjdwUbnFNejANBgkqhkiG9w0BAQsFADApMScwJQYDVQQDEx5BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiMi5jb20wHhcNMTcwMzEzMTgxMTM0WhcNMTgwMzEzMTgxMTM0WjApMScwJQYDVQQDEx5BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiMi5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCYh9E/B76B0jUBedTJCBXYkXd3NVav5GFOI43Cu3mcnQvT82112PY7TSW/GTOJL1oVhAyLR6nNjPKD0f7J+3gkrXYpvDK8g69GWI5Gy3ObA95VhzpWKwqnRJrblJfYvQtiA+dQTGH+x5tSbHieGni71lqanCwGMP4PnLKTAGtRXEm1Yz/l0k57PTiKBybIlhW5YUm/Nl8d1qlp+eQdM/bhkj9imnANghSAcXU//6zeioG63ad0COltc5Eh4bMUNBGdx15gMyu40T2TDojeYIS+3rkYoRmQHJ9LoHt43DgqPBp8K+bdT/5gVUnRcRv0hpTi5JhPyqWm3h4CAJb2NNr7AgMBAAEwDQYJKoZIhvcNAQELBQADggEBAAcvWybsF8Tt/FRahhgWlONayLorboAD3qxUOfzaBQS32Q6AUPWatQCJX6pRC9tLtQr2PdJ4hDWfcGmMuFUzArFqBfAFFvb5CTlplNKQw53iZhEdvKEhY77d+j4BPpHmIvHDq9dshl0oPWV5ywiWJfBmf/6I7DRffXtBkj3IpuVQWrtOycZHsW1ymsURCQEr/YlqPlOCtoMCfl6wvOR/zLF2/sheVBgrwJTVCljBP4DENwK6he3dioyNM9k9ITa0Z2SpXu2+pHJopRc94H+KJG/LTX0y+Fq6bh54/U5ULE/OzDzKRLqJETqOvWNWaMw/tGUTo6gGr4KAbhel/PTIztQ=</X509Certificate></X509Data></KeyInfo></KeyDescriptor><SingleLogoutService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect"" Location=""https://fs.msidlab2.com/adfs/ls/""/><SingleLogoutService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-POST"" Location=""https://fs.msidlab2.com/adfs/ls/""/><NameIDFormat>urn:oasis:names:tc:SAML:1.1:nameid-format:emailAddress</NameIDFormat><NameIDFormat>urn:oasis:names:tc:SAML:2.0:nameid-format:persistent</NameIDFormat><NameIDFormat>urn:oasis:names:tc:SAML:2.0:nameid-format:transient</NameIDFormat><AssertionConsumerService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-POST"" Location=""https://fs.msidlab2.com/adfs/ls/"" index=""0"" isDefault=""true""/><AssertionConsumerService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Artifact"" Location=""https://fs.msidlab2.com/adfs/ls/"" index=""1""/><AssertionConsumerService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect"" Location=""https://fs.msidlab2.com/adfs/ls/"" index=""2""/></SPSSODescriptor><IDPSSODescriptor protocolSupportEnumeration=""urn:oasis:names:tc:SAML:2.0:protocol""><KeyDescriptor use=""encryption""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIC4DCCAcigAwIBAgIQEuEDj9AbNoFF6Zly+fbyhTANBgkqhkiG9w0BAQsFADAsMSowKAYDVQQDEyFBREZTIEVuY3J5cHRpb24gLSBmcy5tc2lkbGFiMi5jb20wHhcNMTcwMzEzMTgxMTM0WhcNMTgwMzEzMTgxMTM0WjAsMSowKAYDVQQDEyFBREZTIEVuY3J5cHRpb24gLSBmcy5tc2lkbGFiMi5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDfhryn2qhHUnqlhJ2qgTvYeF3KgbC2UdwRIQXgCmdRjl/YMbFygQNan9xI+wW6woGpTAKn0GKYtB3zAI+BnC+1fAhuYcOwc/zXDPklfB5gi+jM8bNNokTog7gpSPJahdSPJch6RSNhhA9HCjjhmDYLnQ9zud0bdcZIzDjjA4p1kDrqO5aokLRBjcF6A+53wzweyapy9jZpZNbPTVBMyTIo/CmqIDiLEadoRYOSZlZRfhgUovc9jmYm08rtFrznYXg7H9Fqy06V6y3GoYExgyE2TvSWlaPD4yeUs2YbXiH6UmzakzGybSSz6BIawqitHPE/qWXZmZet7CEZ2hlndYqTAgMBAAEwDQYJKoZIhvcNAQELBQADggEBAFjhxKiA9P5BPDaKNxlVLzz+iEZvV2wwpWtT/5nt613TG0M5aSmLbkRRYDqp+Rcf3mlnz6QR4STIkSVnaQY1SetJmFpRq1C+z26r6oe3pho1fu7ICRTazOC7H+iKP9RXCYZ3hQgUGS7uhoumUzBUbr0PgS1cLnHE3Bte7xjVVyYFtPVootQM/BZ4m5xVLf5Wpj2ntoscwPMSDpp1U8pUx/QRiMdV+lETt6kq+Q4t6QTfAl0iBZFDLm+5BmUjl1jUXSCjxfhYpUkJZRUdqwbXFYk7kXOT+zwJHQs3o+/0jxBzLcKRT8RmV5BjDyjdgY7u98yqSIwhCeJA1/AbrQMzRU8=</X509Certificate></X509Data></KeyInfo></KeyDescriptor><KeyDescriptor use=""signing""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIC2jCCAcKgAwIBAgIQE4Ec5zS+mZ1CjdwUbnFNejANBgkqhkiG9w0BAQsFADApMScwJQYDVQQDEx5BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiMi5jb20wHhcNMTcwMzEzMTgxMTM0WhcNMTgwMzEzMTgxMTM0WjApMScwJQYDVQQDEx5BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiMi5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCYh9E/B76B0jUBedTJCBXYkXd3NVav5GFOI43Cu3mcnQvT82112PY7TSW/GTOJL1oVhAyLR6nNjPKD0f7J+3gkrXYpvDK8g69GWI5Gy3ObA95VhzpWKwqnRJrblJfYvQtiA+dQTGH+x5tSbHieGni71lqanCwGMP4PnLKTAGtRXEm1Yz/l0k57PTiKBybIlhW5YUm/Nl8d1qlp+eQdM/bhkj9imnANghSAcXU//6zeioG63ad0COltc5Eh4bMUNBGdx15gMyu40T2TDojeYIS+3rkYoRmQHJ9LoHt43DgqPBp8K+bdT/5gVUnRcRv0hpTi5JhPyqWm3h4CAJb2NNr7AgMBAAEwDQYJKoZIhvcNAQELBQADggEBAAcvWybsF8Tt/FRahhgWlONayLorboAD3qxUOfzaBQS32Q6AUPWatQCJX6pRC9tLtQr2PdJ4hDWfcGmMuFUzArFqBfAFFvb5CTlplNKQw53iZhEdvKEhY77d+j4BPpHmIvHDq9dshl0oPWV5ywiWJfBmf/6I7DRffXtBkj3IpuVQWrtOycZHsW1ymsURCQEr/YlqPlOCtoMCfl6wvOR/zLF2/sheVBgrwJTVCljBP4DENwK6he3dioyNM9k9ITa0Z2SpXu2+pHJopRc94H+KJG/LTX0y+Fq6bh54/U5ULE/OzDzKRLqJETqOvWNWaMw/tGUTo6gGr4KAbhel/PTIztQ=</X509Certificate></X509Data></KeyInfo></KeyDescriptor><SingleLogoutService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect"" Location=""https://fs.msidlab2.com/adfs/ls/""/><SingleLogoutService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-POST"" Location=""https://fs.msidlab2.com/adfs/ls/""/><NameIDFormat>urn:oasis:names:tc:SAML:1.1:nameid-format:emailAddress</NameIDFormat><NameIDFormat>urn:oasis:names:tc:SAML:2.0:nameid-format:persistent</NameIDFormat><NameIDFormat>urn:oasis:names:tc:SAML:2.0:nameid-format:transient</NameIDFormat><SingleSignOnService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect"" Location=""https://fs.msidlab2.com/adfs/ls/""/><SingleSignOnService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-POST"" Location=""https://fs.msidlab2.com/adfs/ls/""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/emailaddress"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""E-Mail Address"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/givenname"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Given Name"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/name"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Name"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/upn"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""UPN"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/claims/CommonName"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Common Name"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/claims/EmailAddress"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""AD FS 1.x E-Mail Address"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/claims/Group"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Group"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/claims/UPN"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""AD FS 1.x UPN"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/role"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Role"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/surname"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Surname"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/privatepersonalidentifier"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""PPID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/nameidentifier"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Name ID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/authenticationinstant"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Authentication time stamp"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/authenticationmethod"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Authentication method"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/denyonlysid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Deny only group SID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/denyonlyprimarysid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Deny only primary SID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/denyonlyprimarygroupsid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Deny only primary group SID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/groupsid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Group SID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/primarygroupsid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Primary group SID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/primarysid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Primary SID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/windowsaccountname"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Windows account name"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/devicecontext/claims/isregistereduser"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Is Registered User"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/devicecontext/claims/identifier"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Device Identifier"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/devicecontext/claims/registrationid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Device Registration Identifier"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/devicecontext/claims/displayname"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Device Registration DisplayName"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/devicecontext/claims/ostype"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Device OS type"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/devicecontext/claims/osversion"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Device OS Version"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/devicecontext/claims/ismanaged"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Is Managed Device"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-forwarded-client-ip"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Forwarded Client IP"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-client-application"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Client Application"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-client-user-agent"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Client User Agent"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-client-ip"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Client IP"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-endpoint-absolute-path"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Endpoint Path"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-proxy"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Proxy"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/requestcontext/claims/relyingpartytrustid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Application Identifier"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/applicationpolicy"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Application policies"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/authoritykeyidentifier"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Authority Key Identifier"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/basicconstraints"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Basic Constraint"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/eku"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Enhanced Key Usage"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/field/issuer"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Issuer"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/field/issuername"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Issuer Name"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/keyusage"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Key Usage"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/field/notafter"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Not After"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/field/notbefore"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Not Before"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/certificatepolicy"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Certificate Policies"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/rsa"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Public Key"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/field/rawdata"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Certificate Raw Data"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/san"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Subject Alternative Name"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/serialnumber"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Serial Number"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/field/signaturealgorithm"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Signature Algorithm"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/field/subject"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Subject"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/subjectkeyidentifier"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Subject Key Identifier"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/field/subjectname"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Subject Name"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/certificatetemplateinformation"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""V2 Template Name"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/certificatetemplatename"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""V1 Template Name"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/thumbprint"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Thumbprint"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/field/x509version"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""X.509 Version"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2012/01/insidecorporatenetwork"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Inside Corporate Network"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2012/01/passwordexpirationtime"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Password Expiration Time"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2012/01/passwordexpirationdays"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Password Expiration Days"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2012/01/passwordchangeurl"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Update Password URL"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/claims/authnmethodsreferences"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Authentication Methods References"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/requestcontext/claims/client-request-id"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Client Request ID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2013/11/alternateloginid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Alternate Login ID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/></IDPSSODescriptor><ContactPerson contactType=""support""/></EntityDescriptor>"; }

        public static string AdfsV3SignatureValue { get => @"LlgnhZB+Zdtt1bxJpWHhAxzbeDA6Oz8BHMge6MIyfZrSBNLnvg/tGYneOhOl7mu0cg8cCJCLaXWBUfPOWDAebXqKYvApA1yu1ALd2sg03Q/A1ti/kWRhiJempc/vX27upSXppjTfaBbiqeoY6qpFJwzBSl3NfOS9b1Zf6BsxyZ6L8msmcsEv4tj17+DiDPcoGDK4uqyHctQDa1CySWLTQduqZ5IABZ2LipDCsfz7Tg1rGCCbHLhiiP0ge0fact38ZM1XNZ7gtiLCufUUUgG5civp5ifYJEWmIZ/SdwjUeMMHejd36iinRoAAsM/bW/trZVo3PQjCGpHxoqR+FMt/Dw=="; }

        public static string AdfsV3ReferenceDigestValue { get => "/WtSzs2am3kxkT+nYdFWs/+udQe78CzNYMquzKUjUqk="; }

        public static string AdfsV3ReferenceURI { get => "#_7d30a77e-57fa-4c05-93ab-63a2d4e1a336"; }

        public static string AdfsV4Metadata { get => @"<EntityDescriptor ID=""_b9fe7514-a527-4a24-86ef-8849d3ab008a"" entityID=""http://fs.msidlab11.com/adfs/services/trust"" xmlns=""urn:oasis:names:tc:SAML:2.0:metadata""><ds:Signature xmlns:ds=""http://www.w3.org/2000/09/xmldsig#""><ds:SignedInfo><ds:CanonicalizationMethod Algorithm=""http://www.w3.org/2001/10/xml-exc-c14n#""/><ds:SignatureMethod Algorithm=""http://www.w3.org/2001/04/xmldsig-more#rsa-sha256""/><ds:Reference URI=""#_b9fe7514-a527-4a24-86ef-8849d3ab008a""><ds:Transforms><ds:Transform Algorithm=""http://www.w3.org/2000/09/xmldsig#enveloped-signature""/><ds:Transform Algorithm=""http://www.w3.org/2001/10/xml-exc-c14n#""/></ds:Transforms><ds:DigestMethod Algorithm=""http://www.w3.org/2001/04/xmlenc#sha256""/><ds:DigestValue>MuKCkLnC//j4+3h3S1g+lOowG748ljSAjsHAg1bCpHs=</ds:DigestValue></ds:Reference></ds:SignedInfo><ds:SignatureValue>aSJ3IUuYvAhF6i8rgU9gPnNr2YS02dWRtKyGYa+8N+dUPOPpoW6bfeVEO0/HnY4Xe/Pf3SL9UFPhIqvup4N/kKs/OOZbKaYn+hitnDPFBQ9FckA0ZecfmgiOofSWo/6Za994a+ECnuSb+QevF53kGjDHbGTgThBTjo1lWgzOtVDVrRS2SmoHwEXpZszC0r+m9Jm7M7Ub4UjyesleHSYPm1Me9Copu+BQLdJO/d2X+VH8uY2VLScWS97h4UL5AKpY78pOrcaQTbbaC7oKeaR/K2FwnWF+TtmTid3yBNEZkixF5GWvLrWkoTK0lFDbksJ0Y1j4zTRBP85uGxx5iprPAw==</ds:SignatureValue><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIC3DCCAcSgAwIBAgIQR6dE8BwieKVBETXJ7bFh1TANBgkqhkiG9w0BAQsFADAqMSgwJgYDVQQDEx9BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiMTEuY29tMB4XDTE3MDEyMzIxMjgzOVoXDTE4MDEyMzIxMjgzOVowKjEoMCYGA1UEAxMfQURGUyBTaWduaW5nIC0gZnMubXNpZGxhYjExLmNvbTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBANEGVDA2PatZq7yN/F9UrvheLNq6FiSzJdJE3Z55SPooQpl5Oi9dBO9vKDmnTBUB+Ex8h2X9Shpp6h1+Sir014Nkn8VHSfKqV7aVUSE0PxBOCCKakwdISUo7NA9g7VCwUXP6Z6nghxcJr7FF1g5/MwZE22tIu4sivJSsskA8J5lm5O9LzP7TLiRlMmD8qE2hYl55A5OE7mhOcz/2JWV+V7VxIuvl108oRNfD1DFwdNRipQYM1H7mgjHveR6IP1+F04rD+tGBjbwekA4BWMkiZFIm3WLJizsTfLcaX3HyEaf9fOYCiQIHC/PYsN4zHmfKPcP44/StBLZhYqsZh/SZp0UCAwEAATANBgkqhkiG9w0BAQsFAAOCAQEAJEy4+oJEfUI5/g0f11r2M+Aa0BlZcd+BenZ9yGQRFSzfWa6XPM4eh9Kxn3XJoqm3L1ergqeSbHcRn6cJ9ge+2DQkl8N37i8QTrsAdZl5OKcxzk313GBmiKaMEx5VgoB8rWsZqNdbh5mJCQeHK07MvAesIVU2Rk5G3BZz3LRSUo7d9asBPfHg37cR9Ol4iuyQGu93MAlEhWBRbAaT7Km942972kYh2uPc/SjdcK8VKk3JJ+Kyp0QrxedpFd8iDy/KQdL4h+1VuR8eKu+znxS/CqpidysP9rwcwsJxGKvWmSgeuMbgxIBLdm1FUQsYAa0zbNkjRNh1n/+XdFVUprZKsw==</X509Certificate></X509Data></KeyInfo></ds:Signature><RoleDescriptor xsi:type=""fed:ApplicationServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/ws-sx/ws-trust/200512 http://schemas.xmlsoap.org/ws/2005/02/trust http://docs.oasis-open.org/wsfed/federation/200706"" ServiceDisplayName=""fs.msidlab11.com AD FS"" xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706""><KeyDescriptor use=""encryption""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIC4jCCAcqgAwIBAgIQFk52rOV/iaxO5BEqaldWZTANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJBREZTIEVuY3J5cHRpb24gLSBmcy5tc2lkbGFiMTEuY29tMB4XDTE3MDEyMzIxMjgzOVoXDTE4MDEyMzIxMjgzOVowLTErMCkGA1UEAxMiQURGUyBFbmNyeXB0aW9uIC0gZnMubXNpZGxhYjExLmNvbTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAOxNB30Crneh0gO79MNOQJnAoBwOB4Q70HLNk7x3zwktThSQ2ordobC1XktjLH9AkbaC4Y6CY/KgNboHeacbgnrBk4/6VuduUzaJoVBQO1CodUUg7g1QPMCaT8D41tElJ7mUb2IWqhcc+vA4A5jI6NgJ1KbWIdgwbonG29b9meiZ8p0sUqFhtNdeVcoRSv0nfLkETA3lsV+DXGCiOSIcN89tnGaOc53tIvtpQpjOLZWNB2//OJch1/StxI2Q7p2L8nhoFZqpduwePecGORfFrk4Vx0mp0Jv/Y1sPxZT+ziwnuiWRiuMu+2jznvRUdARmK4lasgtg1BXxDRrFHBUoyGcCAwEAATANBgkqhkiG9w0BAQsFAAOCAQEAuas6bd+GgK/s8aVqCAGMJ3G4xYPP6DGcm1yLrr5dlVMepVSYxrotomIfhrXU7KOW5KUswsoz4dlIjboVHcAJsFttxI0CMtrTw/7l87pZ5HKO2ilioRtJsW4Y2R/yAi80SW1UKNUlt+/V7hVxA/Zuv95uS1awceWOEB2x5F0e+jCkGN7pbOes6IyjH7cf7QB1y7PbORgkf5Kh6wcHUMZSWmCmauoBUbsspW2oggXUW6EIAeUtiujcVqoEJ9g7bY31mfTa3u8tLynyjo/XdspCpHI7AXNx2wPC2QRFJSA5jo9O15eT8eZQSnVyMHQt39aEnniKvkf6KAPlOsAbWZCXqA==</X509Certificate></X509Data></KeyInfo></KeyDescriptor><fed:ClaimTypesRequested><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/emailaddress"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>E-Mail Address</auth:DisplayName><auth:Description>The e-mail address of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/givenname"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Given Name</auth:DisplayName><auth:Description>The given name of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/name"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Name</auth:DisplayName><auth:Description>The unique name of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/upn"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>UPN</auth:DisplayName><auth:Description>The user principal name (UPN) of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/CommonName"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Common Name</auth:DisplayName><auth:Description>The common name of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/EmailAddress"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>AD FS 1.x E-Mail Address</auth:DisplayName><auth:Description>The e-mail address of the user when interoperating with AD FS 1.1 or AD FS 1.0</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/Group"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Group</auth:DisplayName><auth:Description>A group that the user is a member of</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/UPN"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>AD FS 1.x UPN</auth:DisplayName><auth:Description>The UPN of the user when interoperating with AD FS 1.1 or AD FS 1.0</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/role"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Role</auth:DisplayName><auth:Description>A role that the user has</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/surname"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Surname</auth:DisplayName><auth:Description>The surname of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/privatepersonalidentifier"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>PPID</auth:DisplayName><auth:Description>The private identifier of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/nameidentifier"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Name ID</auth:DisplayName><auth:Description>The SAML name identifier of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/authenticationinstant"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Authentication time stamp</auth:DisplayName><auth:Description>Used to display the time and date that the user was authenticated</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/authenticationmethod"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Authentication method</auth:DisplayName><auth:Description>The method used to authenticate the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/denyonlysid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Deny only group SID</auth:DisplayName><auth:Description>The deny-only group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/denyonlyprimarysid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Deny only primary SID</auth:DisplayName><auth:Description>The deny-only primary SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/denyonlyprimarygroupsid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Deny only primary group SID</auth:DisplayName><auth:Description>The deny-only primary group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/groupsid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Group SID</auth:DisplayName><auth:Description>The group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/primarygroupsid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Primary group SID</auth:DisplayName><auth:Description>The primary group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/primarysid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Primary SID</auth:DisplayName><auth:Description>The primary SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/windowsaccountname"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Windows account name</auth:DisplayName><auth:Description>The domain account name of the user in the form of domain\user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/isregistereduser"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Is Registered User</auth:DisplayName><auth:Description>User is registered to use this device</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/identifier"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Device Identifier</auth:DisplayName><auth:Description>Identifier of the device</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/registrationid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Device Registration Identifier</auth:DisplayName><auth:Description>Identifier for Device Registration</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/displayname"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Device Registration DisplayName</auth:DisplayName><auth:Description>Display name of Device Registration</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/ostype"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Device OS type</auth:DisplayName><auth:Description>OS type of the device</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/osversion"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Device OS Version</auth:DisplayName><auth:Description>OS version of the device</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/ismanaged"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Is Managed Device</auth:DisplayName><auth:Description>Device is managed by a management service</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-forwarded-client-ip"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Forwarded Client IP</auth:DisplayName><auth:Description>IP address of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-client-application"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Client Application</auth:DisplayName><auth:Description>Type of the Client Application</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-client-user-agent"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Client User Agent</auth:DisplayName><auth:Description>Device type the client is using to access the application</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-client-ip"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Client IP</auth:DisplayName><auth:Description>IP address of the client</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-endpoint-absolute-path"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Endpoint Path</auth:DisplayName><auth:Description>Absolute Endpoint path which can be used to determine active versus passive clients</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-proxy"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Proxy</auth:DisplayName><auth:Description>DNS name of the federation server proxy that passed the request</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/relyingpartytrustid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Application Identifier</auth:DisplayName><auth:Description>Identifier for the Relying Party</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/applicationpolicy"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Application policies</auth:DisplayName><auth:Description>Application policies of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/authoritykeyidentifier"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Authority Key Identifier</auth:DisplayName><auth:Description>The Authority Key Identifier extension of the certificate that signed an issued certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/basicconstraints"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Basic Constraint</auth:DisplayName><auth:Description>One of the basic constraints of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/eku"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Enhanced Key Usage</auth:DisplayName><auth:Description>Describes one of the enhanced key usages of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/issuer"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Issuer</auth:DisplayName><auth:Description>The name of the certificate authority that issued the X.509 certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/issuername"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Issuer Name</auth:DisplayName><auth:Description>The distinguished name of the certificate issuer</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/keyusage"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Key Usage</auth:DisplayName><auth:Description>One of the key usages of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/notafter"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Not After</auth:DisplayName><auth:Description>Date in local time after which a certificate is no longer valid</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/notbefore"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Not Before</auth:DisplayName><auth:Description>The date in local time on which a certificate becomes valid</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/certificatepolicy"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Certificate Policies</auth:DisplayName><auth:Description>The policies under which the certificate has been issued</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/rsa"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Public Key</auth:DisplayName><auth:Description>Public Key of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/rawdata"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Certificate Raw Data</auth:DisplayName><auth:Description>The raw data of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/san"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Subject Alternative Name</auth:DisplayName><auth:Description>One of the alternative names of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/serialnumber"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Serial Number</auth:DisplayName><auth:Description>The serial number of a certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/signaturealgorithm"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Signature Algorithm</auth:DisplayName><auth:Description>The algorithm used to create the signature of a certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/subject"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Subject</auth:DisplayName><auth:Description>The subject from the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/subjectkeyidentifier"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Subject Key Identifier</auth:DisplayName><auth:Description>Describes the subject key identifier of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/subjectname"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Subject Name</auth:DisplayName><auth:Description>The subject distinguished name from a certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/certificatetemplateinformation"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>V2 Template Name</auth:DisplayName><auth:Description>The name of the version 2 certificate template used when issuing or renewing a certificate. The extension is Microsoft specific.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/certificatetemplatename"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>V1 Template Name</auth:DisplayName><auth:Description>The name of the version 1 certificate template used when issuing or renewing a certificate. The extension is Microsoft specific.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/thumbprint"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Thumbprint</auth:DisplayName><auth:Description>Thumbprint of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/x509version"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>X.509 Version</auth:DisplayName><auth:Description>The X.509 format version of a certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2012/01/insidecorporatenetwork"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Inside Corporate Network</auth:DisplayName><auth:Description>Used to indicate if a request originated inside corporate network</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2012/01/passwordexpirationtime"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Password Expiration Time</auth:DisplayName><auth:Description>Used to display the time when the password expires</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2012/01/passwordexpirationdays"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Password Expiration Days</auth:DisplayName><auth:Description>Used to display the number of days to password expiry</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2012/01/passwordchangeurl"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Update Password URL</auth:DisplayName><auth:Description>Used to display the web address of update password service</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/claims/authnmethodsreferences"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Authentication Methods References</auth:DisplayName><auth:Description>Used to indicate all authentication methods used to authenticate the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/client-request-id"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Client Request ID</auth:DisplayName><auth:Description>Identifier for a user session</auth:Description></auth:ClaimType></fed:ClaimTypesRequested><fed:TargetScopes><EndpointReference xmlns=""http://www.w3.org/2005/08/addressing""><Address>https://fs.msidlab11.com/adfs/services/trust/2005/issuedtokenmixedasymmetricbasic256</Address></EndpointReference><EndpointReference xmlns=""http://www.w3.org/2005/08/addressing""><Address>https://fs.msidlab11.com/adfs/services/trust/2005/issuedtokenmixedsymmetricbasic256</Address></EndpointReference><EndpointReference xmlns=""http://www.w3.org/2005/08/addressing""><Address>https://fs.msidlab11.com/adfs/services/trust/13/issuedtokenmixedasymmetricbasic256</Address></EndpointReference><EndpointReference xmlns=""http://www.w3.org/2005/08/addressing""><Address>https://fs.msidlab11.com/adfs/services/trust/13/issuedtokenmixedsymmetricbasic256</Address></EndpointReference><EndpointReference xmlns=""http://www.w3.org/2005/08/addressing""><Address>https://fs.msidlab11.com/adfs/ls/</Address></EndpointReference><EndpointReference xmlns=""http://www.w3.org/2005/08/addressing""><Address>http://fs.msidlab11.com/adfs/services/trust</Address></EndpointReference></fed:TargetScopes><fed:ApplicationServiceEndpoint><EndpointReference xmlns=""http://www.w3.org/2005/08/addressing""><Address>https://fs.msidlab11.com/adfs/services/trust/2005/issuedtokenmixedasymmetricbasic256</Address></EndpointReference></fed:ApplicationServiceEndpoint><fed:PassiveRequestorEndpoint><EndpointReference xmlns=""http://www.w3.org/2005/08/addressing""><Address>https://fs.msidlab11.com/adfs/ls/</Address></EndpointReference></fed:PassiveRequestorEndpoint></RoleDescriptor><RoleDescriptor xsi:type=""fed:SecurityTokenServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/ws-sx/ws-trust/200512 http://schemas.xmlsoap.org/ws/2005/02/trust http://docs.oasis-open.org/wsfed/federation/200706"" ServiceDisplayName=""fs.msidlab11.com AD FS"" xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706""><KeyDescriptor use=""signing""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIC3DCCAcSgAwIBAgIQR6dE8BwieKVBETXJ7bFh1TANBgkqhkiG9w0BAQsFADAqMSgwJgYDVQQDEx9BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiMTEuY29tMB4XDTE3MDEyMzIxMjgzOVoXDTE4MDEyMzIxMjgzOVowKjEoMCYGA1UEAxMfQURGUyBTaWduaW5nIC0gZnMubXNpZGxhYjExLmNvbTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBANEGVDA2PatZq7yN/F9UrvheLNq6FiSzJdJE3Z55SPooQpl5Oi9dBO9vKDmnTBUB+Ex8h2X9Shpp6h1+Sir014Nkn8VHSfKqV7aVUSE0PxBOCCKakwdISUo7NA9g7VCwUXP6Z6nghxcJr7FF1g5/MwZE22tIu4sivJSsskA8J5lm5O9LzP7TLiRlMmD8qE2hYl55A5OE7mhOcz/2JWV+V7VxIuvl108oRNfD1DFwdNRipQYM1H7mgjHveR6IP1+F04rD+tGBjbwekA4BWMkiZFIm3WLJizsTfLcaX3HyEaf9fOYCiQIHC/PYsN4zHmfKPcP44/StBLZhYqsZh/SZp0UCAwEAATANBgkqhkiG9w0BAQsFAAOCAQEAJEy4+oJEfUI5/g0f11r2M+Aa0BlZcd+BenZ9yGQRFSzfWa6XPM4eh9Kxn3XJoqm3L1ergqeSbHcRn6cJ9ge+2DQkl8N37i8QTrsAdZl5OKcxzk313GBmiKaMEx5VgoB8rWsZqNdbh5mJCQeHK07MvAesIVU2Rk5G3BZz3LRSUo7d9asBPfHg37cR9Ol4iuyQGu93MAlEhWBRbAaT7Km942972kYh2uPc/SjdcK8VKk3JJ+Kyp0QrxedpFd8iDy/KQdL4h+1VuR8eKu+znxS/CqpidysP9rwcwsJxGKvWmSgeuMbgxIBLdm1FUQsYAa0zbNkjRNh1n/+XdFVUprZKsw==</X509Certificate></X509Data></KeyInfo></KeyDescriptor><fed:TokenTypesOffered><fed:TokenType Uri=""urn:oasis:names:tc:SAML:2.0:assertion""/><fed:TokenType Uri=""urn:oasis:names:tc:SAML:1.0:assertion""/></fed:TokenTypesOffered><fed:ClaimTypesOffered><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/emailaddress"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>E-Mail Address</auth:DisplayName><auth:Description>The e-mail address of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/givenname"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Given Name</auth:DisplayName><auth:Description>The given name of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/name"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Name</auth:DisplayName><auth:Description>The unique name of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/upn"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>UPN</auth:DisplayName><auth:Description>The user principal name (UPN) of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/CommonName"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Common Name</auth:DisplayName><auth:Description>The common name of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/EmailAddress"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>AD FS 1.x E-Mail Address</auth:DisplayName><auth:Description>The e-mail address of the user when interoperating with AD FS 1.1 or AD FS 1.0</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/Group"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Group</auth:DisplayName><auth:Description>A group that the user is a member of</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/claims/UPN"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>AD FS 1.x UPN</auth:DisplayName><auth:Description>The UPN of the user when interoperating with AD FS 1.1 or AD FS 1.0</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/role"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Role</auth:DisplayName><auth:Description>A role that the user has</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/surname"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Surname</auth:DisplayName><auth:Description>The surname of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/privatepersonalidentifier"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>PPID</auth:DisplayName><auth:Description>The private identifier of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/nameidentifier"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Name ID</auth:DisplayName><auth:Description>The SAML name identifier of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/authenticationinstant"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Authentication time stamp</auth:DisplayName><auth:Description>Used to display the time and date that the user was authenticated</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/authenticationmethod"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Authentication method</auth:DisplayName><auth:Description>The method used to authenticate the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/denyonlysid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Deny only group SID</auth:DisplayName><auth:Description>The deny-only group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/denyonlyprimarysid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Deny only primary SID</auth:DisplayName><auth:Description>The deny-only primary SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/denyonlyprimarygroupsid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Deny only primary group SID</auth:DisplayName><auth:Description>The deny-only primary group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/groupsid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Group SID</auth:DisplayName><auth:Description>The group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/primarygroupsid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Primary group SID</auth:DisplayName><auth:Description>The primary group SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/primarysid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Primary SID</auth:DisplayName><auth:Description>The primary SID of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/windowsaccountname"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Windows account name</auth:DisplayName><auth:Description>The domain account name of the user in the form of domain\user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/isregistereduser"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Is Registered User</auth:DisplayName><auth:Description>User is registered to use this device</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/identifier"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Device Identifier</auth:DisplayName><auth:Description>Identifier of the device</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/registrationid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Device Registration Identifier</auth:DisplayName><auth:Description>Identifier for Device Registration</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/displayname"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Device Registration DisplayName</auth:DisplayName><auth:Description>Display name of Device Registration</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/ostype"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Device OS type</auth:DisplayName><auth:Description>OS type of the device</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/osversion"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Device OS Version</auth:DisplayName><auth:Description>OS version of the device</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/devicecontext/claims/ismanaged"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Is Managed Device</auth:DisplayName><auth:Description>Device is managed by a management service</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-forwarded-client-ip"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Forwarded Client IP</auth:DisplayName><auth:Description>IP address of the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-client-application"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Client Application</auth:DisplayName><auth:Description>Type of the Client Application</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-client-user-agent"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Client User Agent</auth:DisplayName><auth:Description>Device type the client is using to access the application</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-client-ip"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Client IP</auth:DisplayName><auth:Description>IP address of the client</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-endpoint-absolute-path"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Endpoint Path</auth:DisplayName><auth:Description>Absolute Endpoint path which can be used to determine active versus passive clients</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-proxy"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Proxy</auth:DisplayName><auth:Description>DNS name of the federation server proxy that passed the request</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/relyingpartytrustid"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Application Identifier</auth:DisplayName><auth:Description>Identifier for the Relying Party</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/applicationpolicy"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Application policies</auth:DisplayName><auth:Description>Application policies of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/authoritykeyidentifier"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Authority Key Identifier</auth:DisplayName><auth:Description>The Authority Key Identifier extension of the certificate that signed an issued certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/basicconstraints"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Basic Constraint</auth:DisplayName><auth:Description>One of the basic constraints of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/eku"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Enhanced Key Usage</auth:DisplayName><auth:Description>Describes one of the enhanced key usages of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/issuer"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Issuer</auth:DisplayName><auth:Description>The name of the certificate authority that issued the X.509 certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/issuername"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Issuer Name</auth:DisplayName><auth:Description>The distinguished name of the certificate issuer</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/keyusage"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Key Usage</auth:DisplayName><auth:Description>One of the key usages of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/notafter"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Not After</auth:DisplayName><auth:Description>Date in local time after which a certificate is no longer valid</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/notbefore"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Not Before</auth:DisplayName><auth:Description>The date in local time on which a certificate becomes valid</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/certificatepolicy"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Certificate Policies</auth:DisplayName><auth:Description>The policies under which the certificate has been issued</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/rsa"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Public Key</auth:DisplayName><auth:Description>Public Key of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/rawdata"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Certificate Raw Data</auth:DisplayName><auth:Description>The raw data of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/san"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Subject Alternative Name</auth:DisplayName><auth:Description>One of the alternative names of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/serialnumber"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Serial Number</auth:DisplayName><auth:Description>The serial number of a certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/signaturealgorithm"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Signature Algorithm</auth:DisplayName><auth:Description>The algorithm used to create the signature of a certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/subject"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Subject</auth:DisplayName><auth:Description>The subject from the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/subjectkeyidentifier"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Subject Key Identifier</auth:DisplayName><auth:Description>Describes the subject key identifier of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/subjectname"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Subject Name</auth:DisplayName><auth:Description>The subject distinguished name from a certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/certificatetemplateinformation"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>V2 Template Name</auth:DisplayName><auth:Description>The name of the version 2 certificate template used when issuing or renewing a certificate. The extension is Microsoft specific.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/certificatetemplatename"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>V1 Template Name</auth:DisplayName><auth:Description>The name of the version 1 certificate template used when issuing or renewing a certificate. The extension is Microsoft specific.</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/thumbprint"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Thumbprint</auth:DisplayName><auth:Description>Thumbprint of the certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/12/certificatecontext/field/x509version"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>X.509 Version</auth:DisplayName><auth:Description>The X.509 format version of a certificate</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2012/01/insidecorporatenetwork"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Inside Corporate Network</auth:DisplayName><auth:Description>Used to indicate if a request originated inside corporate network</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2012/01/passwordexpirationtime"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Password Expiration Time</auth:DisplayName><auth:Description>Used to display the time when the password expires</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2012/01/passwordexpirationdays"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Password Expiration Days</auth:DisplayName><auth:Description>Used to display the number of days to password expiry</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/ws/2012/01/passwordchangeurl"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Update Password URL</auth:DisplayName><auth:Description>Used to display the web address of update password service</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/claims/authnmethodsreferences"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Authentication Methods References</auth:DisplayName><auth:Description>Used to indicate all authentication methods used to authenticate the user</auth:Description></auth:ClaimType><auth:ClaimType Uri=""http://schemas.microsoft.com/2012/01/requestcontext/claims/client-request-id"" Optional=""true"" xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706""><auth:DisplayName>Client Request ID</auth:DisplayName><auth:Description>Identifier for a user session</auth:Description></auth:ClaimType></fed:ClaimTypesOffered><fed:SecurityTokenServiceEndpoint><EndpointReference xmlns=""http://www.w3.org/2005/08/addressing""><Address>https://fs.msidlab11.com/adfs/services/trust/2005/certificatemixed</Address><Metadata><Metadata xmlns=""http://schemas.xmlsoap.org/ws/2004/09/mex"" xmlns:xsd=""http://www.w3.org/2001/XMLSchema"" xmlns:wsx=""http://schemas.xmlsoap.org/ws/2004/09/mex""><wsx:MetadataSection Dialect=""http://schemas.xmlsoap.org/ws/2004/09/mex"" xmlns=""""><wsx:MetadataReference><Address xmlns=""http://www.w3.org/2005/08/addressing"">https://fs.msidlab11.com/adfs/services/trust/mex</Address></wsx:MetadataReference></wsx:MetadataSection></Metadata></Metadata></EndpointReference></fed:SecurityTokenServiceEndpoint><fed:PassiveRequestorEndpoint><EndpointReference xmlns=""http://www.w3.org/2005/08/addressing""><Address>https://fs.msidlab11.com/adfs/ls/</Address></EndpointReference></fed:PassiveRequestorEndpoint></RoleDescriptor><SPSSODescriptor WantAssertionsSigned=""true"" protocolSupportEnumeration=""urn:oasis:names:tc:SAML:2.0:protocol""><KeyDescriptor use=""encryption""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIC4jCCAcqgAwIBAgIQFk52rOV/iaxO5BEqaldWZTANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJBREZTIEVuY3J5cHRpb24gLSBmcy5tc2lkbGFiMTEuY29tMB4XDTE3MDEyMzIxMjgzOVoXDTE4MDEyMzIxMjgzOVowLTErMCkGA1UEAxMiQURGUyBFbmNyeXB0aW9uIC0gZnMubXNpZGxhYjExLmNvbTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAOxNB30Crneh0gO79MNOQJnAoBwOB4Q70HLNk7x3zwktThSQ2ordobC1XktjLH9AkbaC4Y6CY/KgNboHeacbgnrBk4/6VuduUzaJoVBQO1CodUUg7g1QPMCaT8D41tElJ7mUb2IWqhcc+vA4A5jI6NgJ1KbWIdgwbonG29b9meiZ8p0sUqFhtNdeVcoRSv0nfLkETA3lsV+DXGCiOSIcN89tnGaOc53tIvtpQpjOLZWNB2//OJch1/StxI2Q7p2L8nhoFZqpduwePecGORfFrk4Vx0mp0Jv/Y1sPxZT+ziwnuiWRiuMu+2jznvRUdARmK4lasgtg1BXxDRrFHBUoyGcCAwEAATANBgkqhkiG9w0BAQsFAAOCAQEAuas6bd+GgK/s8aVqCAGMJ3G4xYPP6DGcm1yLrr5dlVMepVSYxrotomIfhrXU7KOW5KUswsoz4dlIjboVHcAJsFttxI0CMtrTw/7l87pZ5HKO2ilioRtJsW4Y2R/yAi80SW1UKNUlt+/V7hVxA/Zuv95uS1awceWOEB2x5F0e+jCkGN7pbOes6IyjH7cf7QB1y7PbORgkf5Kh6wcHUMZSWmCmauoBUbsspW2oggXUW6EIAeUtiujcVqoEJ9g7bY31mfTa3u8tLynyjo/XdspCpHI7AXNx2wPC2QRFJSA5jo9O15eT8eZQSnVyMHQt39aEnniKvkf6KAPlOsAbWZCXqA==</X509Certificate></X509Data></KeyInfo></KeyDescriptor><KeyDescriptor use=""signing""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIC3DCCAcSgAwIBAgIQR6dE8BwieKVBETXJ7bFh1TANBgkqhkiG9w0BAQsFADAqMSgwJgYDVQQDEx9BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiMTEuY29tMB4XDTE3MDEyMzIxMjgzOVoXDTE4MDEyMzIxMjgzOVowKjEoMCYGA1UEAxMfQURGUyBTaWduaW5nIC0gZnMubXNpZGxhYjExLmNvbTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBANEGVDA2PatZq7yN/F9UrvheLNq6FiSzJdJE3Z55SPooQpl5Oi9dBO9vKDmnTBUB+Ex8h2X9Shpp6h1+Sir014Nkn8VHSfKqV7aVUSE0PxBOCCKakwdISUo7NA9g7VCwUXP6Z6nghxcJr7FF1g5/MwZE22tIu4sivJSsskA8J5lm5O9LzP7TLiRlMmD8qE2hYl55A5OE7mhOcz/2JWV+V7VxIuvl108oRNfD1DFwdNRipQYM1H7mgjHveR6IP1+F04rD+tGBjbwekA4BWMkiZFIm3WLJizsTfLcaX3HyEaf9fOYCiQIHC/PYsN4zHmfKPcP44/StBLZhYqsZh/SZp0UCAwEAATANBgkqhkiG9w0BAQsFAAOCAQEAJEy4+oJEfUI5/g0f11r2M+Aa0BlZcd+BenZ9yGQRFSzfWa6XPM4eh9Kxn3XJoqm3L1ergqeSbHcRn6cJ9ge+2DQkl8N37i8QTrsAdZl5OKcxzk313GBmiKaMEx5VgoB8rWsZqNdbh5mJCQeHK07MvAesIVU2Rk5G3BZz3LRSUo7d9asBPfHg37cR9Ol4iuyQGu93MAlEhWBRbAaT7Km942972kYh2uPc/SjdcK8VKk3JJ+Kyp0QrxedpFd8iDy/KQdL4h+1VuR8eKu+znxS/CqpidysP9rwcwsJxGKvWmSgeuMbgxIBLdm1FUQsYAa0zbNkjRNh1n/+XdFVUprZKsw==</X509Certificate></X509Data></KeyInfo></KeyDescriptor><SingleLogoutService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect"" Location=""https://fs.msidlab11.com/adfs/ls/""/><SingleLogoutService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-POST"" Location=""https://fs.msidlab11.com/adfs/ls/""/><NameIDFormat>urn:oasis:names:tc:SAML:1.1:nameid-format:emailAddress</NameIDFormat><NameIDFormat>urn:oasis:names:tc:SAML:2.0:nameid-format:persistent</NameIDFormat><NameIDFormat>urn:oasis:names:tc:SAML:2.0:nameid-format:transient</NameIDFormat><AssertionConsumerService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-POST"" Location=""https://fs.msidlab11.com/adfs/ls/"" index=""0"" isDefault=""true""/><AssertionConsumerService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Artifact"" Location=""https://fs.msidlab11.com/adfs/ls/"" index=""1""/><AssertionConsumerService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect"" Location=""https://fs.msidlab11.com/adfs/ls/"" index=""2""/></SPSSODescriptor><IDPSSODescriptor protocolSupportEnumeration=""urn:oasis:names:tc:SAML:2.0:protocol""><KeyDescriptor use=""encryption""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIC4jCCAcqgAwIBAgIQFk52rOV/iaxO5BEqaldWZTANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJBREZTIEVuY3J5cHRpb24gLSBmcy5tc2lkbGFiMTEuY29tMB4XDTE3MDEyMzIxMjgzOVoXDTE4MDEyMzIxMjgzOVowLTErMCkGA1UEAxMiQURGUyBFbmNyeXB0aW9uIC0gZnMubXNpZGxhYjExLmNvbTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAOxNB30Crneh0gO79MNOQJnAoBwOB4Q70HLNk7x3zwktThSQ2ordobC1XktjLH9AkbaC4Y6CY/KgNboHeacbgnrBk4/6VuduUzaJoVBQO1CodUUg7g1QPMCaT8D41tElJ7mUb2IWqhcc+vA4A5jI6NgJ1KbWIdgwbonG29b9meiZ8p0sUqFhtNdeVcoRSv0nfLkETA3lsV+DXGCiOSIcN89tnGaOc53tIvtpQpjOLZWNB2//OJch1/StxI2Q7p2L8nhoFZqpduwePecGORfFrk4Vx0mp0Jv/Y1sPxZT+ziwnuiWRiuMu+2jznvRUdARmK4lasgtg1BXxDRrFHBUoyGcCAwEAATANBgkqhkiG9w0BAQsFAAOCAQEAuas6bd+GgK/s8aVqCAGMJ3G4xYPP6DGcm1yLrr5dlVMepVSYxrotomIfhrXU7KOW5KUswsoz4dlIjboVHcAJsFttxI0CMtrTw/7l87pZ5HKO2ilioRtJsW4Y2R/yAi80SW1UKNUlt+/V7hVxA/Zuv95uS1awceWOEB2x5F0e+jCkGN7pbOes6IyjH7cf7QB1y7PbORgkf5Kh6wcHUMZSWmCmauoBUbsspW2oggXUW6EIAeUtiujcVqoEJ9g7bY31mfTa3u8tLynyjo/XdspCpHI7AXNx2wPC2QRFJSA5jo9O15eT8eZQSnVyMHQt39aEnniKvkf6KAPlOsAbWZCXqA==</X509Certificate></X509Data></KeyInfo></KeyDescriptor><KeyDescriptor use=""signing""><KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#""><X509Data><X509Certificate>MIIC3DCCAcSgAwIBAgIQR6dE8BwieKVBETXJ7bFh1TANBgkqhkiG9w0BAQsFADAqMSgwJgYDVQQDEx9BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiMTEuY29tMB4XDTE3MDEyMzIxMjgzOVoXDTE4MDEyMzIxMjgzOVowKjEoMCYGA1UEAxMfQURGUyBTaWduaW5nIC0gZnMubXNpZGxhYjExLmNvbTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBANEGVDA2PatZq7yN/F9UrvheLNq6FiSzJdJE3Z55SPooQpl5Oi9dBO9vKDmnTBUB+Ex8h2X9Shpp6h1+Sir014Nkn8VHSfKqV7aVUSE0PxBOCCKakwdISUo7NA9g7VCwUXP6Z6nghxcJr7FF1g5/MwZE22tIu4sivJSsskA8J5lm5O9LzP7TLiRlMmD8qE2hYl55A5OE7mhOcz/2JWV+V7VxIuvl108oRNfD1DFwdNRipQYM1H7mgjHveR6IP1+F04rD+tGBjbwekA4BWMkiZFIm3WLJizsTfLcaX3HyEaf9fOYCiQIHC/PYsN4zHmfKPcP44/StBLZhYqsZh/SZp0UCAwEAATANBgkqhkiG9w0BAQsFAAOCAQEAJEy4+oJEfUI5/g0f11r2M+Aa0BlZcd+BenZ9yGQRFSzfWa6XPM4eh9Kxn3XJoqm3L1ergqeSbHcRn6cJ9ge+2DQkl8N37i8QTrsAdZl5OKcxzk313GBmiKaMEx5VgoB8rWsZqNdbh5mJCQeHK07MvAesIVU2Rk5G3BZz3LRSUo7d9asBPfHg37cR9Ol4iuyQGu93MAlEhWBRbAaT7Km942972kYh2uPc/SjdcK8VKk3JJ+Kyp0QrxedpFd8iDy/KQdL4h+1VuR8eKu+znxS/CqpidysP9rwcwsJxGKvWmSgeuMbgxIBLdm1FUQsYAa0zbNkjRNh1n/+XdFVUprZKsw==</X509Certificate></X509Data></KeyInfo></KeyDescriptor><SingleLogoutService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect"" Location=""https://fs.msidlab11.com/adfs/ls/""/><SingleLogoutService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-POST"" Location=""https://fs.msidlab11.com/adfs/ls/""/><NameIDFormat>urn:oasis:names:tc:SAML:1.1:nameid-format:emailAddress</NameIDFormat><NameIDFormat>urn:oasis:names:tc:SAML:2.0:nameid-format:persistent</NameIDFormat><NameIDFormat>urn:oasis:names:tc:SAML:2.0:nameid-format:transient</NameIDFormat><SingleSignOnService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect"" Location=""https://fs.msidlab11.com/adfs/ls/""/><SingleSignOnService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-POST"" Location=""https://fs.msidlab11.com/adfs/ls/""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/emailaddress"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""E-Mail Address"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/givenname"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Given Name"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/name"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Name"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/upn"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""UPN"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/claims/CommonName"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Common Name"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/claims/EmailAddress"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""AD FS 1.x E-Mail Address"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/claims/Group"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Group"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/claims/UPN"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""AD FS 1.x UPN"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/role"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Role"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/surname"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Surname"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/privatepersonalidentifier"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""PPID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/nameidentifier"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Name ID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/authenticationinstant"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Authentication time stamp"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/authenticationmethod"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Authentication method"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/denyonlysid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Deny only group SID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/denyonlyprimarysid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Deny only primary SID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/denyonlyprimarygroupsid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Deny only primary group SID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/groupsid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Group SID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/primarygroupsid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Primary group SID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/primarysid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Primary SID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/windowsaccountname"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Windows account name"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/devicecontext/claims/isregistereduser"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Is Registered User"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/devicecontext/claims/identifier"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Device Identifier"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/devicecontext/claims/registrationid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Device Registration Identifier"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/devicecontext/claims/displayname"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Device Registration DisplayName"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/devicecontext/claims/ostype"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Device OS type"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/devicecontext/claims/osversion"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Device OS Version"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/devicecontext/claims/ismanaged"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Is Managed Device"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-forwarded-client-ip"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Forwarded Client IP"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-client-application"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Client Application"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-client-user-agent"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Client User Agent"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-client-ip"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Client IP"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-endpoint-absolute-path"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Endpoint Path"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/requestcontext/claims/x-ms-proxy"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Proxy"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/requestcontext/claims/relyingpartytrustid"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Application Identifier"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/applicationpolicy"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Application policies"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/authoritykeyidentifier"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Authority Key Identifier"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/basicconstraints"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Basic Constraint"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/eku"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Enhanced Key Usage"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/field/issuer"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Issuer"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/field/issuername"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Issuer Name"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/keyusage"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Key Usage"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/field/notafter"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Not After"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/field/notbefore"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Not Before"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/certificatepolicy"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Certificate Policies"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/rsa"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Public Key"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/field/rawdata"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Certificate Raw Data"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/san"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Subject Alternative Name"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2008/06/identity/claims/serialnumber"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Serial Number"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/field/signaturealgorithm"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Signature Algorithm"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/field/subject"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Subject"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/subjectkeyidentifier"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Subject Key Identifier"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/field/subjectname"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Subject Name"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/certificatetemplateinformation"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""V2 Template Name"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/extension/certificatetemplatename"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""V1 Template Name"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/thumbprint"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Thumbprint"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/12/certificatecontext/field/x509version"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""X.509 Version"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2012/01/insidecorporatenetwork"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Inside Corporate Network"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2012/01/passwordexpirationtime"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Password Expiration Time"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2012/01/passwordexpirationdays"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Password Expiration Days"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/ws/2012/01/passwordchangeurl"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Update Password URL"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/claims/authnmethodsreferences"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Authentication Methods References"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/><Attribute Name=""http://schemas.microsoft.com/2012/01/requestcontext/claims/client-request-id"" NameFormat=""urn:oasis:names:tc:SAML:2.0:attrname-format:uri"" FriendlyName=""Client Request ID"" xmlns=""urn:oasis:names:tc:SAML:2.0:assertion""/></IDPSSODescriptor><ContactPerson contactType=""support""/></EntityDescriptor>"; }

        public static string AdfsV4SignatureValue { get => @"aSJ3IUuYvAhF6i8rgU9gPnNr2YS02dWRtKyGYa+8N+dUPOPpoW6bfeVEO0/HnY4Xe/Pf3SL9UFPhIqvup4N/kKs/OOZbKaYn+hitnDPFBQ9FckA0ZecfmgiOofSWo/6Za994a+ECnuSb+QevF53kGjDHbGTgThBTjo1lWgzOtVDVrRS2SmoHwEXpZszC0r+m9Jm7M7Ub4UjyesleHSYPm1Me9Copu+BQLdJO/d2X+VH8uY2VLScWS97h4UL5AKpY78pOrcaQTbbaC7oKeaR/K2FwnWF+TtmTid3yBNEZkixF5GWvLrWkoTK0lFDbksJ0Y1j4zTRBP85uGxx5iprPAw=="; }

        public static string AdfsV4ReferenceDigestValue { get => "MuKCkLnC//j4+3h3S1g+lOowG748ljSAjsHAg1bCpHs="; }

        public static string AdfsV4ReferenceURI { get => "#_b9fe7514-a527-4a24-86ef-8849d3ab008a"; }

        public static WsFederationConfiguration AADCommonEndpoint
        {
            get
            {
                var data1 = new X509Data(X509Certificate1);
                var keyInfo1 = new KeyInfo();
                keyInfo1.X509Data.Add(data1);

                var data2 = new X509Data(X509Certificate2);
                var keyInfo2 = new KeyInfo();
                keyInfo2.X509Data.Add(data2);

                var data3 = new X509Data(X509Certificate3);
                var keyInfo3 = new KeyInfo();
                keyInfo3.X509Data.Add(data3);

                // good configuration for common endpoint
                var configuration = new WsFederationConfiguration()
                {
                    Issuer = IssuerForCommon,
                    Signature = new Signature
                    {
                        KeyInfo = keyInfo1,
                        Prefix = "",
                        SignatureValue = AADCommonMetadataSignatureValue,
                        SignedInfo = AADCommonSignedInfo
                    },
                    TokenEndpoint = TokenEndpointForCommon
                };

                configuration.KeyInfos.Add(keyInfo1);

                configuration.KeyInfos.Add(keyInfo2);

                configuration.KeyInfos.Add(keyInfo3);

                foreach(var key in MetadataSigningKeys)
                    configuration.SigningKeys.Add(key);

                return configuration;
            }
        }

        public static WsFederationConfiguration AADCommonFormated
        {
            get
            {
                var data1 = new X509Data(X509Certificate1);
                var keyInfo1 = new KeyInfo();
                keyInfo1.X509Data.Add(data1);

                var data2 = new X509Data(X509Certificate2);
                var keyInfo2 = new KeyInfo();
                keyInfo2.X509Data.Add(data2);

                var data3 = new X509Data(X509Certificate3);
                var keyInfo3 = new KeyInfo();
                keyInfo3.X509Data.Add(data3);

                var configuration = new WsFederationConfiguration()
                {
                    Issuer = "https://sts.windows.net/268da1a1-9db4-48b9-b1fe-683250ba90cc/",
                    Signature = new Signature(AADCommonSignedInfo)
                    {
                        KeyInfo = keyInfo1,
                        SignatureValue = AADCommonMetadataSignatureValue,
                    },
                    TokenEndpoint = "https://login.microsoftonline.com/268da1a1-9db4-48b9-b1fe-683250ba90cc/wsfed"
                };

                configuration.KeyInfos.Add(keyInfo1);

                configuration.KeyInfos.Add(keyInfo2);

                configuration.KeyInfos.Add(keyInfo3);

                foreach (var key in MetadataSigningKeys)
                    configuration.SigningKeys.Add(key);             

                return configuration;
            }
        }

        public static WsFederationConfiguration AADCommonFormatedNoSignature
        {
            get
            {
                var configuration = AADCommonFormated;
                configuration.Signature = null;
                return configuration;
            }
        }

        public static WsFederationConfiguration AADCommonFormatedNoIssuer
        {
            get
            {
                var configuration = AADCommonFormated;
                configuration.Issuer = null;
                return configuration;
            }
        }

        public static WsFederationConfiguration AADCommonFormatedNoTokenEndpoint
        {
            get
            {
                var configuration = AADCommonFormated;
                configuration.TokenEndpoint = null;
                return configuration;
            }
        }

        public static WsFederationConfiguration AdfsV2Endpoint
        {
            get
            {
                var data = new X509Data(X509CertificateAdfsV2);
                var keyInfo = new KeyInfo();
                keyInfo.X509Data.Add(data);

                var configuration = new WsFederationConfiguration()
                {
                    Issuer = "http://fs.msidlab7.com/adfs/services/trust",
                    Signature = new Signature(AdfsV2SignedInfo)
                    {
                        KeyInfo = keyInfo,
                        SignatureValue = AdfsV2SignatureValue,
                    },
                    TokenEndpoint = "https://fs.msidlab7.com/adfs/ls/"
                };

                configuration.KeyInfos.Add(keyInfo);

                configuration.SigningKeys.Add(AdfsV2MetadataSigningKey);

                return configuration;
            }
        }

        public static WsFederationConfiguration AdfsV3Endpoint
        {
            get
            {
                var data = new X509Data(X509CertificateAdfsV3);
                var keyInfo = new KeyInfo();
                keyInfo.X509Data.Add(data);

                var configuration = new WsFederationConfiguration()
                {
                    Issuer = "http://fs.msidlab2.com/adfs/services/trust",
                    Signature = new Signature(AdfsV3SignedInfo)
                    {
                        KeyInfo = keyInfo,
                        Prefix = "ds",
                        SignatureValue = AdfsV3SignatureValue,
                    },
                    TokenEndpoint = "https://fs.msidlab2.com/adfs/ls/"
                };

                configuration.KeyInfos.Add(keyInfo);

                configuration.SigningKeys.Add(AdfsV3MetadataSigningKey);

                return configuration;
            }
        }

        public static WsFederationConfiguration AdfsV4Endpoint
        {
            get
            {
                var data = new X509Data(X509CertificateAdfsV4);
                var keyInfo = new KeyInfo();
                keyInfo.X509Data.Add(data);

                var configuration = new WsFederationConfiguration
                {
                    Issuer = "http://fs.msidlab11.com/adfs/services/trust",
                    Signature = new Signature(AdfsV4SignedInfo)
                    {
                        KeyInfo = keyInfo,
                        Prefix = "ds",
                        SignatureValue = AdfsV4SignatureValue,
                    },
                    TokenEndpoint = "https://fs.msidlab11.com/adfs/ls/"
                };

                configuration.KeyInfos.Add(keyInfo);

                configuration.SigningKeys.Add(AdfsV4MetadataSigningKey);

                return configuration;
            }
        }

        public static WsFederationConfiguration EmptyEntityDescriptor
        {
            get => new WsFederationConfiguration
            {
                Issuer = "https://DevIDPServer/sts"
            };
        }

        public static WsFederationConfiguration NoRoleDescriptor
        {
            get => new WsFederationConfiguration
            {
                Issuer = "https://sts.windows.net/268da1a1-9db4-48b9-b1fe-683250ba90cc/"
            };
        }

        public static Reference AADCommonReference
        {
            get => new Reference(new EnvelopedSignatureTransform(), new ExclusiveCanonicalizationTransform())
            {
                    DigestMethod = AADCommonMetadataReferenceDigestMethod,
                    DigestValue = AADCommonMetadataReferenceDigestValue,
                    TokenStream = new XmlTokenStream(),
                    Uri = AADCommonMetadataReferenceURI,
            };
        }

        public static SignedInfo AADCommonSignedInfo
        {
            get => new SignedInfo(AADCommonReference)
            {
                CanonicalizationMethod = "http://www.w3.org/2001/10/xml-exc-c14n#",
            };
        }

        public static Reference AdfsV2Reference
        {
            get => new Reference(AADCommonMetadataReferenceEnvelopedTransform, AADCommonMetadataReferenceC14nTransform)
            {
                DigestMethod = AADCommonMetadataReferenceDigestMethod,
                DigestValue = AdfsV2ReferenceDigestValue,
                TokenStream = new XmlTokenStream(),
                Uri = AdfsV2ReferenceURI,
            };
        }

        public static SignedInfo AdfsV2SignedInfo
        {
            get => new SignedInfo(AdfsV2Reference)
            {
                CanonicalizationMethod = "http://www.w3.org/2001/10/xml-exc-c14n#"
            };
        }

        public static Reference AdfsV3Reference
        {
            get => new Reference(AADCommonMetadataReferenceEnvelopedTransform, AADCommonMetadataReferenceC14nTransform)
            {
                DigestMethod = AADCommonMetadataReferenceDigestMethod,
                DigestValue = AdfsV3ReferenceDigestValue,
                Prefix = "ds",
                TokenStream = new XmlTokenStream(),
                Uri = AdfsV3ReferenceURI
            };
        }

        public static SignedInfo AdfsV3SignedInfo
        {
            get => new SignedInfo(AdfsV3Reference)
            {
                CanonicalizationMethod = "http://www.w3.org/2001/10/xml-exc-c14n#",
                Prefix = "ds"
            };
        }

        public static Reference AdfsV4Reference
        {
            get => new Reference(AADCommonMetadataReferenceEnvelopedTransform, AADCommonMetadataReferenceC14nTransform)
            {
                DigestMethod = AADCommonMetadataReferenceDigestMethod,
                DigestValue = AdfsV4ReferenceDigestValue,
                Prefix = "ds",
                TokenStream = new XmlTokenStream(),
                Uri = AdfsV4ReferenceURI
            };
        }

        public static SignedInfo AdfsV4SignedInfo
        {
            get => new SignedInfo(AdfsV4Reference)
            {
                CanonicalizationMethod = "http://www.w3.org/2001/10/xml-exc-c14n#",
                Prefix = "ds"
            };
        }

        public static string X509CertificateKeyId1 { get => "6B740DD01652EECE2737E05DAE36C5D18FCB74C3"; }

        public static string X509CertificateKeyId2 { get => "CF4DFDCDDB05BA2CE905F0552B54E7DB940760ED"; }

        public static string X509CertificateKeyId3 { get => "D92E120951ACF1283D2D2E80A8B22AE83A56FA0F"; }

        public static string X509CertificateKeyIdAdfsV2 { get => "28D1BE71EBAB715A8F53CB9FD9D84C4373CD3708"; }

        public static string X509CertificateKeyIdAdfsV3 { get => "8C3B60F1C93FA3E52AFD41885E7B6C6C4A61C65A"; }

        public static string X509CertificateKeyIdAdfsV4 { get => "D5FE73910389B58BBB3B0EBB87FDF110FF79FEBB"; }

        public static string X509CertificateData1 { get => "MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD"; }

        public static string X509CertificateData2 { get => "MIIDBTCCAe2gAwIBAgIQXxLnqm1cOoVGe62j7W7wZzANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDMyNjAwMDAwMFoXDTE5MDMyNzAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKJGarCm4IF0/Gz5Xx/zyZwD2rdJJZtO2Ukk1Oz+Br1sLVY8I5vj5esB+lotmLEblA9N/w188vmTvykaEzUl49NA4s86x44SW6WtdQbGJ0IjpQJUalUMyy91vIBkK/7K3nBXeVBsRk7tm528leoQ05/aZ+1ycJBIU+1oGYThv8MOjyHAlXJmCaGXwXTisZ+hHjcwlMk/+ZEutHflKLIpPUNEi7j4Xw+zp9UKo5pzWIr/iJ4HjvCkFofW90AMF2xp8dMhpbVcfJGS/Ii3J66LuNLCH/HtSZ42FO+tnRL/nNzzFWUhGT92Q5VFVngfWJ3PAg1zz8I1wowLD2fiB2udGXcCAwEAAaMhMB8wHQYDVR0OBBYEFFXPbFXjmMR3BluF+2MeSXd1NQ3rMA0GCSqGSIb3DQEBCwUAA4IBAQAsd3wGVilJxDtbY1K2oAsWLdNJgmCaYdrtdlAsjGlarSQSzBH0Ybf78fcPX//DYaLXlvaEGKVKp0jPq+RnJ17oP/RJpJTwVXPGRIlZopLIgnKpWlS/PS0uKAdNvLmz1zbGSILdcF+Qf41OozD4QNsS1c9YbDO4vpC9v8x3PVjfJvJwPonzNoOsLXA+8IONSXwCApsnmrwepKu8sifsFYSwgrwxRPGTEAjkdzRJ0yMqiY/VoJ7lqJ/FBJqqAjGPGq/yI9rVoG+mbO1amrIDWHHTKgfbKk0bXGtVUbsayyHR5jSgadmkLBh5AaN/HcgDK/jINrnpiQ+/2ewH/8qLaQ3B"; }

        public static string X509CertificateData3 { get => "MIIDKDCCAhCgAwIBAgIQBHJvVNxP1oZO4HYKh+rypDANBgkqhkiG9w0BAQsFADAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXMwHhcNMTYxMTE2MDgwMDAwWhcNMTgxMTE2MDgwMDAwWjAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXMwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQChn5BCs24Hh6L0BNitPrV5s+2/DBhaeytOmnghJKnqeJlhv3ZczShRM2Cp38LW8Y3wn7L3AJtolaSkF/joKN1l6GupzM+HOEdq7xZxFehxIHW7+25mG/WigBnhsBzLv1SR4uIbrQeS5M0kkLwJ9pOnVH3uzMGG6TRXPnK3ivlKl97AiUEKdlRjCQNLXvYf1ZqlC77c/ZCOHSX4kvIKR2uG+LNlSTRq2rn8AgMpFT4DSlEZz4RmFQvQupQzPpzozaz/gadBpJy/jgDmJlQMPXkHp7wClvbIBGiGRaY6eZFxNV96zwSR/GPNkTObdw2S8/SiAgvIhIcqWTPLY6aVTqJfAgMBAAGjWDBWMFQGA1UdAQRNMEuAEDUj0BrjP0RTbmoRPTRMY3WhJTAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXOCEARyb1TcT9aGTuB2Cofq8qQwDQYJKoZIhvcNAQELBQADggEBAGnLhDHVz2gLDiu9L34V3ro/6xZDiSWhGyHcGqky7UlzQH3pT5so8iF5P0WzYqVtogPsyC2LPJYSTt2vmQugD4xlu/wbvMFLcV0hmNoTKCF1QTVtEQiAiy0Aq+eoF7Al5fV1S3Sune0uQHimuUFHCmUuF190MLcHcdWnPAmzIc8fv7quRUUsExXmxSX2ktUYQXzqFyIOSnDCuWFm6tpfK5JXS8fW5bpqTlrysXXz/OW/8NFGq/alfjrya4ojrOYLpunGriEtNPwK7hxj1AlCYEWaRHRXaUIW1ByoSff/6Y6+ZhXPUe0cDlNRt/qIz5aflwO7+W8baTS4O8m/icu7ItE="; }

        public static string X509CertificateDataAdfsV2 { get => @"MIIC2jCCAcKgAwIBAgIQFuP8MbE4ipBGdYQ2Wmg6FDANBgkqhkiG9w0BAQsFADApMScwJQYDVQQDEx5BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiNy5jb20wHhcNMTYxMjAzMDIzNjEwWhcNMTcxMjAzMDIzNjEwWjApMScwJQYDVQQDEx5BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiNy5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCQXc06bWagsWphlP+yFdK/6YPZNvYC5BSzV2bmbYp76YCzpDAaYTBL6toAgjMBoTWHeeO/K8GpKxjK5RzBCWJL0j6Ao+GLGiX430VEvWCxCVbi4H3GY7T2MUangLIBEbutFbgcKxSdLIG9KEUKZXZSnKxx4W6b/m56dswPwpPJ536KWgjovNKy+/XfAONFkg4Rj6HvFN7ylNSBwWfcFdw+C8aZn4swW8msLMm4ridHxR0jnHgqrJ8RQjBQfcWaCuS18EaJxVrCN13OnDNa1SnQqKdePcczeeL6NRK4luxCTNJ6pJ8GLSxNA5rxbv3uMZ0VcaANLFp0KZDJ7xr3oTEdAgMBAAEwDQYJKoZIhvcNAQELBQADggEBAE3plOFUXZ/O/9pEF4ZiBvjttxXz/ntaKlC9WDZQaysjwAvwwQf6lCWBYrOn9muQrDeAriYOZHxRXxzwdqUgHRUQ6NZxNQUgc+ElH1v+E25ns8vP8yS8U9FhQUU8IfkLYcOt3bY0PcGCJhKsHSJoHrsKqQqWWQN4f48ocrLOdAhS0VLTxpa84i8mGNO9Iaavdd6g0vfC0dgBaBRyWDO0MAywiqx1LsxOXBidxn/Z4bm/2+lcAnRXALowifDPkDvIkx+zjU2YvcYWDiNl3sei3NfS68sYycGMaCmwXUyjQy13FMrOuEo2KqMmWHPMyZbbEyVBsytq5+W1Rx+2ElWAzJg="; }

        public static string X509CertificateDataAdfsV3 { get => @"MIIC2jCCAcKgAwIBAgIQE4Ec5zS+mZ1CjdwUbnFNejANBgkqhkiG9w0BAQsFADApMScwJQYDVQQDEx5BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiMi5jb20wHhcNMTcwMzEzMTgxMTM0WhcNMTgwMzEzMTgxMTM0WjApMScwJQYDVQQDEx5BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiMi5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCYh9E/B76B0jUBedTJCBXYkXd3NVav5GFOI43Cu3mcnQvT82112PY7TSW/GTOJL1oVhAyLR6nNjPKD0f7J+3gkrXYpvDK8g69GWI5Gy3ObA95VhzpWKwqnRJrblJfYvQtiA+dQTGH+x5tSbHieGni71lqanCwGMP4PnLKTAGtRXEm1Yz/l0k57PTiKBybIlhW5YUm/Nl8d1qlp+eQdM/bhkj9imnANghSAcXU//6zeioG63ad0COltc5Eh4bMUNBGdx15gMyu40T2TDojeYIS+3rkYoRmQHJ9LoHt43DgqPBp8K+bdT/5gVUnRcRv0hpTi5JhPyqWm3h4CAJb2NNr7AgMBAAEwDQYJKoZIhvcNAQELBQADggEBAAcvWybsF8Tt/FRahhgWlONayLorboAD3qxUOfzaBQS32Q6AUPWatQCJX6pRC9tLtQr2PdJ4hDWfcGmMuFUzArFqBfAFFvb5CTlplNKQw53iZhEdvKEhY77d+j4BPpHmIvHDq9dshl0oPWV5ywiWJfBmf/6I7DRffXtBkj3IpuVQWrtOycZHsW1ymsURCQEr/YlqPlOCtoMCfl6wvOR/zLF2/sheVBgrwJTVCljBP4DENwK6he3dioyNM9k9ITa0Z2SpXu2+pHJopRc94H+KJG/LTX0y+Fq6bh54/U5ULE/OzDzKRLqJETqOvWNWaMw/tGUTo6gGr4KAbhel/PTIztQ="; }

        public static string X509CertificateDataAdfsV4 { get => @"MIIC3DCCAcSgAwIBAgIQR6dE8BwieKVBETXJ7bFh1TANBgkqhkiG9w0BAQsFADAqMSgwJgYDVQQDEx9BREZTIFNpZ25pbmcgLSBmcy5tc2lkbGFiMTEuY29tMB4XDTE3MDEyMzIxMjgzOVoXDTE4MDEyMzIxMjgzOVowKjEoMCYGA1UEAxMfQURGUyBTaWduaW5nIC0gZnMubXNpZGxhYjExLmNvbTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBANEGVDA2PatZq7yN/F9UrvheLNq6FiSzJdJE3Z55SPooQpl5Oi9dBO9vKDmnTBUB+Ex8h2X9Shpp6h1+Sir014Nkn8VHSfKqV7aVUSE0PxBOCCKakwdISUo7NA9g7VCwUXP6Z6nghxcJr7FF1g5/MwZE22tIu4sivJSsskA8J5lm5O9LzP7TLiRlMmD8qE2hYl55A5OE7mhOcz/2JWV+V7VxIuvl108oRNfD1DFwdNRipQYM1H7mgjHveR6IP1+F04rD+tGBjbwekA4BWMkiZFIm3WLJizsTfLcaX3HyEaf9fOYCiQIHC/PYsN4zHmfKPcP44/StBLZhYqsZh/SZp0UCAwEAATANBgkqhkiG9w0BAQsFAAOCAQEAJEy4+oJEfUI5/g0f11r2M+Aa0BlZcd+BenZ9yGQRFSzfWa6XPM4eh9Kxn3XJoqm3L1ergqeSbHcRn6cJ9ge+2DQkl8N37i8QTrsAdZl5OKcxzk313GBmiKaMEx5VgoB8rWsZqNdbh5mJCQeHK07MvAesIVU2Rk5G3BZz3LRSUo7d9asBPfHg37cR9Ol4iuyQGu93MAlEhWBRbAaT7Km942972kYh2uPc/SjdcK8VKk3JJ+Kyp0QrxedpFd8iDy/KQdL4h+1VuR8eKu+znxS/CqpidysP9rwcwsJxGKvWmSgeuMbgxIBLdm1FUQsYAa0zbNkjRNh1n/+XdFVUprZKsw=="; }

        public static string Issuer { get => @"https://sts.windows.net/268da1a1-9db4-48b9-b1fe-683250ba90cc/"; }

        public static string IssuerForCommon { get => @"https://sts.windows.net/{tenantid}/"; }

        public static string TokenEndpoint { get => @"https://login.microsoftonline.com/268da1a1-9db4-48b9-b1fe-683250ba90cc/wsfed"; }

        public static string TokenEndpointForCommon { get => @"https://login.microsoftonline.com/common/wsfed"; }

        public static string KeyDescriptorNoKeyUse
        {
            get
            {
                return
                @"<KeyDescriptor xmlns=""urn:oasis:names:tc:SAML:2.0:metadata"">
                   <KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                     <X509Data>
                       <X509Certificate>
                         MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD
                       </X509Certificate>
                     </X509Data>
                    </KeyInfo>
                  </KeyDescriptor>";
            }
        }

        public static string KeyDescriptorKeyUseNotForSigning
        {
            get
            {
                return
                @"<KeyDescriptor use=""Not for signing"" xmlns=""urn:oasis:names:tc:SAML:2.0:metadata"">
                   <KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                     <X509Data>
                       <X509Certificate>
                         MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD
                       </X509Certificate>
                     </X509Data>
                    </KeyInfo>
                  </KeyDescriptor>";
            }
        }

        public static string EmptyEntityDescriptorMetadata = @"<EntityDescriptor xmlns=""urn:oasis:names:tc:SAML:2.0:metadata"" ID=""_6c4f3672-45c2-47a6-9515-afda95224009"" entityID=""https://sts.windows.net/268da1a1-9db4-48b9-b1fe-683250ba90cc/""/>";

        public static string EmptyKeyDescriptor = @"<KeyDescriptor use=""signing"" xmlns=""urn:oasis:names:tc:SAML:2.0:metadata""/>";

        public static string EmptyRoleDescriptor = @"<RoleDescriptor xmlns=""urn:oasis:names:tc:SAML:2.0:metadata"" xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706"" xsi:type=""fed:SecurityTokenServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/wsfed/federation/200706""/>";

        public static string EmptyPassiveRequestorEndpoint = @"<PassiveRequestorEndpoint xmlns=""http://docs.oasis-open.org/wsfed/federation/200706""/>";

        public static string EndpointWithEmptyEndpointReference = @"<PassiveRequestorEndpoint xmlns=""http://docs.oasis-open.org/wsfed/federation/200706"">
                      <wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing""/>
                    </PassiveRequestorEndpoint>";

        public static string EndpointWithEmptyAddress = @"<PassiveRequestorEndpoint xmlns=""http://docs.oasis-open.org/wsfed/federation/200706"">
                      <wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing"">
                        <wsa:Address/>
                      </wsa:EndpointReference>
                    </PassiveRequestorEndpoint>";

        public static string AADCommonMetadataFormated
        {
            get
            {
                return
                @"<EntityDescriptor xmlns=""urn:oasis:names:tc:SAML:2.0:metadata"" ID=""_6c4f3672-45c2-47a6-9515-afda95224009"" entityID=""https://sts.windows.net/268da1a1-9db4-48b9-b1fe-683250ba90cc/"">
                  <Signature xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                    <SignedInfo>
                      <CanonicalizationMethod Algorithm=""http://www.w3.org/2001/10/xml-exc-c14n#""/>
                      <SignatureMethod Algorithm=""http://www.w3.org/2001/04/xmldsig-more#rsa-sha256"" />
                      <Reference URI=""#_0ded55d8-a72f-4e13-ab9e-f40be80b1476"">
                        <Transforms>
                          <Transform Algorithm=""http://www.w3.org/2000/09/xmldsig#enveloped-signature"" />
                          <Transform Algorithm=""http://www.w3.org/2001/10/xml-exc-c14n#""/>
                        </Transforms>
                        <DigestMethod Algorithm=""http://www.w3.org/2001/04/xmlenc#sha256"" />
                        <DigestValue>lnicj3SNizOF6QI1rWw8JrouoiXMslqtFB5ztWW6qvs=</DigestValue>
                      </Reference>
                    </SignedInfo>
                    <SignatureValue>KD9uWOD/9pvF1NlNCpYoXymUPS1l9uIBgBDe0uOQgQv+tUI/1jJX4UpjADDHCOx6HCl5ZgZSXNmOC2lLSJEwmv21BZzI+PAOxF5hdH99cS/lMC/hxgyWdLVeGnr1I4WbPxGqVmjFNuBdBMaourO4z/5f3D2JZQmgnlu8H+4gv2SpjeZz/YhIN6ZrNfmHwsKZashMGtSmE5uHro+uO5yO17Gr9YfUbtokLRIq5Dk9kqnxG8YZF1C1nC9O0PMdlHb4ubwgO20Cvz5sU2iswn9m68btS5TLF5OVhETzyKir1QA+H1tCgGRqIWd4Geyoucdct1r4zAJGCNIekdKnY3NXwg==</SignatureValue>
                    <KeyInfo>
                      <X509Data>
                        <X509Certificate>MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD</X509Certificate>
                      </X509Data>
                    </KeyInfo>
                  </Signature>
                  <RoleDescriptor xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706"" xsi:type=""fed:SecurityTokenServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/wsfed/federation/200706"">
                    <KeyDescriptor use=""signing"">
                      <KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                        <X509Data>
                          <X509Certificate>MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD</X509Certificate>
                        </X509Data>
                      </KeyInfo>
                    </KeyDescriptor>
                    <KeyDescriptor use=""signing"">
                      <KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                        <X509Data>
                          <X509Certificate>MIIDBTCCAe2gAwIBAgIQXxLnqm1cOoVGe62j7W7wZzANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDMyNjAwMDAwMFoXDTE5MDMyNzAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKJGarCm4IF0/Gz5Xx/zyZwD2rdJJZtO2Ukk1Oz+Br1sLVY8I5vj5esB+lotmLEblA9N/w188vmTvykaEzUl49NA4s86x44SW6WtdQbGJ0IjpQJUalUMyy91vIBkK/7K3nBXeVBsRk7tm528leoQ05/aZ+1ycJBIU+1oGYThv8MOjyHAlXJmCaGXwXTisZ+hHjcwlMk/+ZEutHflKLIpPUNEi7j4Xw+zp9UKo5pzWIr/iJ4HjvCkFofW90AMF2xp8dMhpbVcfJGS/Ii3J66LuNLCH/HtSZ42FO+tnRL/nNzzFWUhGT92Q5VFVngfWJ3PAg1zz8I1wowLD2fiB2udGXcCAwEAAaMhMB8wHQYDVR0OBBYEFFXPbFXjmMR3BluF+2MeSXd1NQ3rMA0GCSqGSIb3DQEBCwUAA4IBAQAsd3wGVilJxDtbY1K2oAsWLdNJgmCaYdrtdlAsjGlarSQSzBH0Ybf78fcPX//DYaLXlvaEGKVKp0jPq+RnJ17oP/RJpJTwVXPGRIlZopLIgnKpWlS/PS0uKAdNvLmz1zbGSILdcF+Qf41OozD4QNsS1c9YbDO4vpC9v8x3PVjfJvJwPonzNoOsLXA+8IONSXwCApsnmrwepKu8sifsFYSwgrwxRPGTEAjkdzRJ0yMqiY/VoJ7lqJ/FBJqqAjGPGq/yI9rVoG+mbO1amrIDWHHTKgfbKk0bXGtVUbsayyHR5jSgadmkLBh5AaN/HcgDK/jINrnpiQ+/2ewH/8qLaQ3B</X509Certificate>
                        </X509Data>
                      </KeyInfo>
                    </KeyDescriptor>
                    <KeyDescriptor use=""signing"">
                      <KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                        <X509Data>
                          <X509Certificate>MIIDKDCCAhCgAwIBAgIQBHJvVNxP1oZO4HYKh+rypDANBgkqhkiG9w0BAQsFADAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXMwHhcNMTYxMTE2MDgwMDAwWhcNMTgxMTE2MDgwMDAwWjAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXMwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQChn5BCs24Hh6L0BNitPrV5s+2/DBhaeytOmnghJKnqeJlhv3ZczShRM2Cp38LW8Y3wn7L3AJtolaSkF/joKN1l6GupzM+HOEdq7xZxFehxIHW7+25mG/WigBnhsBzLv1SR4uIbrQeS5M0kkLwJ9pOnVH3uzMGG6TRXPnK3ivlKl97AiUEKdlRjCQNLXvYf1ZqlC77c/ZCOHSX4kvIKR2uG+LNlSTRq2rn8AgMpFT4DSlEZz4RmFQvQupQzPpzozaz/gadBpJy/jgDmJlQMPXkHp7wClvbIBGiGRaY6eZFxNV96zwSR/GPNkTObdw2S8/SiAgvIhIcqWTPLY6aVTqJfAgMBAAGjWDBWMFQGA1UdAQRNMEuAEDUj0BrjP0RTbmoRPTRMY3WhJTAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXOCEARyb1TcT9aGTuB2Cofq8qQwDQYJKoZIhvcNAQELBQADggEBAGnLhDHVz2gLDiu9L34V3ro/6xZDiSWhGyHcGqky7UlzQH3pT5so8iF5P0WzYqVtogPsyC2LPJYSTt2vmQugD4xlu/wbvMFLcV0hmNoTKCF1QTVtEQiAiy0Aq+eoF7Al5fV1S3Sune0uQHimuUFHCmUuF190MLcHcdWnPAmzIc8fv7quRUUsExXmxSX2ktUYQXzqFyIOSnDCuWFm6tpfK5JXS8fW5bpqTlrysXXz/OW/8NFGq/alfjrya4ojrOYLpunGriEtNPwK7hxj1AlCYEWaRHRXaUIW1ByoSff/6Y6+ZhXPUe0cDlNRt/qIz5aflwO7+W8baTS4O8m/icu7ItE=</X509Certificate>
                        </X509Data>
                      </KeyInfo>
                    </KeyDescriptor>
                    <fed:ClaimTypesOffered>
                      <auth:ClaimType xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706"" Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/name"">
                        <auth:DisplayName>Name</auth:DisplayName>
                        <auth:Description>The mutable display name of the user.</auth:Description>
                      </auth:ClaimType>
                      <auth:ClaimType xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706"" Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/nameidentifier"">
                        <auth:DisplayName>Subject</auth:DisplayName>
                        <auth:Description>
                          An immutable, globally unique, non-reusable identifier of the user that is unique to the application for which a token is issued.
                        </auth:Description>
                      </auth:ClaimType>
                      <auth:ClaimType xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706"" Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/givenname"">
                        <auth:DisplayName>Given Name</auth:DisplayName>
                        <auth:Description>First name of the user.</auth:Description>
                      </auth:ClaimType>
                      <auth:ClaimType xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706"" Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/surname"">
                        <auth:DisplayName>Surname</auth:DisplayName>
                        <auth:Description>Last name of the user.</auth:Description>
                      </auth:ClaimType>
                      <auth:ClaimType xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706"" Uri=""http://schemas.microsoft.com/identity/claims/displayname"">
                        <auth:DisplayName>Display Name</auth:DisplayName>
                        <auth:Description>Display name of the user.</auth:Description>
                      </auth:ClaimType>
                      <auth:ClaimType xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706"" Uri=""http://schemas.microsoft.com/identity/claims/nickname"">
                        <auth:DisplayName>Nick Name</auth:DisplayName>
                        <auth:Description>Nick name of the user.</auth:Description>
                      </auth:ClaimType>
                      <auth:ClaimType xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706"" Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/authenticationinstant"">
                        <auth:DisplayName>Authentication Instant</auth:DisplayName>
                        <auth:Description>
                          The time (UTC) when the user is authenticated to Windows Azure Active Directory.
                        </auth:Description>
                      </auth:ClaimType>
                      <auth:ClaimType xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706"" Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/authenticationmethod"">
                        <auth:DisplayName>Authentication Method</auth:DisplayName>
                        <auth:Description>
                          The method that Windows Azure Active Directory uses to authenticate users.
                        </auth:Description>
                      </auth:ClaimType>
                      <auth:ClaimType xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706"" Uri=""http://schemas.microsoft.com/identity/claims/objectidentifier"">
                        <auth:DisplayName>ObjectIdentifier</auth:DisplayName>
                        <auth:Description>
                          Primary identifier for the user in the directory. Immutable, globally unique, non-reusable.
                        </auth:Description>
                      </auth:ClaimType>
                      <auth:ClaimType xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706"" Uri=""http://schemas.microsoft.com/identity/claims/tenantid"">
                        <auth:DisplayName>TenantId</auth:DisplayName>
                        <auth:Description>Identifier for the user's tenant.</auth:Description>
                      </auth:ClaimType>
                      <auth:ClaimType xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706"" Uri=""http://schemas.microsoft.com/identity/claims/identityprovider"">
                        <auth:DisplayName>IdentityProvider</auth:DisplayName>
                        <auth:Description>Identity provider for the user.</auth:Description>
                      </auth:ClaimType>
                      <auth:ClaimType xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706"" Uri=""http://schemas.xmlsoap.org/ws/2005/05/identity/claims/emailaddress"">
                        <auth:DisplayName>Email</auth:DisplayName>
                        <auth:Description>Email address of the user.</auth:Description>
                      </auth:ClaimType>
                      <auth:ClaimType xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706"" Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/groups"">
                        <auth:DisplayName>Groups</auth:DisplayName>
                        <auth:Description>Groups of the user.</auth:Description>
                      </auth:ClaimType>
                      <auth:ClaimType xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706"" Uri=""http://schemas.microsoft.com/identity/claims/accesstoken"">
                        <auth:DisplayName>External Access Token</auth:DisplayName>
                        <auth:Description>Access token issued by external identity provider.</auth:Description>
                      </auth:ClaimType>
                      <auth:ClaimType xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706"" Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/expiration"">
                        <auth:DisplayName>External Access Token Expiration</auth:DisplayName>
                        <auth:Description>
                          UTC expiration time of access token issued by external identity provider.
                        </auth:Description>
                      </auth:ClaimType>
                      <auth:ClaimType xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706"" Uri=""http://schemas.microsoft.com/identity/claims/openid2_id"">
                        <auth:DisplayName>External OpenID 2.0 Identifier</auth:DisplayName>
                        <auth:Description>
                          OpenID 2.0 identifier issued by external identity provider.
                        </auth:Description>
                      </auth:ClaimType>
                      <auth:ClaimType xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706"" Uri=""http://schemas.microsoft.com/claims/groups.link"">
                        <auth:DisplayName>GroupsOverageClaim</auth:DisplayName>
                        <auth:Description>
                          Issued when number of user's group claims exceeds return limit.
                        </auth:Description>
                      </auth:ClaimType>
                      <auth:ClaimType xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706"" Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/role"">
                        <auth:DisplayName>Role Claim</auth:DisplayName>
                        <auth:Description>
                          Roles that the user or Service Principal is attached to
                        </auth:Description>
                      </auth:ClaimType>
                      <auth:ClaimType xmlns:auth=""http://docs.oasis-open.org/wsfed/authorization/200706"" Uri=""http://schemas.microsoft.com/ws/2008/06/identity/claims/wids"">
                        <auth:DisplayName>RoleTemplate Id Claim</auth:DisplayName>
                        <auth:Description>
                          Role template id of the Built-in Directory Roles that the user is a member of
                        </auth:Description>
                      </auth:ClaimType>
                    </fed:ClaimTypesOffered>
                    <fed:SecurityTokenServiceEndpoint>
                      <wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing"">
                        <wsa:Address>https://login.microsoftonline.com/268da1a1-9db4-48b9-b1fe-683250ba90cc/wsfed</wsa:Address>
                      </wsa:EndpointReference>
                    </fed:SecurityTokenServiceEndpoint>
                    <fed:PassiveRequestorEndpoint>
                      <wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing"">
                        <wsa:Address>https://login.microsoftonline.com/268da1a1-9db4-48b9-b1fe-683250ba90cc/wsfed</wsa:Address>
                      </wsa:EndpointReference>
                    </fed:PassiveRequestorEndpoint>
                  </RoleDescriptor>
                  <RoleDescriptor xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706"" xsi:type=""fed:ApplicationServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/wsfed/federation/200706"">
                    <KeyDescriptor use=""signing"">
                      <KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                        <X509Data>
                          <X509Certificate>MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD</X509Certificate>
                        </X509Data>
                      </KeyInfo>
                    </KeyDescriptor>
                    <KeyDescriptor use=""signing"">
                      <KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                        <X509Data>
                          <X509Certificate>MIIDBTCCAe2gAwIBAgIQXxLnqm1cOoVGe62j7W7wZzANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDMyNjAwMDAwMFoXDTE5MDMyNzAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKJGarCm4IF0/Gz5Xx/zyZwD2rdJJZtO2Ukk1Oz+Br1sLVY8I5vj5esB+lotmLEblA9N/w188vmTvykaEzUl49NA4s86x44SW6WtdQbGJ0IjpQJUalUMyy91vIBkK/7K3nBXeVBsRk7tm528leoQ05/aZ+1ycJBIU+1oGYThv8MOjyHAlXJmCaGXwXTisZ+hHjcwlMk/+ZEutHflKLIpPUNEi7j4Xw+zp9UKo5pzWIr/iJ4HjvCkFofW90AMF2xp8dMhpbVcfJGS/Ii3J66LuNLCH/HtSZ42FO+tnRL/nNzzFWUhGT92Q5VFVngfWJ3PAg1zz8I1wowLD2fiB2udGXcCAwEAAaMhMB8wHQYDVR0OBBYEFFXPbFXjmMR3BluF+2MeSXd1NQ3rMA0GCSqGSIb3DQEBCwUAA4IBAQAsd3wGVilJxDtbY1K2oAsWLdNJgmCaYdrtdlAsjGlarSQSzBH0Ybf78fcPX//DYaLXlvaEGKVKp0jPq+RnJ17oP/RJpJTwVXPGRIlZopLIgnKpWlS/PS0uKAdNvLmz1zbGSILdcF+Qf41OozD4QNsS1c9YbDO4vpC9v8x3PVjfJvJwPonzNoOsLXA+8IONSXwCApsnmrwepKu8sifsFYSwgrwxRPGTEAjkdzRJ0yMqiY/VoJ7lqJ/FBJqqAjGPGq/yI9rVoG+mbO1amrIDWHHTKgfbKk0bXGtVUbsayyHR5jSgadmkLBh5AaN/HcgDK/jINrnpiQ+/2ewH/8qLaQ3B</X509Certificate>
                        </X509Data>
                      </KeyInfo>
                    </KeyDescriptor>
                    <KeyDescriptor use=""signing"">
                      <KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                        <X509Data>
                          <X509Certificate>MIIDKDCCAhCgAwIBAgIQBHJvVNxP1oZO4HYKh+rypDANBgkqhkiG9w0BAQsFADAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXMwHhcNMTYxMTE2MDgwMDAwWhcNMTgxMTE2MDgwMDAwWjAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXMwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQChn5BCs24Hh6L0BNitPrV5s+2/DBhaeytOmnghJKnqeJlhv3ZczShRM2Cp38LW8Y3wn7L3AJtolaSkF/joKN1l6GupzM+HOEdq7xZxFehxIHW7+25mG/WigBnhsBzLv1SR4uIbrQeS5M0kkLwJ9pOnVH3uzMGG6TRXPnK3ivlKl97AiUEKdlRjCQNLXvYf1ZqlC77c/ZCOHSX4kvIKR2uG+LNlSTRq2rn8AgMpFT4DSlEZz4RmFQvQupQzPpzozaz/gadBpJy/jgDmJlQMPXkHp7wClvbIBGiGRaY6eZFxNV96zwSR/GPNkTObdw2S8/SiAgvIhIcqWTPLY6aVTqJfAgMBAAGjWDBWMFQGA1UdAQRNMEuAEDUj0BrjP0RTbmoRPTRMY3WhJTAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXOCEARyb1TcT9aGTuB2Cofq8qQwDQYJKoZIhvcNAQELBQADggEBAGnLhDHVz2gLDiu9L34V3ro/6xZDiSWhGyHcGqky7UlzQH3pT5so8iF5P0WzYqVtogPsyC2LPJYSTt2vmQugD4xlu/wbvMFLcV0hmNoTKCF1QTVtEQiAiy0Aq+eoF7Al5fV1S3Sune0uQHimuUFHCmUuF190MLcHcdWnPAmzIc8fv7quRUUsExXmxSX2ktUYQXzqFyIOSnDCuWFm6tpfK5JXS8fW5bpqTlrysXXz/OW/8NFGq/alfjrya4ojrOYLpunGriEtNPwK7hxj1AlCYEWaRHRXaUIW1ByoSff/6Y6+ZhXPUe0cDlNRt/qIz5aflwO7+W8baTS4O8m/icu7ItE=</X509Certificate>
                        </X509Data>
                      </KeyInfo>
                    </KeyDescriptor>
                    <fed:TargetScopes>
                      <wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing"">
                        <wsa:Address>https://sts.windows.net/268da1a1-9db4-48b9-b1fe-683250ba90cc/</wsa:Address>
                      </wsa:EndpointReference>
                    </fed:TargetScopes>
                    <fed:ApplicationServiceEndpoint>
                      <wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing"">
                        <wsa:Address>https://login.microsoftonline.com/268da1a1-9db4-48b9-b1fe-683250ba90cc/wsfed</wsa:Address>
                      </wsa:EndpointReference>
                    </fed:ApplicationServiceEndpoint>
                    <fed:PassiveRequestorEndpoint>
                      <wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing"">
                        <wsa:Address>https://login.microsoftonline.com/268da1a1-9db4-48b9-b1fe-683250ba90cc/wsfed</wsa:Address>
                      </wsa:EndpointReference>
                    </fed:PassiveRequestorEndpoint>
                  </RoleDescriptor>
                  <IDPSSODescriptor protocolSupportEnumeration=""urn:oasis:names:tc:SAML:2.0:protocol"">
                    <KeyDescriptor use=""signing"">
                      <KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                        <X509Data>
                          <X509Certificate>MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD</X509Certificate>
                        </X509Data>
                      </KeyInfo>
                    </KeyDescriptor>
                    <KeyDescriptor use=""signing"">
                      <KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                        <X509Data>
                          <X509Certificate>MIIDBTCCAe2gAwIBAgIQXxLnqm1cOoVGe62j7W7wZzANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDMyNjAwMDAwMFoXDTE5MDMyNzAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKJGarCm4IF0/Gz5Xx/zyZwD2rdJJZtO2Ukk1Oz+Br1sLVY8I5vj5esB+lotmLEblA9N/w188vmTvykaEzUl49NA4s86x44SW6WtdQbGJ0IjpQJUalUMyy91vIBkK/7K3nBXeVBsRk7tm528leoQ05/aZ+1ycJBIU+1oGYThv8MOjyHAlXJmCaGXwXTisZ+hHjcwlMk/+ZEutHflKLIpPUNEi7j4Xw+zp9UKo5pzWIr/iJ4HjvCkFofW90AMF2xp8dMhpbVcfJGS/Ii3J66LuNLCH/HtSZ42FO+tnRL/nNzzFWUhGT92Q5VFVngfWJ3PAg1zz8I1wowLD2fiB2udGXcCAwEAAaMhMB8wHQYDVR0OBBYEFFXPbFXjmMR3BluF+2MeSXd1NQ3rMA0GCSqGSIb3DQEBCwUAA4IBAQAsd3wGVilJxDtbY1K2oAsWLdNJgmCaYdrtdlAsjGlarSQSzBH0Ybf78fcPX//DYaLXlvaEGKVKp0jPq+RnJ17oP/RJpJTwVXPGRIlZopLIgnKpWlS/PS0uKAdNvLmz1zbGSILdcF+Qf41OozD4QNsS1c9YbDO4vpC9v8x3PVjfJvJwPonzNoOsLXA+8IONSXwCApsnmrwepKu8sifsFYSwgrwxRPGTEAjkdzRJ0yMqiY/VoJ7lqJ/FBJqqAjGPGq/yI9rVoG+mbO1amrIDWHHTKgfbKk0bXGtVUbsayyHR5jSgadmkLBh5AaN/HcgDK/jINrnpiQ+/2ewH/8qLaQ3B</X509Certificate>
                        </X509Data>
                      </KeyInfo>
                    </KeyDescriptor>
                    <KeyDescriptor use=""signing"">
                      <KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                        <X509Data>
                          <X509Certificate>MIIDKDCCAhCgAwIBAgIQBHJvVNxP1oZO4HYKh+rypDANBgkqhkiG9w0BAQsFADAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXMwHhcNMTYxMTE2MDgwMDAwWhcNMTgxMTE2MDgwMDAwWjAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXMwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQChn5BCs24Hh6L0BNitPrV5s+2/DBhaeytOmnghJKnqeJlhv3ZczShRM2Cp38LW8Y3wn7L3AJtolaSkF/joKN1l6GupzM+HOEdq7xZxFehxIHW7+25mG/WigBnhsBzLv1SR4uIbrQeS5M0kkLwJ9pOnVH3uzMGG6TRXPnK3ivlKl97AiUEKdlRjCQNLXvYf1ZqlC77c/ZCOHSX4kvIKR2uG+LNlSTRq2rn8AgMpFT4DSlEZz4RmFQvQupQzPpzozaz/gadBpJy/jgDmJlQMPXkHp7wClvbIBGiGRaY6eZFxNV96zwSR/GPNkTObdw2S8/SiAgvIhIcqWTPLY6aVTqJfAgMBAAGjWDBWMFQGA1UdAQRNMEuAEDUj0BrjP0RTbmoRPTRMY3WhJTAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXOCEARyb1TcT9aGTuB2Cofq8qQwDQYJKoZIhvcNAQELBQADggEBAGnLhDHVz2gLDiu9L34V3ro/6xZDiSWhGyHcGqky7UlzQH3pT5so8iF5P0WzYqVtogPsyC2LPJYSTt2vmQugD4xlu/wbvMFLcV0hmNoTKCF1QTVtEQiAiy0Aq+eoF7Al5fV1S3Sune0uQHimuUFHCmUuF190MLcHcdWnPAmzIc8fv7quRUUsExXmxSX2ktUYQXzqFyIOSnDCuWFm6tpfK5JXS8fW5bpqTlrysXXz/OW/8NFGq/alfjrya4ojrOYLpunGriEtNPwK7hxj1AlCYEWaRHRXaUIW1ByoSff/6Y6+ZhXPUe0cDlNRt/qIz5aflwO7+W8baTS4O8m/icu7ItE=</X509Certificate>
                        </X509Data>
                      </KeyInfo>
                    </KeyDescriptor>
                    <SingleLogoutService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect"" Location=""https://login.microsoftonline.com/268da1a1-9db4-48b9-b1fe-683250ba90cc/saml2"" />
                    <SingleSignOnService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect"" Location=""https://login.microsoftonline.com/268da1a1-9db4-48b9-b1fe-683250ba90cc/saml2"" />
                    <SingleSignOnService Binding=""urn:oasis:names:tc:SAML:2.0:bindings:HTTP-POST"" Location=""https://login.microsoftonline.com/268da1a1-9db4-48b9-b1fe-683250ba90cc/saml2"" />
                  </IDPSSODescriptor>
                </EntityDescriptor>";
            }
        }

        public static string MetadataEmptyEntityDescriptor
        {
            get => @"<?xml version=""1.0""?><md:EntityDescriptor xmlns:md=""urn:oasis:names:tc:SAML:2.0:metadata"" entityID=""https://DevIDPServer/sts""/>";
        }

        public static string MetadataEmptyPassiveRequestorEndpoint
        {
            get
            {
                return
                @"<EntityDescriptor xmlns=""urn:oasis:names:tc:SAML:2.0:metadata"" ID=""_6c4f3672-45c2-47a6-9515-afda95224009"" entityID=""https://sts.windows.net/268da1a1-9db4-48b9-b1fe-683250ba90cc/"">
                  <RoleDescriptor xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706"" xsi:type=""fed:SecurityTokenServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/wsfed/federation/200706"">
                    <fed:PassiveRequestorEndpoint />
                  </RoleDescriptor>
                </EntityDescriptor>";
            }
        }

        public static string MetadataEmptyEndpointReference
        {
            get
            {
                return
                @"<EntityDescriptor xmlns=""urn:oasis:names:tc:SAML:2.0:metadata"" ID=""_6c4f3672-45c2-47a6-9515-afda95224009"" entityID=""https://sts.windows.net/268da1a1-9db4-48b9-b1fe-683250ba90cc/"">
                  <RoleDescriptor xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706"" xsi:type=""fed:SecurityTokenServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/wsfed/federation/200706"">
                    <fed:PassiveRequestorEndpoint>
                      <wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing"" />
                    </fed:PassiveRequestorEndpoint>
                  </_RoleDescriptor>
                </EntityDescriptor>";
            }
        }

        public static string MetadataEmptyEndpointAddress
        {
            get
            {
                return
                @"<EntityDescriptor xmlns=""urn:oasis:names:tc:SAML:2.0:metadata"" ID=""_6c4f3672-45c2-47a6-9515-afda95224009"" entityID=""https://sts.windows.net/268da1a1-9db4-48b9-b1fe-683250ba90cc/"">
                  <RoleDescriptor xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706"" xsi:type=""fed:SecurityTokenServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/wsfed/federation/200706"">
                    <fed:PassiveRequestorEndpoint>
                      <wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing"">
                        <wsa:Address />
                      </wsa:EndpointReference>
                    </fed:PassiveRequestorEndpoint>
                  </RoleDescriptor>
                </EntityDescriptor>";
            }
        }

        public static string MetadataMalformedCertificate
        {
            get
            {
                return
                @"<EntityDescriptor xmlns=""urn:oasis:names:tc:SAML:2.0:metadata"" ID=""_6c4f3672-45c2-47a6-9515-afda95224009"" entityID=""https://sts.windows.net/268da1a1-9db4-48b9-b1fe-683250ba90cc/"">
                  <RoleDescriptor xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706"" xsi:type=""fed:SecurityTokenServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/wsfed/federation/200706"">
                    <KeyDescriptor use=""signing"">
                      <KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                        <X509Data>
                            <X509Certificate>%%MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD</X509Certificate>
                        </X509Data>
                      </KeyInfo>
                    </KeyDescriptor>
                    <fed:PassiveRequestorEndpoint>
                      <wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing"">
                        <wsa:Address>
                          https://login.microsoftonline.com/268da1a1-9db4-48b9-b1fe-683250ba90cc/wsfed
                        </wsa:Address>
                      </wsa:EndpointReference>
                    </fed:PassiveRequestorEndpoint>                    
                  </RoleDescriptor>
                </EntityDescriptor>";
            }
        }

        public static string MetadataNoAddressInEndpointReference
        {
            get
            {
                return
                @"<EntityDescriptor xmlns=""urn:oasis:names:tc:SAML:2.0:metadata"" ID=""_6c4f3672-45c2-47a6-9515-afda95224009"" entityID=""https://sts.windows.net/268da1a1-9db4-48b9-b1fe-683250ba90cc/"">
                  <RoleDescriptor xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706"" xsi:type=""fed:SecurityTokenServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/wsfed/federation/200706"">
                    <KeyDescriptor use=""signing"">
                      <KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                        <X509Data>
                          <X509Certificate>
                            MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD
                          </X509Certificate>
                        </X509Data>
                      </KeyInfo>
                    </KeyDescriptor>                    
                    <fed:PassiveRequestorEndpoint>
                      <wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing"">
                        <wsa:_Address>
                          https://login.microsoftonline.com/268da1a1-9db4-48b9-b1fe-683250ba90cc/wsfed
                        </wsa:_Address>
                      </wsa:EndpointReference>
                    </fed:PassiveRequestorEndpoint>
                  </RoleDescriptor>
                </EntityDescriptor>";
            }
        }

        public static string MetadataNoEndpointReference
        {
            get
            {
                return
                @"<EntityDescriptor xmlns=""urn:oasis:names:tc:SAML:2.0:metadata"" ID=""_6c4f3672-45c2-47a6-9515-afda95224009"" entityID=""https://sts.windows.net/268da1a1-9db4-48b9-b1fe-683250ba90cc/"">
                  <RoleDescriptor xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706"" xsi:type=""fed:SecurityTokenServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/wsfed/federation/200706"">
                    <KeyDescriptor use=""signing"">
                      <KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                        <X509Data>
                          <X509Certificate>
                            MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD
                          </X509Certificate>
                        </X509Data>
                      </KeyInfo>
                    </KeyDescriptor>                    
                    <fed:PassiveRequestorEndpoint>
                      <wsa:_EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing"">
                        <wsa:Address>
                          https://login.microsoftonline.com/268da1a1-9db4-48b9-b1fe-683250ba90cc/wsfed
                        </wsa:Address>
                      </wsa:_EndpointReference>
                    </fed:PassiveRequestorEndpoint>
                  </RoleDescriptor>
                </EntityDescriptor>";
            }
        }

        public static string MetadataNoEntityDescriptor
        {
            get
            {
                return
                @"<_EntityDescriptor xmlns=""urn:oasis:names:tc:SAML:2.0:metadata"" ID=""_6c4f3672-45c2-47a6-9515-afda95224009"" entityID=""https://sts.windows.net/268da1a1-9db4-48b9-b1fe-683250ba90cc/"">
                  </_EntityDescriptor>";
            }
        }

        public static string MetadataNoIssuer
        {
            get
            {
                return @"<EntityDescriptor xmlns=""urn:oasis:names:tc:SAML:2.0:metadata"" ID=""_6c4f3672-45c2-47a6-9515-afda95224009""></EntityDescriptor>";
            }
        }

        public static string MetadataNoKeyDescriptorForSigningInRoleDescriptor
        {
            get
            {
                return
                @"<EntityDescriptor xmlns=""urn:oasis:names:tc:SAML:2.0:metadata"" ID=""_6c4f3672-45c2-47a6-9515-afda95224009"" entityID=""https://sts.windows.net/268da1a1-9db4-48b9-b1fe-683250ba90cc/"">
                  <RoleDescriptor xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706"" xsi:type=""fed:SecurityTokenServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/wsfed/federation/200706"">
                    <_KeyDescriptor use=""signing"">
                      <KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                        <X509Data>
                          <X509Certificate>
                            MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD
                          </X509Certificate>
                        </X509Data>
                      </KeyInfo>
                    </_KeyDescriptor>                    
                    <fed:PassiveRequestorEndpoint>
                      <wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing"">
                        <wsa:Address>
                          https://login.microsoftonline.com/268da1a1-9db4-48b9-b1fe-683250ba90cc/wsfed
                        </wsa:Address>
                      </wsa:EndpointReference>
                    </fed:PassiveRequestorEndpoint>
                  </RoleDescriptor>
                </EntityDescriptor>";
            }
        }

        public static string MetadataNoKeyInfoInKeyDescriptor
        {
            get
            {
                return
                @"<EntityDescriptor xmlns=""urn:oasis:names:tc:SAML:2.0:metadata"" ID=""_6c4f3672-45c2-47a6-9515-afda95224009"" entityID=""https://sts.windows.net/268da1a1-9db4-48b9-b1fe-683250ba90cc/"">
                  <RoleDescriptor xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706"" xsi:type=""fed:SecurityTokenServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/wsfed/federation/200706"">
                    <KeyDescriptor use=""signing"">
                      <_KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                        <X509Data>
                          <X509Certificate>
                            MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD
                          </X509Certificate>
                        </X509Data>
                      </_KeyInfo>
                    </KeyDescriptor>                    
                    <fed:PassiveRequestorEndpoint>
                      <wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing"">
                        <wsa:Address>
                          https://login.microsoftonline.com/268da1a1-9db4-48b9-b1fe-683250ba90cc/wsfed
                        </wsa:Address>
                      </wsa:EndpointReference>
                    </fed:PassiveRequestorEndpoint>
                  </RoleDescriptor>
                </EntityDescriptor>";
            }
        }

        public static string MetadataUnknownElementBeforeSignatureEndElement
        {
            get
            {
                return
                @"<EntityDescriptor xmlns=""urn:oasis:names:tc:SAML:2.0:metadata"" ID=""_6c4f3672-45c2-47a6-9515-afda95224009"" entityID=""https://sts.windows.net/268da1a1-9db4-48b9-b1fe-683250ba90cc/"">
                  <Signature xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                    <SignedInfo>
                      <CanonicalizationMethod Algorithm=""http://www.w3.org/2001/10/xml-exc-c14n#""/>
                      <SignatureMethod Algorithm=""http://www.w3.org/2001/04/xmldsig-more#rsa-sha256"" />
                      <Reference URI=""#_6c4f3672-45c2-47a6-9515-afda95224009"">
                        <Transforms>
                          <Transform Algorithm=""http://www.w3.org/2000/09/xmldsig#enveloped-signature"" />
                          <Transform Algorithm=""http://www.w3.org/2001/10/xml-exc-c14n#""/>
                        </Transforms>
                        <DigestMethod Algorithm=""http://www.w3.org/2001/04/xmlenc#sha256"" />
                        <DigestValue> i6nrvd1p0+HbCCrFBN5z3jrCe/56R3DlWYQanX6cygM=</DigestValue>
                      </Reference>
                    </SignedInfo>
                    <SignatureValue>
                      gdmviHtNhy8FQ6gSbyovhzMBxioMs6hoHYYzoyjS4DxHqhLgaPrRe948NKfXRYe4o1syVp+cZaGTcRzlPmCFOxH1zjY9qPUT2tCsJ1aCUCoiepu0uYGkWKV9CifHt7+aixQEufxM06iwZcMdfXPF3lqqdOoC7pRTcPlBJo6m6odXmjIcHPpsBGtkJuS7W6JULFhzBC9ytS0asrVaEZhVijP95QM0SZRL/pnJp1gOtKYKsQV246lV8tHFfFIddtklVYTvhlagjVUHsUtUhfwrt/5i/Rnr40qMNx/H10ZClTAQXthQH3GnzObAmhfoMNS1hAMpnX4BEhBOAqHHv2jyPA==
                    </SignatureValue>
                    <_KeyInfo>
                      <X509Data>
                        <X509Certificate>
                          MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD
                        </X509Certificate>
                      </X509Data>
                    </_KeyInfo>
                  </Signature>
                  <RoleDescriptor xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706"" xsi:type=""fed:SecurityTokenServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/wsfed/federation/200706"">
                    <fed:PassiveRequestorEndpoint>
                      <wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing"">
                        <wsa:Address>
                          https://login.microsoftonline.com/268da1a1-9db4-48b9-b1fe-683250ba90cc/wsfed
                        </wsa:Address>
                      </wsa:EndpointReference>
                    </fed:PassiveRequestorEndpoint>
                  </RoleDescriptor>
                </EntityDescriptor>";
            }
        }

        public static string MetadataNoRoleDescriptor
        {
            get
            {
                return
                @"<EntityDescriptor xmlns=""urn:oasis:names:tc:SAML:2.0:metadata"" ID=""_6c4f3672-45c2-47a6-9515-afda95224009"" entityID=""https://sts.windows.net/268da1a1-9db4-48b9-b1fe-683250ba90cc/"">
                  <_RoleDescriptor xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706"" xsi:type=""fed:SecurityTokenServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/wsfed/federation/200706"">
                    <fed:PassiveRequestorEndpoint>
                      <wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing"">
                        <wsa:Address>
                          https://login.microsoftonline.com/268da1a1-9db4-48b9-b1fe-683250ba90cc/wsfed
                        </wsa:Address>
                      </wsa:EndpointReference>
                    </fed:PassiveRequestorEndpoint>
                  </_RoleDescriptor>
                </EntityDescriptor>";
            }
        }

        public static string MetadataNoPassiveRequestorEndpointInRoleDescriptor
        {
            get
            {
                return
                @"<EntityDescriptor xmlns=""urn:oasis:names:tc:SAML:2.0:metadata"" ID=""_6c4f3672-45c2-47a6-9515-afda95224009"" entityID=""https://sts.windows.net/268da1a1-9db4-48b9-b1fe-683250ba90cc/"">
                  <RoleDescriptor xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706"" xsi:type=""fed:SecurityTokenServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/wsfed/federation/200706"">                    
                    <fed:_PassiveRequestorEndpoint>
                      <wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing"">
                        <wsa:Address>
                          https://login.microsoftonline.com/268da1a1-9db4-48b9-b1fe-683250ba90cc/wsfed
                        </wsa:Address>
                      </wsa:EndpointReference>
                    </fed:_PassiveRequestorEndpoint>
                  </RoleDescriptor>
                </EntityDescriptor>";
            }
        }

        public static string MetadataNoSignedInfoInSignature
        {
            get
            {
                return
                @"<EntityDescriptor xmlns=""urn:oasis:names:tc:SAML:2.0:metadata"" ID=""_6c4f3672-45c2-47a6-9515-afda95224009"" entityID=""https://sts.windows.net/268da1a1-9db4-48b9-b1fe-683250ba90cc/"">
                  <Signature xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                    <_SignedInfo>
                      <CanonicalizationMethod Algorithm=""http://www.w3.org/2001/10/xml-exc-c14n#""/>
                      <SignatureMethod Algorithm=""http://www.w3.org/2001/04/xmldsig-more#rsa-sha256"" />
                      <Reference URI=""#_6c4f3672-45c2-47a6-9515-afda95224009"">
                        <Transforms>
                          <Transform Algorithm=""http://www.w3.org/2000/09/xmldsig#enveloped-signature"" />
                          <Transform Algorithm=""http://www.w3.org/2001/10/xml-exc-c14n#""/>
                        </Transforms>
                        <DigestMethod Algorithm=""http://www.w3.org/2001/04/xmlenc#sha256"" />
                        <DigestValue> i6nrvd1p0+HbCCrFBN5z3jrCe/56R3DlWYQanX6cygM=</DigestValue>
                      </Reference>
                    </_SignedInfo>
                    <SignatureValue>
                      gdmviHtNhy8FQ6gSbyovhzMBxioMs6hoHYYzoyjS4DxHqhLgaPrRe948NKfXRYe4o1syVp+cZaGTcRzlPmCFOxH1zjY9qPUT2tCsJ1aCUCoiepu0uYGkWKV9CifHt7+aixQEufxM06iwZcMdfXPF3lqqdOoC7pRTcPlBJo6m6odXmjIcHPpsBGtkJuS7W6JULFhzBC9ytS0asrVaEZhVijP95QM0SZRL/pnJp1gOtKYKsQV246lV8tHFfFIddtklVYTvhlagjVUHsUtUhfwrt/5i/Rnr40qMNx/H10ZClTAQXthQH3GnzObAmhfoMNS1hAMpnX4BEhBOAqHHv2jyPA==
                    </SignatureValue>
                    <KeyInfo>
                      <X509Data>
                        <X509Certificate>
                          MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD
                        </X509Certificate>
                      </X509Data>
                    </KeyInfo>
                  </Signature>
                  <RoleDescriptor xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706"" xsi:type=""fed:SecurityTokenServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/wsfed/federation/200706"">
                    <fed:PassiveRequestorEndpoint>
                      <wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing"">
                        <wsa:Address>
                          https://login.microsoftonline.com/268da1a1-9db4-48b9-b1fe-683250ba90cc/wsfed
                        </wsa:Address>
                      </wsa:EndpointReference>
                    </fed:PassiveRequestorEndpoint>
                  </RoleDescriptor>
                </EntityDescriptor>";
            }
        }

        public static string MetadataNoTokenUri
        {
            get
            {
                return
                @"<EntityDescriptor xmlns=""urn:oasis:names:tc:SAML:2.0:metadata"" ID=""_6c4f3672-45c2-47a6-9515-afda95224009"" entityID=""https://sts.windows.net/268da1a1-9db4-48b9-b1fe-683250ba90cc/"">
                  <Signature xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                    <SignedInfo>
                      <CanonicalizationMethod Algorithm=""http://www.w3.org/2001/10/xml-exc-c14n#""/>
                      <SignatureMethod Algorithm=""http://www.w3.org/2001/04/xmldsig-more#rsa-sha256"" />
                      <Reference URI=""#_6c4f3672-45c2-47a6-9515-afda95224009"">
                        <Transforms>
                          <Transform Algorithm=""http://www.w3.org/2000/09/xmldsig#enveloped-signature"" />
                          <Transform Algorithm=""http://www.w3.org/2001/10/xml-exc-c14n#""/>
                        </Transforms>
                        <DigestMethod Algorithm=""http://www.w3.org/2001/04/xmlenc#sha256"" />
                        <DigestValue> i6nrvd1p0+HbCCrFBN5z3jrCe/56R3DlWYQanX6cygM=</DigestValue>
                      </Reference>
                    </SignedInfo>
                    <SignatureValue>
                      gdmviHtNhy8FQ6gSbyovhzMBxioMs6hoHYYzoyjS4DxHqhLgaPrRe948NKfXRYe4o1syVp+cZaGTcRzlPmCFOxH1zjY9qPUT2tCsJ1aCUCoiepu0uYGkWKV9CifHt7+aixQEufxM06iwZcMdfXPF3lqqdOoC7pRTcPlBJo6m6odXmjIcHPpsBGtkJuS7W6JULFhzBC9ytS0asrVaEZhVijP95QM0SZRL/pnJp1gOtKYKsQV246lV8tHFfFIddtklVYTvhlagjVUHsUtUhfwrt/5i/Rnr40qMNx/H10ZClTAQXthQH3GnzObAmhfoMNS1hAMpnX4BEhBOAqHHv2jyPA==
                    </SignatureValue>
                    <KeyInfo>
                      <X509Data>
                        <X509Certificate>
                          MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD
                        </X509Certificate>
                      </X509Data>
                    </KeyInfo>
                  </Signature>
                  <RoleDescriptor xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706"" xsi:type=""fed:SecurityTokenServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/wsfed/federation/200706"">
                    <fed:PassiveRequestorEndpoint>
                      <wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing"">
                        <wsa:Address>
                          
                        </wsa:Address>
                      </wsa:EndpointReference>
                    </fed:PassiveRequestorEndpoint>
                  </RoleDescriptor>
                </EntityDescriptor>";
            }
        }

        public static string MetadataWithBlanks
        {
            get
            {
                return
                @"<EntityDescriptor xmlns=""urn:oasis:names:tc:SAML:2.0:metadata"" ID=""_6c4f3672-45c2-47a6-9515-afda95224009"" entityID=""https://sts.windows.net/268da1a1-9db4-48b9-b1fe-683250ba90cc/"">

                  <Signature xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                    <SignedInfo>
                      <CanonicalizationMethod Algorithm=""http://www.w3.org/2001/10/xml-exc-c14n#""/>
                      <SignatureMethod Algorithm=""http://www.w3.org/2001/04/xmldsig-more#rsa-sha256"" />
                      <Reference URI=""#_0ded55d8-a72f-4e13-ab9e-f40be80b1476"">
                        <Transforms>
                          <Transform Algorithm=""http://www.w3.org/2000/09/xmldsig#enveloped-signature"" />
                          <Transform Algorithm=""http://www.w3.org/2001/10/xml-exc-c14n#""/>
                        </Transforms>
                        <DigestMethod Algorithm=""http://www.w3.org/2001/04/xmlenc#sha256"" />
                        <DigestValue>lnicj3SNizOF6QI1rWw8JrouoiXMslqtFB5ztWW6qvs=</DigestValue>
                      </Reference>
                    </SignedInfo>
                    <SignatureValue>
                      KD9uWOD/9pvF1NlNCpYoXymUPS1l9uIBgBDe0uOQgQv+tUI/1jJX4UpjADDHCOx6HCl5ZgZSXNmOC2lLSJEwmv21BZzI+PAOxF5hdH99cS/lMC/hxgyWdLVeGnr1I4WbPxGqVmjFNuBdBMaourO4z/5f3D2JZQmgnlu8H+4gv2SpjeZz/YhIN6ZrNfmHwsKZashMGtSmE5uHro+uO5yO17Gr9YfUbtokLRIq5Dk9kqnxG8YZF1C1nC9O0PMdlHb4ubwgO20Cvz5sU2iswn9m68btS5TLF5OVhETzyKir1QA+H1tCgGRqIWd4Geyoucdct1r4zAJGCNIekdKnY3NXwg==
                    </SignatureValue>
                    <KeyInfo>
                      <X509Data>
                        <X509Certificate>MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD</X509Certificate>
                      </X509Data>
                    </KeyInfo>
                  </Signature>

                  <RoleDescriptor xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:fed=""http://docs.oasis-open.org/wsfed/federation/200706"" xsi:type=""fed:SecurityTokenServiceType"" protocolSupportEnumeration=""http://docs.oasis-open.org/wsfed/federation/200706"">
                    
                    <KeyDescriptor use=""signing"">

                      <KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                        <X509Data>
                          <X509Certificate>MIIDBTCCAe2gAwIBAgIQY4RNIR0dX6dBZggnkhCRoDANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDIxMzAwMDAwMFoXDTE5MDIxNDAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMBEizU1OJms31S/ry7iav/IICYVtQ2MRPhHhYknHImtU03sgVk1Xxub4GD7R15i9UWIGbzYSGKaUtGU9lP55wrfLpDjQjEgaXi4fE6mcZBwa9qc22is23B6R67KMcVyxyDWei+IP3sKmCcMX7Ibsg+ubZUpvKGxXZ27YgqFTPqCT2znD7K81YKfy+SVg3uW6epW114yZzClTQlarptYuE2mujxjZtx7ZUlwc9AhVi8CeiLwGO1wzTmpd/uctpner6oc335rvdJikNmc1cFKCK+2irew1bgUJHuN+LJA0y5iVXKvojiKZ2Ii7QKXn19Ssg1FoJ3x2NWA06wc0CnruLsCAwEAAaMhMB8wHQYDVR0OBBYEFDAr/HCMaGqmcDJa5oualVdWAEBEMA0GCSqGSIb3DQEBCwUAA4IBAQAiUke5mA86R/X4visjceUlv5jVzCn/SIq6Gm9/wCqtSxYvifRXxwNpQTOyvHhrY/IJLRUp2g9/fDELYd65t9Dp+N8SznhfB6/Cl7P7FRo99rIlj/q7JXa8UB/vLJPDlr+NREvAkMwUs1sDhL3kSuNBoxrbLC5Jo4es+juQLXd9HcRraE4U3UZVhUS2xqjFOfaGsCbJEqqkjihssruofaxdKT1CPzPMANfREFJznNzkpJt4H0aMDgVzq69NxZ7t1JiIuc43xRjeiixQMRGMi1mAB75fTyfFJ/rWQ5J/9kh0HMZVtHsqICBF1tHMTMIK5rwoweY0cuCIpN7A/zMOQtoD</X509Certificate>
                        </X509Data>
                      </KeyInfo>

                    </KeyDescriptor>

                    <KeyDescriptor use=""signing"">

                      <KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                        <X509Data>
                          <X509Certificate>MIIDBTCCAe2gAwIBAgIQXxLnqm1cOoVGe62j7W7wZzANBgkqhkiG9w0BAQsFADAtMSswKQYDVQQDEyJhY2NvdW50cy5hY2Nlc3Njb250cm9sLndpbmRvd3MubmV0MB4XDTE3MDMyNjAwMDAwMFoXDTE5MDMyNzAwMDAwMFowLTErMCkGA1UEAxMiYWNjb3VudHMuYWNjZXNzY29udHJvbC53aW5kb3dzLm5ldDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKJGarCm4IF0/Gz5Xx/zyZwD2rdJJZtO2Ukk1Oz+Br1sLVY8I5vj5esB+lotmLEblA9N/w188vmTvykaEzUl49NA4s86x44SW6WtdQbGJ0IjpQJUalUMyy91vIBkK/7K3nBXeVBsRk7tm528leoQ05/aZ+1ycJBIU+1oGYThv8MOjyHAlXJmCaGXwXTisZ+hHjcwlMk/+ZEutHflKLIpPUNEi7j4Xw+zp9UKo5pzWIr/iJ4HjvCkFofW90AMF2xp8dMhpbVcfJGS/Ii3J66LuNLCH/HtSZ42FO+tnRL/nNzzFWUhGT92Q5VFVngfWJ3PAg1zz8I1wowLD2fiB2udGXcCAwEAAaMhMB8wHQYDVR0OBBYEFFXPbFXjmMR3BluF+2MeSXd1NQ3rMA0GCSqGSIb3DQEBCwUAA4IBAQAsd3wGVilJxDtbY1K2oAsWLdNJgmCaYdrtdlAsjGlarSQSzBH0Ybf78fcPX//DYaLXlvaEGKVKp0jPq+RnJ17oP/RJpJTwVXPGRIlZopLIgnKpWlS/PS0uKAdNvLmz1zbGSILdcF+Qf41OozD4QNsS1c9YbDO4vpC9v8x3PVjfJvJwPonzNoOsLXA+8IONSXwCApsnmrwepKu8sifsFYSwgrwxRPGTEAjkdzRJ0yMqiY/VoJ7lqJ/FBJqqAjGPGq/yI9rVoG+mbO1amrIDWHHTKgfbKk0bXGtVUbsayyHR5jSgadmkLBh5AaN/HcgDK/jINrnpiQ+/2ewH/8qLaQ3B</X509Certificate>
                        </X509Data>
                      </KeyInfo>

                    </KeyDescriptor>

                    <KeyDescriptor use=""signing"">

                      <KeyInfo xmlns=""http://www.w3.org/2000/09/xmldsig#"">
                        <X509Data>
                          <X509Certificate>MIIDKDCCAhCgAwIBAgIQBHJvVNxP1oZO4HYKh+rypDANBgkqhkiG9w0BAQsFADAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXMwHhcNMTYxMTE2MDgwMDAwWhcNMTgxMTE2MDgwMDAwWjAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXMwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQChn5BCs24Hh6L0BNitPrV5s+2/DBhaeytOmnghJKnqeJlhv3ZczShRM2Cp38LW8Y3wn7L3AJtolaSkF/joKN1l6GupzM+HOEdq7xZxFehxIHW7+25mG/WigBnhsBzLv1SR4uIbrQeS5M0kkLwJ9pOnVH3uzMGG6TRXPnK3ivlKl97AiUEKdlRjCQNLXvYf1ZqlC77c/ZCOHSX4kvIKR2uG+LNlSTRq2rn8AgMpFT4DSlEZz4RmFQvQupQzPpzozaz/gadBpJy/jgDmJlQMPXkHp7wClvbIBGiGRaY6eZFxNV96zwSR/GPNkTObdw2S8/SiAgvIhIcqWTPLY6aVTqJfAgMBAAGjWDBWMFQGA1UdAQRNMEuAEDUj0BrjP0RTbmoRPTRMY3WhJTAjMSEwHwYDVQQDExhsb2dpbi5taWNyb3NvZnRvbmxpbmUudXOCEARyb1TcT9aGTuB2Cofq8qQwDQYJKoZIhvcNAQELBQADggEBAGnLhDHVz2gLDiu9L34V3ro/6xZDiSWhGyHcGqky7UlzQH3pT5so8iF5P0WzYqVtogPsyC2LPJYSTt2vmQugD4xlu/wbvMFLcV0hmNoTKCF1QTVtEQiAiy0Aq+eoF7Al5fV1S3Sune0uQHimuUFHCmUuF190MLcHcdWnPAmzIc8fv7quRUUsExXmxSX2ktUYQXzqFyIOSnDCuWFm6tpfK5JXS8fW5bpqTlrysXXz/OW/8NFGq/alfjrya4ojrOYLpunGriEtNPwK7hxj1AlCYEWaRHRXaUIW1ByoSff/6Y6+ZhXPUe0cDlNRt/qIz5aflwO7+W8baTS4O8m/icu7ItE=</X509Certificate>
                        </X509Data>
                      </KeyInfo>

                    </KeyDescriptor>

                    <fed:PassiveRequestorEndpoint>

                      <wsa:EndpointReference xmlns:wsa=""http://www.w3.org/2005/08/addressing"">

                        <wsa:Address>https://login.microsoftonline.com/268da1a1-9db4-48b9-b1fe-683250ba90cc/wsfed</wsa:Address>

                      </wsa:EndpointReference>

                    </fed:PassiveRequestorEndpoint>

                  </RoleDescriptor>

                </EntityDescriptor>";
            }
        }
    }
}
