import { getCurrentTimestamp } from '../utils';
import { BaseTransport } from './base';
import { initializeTransports } from './initialize';
import { TransportItemType } from './types';
class MockTransport extends BaseTransport {
    constructor() {
        super(...arguments);
        this.sentItems = [];
    }
    send(item) {
        this.sentItems.push(item);
    }
}
describe('transports', () => {
    describe('config.ignoreErrors', () => {
        it('will filter out errors by string or regex', () => {
            var _a;
            const transport = new MockTransport();
            const config = {
                transports: [transport],
                ignoreErrors: ['Error: ResizeObserver', /FetchError[:\s\w\/]*pwc/],
            };
            const transports = initializeTransports(config);
            transports.execute(makeExceptionTransportItem('Error', 'ResizeObserver loop limit exceeded'));
            transports.execute(makeExceptionTransportItem('TypeError', '_.viz is undefined'));
            transports.execute(makeExceptionTransportItem('FetchError', '404 \n  Instantiating https://pwc.grafana.net/public/react-router-dom'));
            transports.execute(makeExceptionTransportItem('FetchError', '404 \n  Instantiating https://pwc.grafana.net/public/@emotion/css'));
            expect(transport.sentItems).toHaveLength(1);
            expect(((_a = transport.sentItems[0]) === null || _a === void 0 ? void 0 : _a.payload).type).toEqual('TypeError');
        });
    });
    describe('config.beforeSend', () => {
        it('will not send events that are rejected by beforeSend hook', () => {
            var _a;
            const transport = new MockTransport();
            const hookedItems = [];
            const config = {
                transports: [transport],
                beforeSend: (item) => {
                    hookedItems.push(item);
                    if (item.type === TransportItemType.EXCEPTION && item.payload.type === 'TypeError') {
                        return null;
                    }
                    return item;
                },
            };
            const transports = initializeTransports(config);
            transports.execute(makeExceptionTransportItem('Error', 'ResizeObserver loop limit exceeded'));
            transports.execute(makeExceptionTransportItem('TypeError', '_.viz is undefined'));
            expect(transport.sentItems).toHaveLength(1);
            expect(hookedItems).toHaveLength(2);
            expect(((_a = transport.sentItems[0]) === null || _a === void 0 ? void 0 : _a.payload).type).toEqual('Error');
        });
        it('events can be modified by beforeSend hook', () => {
            var _a;
            const transport = new MockTransport();
            const config = {
                transports: [transport],
                beforeSend: (item) => {
                    if (item.type === TransportItemType.EXCEPTION) {
                        return Object.assign(Object.assign({}, item), { payload: Object.assign(Object.assign({}, item.payload), { type: 'NewType' }) });
                    }
                    return item;
                },
            };
            const transports = initializeTransports(config);
            transports.execute(makeExceptionTransportItem('Error', 'ResizeObserver loop limit exceeded'));
            expect(transport.sentItems).toHaveLength(1);
            expect(((_a = transport.sentItems[0]) === null || _a === void 0 ? void 0 : _a.payload).type).toEqual('NewType');
        });
    });
});
function makeExceptionTransportItem(type, value) {
    return {
        type: TransportItemType.EXCEPTION,
        payload: {
            type,
            value,
            timestamp: getCurrentTimestamp(),
            stacktrace: {
                frames: [],
            },
        },
        meta: {},
    };
}
//# sourceMappingURL=transports.test.js.map