"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.nodeVersionCheckerTask = exports.nodeVersionFiles = void 0;
const tslib_1 = require("tslib");
const chalk_1 = tslib_1.__importDefault(require("chalk"));
const fs_1 = require("fs");
const semver_1 = require("semver");
const task_1 = require("./task");
const pattern = /(circleci\/|FROM )node\:([0-9]+(\.[0-9]+){0,2})/gm;
const packageJsonFile = 'package.json';
const failures = [];
exports.nodeVersionFiles = [packageJsonFile, 'Dockerfile'];
const nodeVersionCheckerRunner = () => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    // Read version from package json and treat that as the expected version in all other locations
    const packageJson = require(`${process.cwd()}/${packageJsonFile}`);
    const expectedVersion = packageJson.engines.node;
    console.log(chalk_1.default.yellow(`Specified node version in package.json is: ${expectedVersion}`));
    for (const file of exports.nodeVersionFiles) {
        const fileContent = (0, fs_1.readFileSync)(`${process.cwd()}/${file}`);
        const matches = fileContent.toString('utf8').match(pattern);
        if (!matches) {
            continue;
        }
        for (const match of matches) {
            const actualVersion = (0, semver_1.coerce)(match);
            if (!actualVersion) {
                failures.push({
                    file,
                    line: match,
                });
                continue;
            }
            const satisfied = (0, semver_1.satisfies)(actualVersion, expectedVersion);
            if (!satisfied) {
                failures.push({
                    file,
                    line: match,
                });
            }
        }
    }
    if (failures.length > 0) {
        console.log(chalk_1.default.red('--------------------------------------------------------------------'));
        console.log(chalk_1.default.red(`These entries don't satisfy the engine version in ${packageJsonFile}`));
        console.log(chalk_1.default.red('--------------------------------------------------------------------'));
        for (let index = 0; index < failures.length; index++) {
            const failure = failures[index];
            console.log(chalk_1.default.green(`\tIn ${failure.file} the line ${failure.line} does not satisfy ${expectedVersion}.`));
        }
        throw new Error('Node versions not in sync');
    }
    console.log(chalk_1.default.yellow('--------------------------------------------------------------------'));
    console.log(chalk_1.default.yellow('All node versions seem ok.'));
    console.log(chalk_1.default.yellow('--------------------------------------------------------------------'));
});
exports.nodeVersionCheckerTask = new task_1.Task('Node Version Checker', nodeVersionCheckerRunner);
//# sourceMappingURL=nodeVersionChecker.js.map