"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.reporter = void 0;
const render_1 = require("@betterer/render");
const components_1 = require("./components");
const state_1 = require("./state");
/**
 * @public The default {@link @betterer/betterer#BettererReporter | `BettererReporter`}.
 * Supports `betterer` and `betterer watch` modes.
 *
 * @remarks The reporter is implemented as an {@link https://github.com/vadimdemedes/ink | Ink } application
 * to handle all the fancy terminal rerendering. It also means we can do cool stuff like dynamic
 * settings in `betterer watch` mode.
 */
exports.reporter = createReporter();
function createReporter() {
    const RENDER_OPTIONS = {
        debug: process.env.NODE_ENV === 'test',
        exitOnCtrlC: false
    };
    let renderer;
    return {
        configError(_, error) {
            renderError(error);
        },
        contextStart(context) {
            renderer = createRenderer(context);
            renderer.render();
        },
        contextEnd(contextSummary) {
            renderer.render((0, state_1.contextEnd)(contextSummary));
            renderer.stop();
        },
        contextError(_, error) {
            renderError(error);
        },
        suiteStart(suite) {
            return new Promise((resolve) => {
                renderer.render((0, state_1.suiteStart)(suite), resolve);
            });
        },
        suiteEnd(suiteSummary) {
            renderer.render((0, state_1.suiteEnd)(suiteSummary));
        }
    };
    function renderError(error) {
        (0, render_1.render)(render_1.React.createElement(components_1.Error, { error: error }), RENDER_OPTIONS);
    }
    function createRenderer(context) {
        let app;
        const dispatch = (0, state_1.createStore)(context);
        return {
            render(action, done) {
                const state = dispatch(action);
                // eslint-disable-next-line no-console
                console.clear();
                const component = render_1.React.createElement(components_1.Reporter, Object.assign({}, state, { done: done }));
                if (!app) {
                    app = (0, render_1.render)(component, RENDER_OPTIONS);
                }
                else {
                    app.rerender(component);
                }
            },
            stop() {
                app.unmount();
            }
        };
    }
}
//# sourceMappingURL=reporter.js.map