"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.eslint = void 0;
const tslib_1 = require("tslib");
const betterer_1 = require("@betterer/betterer");
const errors_1 = require("@betterer/errors");
const assert_1 = (0, tslib_1.__importDefault)(require("assert"));
const eslint_1 = require("eslint");
/**
 * @public Use this test to incrementally introduce new {@link https://eslint.org/ | **ESLint**} rules to
 * your codebase. You can pass as many **ESLint** {@link https://eslint.org/docs/rules/ | rule configurations}
 * as you like:
 *
 * @remarks {@link @betterer/eslint#eslint | `eslint`} is a {@link @betterer/betterer#BettererFileTest | `BettererFileTest`},
 * so you can use {@link @betterer/betterer#BettererFileTest.include | `include()`},
 * {@link @betterer/betterer#BettererFileTest.exclude | `exclude()`}, {@link @betterer/betterer#BettererFileTest.only | `only()`},
 * and {@link @betterer/betterer#BettererFileTest.skip | `skip()`}.
 *
 * @example
 * ```typescript
 * import { eslint } from '@betterer/eslint';
 *
 * export default {
 *   'new eslint rules': () =>
 *     eslint({
 *       'no-debugger': 'error',
 *       'no-unsafe-finally': 'error',
 *     })
 *     .include('./src/*.ts')
 * };
 * ```
 *
 * @param rules - Additional {@link https://eslint.org/ | **ESLint**} {@link https://eslint.org/docs/rules/ | rules}
 * to enable.
 *
 * @throws {@link @betterer/errors#BettererError | `BettererError` }
 * Will throw if the user doesn't pass `rules`.
 */
function eslint(rules) {
    if (!rules) {
        throw new errors_1.BettererError("for `@betterer/eslint` to work, you need to provide rule options, e.g. `{ 'no-debugger': 'error' }`. ❌");
    }
    return new betterer_1.BettererFileTest(async (filePaths, fileTestResult, resolver) => {
        if (!filePaths.length) {
            return;
        }
        const { baseDirectory } = resolver;
        const cli = new eslint_1.ESLint({ cwd: baseDirectory });
        await Promise.all(filePaths.map(async (filePath) => {
            const linterOptions = (await cli.calculateConfigForFile(filePath));
            // Explicitly disable all other configured rules:
            const disabledRules = {};
            Object.keys(linterOptions.rules || {}).forEach((ruleName) => {
                disabledRules[ruleName] = 'off';
            });
            const finalRules = Object.assign(Object.assign({}, disabledRules), rules);
            const runner = new eslint_1.ESLint({
                overrideConfig: { rules: finalRules },
                useEslintrc: true,
                cwd: baseDirectory
            });
            const lintResults = await runner.lintFiles([filePath]);
            lintResults
                .filter((lintResult) => lintResult.source)
                .forEach((lintResult) => {
                const { messages, source } = lintResult;
                (0, assert_1.default)(source);
                const file = fileTestResult.addFile(filePath, source);
                messages.forEach((message) => {
                    const startLine = message.line - 1;
                    const startColumn = message.column - 1;
                    const endLine = message.endLine ? message.endLine - 1 : 0;
                    const endColumn = message.endColumn ? message.endColumn - 1 : 0;
                    file.addIssue(startLine, startColumn, endLine, endColumn, message.message);
                });
            });
        }));
    });
}
exports.eslint = eslint;
//# sourceMappingURL=eslint.js.map