"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.run = void 0;
const errors_1 = require("@betterer/errors");
const tsquery_1 = require("@phenomnomnominal/tsquery");
const fs_1 = require("fs");
const prettier_1 = require("prettier");
const typescript_1 = require("typescript");
const diff_1 = require("./diff");
const upgrade_cjs_1 = require("./upgrade-cjs");
const upgrade_esm_1 = require("./upgrade-esm");
const printer = (0, typescript_1.createPrinter)();
// module.exports
const EXPORT_MODULE_EXPORTS_QUERY = 'PropertyAccessExpression:has(Identifier[name="module"]):has(Identifier[name="exports"])';
async function run(logger, configPath, save) {
    await logger.progress(`upgrading "${configPath}"...`);
    const fileText = await fs_1.promises.readFile(configPath, 'utf8');
    const replaceNewLines = fileText.split('\n\n').join('/* BLANK LINE */');
    const originalSourceFile = tsquery_1.tsquery.ast(replaceNewLines, configPath);
    const moduleType = getModuleType(originalSourceFile);
    if (moduleType == null) {
        throw new errors_1.BettererError('');
    }
    const upgrade = moduleType === typescript_1.ModuleKind.CommonJS ? upgrade_cjs_1.upgradeCJS : upgrade_esm_1.upgradeESM;
    const upgradedSourceFile = upgrade(originalSourceFile, configPath);
    const config = await (0, prettier_1.resolveConfig)(configPath);
    const printed = printUpgraded(originalSourceFile, upgradedSourceFile);
    const formatOptions = Object.assign(Object.assign({}, config), { parser: 'typescript' });
    const fomattedOriginal = (0, prettier_1.format)(fileText, formatOptions);
    const formatted = (0, prettier_1.format)(printed, formatOptions);
    if (fomattedOriginal !== formatted) {
        if (!save) {
            await logger.info('\nBetterer can upgrade your test definition file automatically! ✨');
            await logger.info(`Here's what the changes will look like:\n`);
            await logger.info((0, diff_1.diff)(fileText, formatted));
            await logger.info(`\nIf that looks okay, run \`betterer upgrade --save\` to write the changes to disk. 💾\n`);
        }
    }
    else {
        await logger.success('Nothing to upgrade!');
    }
    if (save) {
        await logger.progress(`Saving upgraded config to "${configPath}"...`);
        await fs_1.promises.writeFile(configPath, formatted);
        await logger.success(`Saved upgraded config to "${configPath}"! ☀️`);
    }
}
exports.run = run;
// export { thing } from 'foo';
// export default {};
const EXPORT_QUERY = 'ExportDeclaration, ExportAssignment';
// export const thing = 'foo';
const EXPORT_VARIABLE_DECLARTION_QUERY = 'VariableStatement:has(ExportKeyword)';
function getModuleType(originalSourceFile) {
    const [exportStatement] = (0, tsquery_1.tsquery)(originalSourceFile, EXPORT_QUERY);
    const [exportVariableDeclaration] = (0, tsquery_1.tsquery)(originalSourceFile, EXPORT_VARIABLE_DECLARTION_QUERY);
    if (exportStatement || exportVariableDeclaration) {
        return typescript_1.ModuleKind.ES2015;
    }
    const [moduleExports] = (0, tsquery_1.tsquery)(originalSourceFile, EXPORT_MODULE_EXPORTS_QUERY);
    if (moduleExports) {
        return typescript_1.ModuleKind.CommonJS;
    }
    return null;
}
function printUpgraded(originalSourceFile, upgradedSourceFile) {
    return [getImport(originalSourceFile, upgradedSourceFile), print(upgradedSourceFile)].join('');
}
function getImport(originalSourceFile, upgradedSourceFile) {
    const needsImport = checkNeedsImport(originalSourceFile, upgradedSourceFile);
    if (!needsImport) {
        return '';
    }
    const [exportStatement] = (0, tsquery_1.tsquery)(originalSourceFile, EXPORT_QUERY);
    const [exportVariableDeclaration] = (0, tsquery_1.tsquery)(originalSourceFile, EXPORT_VARIABLE_DECLARTION_QUERY);
    if (exportStatement || exportVariableDeclaration) {
        return `import { BettererTest } from '@betterer/betterer';`;
    }
    const [moduleExports] = (0, tsquery_1.tsquery)(originalSourceFile, EXPORT_MODULE_EXPORTS_QUERY);
    if (moduleExports) {
        return `const { BettererTest } = require('@betterer/betterer');`;
    }
    return '';
}
// import {} from '@betterer/betterer';
// const {} = require('@betterer/better');
const IMPORT_QUERY = ':matches(ImportDeclaration, VariableStatement:has(CallExpression:has(Identifier[name="require"]))):has(StringLiteral[value="@betterer/betterer"])';
// BettererTest
const BETTERER_TEST_QUERY = 'Identifier[name="BettererTest"]';
function checkNeedsImport(originalSourceFile, upgradedSourceFile) {
    const bettererImports = (0, tsquery_1.tsquery)(originalSourceFile, IMPORT_QUERY);
    const existingImport = bettererImports.some((bettererImport) => (0, tsquery_1.tsquery)(bettererImport, BETTERER_TEST_QUERY).length > 0);
    const usedInUpgrade = (0, tsquery_1.tsquery)(upgradedSourceFile, BETTERER_TEST_QUERY).length > 0;
    return !existingImport && usedInUpgrade;
}
function print(upgradedSourceFile) {
    const printed = printer.printFile(upgradedSourceFile);
    return printed.replace(/\/\* BLANK LINE \*\//g, '\n\n');
}
//# sourceMappingURL=upgrade-config-file.js.map