"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractGlobalFilesFromInputs = exports.getImplicitlyTouchedProjects = exports.getTouchedProjects = void 0;
const minimatch = require("minimatch");
const getTouchedProjects = (touchedFiles, projectGraphNodes) => {
    // sort project names with the most nested first,
    // e.g. ['libs/a/b/c', 'libs/a/b', 'libs/a']
    const projectNames = Object.entries(projectGraphNodes)
        .sort(([name1, p1], [name2, p2]) => p1.data.root.length > p2.data.root.length ? -1 : 1)
        .map(([name]) => name);
    return touchedFiles
        .map((f) => {
        return projectNames.find((projectName) => {
            const p = projectGraphNodes[projectName].data;
            const projectRoot = p.root.endsWith('/') ? p.root : `${p.root}/`;
            return f.file.startsWith(projectRoot);
        });
    })
        .filter(Boolean);
};
exports.getTouchedProjects = getTouchedProjects;
const getImplicitlyTouchedProjects = (fileChanges, projectGraphNodes, nxJson) => {
    const implicits = Object.assign({}, nxJson.implicitDependencies);
    const globalFiles = [
        ...extractGlobalFilesFromInputs(nxJson, projectGraphNodes),
        'nx.json',
        'package.json',
    ];
    globalFiles.forEach((file) => {
        implicits[file] = '*';
    });
    const touched = new Set();
    for (const [pattern, projects] of Object.entries(implicits)) {
        const implicitDependencyWasChanged = fileChanges.some((f) => minimatch(f.file, pattern));
        if (!implicitDependencyWasChanged) {
            continue;
        }
        // File change affects all projects, just return all projects.
        if (projects === '*') {
            return Object.keys(projectGraphNodes);
        }
        else if (Array.isArray(projects)) {
            projects.forEach((project) => touched.add(project));
        }
    }
    return Array.from(touched);
};
exports.getImplicitlyTouchedProjects = getImplicitlyTouchedProjects;
function extractGlobalFilesFromInputs(nxJson, projectGraphNodes) {
    const globalFiles = [];
    globalFiles.push(...extractGlobalFilesFromNamedInputs(nxJson.namedInputs));
    globalFiles.push(...extractGlobalFilesFromTargets(nxJson.targetDefaults));
    Object.values(projectGraphNodes || {}).forEach((node) => {
        globalFiles.push(...extractGlobalFilesFromNamedInputs(node.data.namedInputs));
        globalFiles.push(...extractGlobalFilesFromTargets(node.data.targets));
    });
    return globalFiles;
}
exports.extractGlobalFilesFromInputs = extractGlobalFilesFromInputs;
function extractGlobalFilesFromNamedInputs(namedInputs) {
    const globalFiles = [];
    for (const inputs of Object.values(namedInputs || {})) {
        globalFiles.push(...extractGlobalFiles(inputs));
    }
    return globalFiles;
}
function extractGlobalFilesFromTargets(targets) {
    const globalFiles = [];
    for (const target of Object.values(targets || {})) {
        if (target.inputs) {
            globalFiles.push(...extractGlobalFiles(target.inputs));
        }
    }
    return globalFiles;
}
function extractGlobalFiles(inputs) {
    const globalFiles = [];
    for (const input of inputs) {
        if (typeof input === 'string' && input.startsWith('{workspaceRoot}/')) {
            globalFiles.push(input.substring('{workspaceRoot}/'.length));
        }
        else if (input.fileset && input.fileset.startsWith('{workspaceRoot}/')) {
            globalFiles.push(input.fileset.substring('{workspaceRoot}/'.length));
        }
    }
    return globalFiles;
}
//# sourceMappingURL=workspace-projects.js.map