// SPDX-License-Identifier: GPL-2.0-or-later
/*
 * Virtio memory mapped device driver
 *
 * Copyright 2011-2014, ARM Ltd.
 *
 * This module allows virtio devices to be used over a virtual, memory mapped
 * platform device.
 *
 * The guest device(s) may be instantiated in one of three equivalent ways:
 *
 * 1. Static platform device in board's code, eg.:
 *
 *	static struct platform_device v2m_virtio_device = {
 *		.name = "virtio-mmio",
 *		.id = -1,
 *		.num_resources = 2,
 *		.resource = (struct resource []) {
 *			{
 *				.start = 0x1001e000,
 *				.end = 0x1001e0ff,
 *				.flags = IORESOURCE_MEM,
 *			}, {
 *				.start = 42 + 32,
 *				.end = 42 + 32,
 *				.flags = IORESOURCE_IRQ,
 *			},
 *		}
 *	};
 *
 * 2. Device Tree node, eg.:
 *
 *		virtio_block@1e000 {
 *			compatible = "virtio,mmio";
 *			reg = <0x1e000 0x100>;
 *			interrupts = <42>;
 *		}
 *
 * 3. Kernel module (or command line) parameter. Can be used more than once -
 *    one device will be created for each one. Syntax:
 *
 *		[virtio_mmio.]device=<size>@<baseaddr>:<irq>[:<id>]
 *    where:
 *		<size>     := size (can use standard suffixes like K, M or G)
 *		<baseaddr> := physical base address
 *		<irq>      := interrupt number (as passed to request_irq())
 *		<id>       := (optional) platform device id
 *    eg.:
 *		virtio_mmio.device=0x100@0x100b0000:48 \
 *				virtio_mmio.device=1K@0x1001e000:74
 *
 * Based on Virtio PCI driver by Anthony Liguori, copyright IBM Corp. 2007
 */

#define pr_fmt(fmt) "virtio-mmio: " fmt

#include <linux/acpi.h>
#include <linux/dma-mapping.h>
#include <linux/highmem.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/list.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/pm.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/virtio.h>
#include <linux/virtio_config.h>
#include <uapi/linux/virtio_mmio.h>
#include <linux/virtio_ring.h>
#include <dragonball/dragonball.h>
#ifdef CONFIG_VIRTIO_MMIO_DRAGONBALL
#include <linux/msi.h>
#ifdef CONFIG_X86_64
#include <asm/irqdomain.h>
#endif
#endif

/* The alignment to use between consumer and producer parts of vring.
 * Currently hardcoded to the page size. */
#define VIRTIO_MMIO_VRING_ALIGN		PAGE_SIZE

#ifdef CONFIG_VIRTIO_MMIO_DRAGONBALL
#include "virtio_mmio_db.h"
#endif  /* CONFIG_VIRTIO_MMIO_DRAGONBALL */

#ifdef CONFIG_PVDMA
#include <linux/pvdma.h>
#endif

#define to_virtio_mmio_device(_plat_dev) \
	container_of(_plat_dev, struct virtio_mmio_device, vdev)
#define to_virtio_device(pdev) \
	container_of((pdev), struct virtio_device, dev)

struct virtio_mmio_device {
	struct virtio_device vdev;
	struct platform_device *pdev;

	void __iomem *base;
	unsigned long version;

	/* a list of queues so we can dispatch IRQs */
	spinlock_t lock;
	struct list_head virtqueues;
	int single_irq_enabled;
#ifdef CONFIG_VIRTIO_MMIO_DRAGONBALL
	bool msi_enabled;
	/* Name strings for interrupts. */
	char (*vm_vq_names)[256];
#endif

};

struct virtio_mmio_vq_info {
	/* the actual virtqueue */
	struct virtqueue *vq;

	/* the list node for the virtqueues list */
	struct list_head node;
};

#ifdef CONFIG_VIRTIO_MMIO_DRAGONBALL
static int  msi_irq_vector(struct device *dev, unsigned int nr);
static int  vm_alloc_msi_irqs(struct virtio_device *vdev, int nirqs);
static void vm_free_msi_irqs(struct virtio_device *vdev);
static int  vm_register_msi_irq(struct virtqueue *vq, int index,
				void (*callback)(struct virtqueue *vq),
				const char *name);
static void vm_unregister_msi_irq(struct virtqueue *vq, int index);
#else   /* CONFIG_VIRTIO_MMIO_DRAGONBALL */
static void vm_free_msi_irqs(struct virtio_device *vdev) { }
static void vm_unregister_msi_irq(struct virtqueue *vq, int index) { }
static int  vm_alloc_msi_irqs(struct virtio_device *vdev, int nirqs)
{
	return -EOPNOTSUPP;
}
static int  vm_register_msi_irq(struct virtqueue *vq, int index,
				void (*callback)(struct virtqueue *vq),
				const char *name)
{
	return 0;
}
#endif  /* CONFIG_VIRTIO_MMIO_DRAGONBALL */

/* Configuration interface */

static u64 vm_get_features(struct virtio_device *vdev)
{
	struct virtio_mmio_device *vm_dev = to_virtio_mmio_device(vdev);
	u64 features;

	writel(1, vm_dev->base + VIRTIO_MMIO_DEVICE_FEATURES_SEL);
	features = readl(vm_dev->base + VIRTIO_MMIO_DEVICE_FEATURES);
	features <<= 32;

	writel(0, vm_dev->base + VIRTIO_MMIO_DEVICE_FEATURES_SEL);
	features |= readl(vm_dev->base + VIRTIO_MMIO_DEVICE_FEATURES);

	return features;
}

static int vm_finalize_features(struct virtio_device *vdev)
{
	struct virtio_mmio_device *vm_dev = to_virtio_mmio_device(vdev);

	/* Give virtio_ring a chance to accept features. */
	vring_transport_features(vdev);

	/* Make sure there is are no mixed devices */
	if (vm_dev->version == 2 &&
			!__virtio_test_bit(vdev, VIRTIO_F_VERSION_1)) {
		dev_err(&vdev->dev, "New virtio-mmio devices (version 2) must provide VIRTIO_F_VERSION_1 feature!\n");
		return -EINVAL;
	}

	writel(1, vm_dev->base + VIRTIO_MMIO_DRIVER_FEATURES_SEL);
	writel((u32)(vdev->features >> 32),
			vm_dev->base + VIRTIO_MMIO_DRIVER_FEATURES);

	writel(0, vm_dev->base + VIRTIO_MMIO_DRIVER_FEATURES_SEL);
	writel((u32)vdev->features,
			vm_dev->base + VIRTIO_MMIO_DRIVER_FEATURES);

	return 0;
}

static void vm_get(struct virtio_device *vdev, unsigned offset,
		   void *buf, unsigned len)
{
	struct virtio_mmio_device *vm_dev = to_virtio_mmio_device(vdev);
	void __iomem *base = vm_dev->base + VIRTIO_MMIO_CONFIG;
	u8 b;
	__le16 w;
	__le32 l;

	if (vm_dev->version == 1) {
		u8 *ptr = buf;
		int i;

		for (i = 0; i < len; i++)
			ptr[i] = readb(base + offset + i);
		return;
	}

	switch (len) {
	case 1:
		b = readb(base + offset);
		memcpy(buf, &b, sizeof b);
		break;
	case 2:
		w = cpu_to_le16(readw(base + offset));
		memcpy(buf, &w, sizeof w);
		break;
	case 4:
		l = cpu_to_le32(readl(base + offset));
		memcpy(buf, &l, sizeof l);
		break;
	case 8:
		l = cpu_to_le32(readl(base + offset));
		memcpy(buf, &l, sizeof l);
		l = cpu_to_le32(ioread32(base + offset + sizeof l));
		memcpy(buf + sizeof l, &l, sizeof l);
		break;
	default:
		BUG();
	}
}

static void vm_set(struct virtio_device *vdev, unsigned offset,
		   const void *buf, unsigned len)
{
	struct virtio_mmio_device *vm_dev = to_virtio_mmio_device(vdev);
	void __iomem *base = vm_dev->base + VIRTIO_MMIO_CONFIG;
	u8 b;
	__le16 w;
	__le32 l;

	if (vm_dev->version == 1) {
		const u8 *ptr = buf;
		int i;

		for (i = 0; i < len; i++)
			writeb(ptr[i], base + offset + i);

		return;
	}

	switch (len) {
	case 1:
		memcpy(&b, buf, sizeof b);
		writeb(b, base + offset);
		break;
	case 2:
		memcpy(&w, buf, sizeof w);
		writew(le16_to_cpu(w), base + offset);
		break;
	case 4:
		memcpy(&l, buf, sizeof l);
		writel(le32_to_cpu(l), base + offset);
		break;
	case 8:
		memcpy(&l, buf, sizeof l);
		writel(le32_to_cpu(l), base + offset);
		memcpy(&l, buf + sizeof l, sizeof l);
		writel(le32_to_cpu(l), base + offset + sizeof l);
		break;
	default:
		BUG();
	}
}

static u32 vm_generation(struct virtio_device *vdev)
{
	struct virtio_mmio_device *vm_dev = to_virtio_mmio_device(vdev);

	if (vm_dev->version == 1)
		return 0;
	else
		return readl(vm_dev->base + VIRTIO_MMIO_CONFIG_GENERATION);
}

static u8 vm_get_status(struct virtio_device *vdev)
{
	struct virtio_mmio_device *vm_dev = to_virtio_mmio_device(vdev);

	return readl(vm_dev->base + VIRTIO_MMIO_STATUS) & 0xff;
}

static void vm_set_status(struct virtio_device *vdev, u8 status)
{
	struct virtio_mmio_device *vm_dev = to_virtio_mmio_device(vdev);

	/* We should never be setting status to 0. */
	BUG_ON(status == 0);

	writel(status, vm_dev->base + VIRTIO_MMIO_STATUS);
}

static void vm_reset(struct virtio_device *vdev)
{
	struct virtio_mmio_device *vm_dev = to_virtio_mmio_device(vdev);

	/* 0 status means a reset. */
	writel(0, vm_dev->base + VIRTIO_MMIO_STATUS);
}



/* Transport interface */

/* the notify function used when creating a virt queue */
static bool vm_notify(struct virtqueue *vq)
{
	struct virtio_mmio_device *vm_dev = to_virtio_mmio_device(vq->vdev);

	/* We write the queue's selector into the notification register to
	 * signal the other end */
	writel(vq->index, vm_dev->base + VIRTIO_MMIO_QUEUE_NOTIFY);
	return true;
}

/* Handle a configuration change: Tell driver if it wants to know. */
static irqreturn_t vm_config_changed(int irq, void *opaque)
{
	struct virtio_mmio_device *vm_dev = opaque;

	virtio_config_changed(&vm_dev->vdev);

	return IRQ_HANDLED;
}

/* Notify all virtqueues on an interrupt. */
static irqreturn_t vm_interrupt(int irq, void *opaque)
{
	struct virtio_mmio_device *vm_dev = opaque;
	struct virtio_mmio_vq_info *info;
	unsigned long status;
	unsigned long flags;
	irqreturn_t ret = IRQ_NONE;

	/* Read and acknowledge interrupts */
	status = readl(vm_dev->base + VIRTIO_MMIO_INTERRUPT_STATUS);
	writel(status, vm_dev->base + VIRTIO_MMIO_INTERRUPT_ACK);

	if (unlikely(status & VIRTIO_MMIO_INT_CONFIG)) {
		virtio_config_changed(&vm_dev->vdev);
		ret = IRQ_HANDLED;
	}

	if (likely(status & VIRTIO_MMIO_INT_VRING)) {
		spin_lock_irqsave(&vm_dev->lock, flags);
		list_for_each_entry(info, &vm_dev->virtqueues, node)
			ret |= vring_interrupt(irq, info->vq);
		spin_unlock_irqrestore(&vm_dev->lock, flags);
	}

	return ret;
}



static void vm_del_vq(struct virtqueue *vq)
{
	struct virtio_mmio_device *vm_dev = to_virtio_mmio_device(vq->vdev);
	struct virtio_mmio_vq_info *info = vq->priv;
	unsigned long flags;
	unsigned int index = vq->index;

	vm_unregister_msi_irq(vq, index + 1);

	spin_lock_irqsave(&vm_dev->lock, flags);
	list_del(&info->node);
	spin_unlock_irqrestore(&vm_dev->lock, flags);

	/* Select and deactivate the queue */
	writel(index, vm_dev->base + VIRTIO_MMIO_QUEUE_SEL);
	if (vm_dev->version == 1) {
		writel(0, vm_dev->base + VIRTIO_MMIO_QUEUE_PFN);
	} else {
		writel(0, vm_dev->base + VIRTIO_MMIO_QUEUE_READY);
		WARN_ON(readl(vm_dev->base + VIRTIO_MMIO_QUEUE_READY));
	}

	vring_del_virtqueue(vq);

	kfree(info);
}

static void vm_free_irqs(struct virtio_device *vdev)
{
	struct virtio_mmio_device *vm_dev = to_virtio_mmio_device(vdev);

	if (vm_dev->single_irq_enabled) {
		free_irq(platform_get_irq(vm_dev->pdev, 0), vm_dev);
		vm_dev->single_irq_enabled = 0;
	}
	vm_free_msi_irqs(vdev);
}

static void vm_del_vqs(struct virtio_device *vdev)
{
	struct virtqueue *vq, *n;

	list_for_each_entry_safe(vq, n, &vdev->vqs, list)
		vm_del_vq(vq);

	vm_free_irqs(vdev);
}

static int vm_request_single_irq(struct virtio_device *vdev)
{
	int err;
	struct virtio_mmio_device *vm_dev = to_virtio_mmio_device(vdev);
	int irq = platform_get_irq(vm_dev->pdev, 0);

	err = request_irq(irq, vm_interrupt, IRQF_SHARED,
			  dev_name(&vdev->dev), vm_dev);
	if (!err)
		vm_dev->single_irq_enabled = 1;
	return err;
}

static struct virtqueue *vm_setup_vq(struct virtio_device *vdev, unsigned index,
				  void (*callback)(struct virtqueue *vq),
				  const char *name, bool ctx)
{
	struct virtio_mmio_device *vm_dev = to_virtio_mmio_device(vdev);
	struct virtio_mmio_vq_info *info;
	struct virtqueue *vq;
	unsigned long flags;
	unsigned int num;
	int err;

	if (!name)
		return NULL;

	/* Select the queue we're interested in */
	writel(index, vm_dev->base + VIRTIO_MMIO_QUEUE_SEL);

	/* Queue shouldn't already be set up. */
	if (readl(vm_dev->base + (vm_dev->version == 1 ?
			VIRTIO_MMIO_QUEUE_PFN : VIRTIO_MMIO_QUEUE_READY))) {
		err = -ENOENT;
		goto error_available;
	}

	/* Allocate and fill out our active queue description */
	info = kmalloc(sizeof(*info), GFP_KERNEL);
	if (!info) {
		err = -ENOMEM;
		goto error_kmalloc;
	}

	num = readl(vm_dev->base + VIRTIO_MMIO_QUEUE_NUM_MAX);
	if (num == 0) {
		err = -ENOENT;
		goto error_new_virtqueue;
	}

	/* Create the vring */
	vq = vring_create_virtqueue(index, num, VIRTIO_MMIO_VRING_ALIGN, vdev,
				 true, true, ctx, vm_notify, callback, name);
	if (!vq) {
		err = -ENOMEM;
		goto error_new_virtqueue;
	}

	/* Activate the queue */
	writel(virtqueue_get_vring_size(vq), vm_dev->base + VIRTIO_MMIO_QUEUE_NUM);
	if (vm_dev->version == 1) {
		u64 q_pfn = virtqueue_get_desc_addr(vq) >> PAGE_SHIFT;

		/*
		 * virtio-mmio v1 uses a 32bit QUEUE PFN. If we have something
		 * that doesn't fit in 32bit, fail the setup rather than
		 * pretending to be successful.
		 */
		if (q_pfn >> 32) {
			dev_err(&vdev->dev,
				"platform bug: legacy virtio-mmio must not be used with RAM above 0x%llxGB\n",
				0x1ULL << (32 + PAGE_SHIFT - 30));
			err = -E2BIG;
			goto error_bad_pfn;
		}

		writel(PAGE_SIZE, vm_dev->base + VIRTIO_MMIO_QUEUE_ALIGN);
		writel(q_pfn, vm_dev->base + VIRTIO_MMIO_QUEUE_PFN);
	} else {
		u64 addr;

		addr = virtqueue_get_desc_addr(vq);
		writel((u32)addr, vm_dev->base + VIRTIO_MMIO_QUEUE_DESC_LOW);
		writel((u32)(addr >> 32),
				vm_dev->base + VIRTIO_MMIO_QUEUE_DESC_HIGH);

		addr = virtqueue_get_avail_addr(vq);
		writel((u32)addr, vm_dev->base + VIRTIO_MMIO_QUEUE_AVAIL_LOW);
		writel((u32)(addr >> 32),
				vm_dev->base + VIRTIO_MMIO_QUEUE_AVAIL_HIGH);

		addr = virtqueue_get_used_addr(vq);
		writel((u32)addr, vm_dev->base + VIRTIO_MMIO_QUEUE_USED_LOW);
		writel((u32)(addr >> 32),
				vm_dev->base + VIRTIO_MMIO_QUEUE_USED_HIGH);

		writel(1, vm_dev->base + VIRTIO_MMIO_QUEUE_READY);
	}

	vq->priv = info;
	info->vq = vq;

	spin_lock_irqsave(&vm_dev->lock, flags);
	list_add(&info->node, &vm_dev->virtqueues);
	spin_unlock_irqrestore(&vm_dev->lock, flags);

	err = vm_register_msi_irq(vq, index + 1, callback, name);
	if (err)
		goto error_request_irq;

	return vq;

error_request_irq:
	spin_lock_irqsave(&vm_dev->lock, flags);
	list_del(&info->node);
	spin_unlock_irqrestore(&vm_dev->lock, flags);
error_bad_pfn:
	vring_del_virtqueue(vq);
error_new_virtqueue:
	if (vm_dev->version == 1) {
		writel(0, vm_dev->base + VIRTIO_MMIO_QUEUE_PFN);
	} else {
		writel(0, vm_dev->base + VIRTIO_MMIO_QUEUE_READY);
		WARN_ON(readl(vm_dev->base + VIRTIO_MMIO_QUEUE_READY));
	}
	kfree(info);
error_kmalloc:
error_available:
	return ERR_PTR(err);
}

static int vm_find_vqs(struct virtio_device *vdev, unsigned nvqs,
		       struct virtqueue *vqs[],
		       vq_callback_t *callbacks[],
		       const char * const names[],
		       const bool *ctx,
		       struct irq_affinity *desc)
{
	int i, err, queue_idx = 0;

	/* Allocate one irq for configuration, and one irq for each virtque */
	err = vm_alloc_msi_irqs(vdev, nvqs + 1);
	if (err != 0)
		err = vm_request_single_irq(vdev);
	if (err)
		return err;

	for (i = 0; i < nvqs; ++i) {
		if (!names[i]) {
			vqs[i] = NULL;
			continue;
		}

		vqs[i] = vm_setup_vq(vdev, queue_idx++, callbacks[i], names[i],
				     ctx ? ctx[i] : false);
		if (IS_ERR(vqs[i])) {
			vm_del_vqs(vdev);
			return PTR_ERR(vqs[i]);
		}
	}

	return 0;
}

#ifdef CONFIG_VIRTIO_MMIO_DRAGONBALL
static int msi_irq_vector(struct device *dev, unsigned int nr)
{
	struct msi_desc *entry;

	entry = first_msi_entry(dev);

	return entry->irq + nr;
}

static inline void mmio_msi_set_enable(struct virtio_mmio_device *vm_dev,
					int enable)
{
	u16 csr;

	csr = readw(vm_dev->base + VIRTIO_MMIO_MSI_CSR);
	if (enable)
		csr |= VIRTIO_MMIO_MSI_CSR_ENABLE;
	else
		csr &= ~VIRTIO_MMIO_MSI_CSR_ENABLE;
	writew(csr, vm_dev->base + VIRTIO_MMIO_MSI_CSR);
}
static void mmio_write_msi_msg(struct msi_desc *desc, struct msi_msg *msg)
{
	struct device *dev;
	struct virtio_mmio_device *vm_dev;
	struct virtio_device *vdev;
	void __iomem *pos;
	uint16_t cmd = VIRTIO_MMIO_MSI_CMD(VIRTIO_MMIO_MSI_CMD_UPDATE,
						desc->platform.msi_index);

	dev = desc->dev;
	vdev = to_virtio_device(dev);
	vm_dev = to_virtio_mmio_device(vdev);
	pos = vm_dev->base;

	writel(msg->address_lo, pos + VIRTIO_MMIO_MSI_ADDRESS_LO);
	writel(msg->address_hi, pos + VIRTIO_MMIO_MSI_ADDRESS_HI);
	writel(msg->data, pos + VIRTIO_MMIO_MSI_DATA);
	writew(cmd, pos + VIRTIO_MMIO_MSI_COMMAND);
}

static int vm_alloc_msi_irqs(struct virtio_device *vdev, int nirqs)
{
	struct virtio_mmio_device *vm_dev = to_virtio_mmio_device(vdev);
	int irq, err;
	u16 csr = readw(vm_dev->base + VIRTIO_MMIO_MSI_CSR);

	if (vm_dev->msi_enabled || (csr & VIRTIO_MMIO_MSI_CSR_ENABLE) == 0)
		return -EINVAL;

	vm_dev->vm_vq_names = kmalloc_array(nirqs, sizeof(*vm_dev->vm_vq_names),
					GFP_KERNEL);
	if (!vm_dev->vm_vq_names)
		return -ENOMEM;

	if (!vdev->dev.msi_domain)
		vdev->dev.msi_domain = platform_msi_get_def_irq_domain();
	err = platform_msi_domain_alloc_irqs(&vdev->dev, nirqs,
						mmio_write_msi_msg);
	if (err)
		goto error_free_mem;

	/* The first MSI vector is used for configuration change events. */
	snprintf(vm_dev->vm_vq_names[0], sizeof(*vm_dev->vm_vq_names),
		"%s-config", dev_name(&vdev->dev));
	irq = msi_irq_vector(&vdev->dev, 0);
	err = request_irq(irq, vm_config_changed, 0, vm_dev->vm_vq_names[0],
			  vm_dev);
	if (err)
		goto error_free_mem;
	mmio_msi_set_enable(vm_dev, 1);
	vm_dev->msi_enabled = true;

	return 0;

error_free_mem:
	kfree(vm_dev->vm_vq_names);
	vm_dev->vm_vq_names = NULL;
	return err;
}

static void vm_free_msi_irqs(struct virtio_device *vdev)
{
	struct virtio_mmio_device *vm_dev = to_virtio_mmio_device(vdev);

	if (vm_dev->msi_enabled) {
		mmio_msi_set_enable(vm_dev, 0);
		free_irq(msi_irq_vector(&vdev->dev, 0), vm_dev);
		platform_msi_domain_free_irqs(&vdev->dev);
		kfree(vm_dev->vm_vq_names);
		vm_dev->vm_vq_names = NULL;
		vm_dev->msi_enabled = false;
	}
}

static int vm_register_msi_irq(struct virtqueue *vq, int index,
				void (*callback)(struct virtqueue *vq),
				const char *name)
{
	int err = -EOPNOTSUPP, irq;
	struct virtio_mmio_device *vm_dev = to_virtio_mmio_device(vq->vdev);

	if (!vm_dev->msi_enabled || !callback)
		return 0;

	/* Register interrupt handler for virtque MSI IRQs */
	irq = msi_irq_vector(&vq->vdev->dev, index);
	/* allocate per-vq irq if available and necessary */
	snprintf(vm_dev->vm_vq_names[index],
			sizeof(*vm_dev->vm_vq_names),
			"%s-%s",
			dev_name(&vm_dev->vdev.dev), name);
	err = request_irq(irq, vring_interrupt, 0,
			vm_dev->vm_vq_names[index], vq);

	return err;
}

static void vm_unregister_msi_irq(struct virtqueue *vq, int index)
{
	struct virtio_mmio_device *vm_dev = to_virtio_mmio_device(vq->vdev);

	if (vm_dev->msi_enabled) {
		int irq = msi_irq_vector(&vq->vdev->dev, index);

		free_irq(irq, vq);
	}
}
#endif  /* CONFIG_VIRTIO_MMIO_DRAGONBALL */

static const char *vm_bus_name(struct virtio_device *vdev)
{
	struct virtio_mmio_device *vm_dev = to_virtio_mmio_device(vdev);

	return vm_dev->pdev->name;
}

static bool vm_get_shm_region(struct virtio_device *vdev,
			      struct virtio_shm_region *region, u8 id)
{
	struct virtio_mmio_device *vm_dev = to_virtio_mmio_device(vdev);
	u64 len, addr;

	/* Select the region we're interested in */
	writel(id, vm_dev->base + VIRTIO_MMIO_SHM_SEL);

	/* Read the region size */
	len = (u64) readl(vm_dev->base + VIRTIO_MMIO_SHM_LEN_LOW);
	len |= (u64) readl(vm_dev->base + VIRTIO_MMIO_SHM_LEN_HIGH) << 32;

	region->len = len;

	/* Check if region length is -1. If that's the case, the shared memory
	 * region does not exist and there is no need to proceed further.
	 */
	if (len == ~(u64)0)
		return false;

	/* Read the region base address */
	addr = (u64) readl(vm_dev->base + VIRTIO_MMIO_SHM_BASE_LOW);
	addr |= (u64) readl(vm_dev->base + VIRTIO_MMIO_SHM_BASE_HIGH) << 32;

	region->addr = addr;

	return true;
}

static const struct virtio_config_ops virtio_mmio_config_ops = {
	.get		= vm_get,
	.set		= vm_set,
	.generation	= vm_generation,
	.get_status	= vm_get_status,
	.set_status	= vm_set_status,
	.reset		= vm_reset,
	.find_vqs	= vm_find_vqs,
	.del_vqs	= vm_del_vqs,
	.get_features	= vm_get_features,
	.finalize_features = vm_finalize_features,
	.bus_name	= vm_bus_name,
	.get_shm_region = vm_get_shm_region,
};

#ifdef CONFIG_PM_SLEEP
static int virtio_mmio_freeze(struct device *dev)
{
	struct virtio_mmio_device *vm_dev = dev_get_drvdata(dev);

	return virtio_device_freeze(&vm_dev->vdev);
}

static int virtio_mmio_restore(struct device *dev)
{
	struct virtio_mmio_device *vm_dev = dev_get_drvdata(dev);

	if (vm_dev->version == 1)
		writel(PAGE_SIZE, vm_dev->base + VIRTIO_MMIO_GUEST_PAGE_SIZE);

	return virtio_device_restore(&vm_dev->vdev);
}

static const struct dev_pm_ops virtio_mmio_pm_ops = {
	SET_SYSTEM_SLEEP_PM_OPS(virtio_mmio_freeze, virtio_mmio_restore)
};
#endif

static void virtio_mmio_release_dev(struct device *_d)
{
	struct virtio_device *vdev =
			container_of(_d, struct virtio_device, dev);
	struct virtio_mmio_device *vm_dev =
			container_of(vdev, struct virtio_mmio_device, vdev);
	struct platform_device *pdev = vm_dev->pdev;

	devm_kfree(&pdev->dev, vm_dev);
}

/* Platform device */

static int virtio_mmio_probe(struct platform_device *pdev)
{
	struct virtio_mmio_device *vm_dev;
	unsigned long magic;
	int rc;

	vm_dev = devm_kzalloc(&pdev->dev, sizeof(*vm_dev), GFP_KERNEL);
	if (!vm_dev)
		return -ENOMEM;

	vm_dev->vdev.dev.parent = &pdev->dev;
	vm_dev->vdev.dev.release = virtio_mmio_release_dev;
	vm_dev->vdev.config = &virtio_mmio_config_ops;
#ifdef CONFIG_ARM64
	if (has_dragonball_feature(DB_FEAT_PCI_PASSTHROUGH))
		vm_dev->vdev.dev.of_node = pdev->dev.of_node;
#endif
	vm_dev->pdev = pdev;
	INIT_LIST_HEAD(&vm_dev->virtqueues);
	spin_lock_init(&vm_dev->lock);

	vm_dev->base = devm_platform_ioremap_resource(pdev, 0);
	if (IS_ERR(vm_dev->base))
		return PTR_ERR(vm_dev->base);

	/* Check magic value */
	magic = readl(vm_dev->base + VIRTIO_MMIO_MAGIC_VALUE);
	if (magic != ('v' | 'i' << 8 | 'r' << 16 | 't' << 24)) {
		dev_warn(&pdev->dev, "Wrong magic value 0x%08lx!\n", magic);
		return -ENODEV;
	}

	/* Check device version */
	vm_dev->version = readl(vm_dev->base + VIRTIO_MMIO_VERSION);
	if (vm_dev->version < 1 || vm_dev->version > 2) {
		dev_err(&pdev->dev, "Version %ld not supported!\n",
				vm_dev->version);
		return -ENXIO;
	}

	vm_dev->vdev.id.device = readl(vm_dev->base + VIRTIO_MMIO_DEVICE_ID);
	if (vm_dev->vdev.id.device == 0) {
		/*
		 * virtio-mmio device with an ID 0 is a (dummy) placeholder
		 * with no function. End probing now with no error reported.
		 */
		return -ENODEV;
	}
	vm_dev->vdev.id.vendor = readl(vm_dev->base + VIRTIO_MMIO_VENDOR_ID);

#ifdef CONFIG_PVDMA
	if (pvdma_is_enabled && check_vendor_db_pvdma(vm_dev->vdev.id.vendor)) {
		pr_info("enable pvdma for %s\n", dev_name(&pdev->dev));
		set_dma_ops(&pdev->dev, mmio_dma_ops);
		pdev->dev.dma_mask = &pdev->dev.coherent_dma_mask;
	}
#endif

	if (vm_dev->version == 1) {
		writel(PAGE_SIZE, vm_dev->base + VIRTIO_MMIO_GUEST_PAGE_SIZE);

		rc = dma_set_mask(&pdev->dev, DMA_BIT_MASK(64));
		/*
		 * In the legacy case, ensure our coherently-allocated virtio
		 * ring will be at an address expressable as a 32-bit PFN.
		 */
		if (!rc)
			dma_set_coherent_mask(&pdev->dev,
					      DMA_BIT_MASK(32 + PAGE_SHIFT));
	} else {
		rc = dma_set_mask_and_coherent(&pdev->dev, DMA_BIT_MASK(64));
	}
	if (rc)
		rc = dma_set_mask_and_coherent(&pdev->dev, DMA_BIT_MASK(32));
	if (rc)
		dev_warn(&pdev->dev, "Failed to enable 64-bit or 32-bit DMA.  Trying to continue, but this might not work.\n");

	platform_set_drvdata(pdev, vm_dev);

	rc = register_virtio_device(&vm_dev->vdev);
	if (rc)
		put_device(&vm_dev->vdev.dev);

	return rc;
}

static int virtio_mmio_remove(struct platform_device *pdev)
{
	struct virtio_mmio_device *vm_dev = platform_get_drvdata(pdev);
	unregister_virtio_device(&vm_dev->vdev);

	return 0;
}



/* Devices list parameter */

#if defined(CONFIG_VIRTIO_MMIO_CMDLINE_DEVICES)

static struct device vm_cmdline_parent = {
	.init_name = "virtio-mmio-cmdline",
};

static int vm_cmdline_parent_registered;
static int vm_cmdline_id;

static int vm_add_device(struct resource *resources, size_t res_size)
{

	int err;
	struct platform_device *pdev;

	if (!vm_cmdline_parent_registered) {
		err = device_register(&vm_cmdline_parent);
		if (err) {
			pr_err("Failed to register parent device!\n");
			return err;
		}
		vm_cmdline_parent_registered = 1;
	}

	pr_info("Registering device virtio-mmio.%d at 0x%llx-0x%llx, IRQ %d.\n",
			vm_cmdline_id,
			(unsigned long long)resources[0].start,
			(unsigned long long)resources[0].end,
			(int)resources[1].start);

	pdev = platform_device_register_resndata(&vm_cmdline_parent,
			"virtio-mmio", vm_cmdline_id++,
			resources, res_size, NULL, 0);

	return PTR_ERR_OR_ZERO(pdev);
}

static int vm_cmdline_set(const char *device,
		const struct kernel_param *kp)
{
	struct resource resources[2] = {};
	char *str;
	long long int base, size;
	unsigned int irq;
	int processed, consumed = 0;

	/* Consume "size" part of the command line parameter */
	size = memparse(device, &str);

	/* Get "@<base>:<irq>[:<id>]" chunks */
	processed = sscanf(str, "@%lli:%u%n:%d%n",
			&base, &irq, &consumed,
			&vm_cmdline_id, &consumed);

	/*
	 * sscanf() must process at least 2 chunks; also there
	 * must be no extra characters after the last chunk, so
	 * str[consumed] must be '\0'
	 */
	if (processed < 2 || str[consumed] || irq == 0)
		return -EINVAL;

	resources[0].flags = IORESOURCE_MEM;
	resources[0].start = base;
	resources[0].end = base + size - 1;

	resources[1].flags = IORESOURCE_IRQ;
	resources[1].start = resources[1].end = irq;

	return vm_add_device(resources, ARRAY_SIZE(resources));
}

static int vm_cmdline_get_device(struct device *dev, void *data)
{
	char *buffer = data;
	unsigned int len = strlen(buffer);
	struct platform_device *pdev = to_platform_device(dev);

	snprintf(buffer + len, PAGE_SIZE - len, "0x%llx@0x%llx:%llu:%d\n",
			pdev->resource[0].end - pdev->resource[0].start + 1ULL,
			(unsigned long long)pdev->resource[0].start,
			(unsigned long long)pdev->resource[1].start,
			pdev->id);
	return 0;
}

static int vm_cmdline_get(char *buffer, const struct kernel_param *kp)
{
	buffer[0] = '\0';
	device_for_each_child(&vm_cmdline_parent, buffer,
			vm_cmdline_get_device);
	return strlen(buffer) + 1;
}

static const struct kernel_param_ops vm_cmdline_param_ops = {
	.set = vm_cmdline_set,
	.get = vm_cmdline_get,
};

device_param_cb(device, &vm_cmdline_param_ops, NULL, S_IRUSR);

static int vm_unregister_cmdline_device(struct device *dev,
		void *data)
{
	platform_device_unregister(to_platform_device(dev));

	return 0;
}

static void vm_unregister_cmdline_devices(void)
{
	if (vm_cmdline_parent_registered) {
		device_for_each_child(&vm_cmdline_parent, NULL,
				vm_unregister_cmdline_device);
		device_unregister(&vm_cmdline_parent);
		vm_cmdline_parent_registered = 0;
	}
}

#ifdef CONFIG_DRAGONBALL_DEVICE_MANAGER
static int vm_match_device(struct device *dev, void *data)
{
	struct resource *resource = (struct resource *)data;
	struct platform_device *pdev = to_platform_device(dev);

	if ((pdev->resource[0].start == resource[0].start)  &&
		(pdev->resource[0].end == resource[0].end) &&
		(pdev->resource[1].start == resource[1].start))
		return 1;
	return 0;
}

static struct device *vm_find_device(struct resource *res)
{
	return device_find_child(&vm_cmdline_parent, res, vm_match_device);
}

static int vm_device_overlap(struct device *dev, void *data)
{
	struct resource *res = (struct resource *)data;
	struct platform_device *pdev = to_platform_device(dev);

	/* Detect IRQ number conflicts except shared IRQs. */
	if (!(res[1].flags & IORESOURCE_IRQ_SHAREABLE) &&
		(pdev->resource[1].start == res[1].start)) {
		return 1;
	}

	/* Detect device MMIO addresses overlapping */
	if ((pdev->resource[0].start < res[0].end) &&
		(pdev->resource[0].end > res[0].start)) {
		return 1;
	}

	return 0;
}

static struct device *vm_detect_resource(struct resource *res)
{
	/* return NULL if no resource overlapped */
	return device_find_child(&vm_cmdline_parent, res, vm_device_overlap);
}

/*
 * virtio_mmio_add_device is used to support the device hot-plug feature of dragonball,
 * and hot-plug the specified virtio_mmio device into the guest kernel. The internal
 * process of this function will call bus_probe_device, traverse and bind the
 * corresponding driver for this device.
 */
int virtio_mmio_add_device(struct resource *resources, size_t res_size)
{
	int err;
	struct device *dev;

	if (res_size < 2 || !resources)
		return -EINVAL;

	dev = vm_detect_resource(resources);
	if (dev) {
		put_device(dev);
		return -EEXIST;
	}

	lock_device_hotplug();
	err = vm_add_device(resources, res_size);
	unlock_device_hotplug();

	return err;
}
EXPORT_SYMBOL_GPL(virtio_mmio_add_device);

int virtio_mmio_del_device(struct resource *resources, size_t res_size)
{
	int ret;
	struct device *dev;

	if (res_size < 2 || !resources)
		return -EINVAL;

	dev = vm_find_device(resources);
	if (!dev)
		return -ENODEV;

	put_device(dev);
	lock_device_hotplug();
	ret = vm_unregister_cmdline_device(dev, NULL);
	unlock_device_hotplug();

	return ret;
}
EXPORT_SYMBOL_GPL(virtio_mmio_del_device);
#endif  /* CONFIG_DRAGONBALL_DEVICE_MANAGER */

#else

static void vm_unregister_cmdline_devices(void)
{
}

#endif

/* Platform driver */

static const struct of_device_id virtio_mmio_match[] = {
	{ .compatible = "virtio,mmio", },
	{},
};
MODULE_DEVICE_TABLE(of, virtio_mmio_match);

#ifdef CONFIG_ACPI
static const struct acpi_device_id virtio_mmio_acpi_match[] = {
	{ "LNRO0005", },
	{ }
};
MODULE_DEVICE_TABLE(acpi, virtio_mmio_acpi_match);
#endif

static struct platform_driver virtio_mmio_driver = {
	.probe		= virtio_mmio_probe,
	.remove		= virtio_mmio_remove,
	.driver		= {
		.name	= "virtio-mmio",
		.of_match_table	= virtio_mmio_match,
		.acpi_match_table = ACPI_PTR(virtio_mmio_acpi_match),
#ifdef CONFIG_PM_SLEEP
		.pm	= &virtio_mmio_pm_ops,
#endif
	},
};

static int __init virtio_mmio_init(void)
{
	return platform_driver_register(&virtio_mmio_driver);
}

static void __exit virtio_mmio_exit(void)
{
	platform_driver_unregister(&virtio_mmio_driver);
	vm_unregister_cmdline_devices();
}

module_init(virtio_mmio_init);
module_exit(virtio_mmio_exit);

MODULE_AUTHOR("Pawel Moll <pawel.moll@arm.com>");
MODULE_DESCRIPTION("Platform bus driver for memory mapped virtio devices");
MODULE_LICENSE("GPL");
