// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2020-2022 Alibaba Corporation. All rights reserved.
 * Author: Zelin Deng <zelin.deng@linux.alibaba.com>
 * Author: Guanjun <guanjun@linux.alibaba.com>
 * Author: Jiayu Ni <jiayu.ni@linux.alibaba.com>
 */

#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <endian.h>

#include "../utils/utils.h"
#include "udma_ulib.h"
#include "ycc_algs.h"
#include "rng.h"

static int ycc_rng_done_callback(void *ptr, uint16_t state)
{
	struct rng_ctx *cipher = (struct rng_ctx *)ptr;
	struct ycc_rng_ctx *ctx = (struct ycc_rng_ctx *)cipher->__ctx;

	if (state == CMD_SUCCESS)
		memcpy(cipher->result, ctx->dst_vaddr, cipher->len);

	ycc_udma_free(ctx->dst_vaddr);
	ctx->dst_vaddr = NULL;

	if (cipher->complete)
		cipher->complete(cipher, state == CMD_SUCCESS ? 0 : -EBADMSG);

	return 0;
}

int ycc_rng_generate(struct rng_ctx *cipher)
{
	struct ycc_rng_ctx *ctx = (struct ycc_rng_ctx *)cipher->__ctx;
	struct ycc_rng_cmd *rng_cmd;
	struct ycc_flags *aflags;
	int ret = -EINVAL;

	if (ctx->dst_vaddr || !cipher->result)
		goto out;

	ctx->dst_vaddr = ycc_udma_malloc(cipher->len);
	if (!ctx->dst_vaddr) {
		ret = -ENOMEM;
		goto out;
	}

	aflags = malloc(sizeof(struct ycc_flags));
	if (!aflags) {
		ret = -ENOMEM;
		goto free_dst;
	}

	aflags->ptr = (void *)cipher;
	aflags->ycc_done_callback = ycc_rng_done_callback;

	memset(&ctx->desc, 0, sizeof(ctx->desc));
	ctx->desc.private_ptr = (uint64_t)aflags;

	rng_cmd         = &ctx->desc.cmd.rng_cmd;
	rng_cmd->cmd_id = YCC_CMD_RNG_GEN;
	rng_cmd->dptr   = virt_to_phys(ctx->dst_vaddr);
	rng_cmd->dlen   = cipher->len;

	ret = ycc_enqueue(ctx->ring, (uint8_t *)&ctx->desc);
	if (!ret)
		return -EINPROGRESS;

	free(aflags);
free_dst:
	ycc_udma_free(ctx->dst_vaddr);
	ctx->dst_vaddr = NULL;
out:
	return ret;
}

int ycc_rng_init(struct rng_ctx *cipher)
{
	struct ycc_rng_ctx *ctx = (struct ycc_rng_ctx *)cipher->__ctx;
	struct ycc_ring *ring;

	ring = ycc_crypto_get_ring();
	if (!ring)
		return -EINVAL;

	ctx->ring = ring;
	return 0;
}

void ycc_rng_exit(struct rng_ctx *cipher)
{
	struct ycc_rng_ctx *ctx = (struct ycc_rng_ctx *)cipher->__ctx;

	if (ctx->ring)
		ycc_crypto_free_ring(ctx->ring);

	if (ctx->dst_vaddr)
		ycc_udma_free(ctx->dst_vaddr);
}

static struct rng_alg rng_algs[] = {
	{
		.name = "trng",
		.ctxsize = sizeof(struct ycc_rng_ctx),
		.seedsize = 0,
		.generate = ycc_rng_generate,
		.seed = NULL,
		.init = ycc_rng_init,
		.exit = ycc_rng_exit,
	},
	{},
};

extern struct rng_alg_entry *rng_alg_entries;
void rng_register_algs(void)
{
	struct rng_alg_entry *prev = rng_alg_entries, *cur;
	uint32_t array_size = sizeof(rng_algs) / sizeof(struct rng_alg);
	int i;

	for (i = 0; i < array_size; i++) {
		cur = malloc(sizeof(struct rng_alg_entry));
		if (!cur) {
			ycc_err("Failed to alloc memory for alg:%s\n", rng_algs[i].name);
			break;
		}
		cur->alg = &rng_algs[i];
		cur->next = NULL;
		if (!prev) {
			rng_alg_entries = cur;
			prev = cur;
		} else {
			prev->next = cur;
			prev = cur;
		}
	}
}

void rng_unregister_algs(void)
{
	struct rng_alg_entry *cur = rng_alg_entries, *next;

	while (cur) {
		next = cur->next;
		free(cur);
		cur = next;
	};
	rng_alg_entries = NULL;
}
