// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2020-2022 Alibaba Corporation. All rights reserved.
 * Author: Zelin Deng <zelin.deng@linux.alibaba.com>
 * Author: Guanjun <guanjun@linux.alibaba.com>
 * Author: Jiayu Ni <jiayu.ni@linux.alibaba.com>
 */

#include <pthread.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "ycc_uio.h"
#include "utils.h"
#include "ske.h"

struct skcipher_test {
	const char *name;
	const char *key;
	const char *iv;
	const char *iv_out;
	const char *ptext;
	const char *ctext;
	unsigned int key_len; /* length of key */
	unsigned int len; /* length of ptext or ctext*/
};

static const struct skcipher_test ske = {
	.name = "ske-unit",
	.key = "\x06\xa9\x21\x40\x36\xb8\xa1\x5b"
	       "\x51\x2e\x03\xd5\x34\x12\x00\x06",
	.key_len = 16,
	.iv = "\x3d\xaf\xba\x42\x9d\x9e\xb4\x30"
	      "\xb4\x22\xda\x80\x2c\x9f\xac\x41",
	.ptext = "Single block msg",
	.ctext = "\xe3\x53\x77\x9c\x10\x79\xae\xb8"
		 "\x27\x08\x94\x2d\xbe\x77\x18\x1a",
	.len = 16,
};

static bool inited;

/*
 * Destructor: It is must if driver initialize successfully.
 */
static void __attribute__((destructor)) drv_exit(void)
{
	if (inited)
		ycc_drv_exit();
}

static unsigned int done0, done1;
static void *ske_test0(void *arg)
{
	struct skcipher_ctx *ctx = (struct skcipher_ctx *)arg;
	struct skcipher_req *req;
	unsigned char ptext[16];
	unsigned char ctext[16];
	unsigned char iv[16];
	int i = 100000;
	int ret = 0;

	req = skcipher_alloc_req(ctx);
	if (!req)
		return NULL;

	memcpy(ptext, ske.ptext, ske.len);
	memcpy(iv, ske.iv, sizeof(ske.iv) - 1);
	skcipher_set_req(req, ptext, ctext, ske.len, iv);

	while (i--)
		ret += skcipher_encrypt(req);

	if (!ret)
		printf("Sample code multi-thread-ske thread0 passed\n");

	skcipher_free_req(req);
	done0 = 1;
	return NULL;
}

static void *ske_test1(void *arg)
{
	struct skcipher_ctx *ctx = (struct skcipher_ctx *)arg;
	struct skcipher_req *req;
	unsigned char ptext[16];
	unsigned char ctext[16];
	unsigned char iv[16];
	int i = 100000;
	int ret = 0;

	req = skcipher_alloc_req(ctx);
	if (!req)
		return NULL;

	memcpy(ctext, ske.ctext, ske.len);
	memcpy(iv, ske.iv, sizeof(ske.iv) - 1);
	skcipher_set_req(req, ctext, ptext, ske.len, iv);

	while (i--)
		ret += skcipher_decrypt(req);

	if (!ret)
		printf("Sample code multi-thread-ske thread1 passed\n");

	skcipher_free_req(req);
	done1 = 1;
	return NULL;
}

int main(int argc, char *argv[])
{
	pthread_t thread0, thread1;
	struct skcipher_ctx *ctx;
	unsigned char key[16];
	int ret;

	ret = ycc_drv_init(1);
	if (ret < 0)
		return ret;

	inited = true;
	ret = -1;
	ctx = skcipher_alloc_ctx("cbc(aes)", CRYPTO_SYNC);
	if (!ctx)
		goto out;

	memcpy(key, ske.key, ske.key_len);
	skcipher_setkey(ctx, key, ske.key_len);

	ret = pthread_create(&thread0, NULL, ske_test0, (void *)ctx);
	if (ret)
		done0 = 1;

	ret = pthread_create(&thread1, NULL, ske_test1, (void *)ctx);
	if (ret)
		done1 = 1;

	/* Wait for both two thread completion */
	while (done0 == 0 || done1 == 0);

	skcipher_free_ctx(ctx);
out:
	return ret;
}
