/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.application;

import java.util.EnumMap;
import java.util.Optional;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonar.application.AppReloader;
import org.sonar.application.AppState;
import org.sonar.application.AppStateListener;
import org.sonar.application.NodeLifecycle;
import org.sonar.application.ProcessLauncher;
import org.sonar.application.Scheduler;
import org.sonar.application.command.AbstractCommand;
import org.sonar.application.command.CommandFactory;
import org.sonar.application.config.AppSettings;
import org.sonar.application.config.ClusterSettings;
import org.sonar.application.process.ManagedProcessEventListener;
import org.sonar.application.process.ManagedProcessHandler;
import org.sonar.application.process.ManagedProcessLifecycle;
import org.sonar.application.process.ProcessLifecycleListener;
import org.sonar.process.ProcessId;
import org.sonar.process.ProcessProperties;

public class SchedulerImpl
implements Scheduler,
ManagedProcessEventListener,
ProcessLifecycleListener,
AppStateListener {
    private static final Logger LOG = LoggerFactory.getLogger(SchedulerImpl.class);
    private static final ManagedProcessHandler.Timeout HARD_STOP_TIMEOUT = ManagedProcessHandler.Timeout.newTimeout(1L, TimeUnit.MINUTES);
    private final AppSettings settings;
    private final AppReloader appReloader;
    private final CommandFactory commandFactory;
    private final ProcessLauncher processLauncher;
    private final AppState appState;
    private final NodeLifecycle nodeLifecycle = new NodeLifecycle();
    private final CountDownLatch awaitTermination = new CountDownLatch(1);
    private final AtomicBoolean firstWaitingEsLog = new AtomicBoolean(true);
    private final EnumMap<ProcessId, ManagedProcessHandler> processesById = new EnumMap(ProcessId.class);
    private final AtomicInteger operationalCountDown = new AtomicInteger();
    private final AtomicInteger stopCountDown = new AtomicInteger(0);
    private RestartStopperThread restartStopperThread;
    private HardStopperThread hardStopperThread;
    private RestarterThread restarterThread;
    private long processWatcherDelayMs = 500L;

    public SchedulerImpl(AppSettings settings, AppReloader appReloader, CommandFactory commandFactory, ProcessLauncher processLauncher, AppState appState) {
        this.settings = settings;
        this.appReloader = appReloader;
        this.commandFactory = commandFactory;
        this.processLauncher = processLauncher;
        this.appState = appState;
        this.appState.addListener(this);
    }

    SchedulerImpl setProcessWatcherDelayMs(long l) {
        this.processWatcherDelayMs = l;
        return this;
    }

    @Override
    public void schedule() throws InterruptedException {
        if (!this.nodeLifecycle.tryToMoveTo(NodeLifecycle.State.STARTING)) {
            return;
        }
        this.processesById.clear();
        for (ProcessId processId : ClusterSettings.getEnabledProcesses(this.settings)) {
            ManagedProcessHandler process = ManagedProcessHandler.builder(processId).addProcessLifecycleListener(this).addEventListener(this).setWatcherDelayMs(this.processWatcherDelayMs).setStopTimeout(SchedulerImpl.stopTimeoutFor(processId, this.settings)).setHardStopTimeout(HARD_STOP_TIMEOUT).build();
            this.processesById.put(process.getProcessId(), process);
        }
        this.operationalCountDown.set(this.processesById.size());
        this.tryToStartAll();
    }

    private static ManagedProcessHandler.Timeout stopTimeoutFor(ProcessId processId, AppSettings settings) {
        switch (processId) {
            case ELASTICSEARCH: {
                return HARD_STOP_TIMEOUT;
            }
            case WEB_SERVER: {
                return ManagedProcessHandler.Timeout.newTimeout(SchedulerImpl.getStopTimeoutMs(settings, ProcessProperties.Property.WEB_GRACEFUL_STOP_TIMEOUT), TimeUnit.MILLISECONDS);
            }
            case COMPUTE_ENGINE: {
                return ManagedProcessHandler.Timeout.newTimeout(SchedulerImpl.getStopTimeoutMs(settings, ProcessProperties.Property.CE_GRACEFUL_STOP_TIMEOUT), TimeUnit.MILLISECONDS);
            }
        }
        throw new IllegalArgumentException("Unsupported processId " + processId);
    }

    private static long getStopTimeoutMs(AppSettings settings, ProcessProperties.Property property) {
        String timeoutMs = settings.getValue(property.getKey()).orElse(property.getDefaultValue());
        long gracePeriod = HARD_STOP_TIMEOUT.getUnit().toMillis(HARD_STOP_TIMEOUT.getDuration());
        return ProcessProperties.parseTimeoutMs((ProcessProperties.Property)property, (String)timeoutMs) + gracePeriod;
    }

    private void tryToStartAll() throws InterruptedException {
        this.tryToStartEs();
        this.tryToStartWeb();
        this.tryToStartCe();
    }

    private void tryToStartEs() throws InterruptedException {
        ManagedProcessHandler process = this.processesById.get(ProcessId.ELASTICSEARCH);
        if (process != null) {
            this.tryToStartProcess(process, this.commandFactory::createEsCommand);
        }
    }

    private void tryToStartWeb() throws InterruptedException {
        ManagedProcessHandler process = this.processesById.get(ProcessId.WEB_SERVER);
        if (process == null) {
            return;
        }
        if (!this.isEsClientStartable()) {
            if (this.firstWaitingEsLog.getAndSet(false)) {
                LOG.info("Waiting for Elasticsearch to be up and running");
            }
            return;
        }
        if (this.appState.isOperational(ProcessId.WEB_SERVER, false)) {
            this.tryToStartProcess(process, () -> this.commandFactory.createWebCommand(false));
        } else if (this.appState.tryToLockWebLeader()) {
            this.tryToStartProcess(process, () -> this.commandFactory.createWebCommand(true));
        } else {
            Optional<String> leader = this.appState.getLeaderHostName();
            if (leader.isPresent()) {
                LOG.info("Waiting for initialization from {}", (Object)leader.get());
            } else {
                LOG.error("Initialization failed. All nodes must be restarted");
            }
        }
    }

    private void tryToStartCe() throws InterruptedException {
        ManagedProcessHandler process = this.processesById.get(ProcessId.COMPUTE_ENGINE);
        if (process != null && this.appState.isOperational(ProcessId.WEB_SERVER, true) && this.isEsClientStartable()) {
            this.tryToStartProcess(process, this.commandFactory::createCeCommand);
        }
    }

    private boolean isEsClientStartable() {
        boolean requireLocalEs = ClusterSettings.isLocalElasticsearchEnabled(this.settings);
        return this.appState.isOperational(ProcessId.ELASTICSEARCH, requireLocalEs);
    }

    private void tryToStartProcess(ManagedProcessHandler processHandler, Supplier<AbstractCommand> commandSupplier) throws InterruptedException {
        if (Thread.currentThread().isInterrupted()) {
            throw new InterruptedException();
        }
        try {
            processHandler.start(() -> {
                AbstractCommand command = (AbstractCommand)commandSupplier.get();
                return this.processLauncher.launch(command);
            });
        }
        catch (RuntimeException e) {
            this.hardStop();
            throw e;
        }
    }

    @Override
    public void stop() {
        if (this.nodeLifecycle.tryToMoveTo(NodeLifecycle.State.STOPPING)) {
            LOG.info("Stopping SonarQube");
        }
        this.stopImpl();
    }

    private void stopImpl() {
        try {
            this.stopAll();
            this.finalizeStop();
        }
        catch (InterruptedException e) {
            LOG.debug("Stop interrupted", (Throwable)e);
            Thread.currentThread().interrupt();
        }
    }

    private void stopAll() throws InterruptedException {
        this.stopProcess(ProcessId.COMPUTE_ENGINE);
        this.stopProcess(ProcessId.WEB_SERVER);
        this.stopProcess(ProcessId.ELASTICSEARCH);
    }

    private void stopProcess(ProcessId processId) throws InterruptedException {
        ManagedProcessHandler process = this.processesById.get(processId);
        if (process != null) {
            LOG.debug("Stopping [{}]...", (Object)process.getProcessId().getKey());
            process.stop();
        }
    }

    @Override
    public void hardStop() {
        if (this.nodeLifecycle.tryToMoveTo(NodeLifecycle.State.HARD_STOPPING)) {
            LOG.info("Hard stopping SonarQube");
        }
        this.hardStopImpl();
    }

    private void hardStopImpl() {
        try {
            this.hardStopAll();
            this.finalizeStop();
        }
        catch (InterruptedException e) {
            LOG.debug("Stopping all processes was interrupted in the middle of a hard stop (current thread name is \"{}\")", (Object)Thread.currentThread().getName());
            Thread.currentThread().interrupt();
        }
    }

    private void hardStopAll() throws InterruptedException {
        this.hardStopProcess(ProcessId.COMPUTE_ENGINE);
        this.hardStopProcess(ProcessId.WEB_SERVER);
        this.hardStopProcess(ProcessId.ELASTICSEARCH);
    }

    private void finalizeStop() {
        if (this.nodeLifecycle.getState() != NodeLifecycle.State.RESTARTING) {
            SchedulerImpl.interrupt(this.restartStopperThread);
            SchedulerImpl.interrupt(this.hardStopperThread);
            SchedulerImpl.interrupt(this.restarterThread);
            if (this.nodeLifecycle.tryToMoveTo(NodeLifecycle.State.STOPPED)) {
                LOG.info("SonarQube is stopped");
            }
            this.awaitTermination.countDown();
        }
    }

    private static void interrupt(@Nullable Thread thread) {
        if (thread != null && Thread.currentThread() != thread) {
            thread.interrupt();
        }
    }

    private void hardStopProcess(ProcessId processId) throws InterruptedException {
        ManagedProcessHandler process = this.processesById.get(processId);
        if (process != null) {
            process.hardStop();
        }
    }

    @Override
    public void awaitTermination() {
        try {
            this.awaitTermination.await();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    @Override
    public void onManagedProcessEvent(ProcessId processId, ManagedProcessEventListener.Type type) {
        if (type == ManagedProcessEventListener.Type.OPERATIONAL) {
            this.onProcessOperational(processId);
        } else if (type == ManagedProcessEventListener.Type.ASK_FOR_RESTART && this.nodeLifecycle.tryToMoveTo(NodeLifecycle.State.RESTARTING)) {
            LOG.info("SQ restart requested by Process[{}]", (Object)processId.getKey());
            this.stopAsyncForRestart();
        }
    }

    private void onProcessOperational(ProcessId processId) {
        boolean lastProcessStarted;
        LOG.info("Process[{}] is up", (Object)processId.getKey());
        this.appState.setOperational(processId);
        boolean bl = lastProcessStarted = this.operationalCountDown.decrementAndGet() == 0;
        if (lastProcessStarted && this.nodeLifecycle.tryToMoveTo(NodeLifecycle.State.OPERATIONAL)) {
            LOG.info("SonarQube is up");
        }
    }

    @Override
    public void onAppStateOperational(ProcessId processId) {
        if (this.nodeLifecycle.getState() == NodeLifecycle.State.STARTING) {
            try {
                this.tryToStartAll();
            }
            catch (InterruptedException e) {
                LOG.debug("Startup process was interrupted on notification that process [{}] was operation", (Object)processId.getKey(), (Object)e);
                this.hardStopAsync();
                Thread.currentThread().interrupt();
            }
        }
    }

    @Override
    public void onProcessState(ProcessId processId, ManagedProcessLifecycle.State to) {
        switch (to) {
            case STOPPED: {
                this.onProcessStop(processId);
                break;
            }
            case STARTING: {
                this.stopCountDown.incrementAndGet();
                break;
            }
        }
    }

    private void onProcessStop(ProcessId processId) {
        LOG.info("Process[{}] is stopped", (Object)processId.getKey());
        boolean lastProcessStopped = this.stopCountDown.decrementAndGet() == 0;
        switch (this.nodeLifecycle.getState()) {
            case RESTARTING: {
                if (!lastProcessStopped) break;
                LOG.info("SonarQube is restarting");
                this.restartAsync();
                break;
            }
            case HARD_STOPPING: 
            case STOPPING: {
                if (!lastProcessStopped) break;
                this.finalizeStop();
                break;
            }
            default: {
                this.hardStopAsync();
            }
        }
    }

    private void hardStopAsync() {
        if (this.hardStopperThread != null) {
            LOG.debug("Hard stopper thread was not null (name is \"{}\")", (Object)this.hardStopperThread.getName(), (Object)new Exception());
            this.hardStopperThread.interrupt();
        }
        this.hardStopperThread = new HardStopperThread();
        this.hardStopperThread.start();
    }

    private void stopAsyncForRestart() {
        if (this.restartStopperThread != null) {
            LOG.debug("Restart stopper thread was not null", (Throwable)new Exception());
            this.restartStopperThread.interrupt();
        }
        this.restartStopperThread = new RestartStopperThread();
        this.restartStopperThread.start();
    }

    private void restartAsync() {
        if (this.restarterThread != null) {
            LOG.debug("Restarter thread was not null (name is \"{}\")", (Object)this.restarterThread.getName(), (Object)new Exception());
            this.restarterThread.interrupt();
        }
        this.restarterThread = new RestarterThread();
        this.restarterThread.start();
    }

    private class HardStopperThread
    extends Thread {
        private HardStopperThread() {
            super("Hard stopper");
        }

        @Override
        public void run() {
            if (SchedulerImpl.this.nodeLifecycle.tryToMoveTo(NodeLifecycle.State.HARD_STOPPING)) {
                SchedulerImpl.this.hardStopImpl();
            }
        }
    }

    private class RestartStopperThread
    extends Thread {
        private RestartStopperThread() {
            super("Restart stopper");
        }

        @Override
        public void run() {
            SchedulerImpl.this.stopImpl();
        }
    }

    private class RestarterThread
    extends Thread {
        private RestarterThread() {
            super("Restarter");
        }

        @Override
        public void run() {
            try {
                SchedulerImpl.this.appReloader.reload(SchedulerImpl.this.settings);
                SchedulerImpl.this.schedule();
            }
            catch (InterruptedException e) {
                LOG.debug("{} thread was interrupted", (Object)this.getName(), (Object)e);
                super.interrupt();
            }
            catch (Exception e) {
                LOG.error("Failed to restart", (Throwable)e);
                SchedulerImpl.this.hardStop();
            }
        }
    }
}

