/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.ce.taskprocessor;

import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableScheduledFuture;
import java.util.Arrays;
import java.util.concurrent.TimeUnit;
import javax.annotation.CheckForNull;
import javax.annotation.Nullable;
import org.sonar.api.utils.log.Logger;
import org.sonar.api.utils.log.Loggers;
import org.sonar.ce.configuration.CeConfiguration;
import org.sonar.ce.taskprocessor.CeProcessingScheduler;
import org.sonar.ce.taskprocessor.CeProcessingSchedulerExecutorService;
import org.sonar.ce.taskprocessor.CeWorker;
import org.sonar.ce.taskprocessor.CeWorkerController;
import org.sonar.ce.taskprocessor.CeWorkerFactory;

public class CeProcessingSchedulerImpl
implements CeProcessingScheduler {
    private static final Logger LOG = Loggers.get(CeProcessingSchedulerImpl.class);
    private static final long DELAY_BETWEEN_DISABLED_TASKS = 30000L;
    private final CeProcessingSchedulerExecutorService executorService;
    private final long delayBetweenEnabledTasks;
    private final TimeUnit timeUnit;
    private final ChainingCallback[] chainingCallbacks;
    private final CeWorkerController ceWorkerController;
    private final long gracefulStopTimeoutInMs;

    public CeProcessingSchedulerImpl(CeConfiguration ceConfiguration, CeProcessingSchedulerExecutorService processingExecutorService, CeWorkerFactory ceCeWorkerFactory, CeWorkerController ceWorkerController) {
        this.executorService = processingExecutorService;
        this.delayBetweenEnabledTasks = ceConfiguration.getQueuePollingDelay();
        this.gracefulStopTimeoutInMs = ceConfiguration.getGracefulStopTimeoutInMs();
        this.ceWorkerController = ceWorkerController;
        this.timeUnit = TimeUnit.MILLISECONDS;
        int threadWorkerCount = ceConfiguration.getWorkerMaxCount();
        this.chainingCallbacks = new ChainingCallback[threadWorkerCount];
        for (int i = 0; i < threadWorkerCount; ++i) {
            CeWorker worker = ceCeWorkerFactory.create(i);
            this.chainingCallbacks[i] = new ChainingCallback(worker);
        }
    }

    @Override
    public void startScheduling() {
        for (ChainingCallback chainingCallback : this.chainingCallbacks) {
            ListenableScheduledFuture future = this.executorService.schedule(chainingCallback.worker, this.delayBetweenEnabledTasks, this.timeUnit);
            Futures.addCallback((ListenableFuture)future, (FutureCallback)chainingCallback);
        }
    }

    @Override
    public void gracefulStopScheduling() {
        LOG.info("Gracefully stopping workers...");
        this.requestAllWorkersToStop();
        try {
            this.waitForInProgressWorkersToFinish(this.gracefulStopTimeoutInMs);
            if (this.ceWorkerController.hasAtLeastOneProcessingWorker()) {
                LOG.info("Graceful stop period ended but some in-progress task did not finish. Tasks will be interrupted.");
            }
            this.interruptAllWorkers();
        }
        catch (InterruptedException e) {
            LOG.debug("Graceful stop was interrupted");
            Thread.currentThread().interrupt();
        }
    }

    @Override
    public void hardStopScheduling() {
        if (Arrays.stream(this.chainingCallbacks).allMatch(ChainingCallback::isInterrupted)) {
            return;
        }
        LOG.info("Hard stopping workers...");
        this.requestAllWorkersToStop();
        try {
            this.waitForInProgressWorkersToFinish(350L);
        }
        catch (InterruptedException e) {
            LOG.debug("Grace period of hard stop has been interrupted: {}", (Object)e);
            Thread.currentThread().interrupt();
        }
        if (this.ceWorkerController.hasAtLeastOneProcessingWorker()) {
            LOG.info("Some in-progress tasks are getting killed.");
        }
        this.interruptAllWorkers();
    }

    private void interruptAllWorkers() {
        Arrays.stream(this.chainingCallbacks).forEach(t -> t.stop(true));
    }

    private void waitForInProgressWorkersToFinish(long shutdownTimeoutInMs) throws InterruptedException {
        long until = System.currentTimeMillis() + shutdownTimeoutInMs;
        LOG.debug("Waiting for workers to finish in-progress tasks for at most {}ms", (Object)shutdownTimeoutInMs);
        while (System.currentTimeMillis() < until && this.ceWorkerController.hasAtLeastOneProcessingWorker()) {
            Thread.sleep(200L);
        }
    }

    private void requestAllWorkersToStop() {
        Arrays.stream(this.chainingCallbacks).forEach(t -> t.stop(false));
    }

    private class ChainingCallback
    implements FutureCallback<CeWorker.Result> {
        private volatile boolean keepRunning = true;
        private volatile boolean interrupted = false;
        private final CeWorker worker;
        @CheckForNull
        private ListenableFuture<CeWorker.Result> workerFuture;

        public ChainingCallback(CeWorker worker) {
            this.worker = worker;
        }

        public void onSuccess(@Nullable CeWorker.Result result) {
            if (this.keepRunning) {
                if (result == null) {
                    this.chainWithEnabledTaskDelay();
                } else {
                    switch (result) {
                        case DISABLED: {
                            this.chainWithDisabledTaskDelay();
                            break;
                        }
                        case NO_TASK: {
                            this.chainWithEnabledTaskDelay();
                            break;
                        }
                        default: {
                            this.chainWithoutDelay();
                        }
                    }
                }
            }
        }

        public void onFailure(Throwable t) {
            if (t instanceof Error) {
                LOG.error("Compute Engine execution failed. Scheduled processing interrupted.", t);
            } else if (this.keepRunning) {
                this.chainWithoutDelay();
            }
        }

        private void chainWithoutDelay() {
            this.workerFuture = CeProcessingSchedulerImpl.this.executorService.submit(this.worker);
            this.addCallback();
        }

        private void chainWithEnabledTaskDelay() {
            this.workerFuture = CeProcessingSchedulerImpl.this.executorService.schedule(this.worker, CeProcessingSchedulerImpl.this.delayBetweenEnabledTasks, CeProcessingSchedulerImpl.this.timeUnit);
            this.addCallback();
        }

        private void chainWithDisabledTaskDelay() {
            this.workerFuture = CeProcessingSchedulerImpl.this.executorService.schedule(this.worker, 30000L, CeProcessingSchedulerImpl.this.timeUnit);
            this.addCallback();
        }

        private void addCallback() {
            if (this.workerFuture != null) {
                Futures.addCallback(this.workerFuture, (FutureCallback)this);
            }
        }

        public void stop(boolean interrupt) {
            this.keepRunning = false;
            if (this.workerFuture != null) {
                this.interrupted = true;
                this.workerFuture.cancel(interrupt);
            }
        }

        public boolean isInterrupted() {
            return this.interrupted;
        }
    }
}

