/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.core.util;

import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.EnumSet;
import java.util.Objects;
import javax.annotation.Nullable;

public final class FileUtils {
    private static final String DIRECTORY_CAN_NOT_BE_NULL = "Directory can not be null";
    private static final EnumSet<FileVisitOption> FOLLOW_LINKS = EnumSet.of(FileVisitOption.FOLLOW_LINKS);

    private FileUtils() {
    }

    public static void deleteDirectory(File directory) throws IOException {
        Objects.requireNonNull(directory, DIRECTORY_CAN_NOT_BE_NULL);
        FileUtils.deleteDirectoryImpl(directory.toPath());
    }

    public static void deleteDirectory(Path directory) throws IOException {
        Objects.requireNonNull(directory, DIRECTORY_CAN_NOT_BE_NULL);
        FileUtils.deleteDirectoryImpl(directory);
    }

    public static void cleanDirectory(File directory) throws IOException {
        Objects.requireNonNull(directory, DIRECTORY_CAN_NOT_BE_NULL);
        Path path = directory.toPath();
        if (!path.toFile().exists()) {
            return;
        }
        FileUtils.cleanDirectoryImpl(path);
    }

    public static boolean deleteQuietly(@Nullable File file) {
        if (file == null) {
            return false;
        }
        return FileUtils.deleteQuietly(file.toPath());
    }

    public static boolean deleteQuietly(@Nullable Path path) {
        if (path == null) {
            return false;
        }
        try {
            if (path.toFile().isDirectory()) {
                FileUtils.deleteDirectory(path);
            } else {
                Files.delete(path);
            }
            return true;
        }
        catch (IOException | SecurityException ignored) {
            return false;
        }
    }

    private static void checkIO(boolean condition, String pattern, Object ... arguments) throws IOException {
        if (!condition) {
            throw new IOException(String.format(pattern, arguments));
        }
    }

    private static void cleanDirectoryImpl(Path path) throws IOException {
        Preconditions.checkArgument(path.toFile().isDirectory(), "'%s' is not a directory", (Object)path);
        Files.walkFileTree(path, FOLLOW_LINKS, 1, new CleanDirectoryFileVisitor(path));
    }

    private static void deleteDirectoryImpl(Path path) throws IOException {
        Objects.requireNonNull(path, DIRECTORY_CAN_NOT_BE_NULL);
        File file = path.toFile();
        if (!file.exists()) {
            return;
        }
        FileUtils.checkIO(!Files.isSymbolicLink(path), "Directory '%s' is a symbolic link", path);
        FileUtils.checkIO(!file.isFile(), "Directory '%s' is a file", path);
        Files.walkFileTree(path, DeleteRecursivelyFileVisitor.INSTANCE);
        FileUtils.checkIO(!file.exists(), "Unable to delete directory '%s'", path);
    }

    private static final class DeleteRecursivelyFileVisitor
    extends SimpleFileVisitor<Path> {
        public static final DeleteRecursivelyFileVisitor INSTANCE = new DeleteRecursivelyFileVisitor();

        private DeleteRecursivelyFileVisitor() {
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            Files.delete(file);
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
            Files.delete(dir);
            return FileVisitResult.CONTINUE;
        }
    }

    private static class CleanDirectoryFileVisitor
    extends SimpleFileVisitor<Path> {
        private static final int VISIT_MAX_DEPTH = 1;
        private final Path path;
        private final boolean symLink;

        private CleanDirectoryFileVisitor(Path path) {
            this.path = path;
            this.symLink = Files.isSymbolicLink(path);
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            if (file.toFile().isDirectory()) {
                FileUtils.deleteDirectoryImpl(file);
            } else if (!this.symLink || !file.equals(this.path)) {
                Files.delete(file);
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
            if (!dir.equals(this.path)) {
                FileUtils.deleteDirectoryImpl(dir);
            }
            return FileVisitResult.CONTINUE;
        }
    }
}

