// Copyright (c) 2024 Alibaba Cloud
//
// SPDX-License-Identifier: Apache-2.0
//

use async_trait::async_trait;
use const_format::concatcp;
use oidc_with_ram::OidcRamClient;
use serde_json::json;
use sts_token_client::StsTokenClient;

mod client_key_client;
mod ecs_ram_role_client;
pub mod oidc_with_ram;
mod sts_token_client;

use crate::kms::plugins::_IN_GUEST_DEFAULT_KEY_PATH;
use crate::kms::{Annotations, Decrypter, Encrypter, Getter, ProviderSettings};
use crate::kms::{Error, Result};

use client_key_client::ClientKeyClient;
use ecs_ram_role_client::EcsRamRoleClient;

#[derive(Clone, Debug)]
pub enum AliyunKmsClient {
    ClientKey {
        inner: ClientKeyClient,
    },
    EcsRamRole {
        ecs_ram_role_client: EcsRamRoleClient,
    },
    StsToken {
        client: StsTokenClient,
    },
    OidcRam {
        client: OidcRamClient,
    },
}

const ALIYUN_IN_GUEST_DEFAULT_KEY_PATH: &str = concatcp!(_IN_GUEST_DEFAULT_KEY_PATH, "/aliyun");

impl AliyunKmsClient {
    pub fn new(
        client_key: &str,
        kms_instance_id: &str,
        password: &str,
        cert_pem: &str,
    ) -> Result<Self> {
        Self::new_client_key_client(client_key, kms_instance_id, password, cert_pem)
    }

    pub fn new_client_key_client(
        client_key: &str,
        kms_instance_id: &str,
        password: &str,
        cert_pem: &str,
    ) -> Result<Self> {
        let inner = ClientKeyClient::new(client_key, kms_instance_id, password, cert_pem)?;

        Ok(Self::ClientKey { inner })
    }

    pub fn new_ecs_ram_role_client(ecs_ram_role_name: &str, region_id: &str) -> Self {
        let ecs_ram_role_client =
            EcsRamRoleClient::new(ecs_ram_role_name.to_string(), region_id.to_string());

        Self::EcsRamRole {
            ecs_ram_role_client,
        }
    }

    /// This new function is used by a in-pod client. The side-effect is to read the
    /// [`ALIYUN_IN_GUEST_DEFAULT_KEY_PATH`] which is the by default path where the credential
    /// to access kms is saved.
    pub async fn from_provider_settings(provider_settings: &ProviderSettings) -> Result<Self> {
        let client_type = if let Some(client_type_value) = provider_settings.get("client_type") {
            match client_type_value.as_str() {
                Some(client_type) => client_type,
                None => {
                    return Err(Error::AliyunKmsError(
                        "client type value is not str.".to_string(),
                    ))
                }
            }
        } else {
            return Err(Error::AliyunKmsError("client type not exist.".to_string()));
        };

        let inner_client = match client_type {
            "client_key" => AliyunKmsClient::ClientKey {
                inner: ClientKeyClient::from_provider_settings(provider_settings)
                    .await
                    .map_err(|e| {
                        Error::AliyunKmsError(format!(
                            "build ClientKeyClient with `from_provider_settings()` failed: {e}"
                        ))
                    })?,
            },
            "ecs_ram_role" => AliyunKmsClient::EcsRamRole {
                ecs_ram_role_client: EcsRamRoleClient::from_provider_settings(provider_settings)
                    .await
                    .map_err(|e| {
                        Error::AliyunKmsError(format!(
                            "build EcsRamRoleClient with `from_provider_settings()` failed: {e}"
                        ))
                    })?,
            },
            "sts_token" => AliyunKmsClient::StsToken {
                client: StsTokenClient::from_provider_settings(provider_settings)
                    .await
                    .map_err(|e| {
                        Error::AliyunKmsError(format!(
                            "build EcsRamRoleClient with `from_provider_settings()` failed: {e}"
                        ))
                    })?,
            },
            "oidc_ram" => AliyunKmsClient::OidcRam {
                client: OidcRamClient::from_provider_settings(provider_settings).map_err(|e| {
                    Error::AliyunKmsError(format!(
                        "build OidcRamClient with `from_provider_settings()` failed: {e}"
                    ))
                })?,
            },
            _ => return Err(Error::AliyunKmsError("client type invalid.".to_string())),
        };

        Ok(inner_client)
    }

    /// Export the [`ProviderSettings`] of the current client. This function is to be used
    /// in the encryptor side. The [`ProviderSettings`] will be used to initial a client
    /// in the decryptor side.
    pub fn export_provider_settings(&self) -> Result<ProviderSettings> {
        match self {
            AliyunKmsClient::ClientKey { inner } => {
                let mut provider_settings = inner.export_provider_settings().map_err(|e| {
                    Error::AliyunKmsError(format!(
                        "ClientKeyClient `export_provider_settings()` failed: {e}"
                    ))
                })?;

                provider_settings.insert(String::from("client_type"), json!("client_key"));

                Ok(provider_settings)
            }
            AliyunKmsClient::EcsRamRole {
                ecs_ram_role_client,
            } => {
                let mut provider_settings = ecs_ram_role_client.export_provider_settings();

                provider_settings.insert(String::from("client_type"), json!("ecs_ram_role"));

                Ok(provider_settings)
            }
            AliyunKmsClient::StsToken { client } => {
                let mut provider_settings = client.export_provider_settings();

                provider_settings.insert(String::from("client_type"), json!("sts_token"));

                Ok(provider_settings)
            }
            AliyunKmsClient::OidcRam { client } => {
                let mut provider_settings = client.export_provider_settings().map_err(|e| {
                    Error::AliyunKmsError(format!(
                        "OidcRamClient `export_provider_settings()` failed: {e}"
                    ))
                })?;

                provider_settings.insert(String::from("client_type"), json!("oidc_ram"));

                Ok(provider_settings)
            }
        }
    }
}

#[async_trait]
impl Encrypter for AliyunKmsClient {
    async fn encrypt(&mut self, data: &[u8], key_id: &str) -> Result<(Vec<u8>, Annotations)> {
        match &mut self {
            AliyunKmsClient::ClientKey { ref mut inner } => inner.encrypt(data, key_id).await,
            AliyunKmsClient::EcsRamRole { .. } => Err(Error::AliyunKmsError(
                "Encrypter does not support accessing through Aliyun EcsRamRole".to_string(),
            )),
            AliyunKmsClient::StsToken { .. } => Err(Error::AliyunKmsError(
                "Encrypter does not support accessing through Aliyun StsToken".to_string(),
            )),
            AliyunKmsClient::OidcRam { .. } => Err(Error::AliyunKmsError(
                "Encrypter does not support accessing through Aliyun OidcRam".to_string(),
            )),
        }
    }
}

#[async_trait]
impl Decrypter for AliyunKmsClient {
    async fn decrypt(
        &mut self,
        ciphertext: &[u8],
        key_id: &str,
        annotations: &Annotations,
    ) -> Result<Vec<u8>> {
        match &mut self {
            AliyunKmsClient::ClientKey { ref mut inner } => {
                inner.decrypt(ciphertext, key_id, annotations).await
            }
            AliyunKmsClient::EcsRamRole { .. } => Err(Error::AliyunKmsError(
                "Decrypter does not support accessing through Aliyun EcsRamRole".to_string(),
            )),
            AliyunKmsClient::StsToken { .. } => Err(Error::AliyunKmsError(
                "Decrypter does not support accessing through Aliyun StsToken".to_string(),
            )),
            AliyunKmsClient::OidcRam { .. } => Err(Error::AliyunKmsError(
                "Decrypter does not support accessing through Aliyun OidcRam".to_string(),
            )),
        }
    }
}

#[async_trait]
impl Getter for AliyunKmsClient {
    async fn get_secret(&self, name: &str, annotations: &Annotations) -> Result<Vec<u8>> {
        match &self {
            AliyunKmsClient::ClientKey { ref inner } => inner.get_secret(name, annotations).await,
            AliyunKmsClient::EcsRamRole {
                ref ecs_ram_role_client,
            } => ecs_ram_role_client.get_secret(name, annotations).await,
            AliyunKmsClient::StsToken { ref client } => client.get_secret(name, annotations).await,
            AliyunKmsClient::OidcRam { ref client } => client
                .get_secret(name, annotations)
                .await
                .map_err(|e| Error::AliyunKmsError(format!("failed to get secret: {e:#?}"))),
        }
    }
}

#[cfg(test)]
mod tests {
    use rstest::rstest;
    use serde_json::{json, Map, Value};

    #[rstest]
    #[ignore]
    #[case(b"this is a test plaintext")]
    #[ignore]
    #[case(b"this is a another test plaintext")]
    #[tokio::test]
    async fn key_lifetime(#[case] plaintext: &[u8]) {
        use crate::kms::{plugins::aliyun::AliyunKmsClient, Decrypter, Encrypter};

        let kid = "alias/test_key_id";
        let provider_settings = json!({
            "client_type": "client_key",
            "client_key_id": "KAAP.f4c8****",
            "kms_instance_id": "kst-shh6****",
        });
        // init encrypter at user side
        let provider_settings = provider_settings.as_object().unwrap().to_owned();
        let mut encryptor = AliyunKmsClient::from_provider_settings(&provider_settings)
            .await
            .unwrap();

        // do encryption
        let (ciphertext, secret_settings) =
            encryptor.encrypt(plaintext, kid).await.expect("encrypt");
        let provider_settings = encryptor.export_provider_settings().unwrap();

        // init decrypter in a guest
        let mut decryptor = AliyunKmsClient::from_provider_settings(&provider_settings)
            .await
            .unwrap();

        // do decryption
        let decrypted = decryptor
            .decrypt(&ciphertext, kid, &secret_settings)
            .await
            .expect("decrypt");

        assert_eq!(decrypted, plaintext);
    }

    #[rstest]
    #[ignore]
    #[case("client_key")]
    #[ignore]
    #[case("ecs_ram_role")]
    #[tokio::test]
    async fn get_secret(#[case] client_type: &str) {
        use crate::kms::{plugins::aliyun::AliyunKmsClient, Annotations, Getter};

        let secret_name = "test_secret";
        let provider_settings = json!({
            "client_type": client_type,
            "client_key_id": "KAAP.f4c8****",
            "kms_instance_id": "kst-shh6****",
        });
        // init getter at user side
        let provider_settings = provider_settings.as_object().unwrap().to_owned();
        let getter = AliyunKmsClient::from_provider_settings(&provider_settings)
            .await
            .unwrap();

        // do get
        let mut annotations: Annotations = Map::<String, Value>::new();
        annotations.insert("version_stage".to_string(), Value::String("".to_string()));
        annotations.insert("version_id".to_string(), Value::String("".to_string()));
        let secret_value = getter
            .get_secret(secret_name, &annotations)
            .await
            .expect("get_secret_with_client_key");

        // We have set "test_secret_value" as secret on Aliyun KMS console.
        assert_eq!(String::from_utf8_lossy(&secret_value), "test_secret_value");
    }
}
