package backend

import (
	"fmt"
	"unsafe"

	"github.com/grafana/grafana-plugin-sdk-go/data"
	jsoniter "github.com/json-iterator/go"
)

func init() { //nolint:gochecknoinits
	jsoniter.RegisterTypeEncoder("backend.DataResponse", &dataResponseCodec{})
	jsoniter.RegisterTypeEncoder("backend.QueryDataResponse", &queryDataResponseCodec{})
}

type dataResponseCodec struct{}

func (codec *dataResponseCodec) IsEmpty(ptr unsafe.Pointer) bool {
	dr := (*DataResponse)(ptr)
	return dr.Error == nil && dr.Frames == nil
}

func (codec *dataResponseCodec) Encode(ptr unsafe.Pointer, stream *jsoniter.Stream) {
	dr := (*DataResponse)(ptr)
	writeDataResponseJSON(dr, stream)
}

type queryDataResponseCodec struct{}

func (codec *queryDataResponseCodec) IsEmpty(ptr unsafe.Pointer) bool {
	qdr := *((*QueryDataResponse)(ptr))
	return qdr.Responses == nil
}

func (codec *queryDataResponseCodec) Encode(ptr unsafe.Pointer, stream *jsoniter.Stream) {
	qdr := (*QueryDataResponse)(ptr)
	writeQueryDataResponseJSON(qdr, stream)
}

func (codec *queryDataResponseCodec) Decode(ptr unsafe.Pointer, iter *jsoniter.Iterator) {
	qdr := QueryDataResponse{}
	readQueryDataResultsJSON(&qdr, iter)
	*((*QueryDataResponse)(ptr)) = qdr
}

//-----------------------------------------------------------------
// Private stream readers
//-----------------------------------------------------------------

func writeDataResponseJSON(dr *DataResponse, stream *jsoniter.Stream) {
	stream.WriteObjectStart()
	started := false

	if dr.Error != nil {
		stream.WriteObjectField("error")
		stream.WriteString(dr.Error.Error())
		started = true
	}

	if dr.Frames != nil {
		if started {
			stream.WriteMore()
		}

		started = false
		stream.WriteObjectField("frames")
		stream.WriteArrayStart()
		for _, frame := range dr.Frames {
			if started {
				stream.WriteMore()
			}
			stream.WriteVal(frame)
			started = true
		}
		stream.WriteArrayEnd()
	}

	stream.WriteObjectEnd()
}

func writeQueryDataResponseJSON(qdr *QueryDataResponse, stream *jsoniter.Stream) {
	stream.WriteObjectStart()
	stream.WriteObjectField("results")
	stream.WriteObjectStart()
	started := false

	// Make sure all keys in the result are written
	for id, res := range qdr.Responses {
		if started {
			stream.WriteMore()
		}
		stream.WriteObjectField(id)
		obj := res // avoid implicit memory
		writeDataResponseJSON(&obj, stream)
		started = true
	}
	stream.WriteObjectEnd()

	stream.WriteObjectEnd()
}

//-----------------------------------------------------------------
// Private stream readers
//-----------------------------------------------------------------

func readQueryDataResultsJSON(qdr *QueryDataResponse, iter *jsoniter.Iterator) {
	found := false

	for l1Field := iter.ReadObject(); l1Field != ""; l1Field = iter.ReadObject() {
		switch l1Field {
		case "results":
			if found {
				iter.ReportError("read results", "already found results")
				return
			}
			found = true

			qdr.Responses = make(Responses)

			for l2Field := iter.ReadObject(); l2Field != ""; l2Field = iter.ReadObject() {
				dr := DataResponse{}
				readDataResponseJSON(&dr, iter)
				qdr.Responses[l2Field] = dr
			}

		default:
			iter.ReportError("bind l1", "unexpected field: "+l1Field)
			return
		}
	}
}

func readDataResponseJSON(rsp *DataResponse, iter *jsoniter.Iterator) {
	for l2Field := iter.ReadObject(); l2Field != ""; l2Field = iter.ReadObject() {
		switch l2Field {
		case "error":
			rsp.Error = fmt.Errorf(iter.ReadString())

		case "frames":
			for iter.ReadArray() {
				frame := &data.Frame{}
				iter.ReadVal(frame)
				if iter.Error != nil {
					return
				}
				rsp.Frames = append(rsp.Frames, frame)
			}

		default:
			iter.ReportError("bind l2", "unexpected field: "+l2Field)
			return
		}
	}
}
