package org.bouncycastle.checkstyle;

import java.util.HashSet;
import java.util.Set;

import com.puppycrawl.tools.checkstyle.api.AbstractCheck;
import com.puppycrawl.tools.checkstyle.api.DetailAST;
import com.puppycrawl.tools.checkstyle.api.TokenTypes;


public class DebugMethodChecker
    extends AbstractCheck
{
    public DebugMethodChecker()
    {
    }

    public String contains = null;
    public String ignorePackages = null;

    private String packageName;
    private String className;

    private static int[] tokens = new int[]{
        TokenTypes.COMMENT_CONTENT,
        TokenTypes.CLASS_DEF,
        TokenTypes.PACKAGE_DEF,
        TokenTypes.STATIC_IMPORT,
        TokenTypes.IMPORT,
        TokenTypes.METHOD_CALL, TokenTypes.METHOD_DEF, TokenTypes.SINGLE_LINE_COMMENT};

    @Override
    public int[] getDefaultTokens()
    {
        return tokens;
    }

    @Override
    public int[] getAcceptableTokens()
    {
        return tokens;
    }

    @Override
    public int[] getRequiredTokens()
    {
        return tokens;
    }

    private int skipCount = 0;

    @Override
    public boolean isCommentNodesRequired()
    {
        return true;
    }


    private int mode = 0;
    private Set<DetailAST> comments = new HashSet<DetailAST>();


    @Override
    public void leaveToken(DetailAST ast)
    {
        if (ast.getType() == TokenTypes.METHOD_DEF)
        {
            //
            // Reset skip count after method def has been processed.
            //

            super.visitToken(ast);
            if (skipCount > 0)
            {
                log(ast.getLineNo(), "Number of squelches ( -DM <...> ) did not match number of method calls found");
            }
            skipCount = 0;
            comments.clear();
        }
    }

    @Override
    public void visitToken(DetailAST ast)
    {
        if (ast.getType() == TokenTypes.PACKAGE_DEF)
        {
            StringBuffer accumulator = new StringBuffer();

            DetailAST p = ast.findFirstToken(TokenTypes.DOT);
            if (p == null)
            {
                p = ast.findFirstToken(TokenTypes.IDENT);
            }
            LeftRightParent(p, accumulator);
            packageName = accumulator.toString();
        }
        else if (ast.getType() == TokenTypes.CLASS_DEF)
        {
            StringBuffer accumulator = new StringBuffer();

            DetailAST p = ast.findFirstToken(TokenTypes.DOT);
            if (p == null)
            {
                p = ast.findFirstToken(TokenTypes.IDENT);
            }
            LeftRightParent(p, accumulator);
            className = accumulator.toString();
        }
        else if (ast.getType() == TokenTypes.COMMENT_CONTENT && !comments.contains(ast))
        {
            //
            // The comment "-DM <methodName>" is used to skip a single call to the method.
            // The value always resets when a new METHOD_DEF is encountered in the source file.
            //

            handleComment(ast);

        }
        else if (ast.getType() == TokenTypes.METHOD_CALL)
        {
            if (contains != null)
            {
                StringBuffer accumulator = new StringBuffer();
                DetailAST p = ast.findFirstToken(TokenTypes.DOT);
                if (p == null)
                {
                    p = ast.findFirstToken(TokenTypes.IDENT);
                }
                LeftRightParent(p, accumulator);
                if (accumulator.toString().matches(contains))
                {
                    if (!onIgnoreList())
                    {
                        if (skipCount == 0)
                        {
                            log(ast.getLineNo(), String.format("Call matching '%s' found use ' -DM <method name> ' in a nearby comment to squash it.", contains));
                        }
                        else
                        {
                            skipCount--;
                        }
                    }
                }
            }
        }

    }

    private void handleComment(DetailAST ast)
    {
        if (ast == null || comments.contains(ast))
        {
            return;
        }

        String text = ast.getText();
        int i = 0;
        for (; ; )
        {
            i = text.indexOf("-DM", i);
            if (i >= 0)
            {
                i += 3;
                StringBuffer number = new StringBuffer();
                for (; i < text.length() && ((text.charAt(i) >= '0' && text.charAt(i) <= '9') || text.charAt(i) == ' '); i++)
                {
                    number.append(text.charAt(i));
                }

                int delta;
                try
                {
                    delta = Integer.parseInt(number.toString().trim());
                }
                catch (NumberFormatException nfe)
                {
                    delta = 1;
                }

                if (i >= text.length())
                {
                    log(ast.getLineNo(), "Invalid -DM comment entry");
                }
                text = text.substring(i).trim();
                if (text.matches(contains))
                {
                    skipCount += delta;
                    comments.add(ast);
                    break;
                }
                continue;
            }
            break;
        }
    }

    private boolean onIgnoreList()
    {
        if (ignorePackages == null || ignorePackages.trim().isEmpty())
        {
            return false;
        }

        String candidate = (packageName + "." + className).trim();
        return candidate.matches(ignorePackages);
    }


    private void LeftRightParent(DetailAST start, StringBuffer accumulator)
    {

        DetailAST original = start;

        if (start == null)
        {
            return;
        }

        while (start != null && start.getType() != TokenTypes.DOT && start.getType() != TokenTypes.IDENT)
        {
            switch (start.getType())
            {
            case TokenTypes.SINGLE_LINE_COMMENT:
                handleComment(start.getFirstChild());
                break;
            case TokenTypes.BLOCK_COMMENT_BEGIN:
                DetailAST child = start.getFirstChild();
                while (child != null)
                {
                    if (child.getType() == TokenTypes.COMMENT_CONTENT)
                    {
                        handleComment(child);
                    }
                    child = child.getNextSibling();
                }
                break;
            }

            start = start.getNextSibling();

        }

        if (start == null) {
            return;
        }

        if (start.getFirstChild() != null)
        {
            LeftRightParent(start.getFirstChild(), accumulator);
        }
        if (start.getType() == TokenTypes.IDENT || start.getType() == TokenTypes.DOT)
        {
            accumulator.append(start.getText());
        }

        if (start.getLastChild() != null)
        {
            LeftRightParent(start.getLastChild(), accumulator);
        }
    }


    public String getContains()
    {
        return contains;
    }

    public void setContains(String contains)
    {
        this.contains = contains;
    }

    public String getIgnorePackages()
    {
        return "";
    }

    public void setIgnorePackages(String ignorePackages)
    {
        this.ignorePackages = ignorePackages.trim();
    }


}

