/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.ce.task.projectanalysis.step;

import com.google.common.base.Preconditions;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.function.Supplier;
import javax.annotation.CheckForNull;
import javax.annotation.Nullable;
import org.sonar.api.utils.DateUtils;
import org.sonar.api.utils.MessageException;
import org.sonar.api.utils.System2;
import org.sonar.api.utils.log.Logger;
import org.sonar.api.utils.log.Loggers;
import org.sonar.ce.task.projectanalysis.analysis.AnalysisMetadataHolder;
import org.sonar.ce.task.projectanalysis.analysis.Branch;
import org.sonar.ce.task.projectanalysis.component.Component;
import org.sonar.ce.task.projectanalysis.component.ConfigurationRepository;
import org.sonar.ce.task.projectanalysis.component.TreeRootHolder;
import org.sonar.ce.task.projectanalysis.period.Period;
import org.sonar.ce.task.projectanalysis.period.PeriodHolderImpl;
import org.sonar.ce.task.step.ComputationStep;
import org.sonar.db.DbClient;
import org.sonar.db.DbSession;
import org.sonar.db.component.BranchDto;
import org.sonar.db.component.BranchType;
import org.sonar.db.component.SnapshotDto;
import org.sonar.db.component.SnapshotQuery;
import org.sonar.db.event.EventDto;

public class LoadPeriodsStep
implements ComputationStep {
    private static final Logger LOG = Loggers.get(LoadPeriodsStep.class);
    private final AnalysisMetadataHolder analysisMetadataHolder;
    private final TreeRootHolder treeRootHolder;
    private final PeriodHolderImpl periodsHolder;
    private final System2 system2;
    private final DbClient dbClient;
    private final ConfigurationRepository configRepository;

    public LoadPeriodsStep(AnalysisMetadataHolder analysisMetadataHolder, TreeRootHolder treeRootHolder, PeriodHolderImpl periodsHolder, System2 system2, DbClient dbClient, ConfigurationRepository configRepository) {
        this.analysisMetadataHolder = analysisMetadataHolder;
        this.treeRootHolder = treeRootHolder;
        this.periodsHolder = periodsHolder;
        this.system2 = system2;
        this.dbClient = dbClient;
        this.configRepository = configRepository;
    }

    public String getDescription() {
        return "Load new code period";
    }

    public void execute(ComputationStep.Context context) {
        if (this.analysisMetadataHolder.isFirstAnalysis() || !this.analysisMetadataHolder.isLongLivingBranch()) {
            this.periodsHolder.setPeriod(null);
            return;
        }
        this.periodsHolder.setPeriod(this.resolvePeriod(this.treeRootHolder.getRoot()).orElse(null));
    }

    private Optional<Period> resolvePeriod(Component projectOrView) {
        String currentVersion = projectOrView.getProjectAttributes().getProjectVersion();
        Optional<String> propertyValue = this.configRepository.getConfiguration().get("sonar.leak.period").filter(t -> !t.isEmpty());
        LoadPeriodsStep.checkPeriodProperty(propertyValue.isPresent(), "", "property is undefined or value is empty", new Object[0]);
        try (DbSession dbSession = this.dbClient.openSession(false);){
            Optional<Period> manualBaselineOpt = this.resolveByManualBaseline(dbSession, projectOrView.getUuid());
            if (manualBaselineOpt.isPresent()) {
                Optional<Period> optional = manualBaselineOpt;
                return optional;
            }
            Optional<Period> optional = this.resolve(dbSession, projectOrView.getUuid(), currentVersion, propertyValue.get());
            return optional;
        }
    }

    private Optional<Period> resolveByManualBaseline(DbSession dbSession, String projectUuid) {
        Branch branch = this.analysisMetadataHolder.getBranch();
        if (branch.getType() != BranchType.LONG) {
            return Optional.empty();
        }
        return this.dbClient.branchDao().selectByUuid(dbSession, projectUuid).map(branchDto -> this.resolveByManualBaseline(dbSession, projectUuid, (BranchDto)branchDto));
    }

    private Period resolveByManualBaseline(DbSession dbSession, String projectUuid, BranchDto branchDto) {
        String baselineAnalysisUuid = branchDto.getManualBaseline();
        if (baselineAnalysisUuid == null) {
            return null;
        }
        LOG.debug("Resolving new code period by manual baseline");
        SnapshotDto baseline = this.dbClient.snapshotDao().selectByUuid(dbSession, baselineAnalysisUuid).filter(t -> t.getComponentUuid().equals(projectUuid)).orElseThrow(() -> new IllegalStateException("Analysis '" + baselineAnalysisUuid + "' of project '" + projectUuid + "' defined as manual baseline does not exist"));
        return LoadPeriodsStep.newPeriod("manual_baseline", null, baseline);
    }

    private Optional<Period> resolve(DbSession dbSession, String projectUuid, String analysisProjectVersion, String propertyValue) {
        Integer days = LoadPeriodsStep.parseDaysQuietly(propertyValue);
        if (days != null) {
            return this.resolveByDays(dbSession, projectUuid, days, propertyValue);
        }
        Instant date = LoadPeriodsStep.parseDate(propertyValue);
        if (date != null) {
            return this.resolveByDate(dbSession, projectUuid, date, propertyValue);
        }
        List versions = this.dbClient.eventDao().selectVersionsByMostRecentFirst(dbSession, projectUuid);
        if (versions.isEmpty()) {
            return this.resolveWhenNoExistingVersion(dbSession, projectUuid, analysisProjectVersion, propertyValue);
        }
        String mostRecentVersion = Optional.ofNullable(((EventDto)versions.iterator().next()).getName()).orElseThrow(() -> new IllegalStateException("selectVersionsByMostRecentFirst returned a DTO which didn't have a name"));
        boolean previousVersionPeriod = "previous_version".equals(propertyValue);
        if (previousVersionPeriod) {
            if (versions.size() == 1) {
                return this.resolvePreviousVersionWithOnlyOneExistingVersion(dbSession, projectUuid);
            }
            return this.resolvePreviousVersion(dbSession, analysisProjectVersion, versions, mostRecentVersion);
        }
        return this.resolveVersion(dbSession, versions, propertyValue);
    }

    @CheckForNull
    private static Instant parseDate(String propertyValue) {
        try {
            LocalDate localDate = LocalDate.parse(propertyValue);
            return localDate.atStartOfDay(ZoneId.systemDefault()).toInstant();
        }
        catch (DateTimeParseException e) {
            boolean invalidDate = e.getCause() == null || e.getCause() == e || !e.getCause().getMessage().contains("Invalid date");
            LoadPeriodsStep.checkPeriodProperty(invalidDate, propertyValue, "Invalid date", new Object[0]);
            return null;
        }
    }

    private Optional<Period> resolveByDays(DbSession dbSession, String projectUuid, Integer days, String propertyValue) {
        LoadPeriodsStep.checkPeriodProperty(days > 0, propertyValue, "number of days is <= 0", new Object[0]);
        long analysisDate = this.analysisMetadataHolder.getAnalysisDate();
        List snapshots = this.dbClient.snapshotDao().selectAnalysesByQuery(dbSession, LoadPeriodsStep.createCommonQuery(projectUuid).setCreatedBefore(Long.valueOf(analysisDate)).setSort(SnapshotQuery.SORT_FIELD.BY_DATE, SnapshotQuery.SORT_ORDER.ASC));
        LoadPeriodsStep.ensureNotOnFirstAnalysis(!snapshots.isEmpty());
        Instant targetDate = DateUtils.addDays((Instant)Instant.ofEpochMilli(analysisDate), (int)(-days.intValue()));
        LOG.debug("Resolving new code period by {} days: {}", (Object)days, LoadPeriodsStep.supplierToString(() -> LoadPeriodsStep.logDate(targetDate)));
        SnapshotDto snapshot = LoadPeriodsStep.findNearestSnapshotToTargetDate(snapshots, targetDate);
        return Optional.of(LoadPeriodsStep.newPeriod("days", String.valueOf(days), snapshot));
    }

    private Optional<Period> resolveByDate(DbSession dbSession, String projectUuid, Instant date, String propertyValue) {
        Instant now = Instant.ofEpochMilli(this.system2.now());
        LoadPeriodsStep.checkPeriodProperty(date.compareTo(now) <= 0, propertyValue, "date is in the future (now: '%s')", LoadPeriodsStep.supplierToString(() -> LoadPeriodsStep.logDate(now)));
        LOG.debug("Resolving new code period by date: {}", LoadPeriodsStep.supplierToString(() -> LoadPeriodsStep.logDate(date)));
        Optional<Period> period = this.findFirstSnapshot(dbSession, LoadPeriodsStep.createCommonQuery(projectUuid).setCreatedAfter(Long.valueOf(date.toEpochMilli())).setSort(SnapshotQuery.SORT_FIELD.BY_DATE, SnapshotQuery.SORT_ORDER.ASC)).map(dto -> LoadPeriodsStep.newPeriod("date", DateUtils.formatDate((Instant)date), dto));
        LoadPeriodsStep.checkPeriodProperty(period.isPresent(), propertyValue, "No analysis found created after date '%s'", LoadPeriodsStep.supplierToString(() -> LoadPeriodsStep.logDate(date)));
        return period;
    }

    private Optional<Period> resolveWhenNoExistingVersion(DbSession dbSession, String projectUuid, String currentVersion, String propertyValue) {
        LOG.debug("Resolving first analysis as new code period as there is no existing version");
        boolean previousVersionPeriod = "previous_version".equals(propertyValue);
        boolean currentVersionPeriod = currentVersion.equals(propertyValue);
        LoadPeriodsStep.checkPeriodProperty(previousVersionPeriod || currentVersionPeriod, propertyValue, "No existing version. Property should be either '%s' or the current version '%s' (actual: '%s')", "previous_version", currentVersion, propertyValue);
        String periodMode = previousVersionPeriod ? "previous_version" : "version";
        return this.findOldestAnalysis(dbSession, periodMode, projectUuid);
    }

    private Optional<Period> resolvePreviousVersionWithOnlyOneExistingVersion(DbSession dbSession, String projectUuid) {
        LOG.debug("Resolving first analysis as new code period as there is only one existing version");
        return this.findOldestAnalysis(dbSession, "previous_version", projectUuid);
    }

    private Optional<Period> findOldestAnalysis(DbSession dbSession, String periodMode, String projectUuid) {
        Optional<Period> period = this.dbClient.snapshotDao().selectOldestSnapshot(dbSession, projectUuid).map(dto -> LoadPeriodsStep.newPeriod(periodMode, null, dto));
        LoadPeriodsStep.ensureNotOnFirstAnalysis(period.isPresent());
        return period;
    }

    private Optional<Period> resolvePreviousVersion(DbSession dbSession, String currentVersion, List<EventDto> versions, String mostRecentVersion) {
        EventDto previousVersion = versions.get(currentVersion.equals(mostRecentVersion) ? 1 : 0);
        LOG.debug("Resolving new code period by previous version: {}", (Object)previousVersion.getName());
        return this.newPeriod(dbSession, "previous_version", previousVersion);
    }

    private Optional<Period> resolveVersion(DbSession dbSession, List<EventDto> versions, String propertyValue) {
        LOG.debug("Resolving new code period by version: {}", (Object)propertyValue);
        Optional<EventDto> version = versions.stream().filter(t -> propertyValue.equals(t.getName())).findFirst();
        LoadPeriodsStep.checkPeriodProperty(version.isPresent(), propertyValue, "version is none of the existing ones: %s", LoadPeriodsStep.supplierToString(() -> LoadPeriodsStep.toVersions(versions)));
        return this.newPeriod(dbSession, "version", version.get());
    }

    private Optional<Period> newPeriod(DbSession dbSession, String periodMode, EventDto previousVersion) {
        Optional<Period> period = this.dbClient.snapshotDao().selectByUuid(dbSession, previousVersion.getAnalysisUuid()).map(dto -> LoadPeriodsStep.newPeriod(periodMode, previousVersion.getName(), dto));
        if (!period.isPresent()) {
            throw new IllegalStateException(String.format("Analysis '%s' for version event '%s' has been deleted", previousVersion.getAnalysisUuid(), previousVersion.getName()));
        }
        return period;
    }

    private static String toVersions(List<EventDto> versions) {
        return Arrays.toString(versions.stream().map(EventDto::getName).toArray(String[]::new));
    }

    private static Object supplierToString(final Supplier<String> s) {
        return new Object(){

            public String toString() {
                return (String)s.get();
            }
        };
    }

    private static Period newPeriod(String mode, @Nullable String modeParameter, SnapshotDto dto) {
        return new Period(mode, modeParameter, dto.getCreatedAt(), dto.getUuid());
    }

    private static void checkPeriodProperty(boolean test, String propertyValue, String testDescription, Object ... args) {
        if (!test) {
            LOG.debug("Invalid code period '{}': {}", (Object)propertyValue, LoadPeriodsStep.supplierToString(() -> String.format(testDescription, args)));
            throw MessageException.of((String)String.format("Invalid new code period. '%s' is not one of: integer > 0, date before current analysis j, \"previous_version\", or version string that exists in the project' \nPlease contact a project administrator to correct this setting", propertyValue));
        }
    }

    private Optional<SnapshotDto> findFirstSnapshot(DbSession session, SnapshotQuery query) {
        return this.dbClient.snapshotDao().selectAnalysesByQuery(session, query).stream().findFirst();
    }

    private static void ensureNotOnFirstAnalysis(boolean expression) {
        Preconditions.checkState((boolean)expression, (Object)"Attempting to resolve period while no analysis exist for project");
    }

    @CheckForNull
    private static Integer parseDaysQuietly(String property) {
        try {
            return Integer.parseInt(property);
        }
        catch (NumberFormatException e) {
            return null;
        }
    }

    private static SnapshotDto findNearestSnapshotToTargetDate(List<SnapshotDto> snapshots, Instant targetDate) {
        Duration bestDuration = null;
        SnapshotDto nearest = null;
        for (SnapshotDto snapshot : snapshots) {
            Instant createdAt = Instant.ofEpochMilli(snapshot.getCreatedAt());
            Duration duration = Duration.between(targetDate, createdAt).abs();
            if (bestDuration != null && duration.compareTo(bestDuration) > 0) continue;
            bestDuration = duration;
            nearest = snapshot;
        }
        return nearest;
    }

    private static SnapshotQuery createCommonQuery(String projectUuid) {
        return new SnapshotQuery().setComponentUuid(projectUuid).setStatus("P");
    }

    private static String logDate(Instant instant) {
        return DateUtils.formatDate((Instant)instant.truncatedTo(ChronoUnit.SECONDS));
    }
}

