// SPDX-License-Identifier: (BSD-3-Clause OR GPL-2.0-only)
/* Copyright(c) 2020 - 2021 Intel Corporation */

#include <adf_accel_devices.h>
#include <adf_common_drv.h>
#include <adf_cfg.h>
#include "adf_c4xxx_hw_data.h"
#include "adf_c4xxx_inline.h"
#include "adf_c4xxx_accel_units.h"
#include "icp_qat_hw.h"

static void
adf_init_error_notification_configuration(struct adf_accel_dev *accel_dev,
					  u32 offset)
{
	void __iomem *aram_csr_base;

	aram_csr_base = (&GET_BARS(accel_dev)[ADF_C4XXX_SRAM_BAR])->virt_addr;

	/* configure error notification configuration registers
	 * Set CD Parity error
	 */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_CD_RF_PARITY_ERR_0 + offset,
		   ADF_C4XXX_CD_RF_PARITY_ERR_0_VAL);
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_CD_RF_PARITY_ERR_1 + offset,
		   ADF_C4XXX_CD_RF_PARITY_ERR_1_VAL);
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_CD_RF_PARITY_ERR_2 + offset,
		   ADF_C4XXX_CD_RF_PARITY_ERR_2_VAL);
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_CD_RF_PARITY_ERR_3 + offset,
		   ADF_C4XXX_CD_RF_PARITY_ERR_3_VAL);
	/* Set CD RAM ECC Correctable Error */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_CD_CERR + offset,
		   ADF_C4XXX_CD_CERR_VAL);
	/* Set CD RAM ECC UnCorrectable Error */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_CD_UERR + offset,
		   ADF_C4XXX_CD_UERR_VAL);
	/* Set Inline (excl cmd_dis) Parity Error */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_INLN_RF_PARITY_ERR_0 + offset,
		   ADF_C4XXX_INLN_RF_PARITY_ERR_0_VAL);
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_INLN_RF_PARITY_ERR_1 + offset,
		   ADF_C4XXX_INLN_RF_PARITY_ERR_1_VAL);
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_INLN_RF_PARITY_ERR_2 + offset,
		   ADF_C4XXX_INLN_RF_PARITY_ERR_2_VAL);
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_INLN_RF_PARITY_ERR_3 + offset,
		   ADF_C4XXX_INLN_RF_PARITY_ERR_3_VAL);
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_INLN_RF_PARITY_ERR_4 + offset,
		   ADF_C4XXX_INLN_RF_PARITY_ERR_4_VAL);
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_INLN_RF_PARITY_ERR_5 + offset,
		   ADF_C4XXX_INLN_RF_PARITY_ERR_5_VAL);
	/* Set Parser RAM ECC Correctable Error */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSER_CERR + offset,
		   ADF_C4XXX_PARSER_CERR_VAL);
	/* Set Parser RAM ECC UnCorrectable Error */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSER_UERR + offset,
		   ADF_C4XXX_PARSER_UERR_VAL);
	/* Set CTPB RAM ECC Correctable Error */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_CTPB_CERR + offset,
		   ADF_C4XXX_CTPB_CERR_VAL);
	/* Set CTPB RAM ECC UnCorrectable Error */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_CTPB_UERR + offset,
		   ADF_C4XXX_CTPB_UERR_VAL);
	/* Set CPP Interface Status */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_CPPM_ERR_STAT + offset,
		   ADF_C4XXX_CPPM_ERR_STAT_VAL);
	/* Set CGST_MGMT_INT */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_CONGESTION_MGMT_INT + offset,
		   ADF_C4XXX_CONGESTION_MGMT_INI_VAL);
	/* CPP Interface Status */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_CPPT_ERR_STAT + offset,
		   ADF_C4XXX_CPPT_ERR_STAT_VAL);
	/* MAC Interrupt Mask */
	ADF_CSR_WR64(aram_csr_base, ADF_C4XXX_IC_MAC_IM + offset,
		     ADF_C4XXX_MAC_IM_VAL);
}

static void adf_enable_parse_extraction(struct adf_accel_dev *accel_dev,
					u32 ipsec_spi_mask_mode)
{
	void __iomem *aram_csr_base;

	aram_csr_base = (&GET_BARS(accel_dev)[ADF_C4XXX_SRAM_BAR])->virt_addr;

	/* Enable Inline Parse Extraction CSRs
	 * Set IC_PARSE_CTRL register
	 */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_CTRL_OFFSET,
		   ADF_C4XXX_IC_PARSE_CTRL_OFFSET_DEFAULT_VALUE);

	/* Configure common ESP protocol for both IPv4 and IPv6
	 * Set IC_PARSE_FIXED_DATA(0)
	 */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_FIXED_DATA(0),
		   ADF_C4XXX_DEFAULT_IC_PARSE_FIXED_DATA_0);

	/* Set IC_PARSE_FIXED_LENGTH for 24-bits and 32-bits SPI */
	if (ipsec_spi_mask_mode == IPSEC_SPI_MASK_MODE_1) {
		ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_FIXED_LENGTH,
			   ADF_C4XXX_24BIT_SPI_IC_PARSE_FIXED_LEN);
	} else {
		ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_FIXED_LENGTH,
			   ADF_C4XXX_DEFAULT_IC_PARSE_FIXED_LEN);
	}

	/* Configure protocol extraction field from an IPv4 header */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_IPV4_OFFSET_0,
		   ADF_C4XXX_DEFAULT_IC_PARSE_IPV4_OFFS_0_VALUE);
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_IPV4_LENGTH_0,
		   ADF_C4XXX_DEFAULT_IC_PARSE_IPV4_LEN_0_VALUE);

	/* Configure SPI extraction field from an IPv4 header */
	if (ipsec_spi_mask_mode == IPSEC_SPI_MASK_MODE_1) {
		ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_IPV4_OFFSET_1,
			   ADF_C4XXX_24BIT_SPI_IC_PARSE_IPV4_OFFS_1_VALUE);
		ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_IPV4_LENGTH_1,
			   ADF_C4XXX_24BIT_SPI_IC_PARSE_IPV4_LEN_1_VALUE);
	} else {
		ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_IPV4_OFFSET_1,
			   ADF_C4XXX_DEFAULT_IC_PARSE_IPV4_OFFS_1_VALUE);
		ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_IPV4_LENGTH_1,
			   ADF_C4XXX_DEFAULT_IC_PARSE_IPV4_LEN_1_VALUE);
	}

	 /* Configure destination field IP address from an IPv4 header */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_IPV4_OFFSET_2,
		   ADF_C4XXX_DEFAULT_IC_PARSE_IPV4_OFFS_2_VALUE);
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_IPV4_LENGTH_2,
		   ADF_C4XXX_DEFAULT_IC_PARSE_IPV4_LEN_2_VALUE);

	/* Configure function number extraction field from an IPv4 header */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_IPV4_OFFSET_3,
		   ADF_C4XXX_DEFAULT_IC_PARSE_IPV4_OFFS_3_VALUE);
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_IPV4_LENGTH_3,
		   ADF_C4XXX_DEFAULT_IC_PARSE_IPV4_LEN_3_VALUE);

	/* Configure protocol extraction field from an IPv6 header */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_IPV6_OFFSET_0,
		   ADF_C4XXX_DEFAULT_IC_PARSE_IPV6_OFFS_0_VALUE);
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_IPV6_LENGTH_0,
		   ADF_C4XXX_DEFAULT_IC_PARSE_IPV6_LEN_0_VALUE);

	/* Configure SPI extraction field from an IPv6 header */
	if (ipsec_spi_mask_mode == IPSEC_SPI_MASK_MODE_1) {
		ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_IPV6_OFFSET_1,
			   ADF_C4XXX_24BIT_SPI_IC_PARSE_IPV6_OFFS_1_VALUE);
		ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_IPV6_LENGTH_1,
			   ADF_C4XXX_24BIT_SPI_IC_PARSE_IPV6_LEN_1_VALUE);
	} else {
		ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_IPV6_OFFSET_1,
			   ADF_C4XXX_DEFAULT_IC_PARSE_IPV6_OFFS_1_VALUE);
		ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_IPV6_LENGTH_1,
			   ADF_C4XXX_DEFAULT_IC_PARSE_IPV6_LEN_1_VALUE);
	}

	/* Configure destination field IP address from an IPv6 header */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_IPV6_OFFSET_2,
		   ADF_C4XXX_DEFAULT_IC_PARSE_IPV6_OFFS_2_VALUE);
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_IPV6_LENGTH_2,
		   ADF_C4XXX_DEFAULT_IC_PARSE_IPV6_LEN_2_VALUE);

	/* Configure function number extraction field from an IPv6 header */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_IPV6_OFFSET_3,
		   ADF_C4XXX_DEFAULT_IC_PARSE_IPV6_OFFS_3_VALUE);
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_IC_PARSE_IPV6_LENGTH_3,
		   ADF_C4XXX_DEFAULT_IC_PARSE_IPV6_LEN_3_VALUE);
}

int adf_get_inline_ipsec_algo_group(struct adf_accel_dev *accel_dev,
				    unsigned long *ipsec_algo_group)
{
	char val[ADF_CFG_MAX_VAL_LEN_IN_BYTES];

	if (adf_cfg_get_param_value(accel_dev, ADF_INLINE_SEC,
				    ADF_INLINE_IPSEC_ALGO_GROUP, val))
		return -EFAULT;
	if (kstrtoul(val, 0, ipsec_algo_group))
		return -EFAULT;

	/* Verify the ipsec_algo_group */
	if (*ipsec_algo_group >= IPSEC_ALGO_GROUP_DELIMITER) {
		dev_err(&GET_DEV(accel_dev),
			"Unsupported IPSEC algo group %lu in config file!\n",
			*ipsec_algo_group);
		return -EFAULT;
	}

	return 0;
}

static int adf_get_inline_ipsec_spi_mask_mode(struct adf_accel_dev *accel_dev,
					      u32 *ipsec_spi_mask_mode)
{
	int idx;
	char val[ADF_CFG_MAX_VAL_LEN_IN_BYTES];
	unsigned int spi_mask_mode_config_val;
	struct pci_dev *pdev = NULL;
	int spi_mask_mode1_dev_ids[] = {0x1895, 0x188F};

	if (!ipsec_spi_mask_mode)
		goto err;

	if (adf_cfg_get_param_value(accel_dev, ADF_INLINE_SEC,
				    ADF_INLINE_SPI_MASK_MODE, val))
		goto err;

	if (kstrtouint(val, 10, &spi_mask_mode_config_val))
		goto err;

	if (spi_mask_mode_config_val >= IPSEC_SPI_MASK_MODE_DELIMITER) {
		dev_err(&GET_DEV(accel_dev),
			"Unsupported IPsec SPI mask mode %u in config file!\n",
			spi_mask_mode_config_val);
		goto err;
	}

	if (spi_mask_mode_config_val == IPSEC_DEFAULT_SPI_MASK_MODE) {
		*ipsec_spi_mask_mode = IPSEC_DEFAULT_SPI_MASK_MODE;
		return 0;
	}

	if (spi_mask_mode_config_val == IPSEC_SPI_MASK_MODE_1) {
		for (idx = 0; idx < ARRAY_SIZE(spi_mask_mode1_dev_ids); idx++) {
			pdev = pci_get_device(PCI_VENDOR_ID_INTEL,
					      spi_mask_mode1_dev_ids[idx],
					      NULL);
			if (pdev) {
				pci_dev_put(pdev);
				*ipsec_spi_mask_mode = IPSEC_SPI_MASK_MODE_1;
				return 0;
			}
		}
		dev_err(&GET_DEV(accel_dev),
			"24-bits SPI unsupported on this platform\n");
	}
err:
	return -EFAULT;
}

static int adf_init_inline_hw(struct adf_accel_dev *accel_dev)
{
	u32 sa_entry_reg_value = 0;
	u32 sa_func_limit = 0;
	u32 supported_algo = 0;
	u32 spi_mask_mode = 0;
	void __iomem *aram_csr_base;
	u32 offset;
	int ret;
	unsigned long ipsec_algo_group = IPSEC_DEFAUL_ALGO_GROUP;

	aram_csr_base = (&GET_BARS(accel_dev)[ADF_C4XXX_SRAM_BAR])->virt_addr;

	if (adf_get_inline_ipsec_algo_group(accel_dev, &ipsec_algo_group))
		return -EFAULT;

	sa_entry_reg_value |= (ADF_C4XXX_DEFAULT_LU_KEY_LEN
				<< ADF_C4XXX_LU_KEY_LEN_BIT_OFFSET);
	if (ipsec_algo_group == IPSEC_DEFAUL_ALGO_GROUP) {
		sa_entry_reg_value |= ADF_C4XXX_DEFAULT_SA_SIZE;
		sa_func_limit = ADF_C4XXX_FUNC_LIMIT(accel_dev,
						     ADF_C4XXX_DEFAULT_SA_SIZE);
		supported_algo = ADF_C4XXX_DEFAULT_SUPPORTED_ALGORITHMS;
	} else if (ipsec_algo_group == IPSEC_ALGO_GROUP1) {
		sa_entry_reg_value |= ADF_C4XXX_ALGO_GROUP1_SA_SIZE;
		sa_func_limit =
			ADF_C4XXX_FUNC_LIMIT(accel_dev,
					     ADF_C4XXX_ALGO_GROUP1_SA_SIZE);
		supported_algo = ADF_C4XXX_SUPPORTED_ALGORITHMS_GROUP1;
	} else {
		return -EFAULT;
	}

	/* REG_SA_ENTRY_CTRL register initialisation */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_REG_SA_ENTRY_CTRL,
		   sa_entry_reg_value);

	/* REG_SAL_FUNC_LIMITS register initialisation. Only the first register
	 * needs to be initialised to enable as it is assigned to a physical
	 * function. Other registers will be initialised by the LAN PF driver.
	 * The function limits is initialised to its maximal value.
	 */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_REG_SA_FUNC_LIMITS,
		   sa_func_limit);

	/* Initialize REG_SA_SCRATCH[0] register to
	 * advertise supported crypto algorithms
	 */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_REG_SA_SCRATCH_0,
		   supported_algo);

	/* Initialize REG_SA_SCRATCH[1] register to advertise SPI mask mode */
	ret = adf_get_inline_ipsec_spi_mask_mode(accel_dev, &spi_mask_mode);
	if (ret) {
		dev_err(&GET_DEV(accel_dev),
			"Failed to get inline IPsec spi mask mode\n");
		return ret;
	}
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_REG_SA_SCRATCH_1,
		   spi_mask_mode);

	/* REG_SA_SCRATCH[2] register initialisation
	 * to advertise supported crypto offload features.
	 */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_REG_SA_SCRATCH_2,
		   ADF_C4XXX_DEFAULT_CY_OFFLOAD_FEATURES);

	/* Overwrite default MAC_CFG register in ingress offset */
	ADF_CSR_WR64(aram_csr_base,
		     ADF_C4XXX_MAC_CFG + ADF_C4XXX_INLINE_INGRESS_OFFSET,
		     ADF_C4XXX_MAC_CFG_VALUE);

	/* Overwrite default MAC_CFG register in egress offset */
	ADF_CSR_WR64(aram_csr_base,
		     ADF_C4XXX_MAC_CFG + ADF_C4XXX_INLINE_EGRESS_OFFSET,
		     ADF_C4XXX_MAC_CFG_VALUE);

	/* Overwrite default MAC_PIA_CFG
	 * (Packet Interface Adapter Configuration) registers
	 * in ingress offset
	 */
	ADF_CSR_WR64(aram_csr_base,
		     ADF_C4XXX_MAC_PIA_CFG + ADF_C4XXX_INLINE_INGRESS_OFFSET,
		     ADF_C4XXX_MAC_PIA_CFG_VALUE);

	/* Overwrite default MAC_PIA_CFG in egress offset */
	ADF_CSR_WR64(aram_csr_base,
		     ADF_C4XXX_MAC_PIA_CFG + ADF_C4XXX_INLINE_EGRESS_OFFSET,
		     ADF_C4XXX_MAC_PIA_CFG_VALUE);

	/* Initialise congestion management registers */
	ret = adf_init_congestion_management_c4xxx(accel_dev);
	if (ret) {
		dev_err(&GET_DEV(accel_dev),
			"Failed to set congestion management profile\n");
		return ret;
	}

	adf_enable_parse_extraction(accel_dev, spi_mask_mode);

	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_INGRESS_CMD_DIS_MISC,
		   ADF_C4XXX_REG_CMD_DIS_MISC_DEFAULT_VALUE);

	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_EGRESS_CMD_DIS_MISC,
		   ADF_C4XXX_REG_CMD_DIS_MISC_DEFAULT_VALUE);

	/* Set bits<1:0> in ADF_C4XXX_INLINE_CAPABILITY register to
	 * advertise that both ingress and egress directions are available
	 */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_REG_SA_INLINE_CAPABILITY,
		   ADF_C4XXX_INLINE_CAPABILITIES);

	/* Set error notification configuration of ingress */
	offset = ADF_C4XXX_INLINE_INGRESS_OFFSET;
	adf_init_error_notification_configuration(accel_dev, offset);
	/* Set error notification configuration of egress */
	offset = ADF_C4XXX_INLINE_EGRESS_OFFSET;
	adf_init_error_notification_configuration(accel_dev, offset);

	return ret;
}

static void adf_enable_inline_notification(struct adf_accel_dev *accel_dev)
{
	void __iomem *aram_csr_base;

	aram_csr_base = (&GET_BARS(accel_dev)[ADF_C4XXX_SRAM_BAR])->virt_addr;

	/* Set bit<0> in ADF_C4XXX_REG_SA_INLINE_ENABLE to advertise
	 * that inline is enabled.
	 */
	ADF_CSR_WR(aram_csr_base, ADF_C4XXX_REG_SA_INLINE_ENABLE,
		   ADF_C4XXX_INLINE_ENABLED);
}

/* Function reads the inline ingress/egress configuration
 * and returns the number of AEs reserved for ingress
 * and egress for accel units which are allocated for
 * inline service
 */
static int adf_get_inline_config(struct adf_accel_dev *accel_dev,
				 u32 *num_ingress_aes)
{
	struct adf_hw_device_data *hw_data = accel_dev->hw_device;
	char key[ADF_CFG_MAX_KEY_LEN_IN_BYTES];
	char val[ADF_CFG_MAX_VAL_LEN_IN_BYTES];
	char *value;
	u32 num_au = get_num_accel_units_c4xxx(hw_data);
	unsigned long ingress, egress = 0;
	struct adf_accel_unit *accel_unit = accel_dev->au_info->au;
	u32 num_inline_aes = 0, num_ingress_ae = 0;
	u32 i = 0;

	snprintf(key, sizeof(key), ADF_INLINE_INGRESS);
	if (adf_cfg_get_param_value(accel_dev, ADF_INLINE_SEC, key, val)) {
		dev_err(&GET_DEV(accel_dev), "Failed to find ingress\n");
		return -EFAULT;
	}
	value = val;
	value = strsep(&value, ADF_C4XXX_PERCENTAGE);
	if (kstrtoul(value, 10, &ingress))
		return -EFAULT;

	snprintf(key, sizeof(key), ADF_INLINE_EGRESS);
	if (adf_cfg_get_param_value(accel_dev, ADF_INLINE_SEC, key, val)) {
		dev_err(&GET_DEV(accel_dev), "Failed to find egress\n");
		return -EFAULT;
	}
	value = val;
	value = strsep(&value, ADF_C4XXX_PERCENTAGE);
	if (kstrtoul(value, 10, &egress))
		return -EFAULT;

	if (ingress + egress != ADF_C4XXX_100) {
		dev_err(&GET_DEV(accel_dev),
			"The sum of ingress and egress should be 100\n");
		return -EFAULT;
	}

	for (i = 0; i < num_au; i++) {
		if (accel_unit[i].services == ADF_ACCEL_INLINE_CRYPTO)
			num_inline_aes += accel_unit[i].num_ae;
	}

	num_ingress_ae = num_inline_aes * ingress / ADF_C4XXX_100;
	if (((num_inline_aes * ingress) % ADF_C4XXX_100) >
	      ADF_C4XXX_ROUND_LIMIT)
		num_ingress_ae++;

	*num_ingress_aes = num_ingress_ae;
	return 0;
}

int adf_set_inline_ae_mask_c4xxx(struct adf_accel_dev *accel_dev)
{
	struct adf_hw_device_data *hw_data = accel_dev->hw_device;
	u32 num_au = get_num_accel_units_c4xxx(hw_data);
	struct adf_accel_unit_info *au_info = accel_dev->au_info;
	struct adf_accel_unit *accel_unit = accel_dev->au_info->au;
	u32 num_ingress_ae = 0;
	u32 ingress_msk = 0;
	u32 i, j, ae_mask;

	if (adf_get_inline_config(accel_dev, &num_ingress_ae))
		return -EFAULT;

	for (i = 0 ; i < num_au; i++) {
		j = 0;
		if (accel_unit[i].services == ADF_ACCEL_INLINE_CRYPTO) {
			/* AEs with inline service enabled are also used
			 * for asymmetric crypto
			 */
			au_info->asym_ae_msk |= accel_unit[i].ae_mask;
			ae_mask = accel_unit[i].ae_mask;
			while (num_ingress_ae && ae_mask) {
				if (ae_mask & 1) {
					ingress_msk |= BIT(j);
					num_ingress_ae--;
				}
				ae_mask = ae_mask >> 1;
				j++;
			}
			au_info->inline_ingress_msk |= ingress_msk;

			au_info->inline_egress_msk |=
				~(au_info->inline_ingress_msk) &
				accel_unit[i].ae_mask;
		}
	}

	return 0;
}

int adf_init_inline_c4xxx(struct adf_accel_dev *accel_dev)
{
	void __iomem *csr_base;

	csr_base = (&GET_BARS(accel_dev)[ADF_C4XXX_PMISC_BAR])->virt_addr;

	if (accel_dev->au_info->num_inline_au > 0)
		if (adf_init_inline_hw(accel_dev))
			return -EFAULT;

	if (accel_dev->au_info->num_inline_au > 0) {
		/* ici/ice interrupt shall be enabled after msi-x enabled */
		ADF_CSR_WR(csr_base, ADF_C4XXX_ERRMSK11,
			   ADF_C4XXX_ERRMSK11_ERR_DISABLE_ICI_ICE_INTR);
		adf_enable_inline_notification(accel_dev);
	}

	return 0;
}
